from mmcv.ops.multi_scale_deform_attn import multi_scale_deformable_attn_pytorch
import warnings
import torch
import torch.nn as nn
import torch.nn.functional as F
from mmengine.model.weight_init import xavier_init, constant_init
from mmcv.cnn.bricks.transformer import build_attention
import math

from mmengine.model import BaseModule, ModuleList, Sequential
from mmengine.registry import MODELS

from mmcv.utils import ext_loader
from ..multi_scale_deformable_attn_function import MultiScaleDeformableAttnFunction_fp32

ext_module = ext_loader.load_ext(
    '_ext', ['ms_deform_attn_backward', 'ms_deform_attn_forward'])


@MODELS.register_module()
class SpatialDeformableCrossAttention(BaseModule):
    """An attention module used in MM_BEVFormerLayer
    Args:
        init_dims (int): 
        embed_dims (int): The embedding dimension of Attention.
            Default: 256.
        num_cams (int): The number of cameras
        dropout (float): A Dropout layer on `inp_residual`.
            Default: 0..
        init_cfg (obj:`mmcv.ConfigDict`): The Config for initialization.
            Default: None.
        deformable_attention: (dict): The config for the deformable attention used in SCA.
    """

    def __init__(self,
                 init_dims,
                 embed_dims=256,
                 dropout=0.1,
                 batch_first=True,
                 num_heads=8,
                 num_levels=1,
                 num_points=8,
                 im2col_step=64,
                 init_cfg=None,
                 norm_cfg=None,
                 dilated_stride=1
                 ):
        super(SpatialDeformableCrossAttention, self).__init__(init_cfg)
        self.init_dims = init_dims
        self.init_cfg = init_cfg
        self.dropout = nn.Dropout(dropout) \
 \
        # you'd better set dim_per_head to a power of 2
        # which is more efficient in the CUDA implementation
        def _is_power_of_2(n):
            if (not isinstance(n, int)) or (n < 0):
                raise ValueError(
                    'invalid input for _is_power_of_2: {} (type: {})'.format(
                        n, type(n)))
            return (n & (n - 1) == 0) and n != 0

        dim_per_head = embed_dims // num_heads
        if not _is_power_of_2(dim_per_head):
            warnings.warn(
                "You'd better set embed_dims in "
                'MultiScaleDeformAttention to make '
                'the dimension of each attention head a power of 2 '
                'which is more efficient in our CUDA implementation.')

        self.embed_dims = embed_dims
        self.batch_first = batch_first
        self.norm_cfg = norm_cfg

        self.im2col_step = im2col_step
        self.embed_dims = embed_dims
        self.num_levels = num_levels
        self.num_heads = num_heads
        self.num_points = num_points
        self.sampling_offsets = nn.Linear(
            embed_dims+init_dims, num_heads * num_levels * num_points * 2)
        self.attention_weights = nn.Linear(embed_dims+init_dims,
                                           num_heads * num_levels * num_points)
        self.value_proj = nn.Linear(init_dims, embed_dims)
        self.output_proj = nn.Linear(embed_dims, embed_dims)
        self.dilated_stride = dilated_stride

        self.init_weights(self.dilated_stride)
        self._is_init = True

    def init_weights(self, dilated_stride):
        """Default initialization for Parameters of Module."""
        xavier_init(self.output_proj, distribution='uniform', bias=0.)
        xavier_init(self.value_proj, distribution='uniform', bias=0.)
        constant_init(self.sampling_offsets, 0.)
        constant_init(self.attention_weights, val=0., bias=0.)

        thetas = torch.arange(
            self.num_heads,
            dtype=torch.float32) * (2.0 * math.pi / self.num_heads)
        grid_init = torch.stack([thetas.cos(), thetas.sin()], -1)
        grid_init = (grid_init /
                     grid_init.abs().max(-1, keepdim=True)[0]).view(
            self.num_heads, 1, 1,
            2).repeat(1, self.num_levels, self.num_points, 1)
        for i in range(self.num_points):
            if i <= 0.5*self.num_points:
                grid_init[:, :, i, :] *= (i + 1)
            else:
                grid_init[:, :, i, :] *= (3*(i-0.5*self.num_points) + 0.5*self.num_points)

        self.sampling_offsets.bias.data = grid_init.view(-1)

    def forward(self,
                query,
                key,
                value,
                identity=None,
                residual=None,
                reference_points=None,
                spatial_shapes=None,
                reference_points_cam=None,
                level_start_index=None,
                bev_h=None,
                bev_w=None,
                num_points_sample=None,
                attn_masks=None,
                positional_encoding=None,
                **kwargs):
        """Forward Function of Detr3DCrossAtten.
        Args:
            query (Tensor): Query of Transformer with shape
                (bs, num_queries, embed_dims).
            key (Tensor): The key tensor with shape
                `(bs, h, w, embed_dims)`.
            value (Tensor): The value tensor with shape
                `(num_key, bs, embed_dims)`. (B, N, C, H, W)
            residual (Tensor): The tensor used for addition, with the
                same shape as `x`. Default None. If None, `x` will be used.
            reference_points (Tensor):  The normalized reference
                points with shape (bs, num_query, 2),
                all elements is range in [0, 1], top-left (0,0),
                bottom-right (1, 1), including padding area.
                or (N, Length_{query}, num_levels, 4), add
                additional two dimensions is (w, h) to
                form reference boxes.
            spatial_shapes (Tensor): Spatial shape of features in
                different level. With shape  (num_levels, 2),
                last dimension represent (h, w).
            level_start_index (Tensor): The start index of each level.
                A tensor has shape (num_levels) and can be represented
                as [0, h_0*w_0, h_0*w_0+h_1*w_1, ...].
        Returns:
             Tensor: forwarded results with shape [num_query, bs, embed_dims].
        """

        if key is None:
            key = query
        if value is None:
            value = key

        if residual is None:
            inp_residual = query
        
        if positional_encoding is not None:
            positional_encoding = positional_encoding.flatten(2).permute(0, 2, 1)
            query = query + positional_encoding

        bs, num_query, _ = query.size()

        bs, h, w, embed_dims = key.shape

        key = key.reshape(
            bs, h * w, self.init_dims)
        value = value.reshape(
            bs, h * w, self.init_dims)

        # queries = self.deformable_attention(query=queries_rebatch.view(bs*self.num_cams, max_len, self.embed_dims),
        # key=key, value=value, reference_points=reference_points_rebatch.view(bs*self.num_cams, max_len, D, 2),
        # spatial_shapes=spatial_shapes, level_start_index=level_start_index).view(bs, self.num_cams, max_len,
        # self.embed_dims)

        bs, num_query, _ = query.shape
        bs, num_value, _ = value.shape

        query = torch.cat([value, query], -1)
        assert (spatial_shapes[:, 0] * spatial_shapes[:, 1]).sum() == num_value

        value = self.value_proj(value)
        value = value.view(bs, num_value, self.num_heads, -1)
        sampling_offsets = self.sampling_offsets(query).view(
            bs, num_query, self.num_heads, self.num_levels, self.num_points, 2).to(key.device)
        attention_weights = self.attention_weights(query).view(
            bs, num_query, self.num_heads, self.num_levels * self.num_points)

        attention_weights = attention_weights.softmax(-1)

        attention_weights = attention_weights.view(bs, num_query,
                                                   self.num_heads,
                                                   self.num_levels,
                                                   self.num_points)
        if reference_points.shape[-1] == 2:
            """
            For each BEV query, it owns `num_Z_anchors` in 3D space that having different heights.
            After proejcting, each BEV query has `num_Z_anchors` reference points in each 2D image.
            For each referent point, we sample `num_points` sampling points.
            For `num_Z_anchors` reference points,  it has overall `num_points * num_Z_anchors` sampling points.
            """
            offset_normalizer = torch.stack(
                [spatial_shapes[..., 1], spatial_shapes[..., 0]], -1).to(key.device)

            bs, num_query, xy = reference_points.shape
            reference_points = reference_points[:, :, None, None, None, :]
            # bs, num_queries, heads, level, points, 2
            sampling_offsets = sampling_offsets / \
                               offset_normalizer[None, None, None, :, None, :]
            bs, num_query, num_heads, num_levels, num_all_points, xy = sampling_offsets.shape
            sampling_locations = reference_points + sampling_offsets
        else:
            assert False

        if torch.cuda.is_available() and value.is_cuda:
            MultiScaleDeformableAttnFunction = MultiScaleDeformableAttnFunction_fp32
            queries = MultiScaleDeformableAttnFunction.apply(
                value, spatial_shapes, level_start_index, sampling_locations,
                attention_weights, self.im2col_step)
        else:
            queries = multi_scale_deformable_attn_pytorch(
                value, spatial_shapes, sampling_locations, attention_weights)

        if not self.batch_first:
            queries = queries.permute(1, 0, 2)

        return queries
