'''
 *
 *     ICTP: Irreducible Cartesian Tensor Potentials
 *
 *        File:  o3.py
 *
 *     Authors: Deleted for purposes of anonymity 
 *
 *     Proprietor: Deleted for purposes of anonymity --- PROPRIETARY INFORMATION
 * 
 * The software and its source code contain valuable trade secrets and shall be maintained in
 * confidence and treated as confidential information. The software may only be used for 
 * evaluation and/or testing purposes, unless otherwise explicitly stated in the terms of a
 * license agreement or nondisclosure agreement with the proprietor of the software. 
 * Any unauthorized publication, transfer to third parties, or duplication of the object or
 * source code---either totally or in part---is strictly prohibited.
 *
 *     Copyright (c) 2024 Proprietor: Deleted for purposes of anonymity
 *     All Rights Reserved.
 *
 * THE PROPRIETOR DISCLAIMS ALL WARRANTIES, EITHER EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO IMPLIED WARRANTIES OF MERCHANTABILITY 
 * AND FITNESS FOR A PARTICULAR PURPOSE AND THE WARRANTY AGAINST LATENT 
 * DEFECTS, WITH RESPECT TO THE PROGRAM AND ANY ACCOMPANYING DOCUMENTATION. 
 * 
 * NO LIABILITY FOR CONSEQUENTIAL DAMAGES:
 * IN NO EVENT SHALL THE PROPRIETOR OR ANY OF ITS SUBSIDIARIES BE 
 * LIABLE FOR ANY DAMAGES WHATSOEVER (INCLUDING, WITHOUT LIMITATION, DAMAGES
 * FOR LOSS OF BUSINESS PROFITS, BUSINESS INTERRUPTION, LOSS OF INFORMATION, OR
 * OTHER PECUNIARY LOSS AND INDIRECT, CONSEQUENTIAL, INCIDENTAL,
 * ECONOMIC OR PUNITIVE DAMAGES) ARISING OUT OF THE USE OF OR INABILITY
 * TO USE THIS PROGRAM, EVEN IF the proprietor HAS BEEN ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGES.
 * 
 * For purposes of anonymity, the identity of the proprietor is not given herewith. 
 * The identity of the proprietor will be given once the review of the 
 * conference submission is completed. 
 *
 * THIS HEADER MAY NOT BE EXTRACTED OR MODIFIED IN ANY WAY.
 *
'''
import math

from typing import List, Optional


def get_slices(l_max: int, 
               n_feats: int,
               n_paths: Optional[List[int]] = None) -> List[slice]:
    """Returns slices for the tensor containing Cartesian harmonics of the rank `l <= l_max`.

    Args:
        l_max (int): Maximal rank of the input tensor.
        n_feats (int): Number of features in the input tensor.
        n_paths (List[int]): Provides the number of paths used to generate Cartesian harmonics 
                             of a particular rank provided in the first input tensor.

    Returns:
        List[slice]: List of slices used to get Cartesian harmonics of the rank `l <= l_max`.
    """
    if n_paths is None:
        n_paths = [1 for _ in range(l_max + 1)]
    else:
        n_paths = n_paths
    
    slices = []
    k = 0
    for l in range(l_max + 1):
        slices.append(slice(k,  k + 3 ** l * n_feats * n_paths[l], None))
        k += 3 ** l * n_feats * n_paths[l]
    return slices


def get_shapes(l_max: int, 
               n_feats: int,
               n_paths: Optional[List[int]] = None,
               use_prod: bool = True) -> List[List[int]]:
    """Returns shapes of Cartesian harmonics of the rank `l <= l_max` comprised in the input 
    tensor.

    Args:
        l_max (int): Maximal rank of the input tensor.
        n_feats (int): Number of features in the input tensor.
        n_paths (List[int]): Provides the number of paths used to generate Cartesian harmonics 
                             of a particular rank provided in the first input tensor.
        use_prod (bool, optional): If True, computes the product between `n_feats` and `n_paths`.
                                   Defaults to True.

    Returns:
        List[List[int]]: List of the shapes Cartesian harmonics of the rank `l <= l_max` 
                         comprised in the input tensor.
    """
    if n_paths is None:
        n_paths = [1 for _ in range(l_max + 1)]
    else:
        n_paths = n_paths
    
    shapes = []
    for l in range(l_max + 1):
        feats_shape = [math.prod([n_feats, n_paths[l]])] if use_prod else [n_feats, n_paths[l]]
        if l == 0:
            shapes.append(feats_shape)
        else:
            shapes.append([3 for _ in range(l)] + feats_shape)
    return shapes
