#MULTI-DIMENSIONAL PROBLEM: LQG with internal noise, COMPARISON OF ACCUMULATED COSTS AND PLOTS

#PACKAGES -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------

using Optim, Plots, DelimitedFiles, LinearAlgebra, Random, StatsBase, FiniteDifferences, LaTeXStrings , EasyFit, Printf, FFTW, Pkg, Noise, Clustering, Dierckx, BSplineKit, MultivariateStats, Flux, Combinatorics, Bigsimr, DataFrames, JLD

#FUNCTIONS -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------

#functions for Todorov's and numerical GD optimization, with useful functions to get model predictions.
cd(@__DIR__)#to go back to the directory of the script
include("../../MyFunctions/functions_sensory_motor_task.jl")

#Functions for full GD algorithms (1D and multiple dimensions)
cd(@__DIR__)#to go back to the directory of the script
include("../../MyFunctions/functions_full_GD_algorithms.jl")

#quality of figures
dpi_value = 500 # A higher dpi value will result in a sharper image but may also increase the file size.
default(dpi=dpi_value)

#TASK PARAMETERS and matrices, we follow (Todorov, 2005) -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------

dimension_of_state = 4 #We chose a system with a 4D state vector, instead of the 5D system of Todorov: see comments about that in the document "functions_Reaching_task_application".
dimension_of_control = 1 #1D control.
observation_dimension = 1 #We chose a 1D sensory feedback for simplicity and to give major emphasis to the visual stimuli (for which the multiplicative and internal noise 
#description could be more significant: eccentricity for multiplicative noise and memory leak for what concerns the internal noise - considering a case in which we need to
#remember the position of the target, like in the navigation task).

Δt = 0.010 #seconds
N_steps = 50 #N_steps = 100 #we use the same time steps as the estimation task of (Todorov, 2005)
m = 1 #Kg, mass of the hand, modelled as a point mass
target_position_mean = 0.15 #meters; mean of the gaussian distribution from which the target position is extracted. We use here the params used in the estimation task of 
#(Todorov, 2005), so that the target is in the periphery and multiplicative sensory noise plays a role. Note that three more means are used: 0.05m and 0.25m.
#the moment. In (Todorov, 2005) they use (0.05m)^2
tau_1 = 0.04 #seconds; tau_1 and tau_2 are the time constants of the second oreder low pass filter to map the control signal into the force acting on the hand
tau_2 = 0.04 #seconds
r = 0.00001 #as in the reaching atsk of (Todorov, 2005)
w_v = 0.2 #w_v and #w_f are computed by considering Todorov's choices in (Todorov, 2005) and rescaling them to our case in which target_displacement is 0.15 and N_steps = 100 
#(we thus just divided by 2 the values used in the paper)
w_f = 0.01 

#NOISE terms 
#here we consider only additive noise to test the full order GD for the LQG problem
σ_ξ = 0.5 #we only use control dependent noise acting on the dynamics to mimic the reaching task of (Todorov, 2005).  
σ_ω_add_sensory_noise = 0.5 #we only use sensory dependent noise acting on the dynamics to mimic the estimation task of (Todorov, 2005).  
σ_ρ_mult_sensory_noise = 0.0 #as in the estimation task of (Todorov, 2005).
σ_ϵ_mult_control_noise = 0.0 #as in the reaching task of (Todorov, 2005).

σ_η_internal_noise_position_vec = [0.0, 1.0, 2.0] #meters, as in the estimation task of (Todorov, 2005).
#we also consider here possible internal fluctuations on the other components of the state estimate 
σ_η_internal_noise_velocity = 0.1
σ_η_internal_noise_force = 0.1
σ_η_internal_noise_filtered_control = 0.1

#NOTE THAT IN THE PREVIOUS SIMULATIONS WE HAVE ALWAYS USED σ_η_internal_noise_velocity = σ_η_internal_noise_force = σ_η_internal_noise_filtered_control = 0.1, CHANGING ONLY 
#σ_η_internal_noise_position. ONLY FOR THE SIMULATION WITHOUT INTERNAL NOISE WE SET ALL THE MENTIONED VARIABLES AT ZERO.

#Initial conditions (mean state and state estimate, and their covariances) -- Note that initial state and state estimate are considered to be uncorrelated at t=1 (initial time step)

#Mean initial state and state estimates -- Note that we assume <x_1> = <z_1>

#note that x_state_mean_0 is the initial mean state of the vector
x_state_mean_0 = zeros(dimension_of_state) #meters. 
x_state_mean_0[1] = target_position_mean 
x_state_mean_0[2] = 0
x_state_mean_0[3] = 0
x_state_mean_0[4] = 0

x_state_estimate_mean_0 = zeros(dimension_of_state)
x_state_estimate_mean_0[:] .= x_state_mean_0[:]

#State covariance 
target_position_variance = 0
velocity_initial_variance = 0
force_initial_variance = 0
filtered_control_initial_variance = 0

Σ_1_x = Diagonal(ones(dimension_of_state)) #intial covariance of the state

Σ_1_x[1,1] = target_position_variance
Σ_1_x[2,2] = velocity_initial_variance
Σ_1_x[3,3] = force_initial_variance
Σ_1_x[4,4] = filtered_control_initial_variance

#State estimate covariance 
target_position_estimate_initial_variance = 0.0
velocity_estimate_initial_variance = 0.0
force_estimate_initial_variance = 0.0
filtered_control_estimate_initial_variance = 0.0

Σ_1_z = Diagonal(ones(dimension_of_state)) #intial covariance of the state estimate 

Σ_1_z[1,1] = target_position_estimate_initial_variance
Σ_1_z[2,2] = velocity_estimate_initial_variance
Σ_1_z[3,3] = force_estimate_initial_variance
Σ_1_z[4,4] = filtered_control_estimate_initial_variance

#Initial condition of raw moments.
#S_0 is a 4x4 block matrix containing the inital condition for the propagation of the raw moments (mxm matrices, where m is the dimension of the state), and it's defined this way: 
#S_0 = [<x_1*transpose(x_1)>, <x_1*transpose(z_1)>; <z_1*transpose(x_1)>, <z_1*transpose(z_1)>], where x is the state and z the state estimate. 
#This variable is used as an argument for "raw_moments_propagation_for_full_GD", used for the control (and estimation) optimization of the full GD.

S_0 = [zeros(dimension_of_state, dimension_of_state) for _ in 1:2, _ in 1:2]

S_0[1,1] .= Σ_1_x .+ x_state_mean_0*transpose(x_state_mean_0)
S_0[1,2] .= x_state_estimate_mean_0*transpose(x_state_mean_0)
S_0[2,1] .= x_state_mean_0*transpose(x_state_estimate_mean_0)
S_0[2,2] .= Σ_1_z .+ x_state_estimate_mean_0*transpose(x_state_estimate_mean_0)

#Matrices of the problem 
#Define the dynamical system
A = zeros(dimension_of_state, dimension_of_state)
B = zeros(dimension_of_state, dimension_of_control)
H = zeros(observation_dimension, dimension_of_state)
D = zeros(observation_dimension, dimension_of_state)

Ω_ξ = Diagonal([0, σ_ξ^2, 0, 0]) #we consider the sqrt of the covariance matrix, so we don't need to call the function sqrt at each time step iteration
Ω_ω = σ_ω_add_sensory_noise^2 #Additive sensory noise (1D sensory feedback)
σ_η_internal_noise_position = σ_η_internal_noise_position_vec[1]
Ω_η = Diagonal([σ_η_internal_noise_position^2, σ_η_internal_noise_velocity^2, σ_η_internal_noise_force^2, σ_η_internal_noise_filtered_control^2]) 

A[1,:] = [1, Δt, 0, 0]
A[2,:] = [0, 1, Δt/m, 0]
A[3,:] = [0, 0, 1-Δt/tau_2, Δt/tau_2]
A[4,:] = [0, 0, 0, 1-Δt/tau_1]

B[:] = [0, 0, 0, Δt/tau_1]
C = σ_ϵ_mult_control_noise .* B
H[1,:] = [1, 0, 0, 0] #we're selecting only the position as observable 

D[1,:] = σ_ρ_mult_sensory_noise .* H[1,:]

#Matrix for cost function (quadratic form)
p_vec = [1; 0; 0; 0]
v_vec = [0; w_v; 0; 0]
f_vec = [0; 0; w_f; 0]

R_matrix = zeros(dimension_of_control, dimension_of_control, N_steps)
for i in 1:N_steps-1
    R_matrix[:,:,i] .= r/(N_steps-1)
end

Q_matrix = zeros(dimension_of_state, dimension_of_state, N_steps)
Q_N = p_vec*transpose(p_vec) .+ v_vec*transpose(v_vec) .+ f_vec*transpose(f_vec) #Matrix for the cost at the last time step 
Q_matrix[:,:,N_steps] .= Q_N[:,:]

#Load optimal solutions -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------

#Folder
#functions for Todorov's and numerical GD optimization, with useful functions to get model predictions.
cd(@__DIR__)#to go back to the directory of the script

folder_location = "../../Data/Multi_D_problems/LQG_with_internal_noise"

#σ_η = 0.0 
#Tod
file_name_K_TOD_s_0 = "Reaching_task_NEURIPS_TOD_K_s_eta_0_0.jld"
file_path_K_TOD_s_0 = folder_location * "/" * file_name_K_TOD_s_0
K_TOD_s_0 = load(file_path_K_TOD_s_0, "K_TOD")

file_name_L_TOD_s_0 = "Reaching_task_NEURIPS_TOD_L_s_eta_0_0.jld"
file_path_L_TOD_s_0 = folder_location * "/" * file_name_L_TOD_s_0
L_TOD_s_0 = load(file_path_L_TOD_s_0, "L_TOD")

#GD
file_name_K_GD_s_0 = "Reaching_task_NEURIPS_GD_K_s_eta_0_0.jld"
file_path_K_GD_s_0 = folder_location * "/" * file_name_K_GD_s_0
K_GD_s_0 = load(file_path_K_GD_s_0, "K_GD")

file_name_L_GD_s_0 = "Reaching_task_NEURIPS_GD_L_s_eta_0_0.jld"
file_path_L_GD_s_0 = folder_location * "/" * file_name_L_GD_s_0
L_GD_s_0 = load(file_path_L_GD_s_0, "L_GD")

#Iterative-GD with TOD estimator
file_name_L_iterative_Full_GD_with_TOD_est_s_0 = "Reaching_task_NEURIPS_Full_GD_with_TOD_estimator_L_s_eta_0_0.jld"
file_path_L_iterative_Full_GD_with_TOD_est_s_0 = folder_location * "/" * file_name_L_iterative_Full_GD_with_TOD_est_s_0
L_iterative_Full_GD_with_TOD_est_s_0 = load(file_path_L_iterative_Full_GD_with_TOD_est_s_0, "L_matrix_optimal_full_GD")

#Iterative-GD with GD estimator
file_name_L_iterative_Full_GD_with_GD_est_s_0 = "Reaching_task_NEURIPS_Full_GD_with_GD_estimator_L_s_eta_0_0.jld"
file_path_L_iterative_Full_GD_with_GD_est_s_0 = folder_location * "/" * file_name_L_iterative_Full_GD_with_GD_est_s_0
L_iterative_Full_GD_with_GD_est_s_0 = load(file_path_L_iterative_Full_GD_with_GD_est_s_0, "L_matrix_optimal_full_GD_reoptimization")

#σ_η = 1.0
#Tod
file_name_K_TOD_s_1 = "Reaching_task_NEURIPS_TOD_K_s_eta_1_0.jld"
file_path_K_TOD_s_1 = folder_location * "/" * file_name_K_TOD_s_1
K_TOD_s_1 = load(file_path_K_TOD_s_1, "K_TOD")

file_name_L_TOD_s_1 = "Reaching_task_NEURIPS_TOD_L_s_eta_1_0.jld"
file_path_L_TOD_s_1 = folder_location * "/" * file_name_L_TOD_s_1
L_TOD_s_1 = load(file_path_L_TOD_s_1, "L_TOD")
#GD
file_name_K_GD_s_1 = "Reaching_task_NEURIPS_GD_K_s_eta_1_0.jld"
file_path_K_GD_s_1 = folder_location * "/" * file_name_K_GD_s_1
K_GD_s_1 = load(file_path_K_GD_s_1, "K_GD")

file_name_L_GD_s_1 = "Reaching_task_NEURIPS_GD_L_s_eta_1_0.jld"
file_path_L_GD_s_1 = folder_location * "/" * file_name_L_GD_s_1
L_GD_s_1 = load(file_path_L_GD_s_1, "L_GD")

#Iterative-GD with TOD estimator
file_name_L_iterative_Full_GD_with_TOD_est_s_1 = "Reaching_task_NEURIPS_Full_GD_with_TOD_estimator_L_s_eta_1_0.jld"
file_path_L_iterative_Full_GD_with_TOD_est_s_1 = folder_location * "/" * file_name_L_iterative_Full_GD_with_TOD_est_s_1
L_iterative_Full_GD_with_TOD_est_s_1 = load(file_path_L_iterative_Full_GD_with_TOD_est_s_1, "L_matrix_optimal_full_GD")

#Iterative-GD with GD estimator
file_name_L_iterative_Full_GD_with_GD_est_s_1 = "Reaching_task_NEURIPS_Full_GD_with_GD_estimator_L_s_eta_1_0.jld"
file_path_L_iterative_Full_GD_with_GD_est_s_1 = folder_location * "/" * file_name_L_iterative_Full_GD_with_GD_est_s_1
L_iterative_Full_GD_with_GD_est_s_1 = load(file_path_L_iterative_Full_GD_with_GD_est_s_1, "L_matrix_optimal_full_GD_reoptimization")

#σ_η = 2.0
#Tod
file_name_K_TOD_s_2 = "Reaching_task_NEURIPS_TOD_K_s_eta_2_0.jld"
file_path_K_TOD_s_2 = folder_location * "/" * file_name_K_TOD_s_2
K_TOD_s_2 = load(file_path_K_TOD_s_2, "K_TOD")

file_name_L_TOD_s_2 = "Reaching_task_NEURIPS_TOD_L_s_eta_2_0.jld"
file_path_L_TOD_s_2 = folder_location * "/" * file_name_L_TOD_s_2
L_TOD_s_2 = load(file_path_L_TOD_s_2, "L_TOD")

#GD
file_name_K_GD_s_2 = "Reaching_task_NEURIPS_GD_K_s_eta_2_0.jld"
file_path_K_GD_s_2 = folder_location * "/" * file_name_K_GD_s_2
K_GD_s_2 = load(file_path_K_GD_s_2, "K_GD")

file_name_L_GD_s_2 = "Reaching_task_NEURIPS_GD_L_s_eta_2_0.jld"
file_path_L_GD_s_2 = folder_location * "/" * file_name_L_GD_s_2
L_GD_s_2 = load(file_path_L_GD_s_2, "L_GD")

#Iterative-GD with TOD estimator
file_name_L_iterative_Full_GD_with_TOD_est_s_2 = "Reaching_task_NEURIPS_Full_GD_with_TOD_estimator_L_s_eta_2_0.jld"
file_path_L_iterative_Full_GD_with_TOD_est_s_2 = folder_location * "/" * file_name_L_iterative_Full_GD_with_TOD_est_s_2
L_iterative_Full_GD_with_TOD_est_s_2 = load(file_path_L_iterative_Full_GD_with_TOD_est_s_2, "L_matrix_optimal_full_GD")

#Iterative-GD with GD estimator
file_name_L_iterative_Full_GD_with_GD_est_s_2 = "Reaching_task_NEURIPS_Full_GD_with_GD_estimator_L_s_eta_2_0.jld"
file_path_L_iterative_Full_GD_with_GD_est_s_2 = folder_location * "/" * file_name_L_iterative_Full_GD_with_GD_est_s_2
L_iterative_Full_GD_with_GD_est_s_2 = load(file_path_L_iterative_Full_GD_with_GD_est_s_2, "L_matrix_optimal_full_GD_reoptimization")

#MONTECARLO SIMULATIONS PARAMETERS -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------

num_trials_MC_runs = 50000
random_seed_MC = rand(1:1000000)

#EXPECTED COSTS -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------

σ_η_tmp = 0.0
println("\n------- (1) σ_η = $σ_η_tmp --------------------------\n")

Ω_η = Diagonal([σ_η_internal_noise_position_vec[1]^2, 0, 0, 0]) 

#TODOROV'S APPROACH

#Montecarlo simulations
final_cost_average_TOD_s_0, final_cost_average_std_TOD_s_0, p_t_average_TOD_s_0, p_t_average_std_TOD_s_0, p_hat_t_average_TOD_s_0, p_hat_t_average_std_TOD_s_0, control_t_average_TOD_s_0, control_t_average_std_TOD_s_0, p_t_TOD_s_0, p_hat_t_TOD_s_0, control_t_TOD_s_0, accumulated_cost_t_TOD_s_0 = get_model_predictions_reaching_task_application_4D_state(num_trials_MC_runs, K_TOD_s_0, L_TOD_s_0, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η, random_seed_MC)

#Expected cost using moment's propagation
expected_cost_mom_prop_TOD_s_0 = moments_propagation_reaching_task_application_4D_state(K_TOD_s_0, L_TOD_s_0, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η)

println("TODOROV's OPTIMIZATION\n")
println("- Expected cost using moment propagation (Todorov): $expected_cost_mom_prop_TOD_s_0\n")
println("- Montecarlo - cost function (Todorov): $final_cost_average_TOD_s_0  (error: $final_cost_average_std_TOD_s_0)\n")

#GD APPROACH
#Montecarlo simulations
final_cost_average_GD_s_0, final_cost_average_std_GD_s_0, p_t_average_GD_s_0, p_t_average_std_GD_s_0, p_hat_t_average_GD_s_0, p_hat_t_average_std_GD_s_0, control_t_average_GD_s_0, control_t_average_std_GD_s_0, p_t_GD_s_0, p_hat_t_GD_s_0, control_t_GD_s_0, accumulated_cost_t_GD_s_0 = get_model_predictions_reaching_task_application_4D_state(num_trials_MC_runs, K_GD_s_0, L_GD_s_0, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η, random_seed_MC)

#Expected cost using moment's propagation
expected_cost_mom_prop_GD_s_0 = moments_propagation_reaching_task_application_4D_state(K_GD_s_0, L_GD_s_0, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η)

println("GD OPTIMIZATION\n")
println("- Expected cost using moment propagation (GD): $expected_cost_mom_prop_GD_s_0\n")
println("- Montecarlo - cost function (GD): $final_cost_average_GD_s_0  (error: $final_cost_average_std_GD_s_0)\n")

#Iterative-GD with TOD estimator
#Montecarlo simulations
final_cost_average_iterative_Full_GD_with_TOD_est_s_0, final_cost_average_std_iterative_Full_GD_with_TOD_est_s_0, p_t_average_iterative_Full_GD_with_TOD_est_s_0, p_t_average_std_iterative_Full_GD_with_TOD_est_s_0, p_hat_t_average_iterative_Full_GD_with_TOD_est_s_0, p_hat_t_average_std_iterative_Full_GD_with_TOD_est_s_0, control_t_average_iterative_Full_GD_with_TOD_est_s_0, control_t_average_std_iterative_Full_GD_with_TOD_est_s_0, p_t_iterative_Full_GD_with_TOD_est_s_0, p_hat_t_iterative_Full_GD_with_TOD_est_s_0, control_t_iterative_Full_GD_with_TOD_est_s_0, accumulated_cost_t_iterative_Full_GD_with_TOD_est_s_0 = get_model_predictions_reaching_task_application_4D_state(num_trials_MC_runs, K_TOD_s_0, L_iterative_Full_GD_with_TOD_est_s_0, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η, random_seed_MC)

#Expected cost using moment's propagation
expected_cost_mom_prop_iterative_Full_GD_with_TOD_est_s_0 = moments_propagation_reaching_task_application_4D_state(K_TOD_s_0, L_iterative_Full_GD_with_TOD_est_s_0, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η)

println("Iterative-GD OPTIMIZATION with TOD estimator\n")
println("- Expected cost using moment propagation (It-GD with TOD est): $expected_cost_mom_prop_iterative_Full_GD_with_TOD_est_s_0\n")
println("- Montecarlo - cost function (It-GD with TOD est): $final_cost_average_iterative_Full_GD_with_TOD_est_s_0  (error: $final_cost_average_std_iterative_Full_GD_with_TOD_est_s_0)\n")

#Iterative-GD with GD estimator
#Montecarlo simulations
final_cost_average_iterative_Full_GD_with_GD_est_s_0, final_cost_average_std_iterative_Full_GD_with_GD_est_s_0, p_t_average_iterative_Full_GD_with_GD_est_s_0, p_t_average_std_iterative_Full_GD_with_GD_est_s_0, p_hat_t_average_iterative_Full_GD_with_GD_est_s_0, p_hat_t_average_std_iterative_Full_GD_with_GD_est_s_0, control_t_average_iterative_Full_GD_with_GD_est_s_0, control_t_average_std_iterative_Full_GD_with_GD_est_s_0, p_t_iterative_Full_GD_with_GD_est_s_0, p_hat_t_iterative_Full_GD_with_GD_est_s_0, control_t_iterative_Full_GD_with_GD_est_s_0, accumulated_cost_t_iterative_Full_GD_with_GD_est_s_0 = get_model_predictions_reaching_task_application_4D_state(num_trials_MC_runs, K_GD_s_0, L_iterative_Full_GD_with_GD_est_s_0, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η, random_seed_MC)

#Expected cost using moment's propagation
expected_cost_mom_prop_iterative_Full_GD_with_GD_est_s_0 = moments_propagation_reaching_task_application_4D_state(K_GD_s_0, L_iterative_Full_GD_with_GD_est_s_0, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η)

println("Iterative-GD OPTIMIZATION with GD estimator\n")
println("- Expected cost using moment propagation (It-GD with GD est): $expected_cost_mom_prop_iterative_Full_GD_with_GD_est_s_0\n")
println("- Montecarlo - cost function (It-GD with GD est): $final_cost_average_iterative_Full_GD_with_GD_est_s_0  (error: $final_cost_average_std_iterative_Full_GD_with_GD_est_s_0)\n")

σ_η_tmp = 1.0
println("\n------- (2) σ_η = $σ_η_tmp --------------------------\n")

Ω_η = Diagonal([σ_η_internal_noise_position_vec[2]^2, σ_η_internal_noise_velocity^2, σ_η_internal_noise_force^2, σ_η_internal_noise_filtered_control^2]) 

#TODOROV'S APPROACH

#Montecarlo simulations
final_cost_average_TOD_s_1, final_cost_average_std_TOD_s_1, p_t_average_TOD_s_1, p_t_average_std_TOD_s_1, p_hat_t_average_TOD_s_1, p_hat_t_average_std_TOD_s_1, control_t_average_TOD_s_1, control_t_average_std_TOD_s_1, p_t_TOD_s_1, p_hat_t_TOD_s_1, control_t_TOD_s_1, accumulated_cost_t_TOD_s_1 = get_model_predictions_reaching_task_application_4D_state(num_trials_MC_runs, K_TOD_s_1, L_TOD_s_1, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η, random_seed_MC)

#Expected cost using moment's propagation
expected_cost_mom_prop_TOD_s_1 = moments_propagation_reaching_task_application_4D_state(K_TOD_s_1, L_TOD_s_1, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η)

println("TODOROV's OPTIMIZATION\n")
println("- Expected cost using moment propagation (Todorov): $expected_cost_mom_prop_TOD_s_1\n")
println("- Montecarlo - cost function (Todorov): $final_cost_average_TOD_s_1  (error: $final_cost_average_std_TOD_s_1)\n")

#GD APPROACH
#Montecarlo simulations
final_cost_average_GD_s_1, final_cost_average_std_GD_s_1, p_t_average_GD_s_1, p_t_average_std_GD_s_1, p_hat_t_average_GD_s_1, p_hat_t_average_std_GD_s_1, control_t_average_GD_s_1, control_t_average_std_GD_s_1, p_t_GD_s_1, p_hat_t_GD_s_1, control_t_GD_s_1, accumulated_cost_t_GD_s_1 = get_model_predictions_reaching_task_application_4D_state(num_trials_MC_runs, K_GD_s_1, L_GD_s_1, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η, random_seed_MC)

#Expected cost using moment's propagation
expected_cost_mom_prop_GD_s_1 = moments_propagation_reaching_task_application_4D_state(K_GD_s_1, L_GD_s_1, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η)

println("GD OPTIMIZATION\n")
println("- Expected cost using moment propagation (GD): $expected_cost_mom_prop_GD_s_1\n")
println("- Montecarlo - cost function (GD): $final_cost_average_GD_s_1  (error: $final_cost_average_std_GD_s_1)\n")

#Iterative-GD with TOD estimator
#Montecarlo simulations
final_cost_average_iterative_Full_GD_with_TOD_est_s_1, final_cost_average_std_iterative_Full_GD_with_TOD_est_s_1, p_t_average_iterative_Full_GD_with_TOD_est_s_1, p_t_average_std_iterative_Full_GD_with_TOD_est_s_1, p_hat_t_average_iterative_Full_GD_with_TOD_est_s_1, p_hat_t_average_std_iterative_Full_GD_with_TOD_est_s_1, control_t_average_iterative_Full_GD_with_TOD_est_s_1, control_t_average_std_iterative_Full_GD_with_TOD_est_s_1, p_t_iterative_Full_GD_with_TOD_est_s_1, p_hat_t_iterative_Full_GD_with_TOD_est_s_1, control_t_iterative_Full_GD_with_TOD_est_s_1, accumulated_cost_t_iterative_Full_GD_with_TOD_est_s_1 = get_model_predictions_reaching_task_application_4D_state(num_trials_MC_runs, K_TOD_s_1, L_iterative_Full_GD_with_TOD_est_s_1, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η, random_seed_MC)

#Expected cost using moment's propagation
expected_cost_mom_prop_iterative_Full_GD_with_TOD_est_s_1 = moments_propagation_reaching_task_application_4D_state(K_TOD_s_1, L_iterative_Full_GD_with_TOD_est_s_1, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η)

println("Iterative-GD OPTIMIZATION with TOD estimator\n")
println("- Expected cost using moment propagation (It-GD with TOD est): $expected_cost_mom_prop_iterative_Full_GD_with_TOD_est_s_1\n")
println("- Montecarlo - cost function (It-GD with TOD est): $final_cost_average_iterative_Full_GD_with_TOD_est_s_1  (error: $final_cost_average_std_iterative_Full_GD_with_TOD_est_s_1)\n")

#Iterative-GD with GD estimator
#Montecarlo simulations
final_cost_average_iterative_Full_GD_with_GD_est_s_1, final_cost_average_std_iterative_Full_GD_with_GD_est_s_1, p_t_average_iterative_Full_GD_with_GD_est_s_1, p_t_average_std_iterative_Full_GD_with_GD_est_s_1, p_hat_t_average_iterative_Full_GD_with_GD_est_s_1, p_hat_t_average_std_iterative_Full_GD_with_GD_est_s_1, control_t_average_iterative_Full_GD_with_GD_est_s_1, control_t_average_std_iterative_Full_GD_with_GD_est_s_1, p_t_iterative_Full_GD_with_GD_est_s_1, p_hat_t_iterative_Full_GD_with_GD_est_s_1, control_t_iterative_Full_GD_with_GD_est_s_1, accumulated_cost_t_iterative_Full_GD_with_GD_est_s_1 = get_model_predictions_reaching_task_application_4D_state(num_trials_MC_runs, K_GD_s_1, L_iterative_Full_GD_with_GD_est_s_1, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η, random_seed_MC)

#Expected cost using moment's propagation
expected_cost_mom_prop_iterative_Full_GD_with_GD_est_s_1 = moments_propagation_reaching_task_application_4D_state(K_GD_s_1, L_iterative_Full_GD_with_GD_est_s_1, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η)

println("Iterative-GD OPTIMIZATION with GD estimator\n")
println("- Expected cost using moment propagation (It-GD with GD est): $expected_cost_mom_prop_iterative_Full_GD_with_GD_est_s_1\n")
println("- Montecarlo - cost function (It-GD with GD est): $final_cost_average_iterative_Full_GD_with_GD_est_s_1  (error: $final_cost_average_std_iterative_Full_GD_with_GD_est_s_1)\n")

σ_η_tmp = 2.0
println("\n------- (2) σ_η = $σ_η_tmp --------------------------\n")

Ω_η = Diagonal([σ_η_internal_noise_position_vec[3]^2, σ_η_internal_noise_velocity^2, σ_η_internal_noise_force^2, σ_η_internal_noise_filtered_control^2]) 

#TODOROV'S APPROACH

#Montecarlo simulations
final_cost_average_TOD_s_2, final_cost_average_std_TOD_s_2, p_t_average_TOD_s_2, p_t_average_std_TOD_s_2, p_hat_t_average_TOD_s_2, p_hat_t_average_std_TOD_s_2, control_t_average_TOD_s_2, control_t_average_std_TOD_s_2, p_t_TOD_s_2, p_hat_t_TOD_s_2, control_t_TOD_s_2, accumulated_cost_t_TOD_s_2 = get_model_predictions_reaching_task_application_4D_state(num_trials_MC_runs, K_TOD_s_2, L_TOD_s_2, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η, random_seed_MC)

#Expected cost using moment's propagation
expected_cost_mom_prop_TOD_s_2 = moments_propagation_reaching_task_application_4D_state(K_TOD_s_2, L_TOD_s_2, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η)

println("TODOROV's OPTIMIZATION\n")
println("- Expected cost using moment propagation (Todorov): $expected_cost_mom_prop_TOD_s_2\n")
println("- Montecarlo - cost function (Todorov): $final_cost_average_TOD_s_2  (error: $final_cost_average_std_TOD_s_2)\n")

#GD APPROACH
#Montecarlo simulations
final_cost_average_GD_s_2, final_cost_average_std_GD_s_2, p_t_average_GD_s_2, p_t_average_std_GD_s_2, p_hat_t_average_GD_s_2, p_hat_t_average_std_GD_s_2, control_t_average_GD_s_2, control_t_average_std_GD_s_2, p_t_GD_s_2, p_hat_t_GD_s_2, control_t_GD_s_2, accumulated_cost_t_GD_s_2 = get_model_predictions_reaching_task_application_4D_state(num_trials_MC_runs, K_GD_s_2, L_GD_s_2, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η, random_seed_MC)

#Expected cost using moment's propagation
expected_cost_mom_prop_GD_s_2 = moments_propagation_reaching_task_application_4D_state(K_GD_s_2, L_GD_s_2, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η)

println("GD OPTIMIZATION\n")
println("- Expected cost using moment propagation (GD): $expected_cost_mom_prop_GD_s_2\n")
println("- Montecarlo - cost function (GD): $final_cost_average_GD_s_2  (error: $final_cost_average_std_GD_s_2)\n")

#Iterative-GD with TOD estimator
#Montecarlo simulations
final_cost_average_iterative_Full_GD_with_TOD_est_s_2, final_cost_average_std_iterative_Full_GD_with_TOD_est_s_2, p_t_average_iterative_Full_GD_with_TOD_est_s_2, p_t_average_std_iterative_Full_GD_with_TOD_est_s_2, p_hat_t_average_iterative_Full_GD_with_TOD_est_s_2, p_hat_t_average_std_iterative_Full_GD_with_TOD_est_s_2, control_t_average_iterative_Full_GD_with_TOD_est_s_2, control_t_average_std_iterative_Full_GD_with_TOD_est_s_2, p_t_iterative_Full_GD_with_TOD_est_s_2, p_hat_t_iterative_Full_GD_with_TOD_est_s_2, control_t_iterative_Full_GD_with_TOD_est_s_2, accumulated_cost_t_iterative_Full_GD_with_TOD_est_s_2 = get_model_predictions_reaching_task_application_4D_state(num_trials_MC_runs, K_TOD_s_2, L_iterative_Full_GD_with_TOD_est_s_2, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η, random_seed_MC)

#Expected cost using moment's propagation
expected_cost_mom_prop_iterative_Full_GD_with_TOD_est_s_2 = moments_propagation_reaching_task_application_4D_state(K_TOD_s_2, L_iterative_Full_GD_with_TOD_est_s_2, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η)

println("Iterative-GD OPTIMIZATION with TOD estimator\n")
println("- Expected cost using moment propagation (It-GD with TOD est): $expected_cost_mom_prop_iterative_Full_GD_with_TOD_est_s_2\n")
println("- Montecarlo - cost function (It-GD with TOD est): $final_cost_average_iterative_Full_GD_with_TOD_est_s_2  (error: $final_cost_average_std_iterative_Full_GD_with_TOD_est_s_2)\n")

#Iterative-GD with GD estimator
#Montecarlo simulations
final_cost_average_iterative_Full_GD_with_GD_est_s_2, final_cost_average_std_iterative_Full_GD_with_GD_est_s_2, p_t_average_iterative_Full_GD_with_GD_est_s_2, p_t_average_std_iterative_Full_GD_with_GD_est_s_2, p_hat_t_average_iterative_Full_GD_with_GD_est_s_2, p_hat_t_average_std_iterative_Full_GD_with_GD_est_s_2, control_t_average_iterative_Full_GD_with_GD_est_s_2, control_t_average_std_iterative_Full_GD_with_GD_est_s_2, p_t_iterative_Full_GD_with_GD_est_s_2, p_hat_t_iterative_Full_GD_with_GD_est_s_2, control_t_iterative_Full_GD_with_GD_est_s_2, accumulated_cost_t_iterative_Full_GD_with_GD_est_s_2 = get_model_predictions_reaching_task_application_4D_state(num_trials_MC_runs, K_GD_s_2, L_iterative_Full_GD_with_GD_est_s_2, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η, random_seed_MC)

#Expected cost using moment's propagation
expected_cost_mom_prop_iterative_Full_GD_with_GD_est_s_2 = moments_propagation_reaching_task_application_4D_state(K_GD_s_2, L_iterative_Full_GD_with_GD_est_s_2, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η)

println("Iterative-GD OPTIMIZATION with GD estimator\n")
println("- Expected cost using moment propagation (It-GD with GD est): $expected_cost_mom_prop_iterative_Full_GD_with_GD_est_s_2\n")
println("- Montecarlo - cost function (It-GD with GD est): $final_cost_average_iterative_Full_GD_with_GD_est_s_2  (error: $final_cost_average_std_iterative_Full_GD_with_GD_est_s_2)\n")

#FOLDER TO SAVE PLOTS 

name_plots = ""

cd(@__DIR__)#to go back to the directory of the script
full_folder_save_plots = "../Multi_D_Plots/Figures_LQG_with_internal_noise"

#PLOTS -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------

#norms of L and K vectors
norms_control_TOD_s_0 = zeros(N_steps-1)
norms_control_TOD_s_1 = zeros(N_steps-1)
norms_control_TOD_s_2 = zeros(N_steps-1)

norms_control_GD_s_0 = zeros(N_steps-1)
norms_control_GD_s_1 = zeros(N_steps-1)
norms_control_GD_s_2 = zeros(N_steps-1)

norms_estimator_TOD_s_0 = zeros(N_steps-1)
norms_estimator_TOD_s_1 = zeros(N_steps-1)
norms_estimator_TOD_s_2 = zeros(N_steps-1)

norms_estimator_GD_s_0 = zeros(N_steps-1)
norms_estimator_GD_s_1 = zeros(N_steps-1)
norms_estimator_GD_s_2 = zeros(N_steps-1)

norms_control_iterative_Full_GD_with_GD_est_s_0 = zeros(N_steps-1)
norms_control_iterative_Full_GD_with_GD_est_s_1 = zeros(N_steps-1)
norms_control_iterative_Full_GD_with_GD_est_s_2 = zeros(N_steps-1)

norms_control_iterative_Full_GD_with_TOD_est_s_0 = zeros(N_steps-1)
norms_control_iterative_Full_GD_with_TOD_est_s_1 = zeros(N_steps-1)
norms_control_iterative_Full_GD_with_TOD_est_s_2 = zeros(N_steps-1)

for i in 1:N_steps-1

    norms_control_GD_s_0[i] = norm(L_GD_s_0[:, :, i])
    norms_control_GD_s_1[i] = norm(L_GD_s_1[:, :, i])
    norms_control_GD_s_2[i] = norm(L_GD_s_2[:, :, i])

    norms_control_TOD_s_0[i] = norm(L_TOD_s_0[:, :, i])
    norms_control_TOD_s_1[i] = norm(L_TOD_s_1[:, :, i])
    norms_control_TOD_s_2[i] = norm(L_TOD_s_2[:, :, i])

    norms_estimator_GD_s_0[i] = norm(K_GD_s_0[:, :, i])
    norms_estimator_GD_s_1[i] = norm(K_GD_s_1[:, :, i])
    norms_estimator_GD_s_2[i] = norm(K_GD_s_2[:, :, i])

    norms_estimator_TOD_s_0[i] = norm(K_TOD_s_0[:, :, i])
    norms_estimator_TOD_s_1[i] = norm(K_TOD_s_1[:, :, i])
    norms_estimator_TOD_s_2[i] = norm(K_TOD_s_2[:, :, i])

    norms_control_iterative_Full_GD_with_GD_est_s_0[i] = norm(L_iterative_Full_GD_with_GD_est_s_0[:, :, i])
    norms_control_iterative_Full_GD_with_GD_est_s_1[i] = norm(L_iterative_Full_GD_with_GD_est_s_1[:, :, i])
    norms_control_iterative_Full_GD_with_GD_est_s_2[i] = norm(L_iterative_Full_GD_with_GD_est_s_2[:, :, i])

    norms_control_iterative_Full_GD_with_TOD_est_s_0[i] = norm(L_iterative_Full_GD_with_TOD_est_s_0[:, :, i])
    norms_control_iterative_Full_GD_with_TOD_est_s_1[i] = norm(L_iterative_Full_GD_with_TOD_est_s_1[:, :, i])
    norms_control_iterative_Full_GD_with_TOD_est_s_2[i] = norm(L_iterative_Full_GD_with_TOD_est_s_2[:, :, i])

end

#plot norms of L vectors for Iterative GD (with TOD estimator) and TOD
ylims_norm = (-100, 100 + maximum([maximum(norms_control_iterative_Full_GD_with_TOD_est_s_0), maximum(norms_control_iterative_Full_GD_with_TOD_est_s_1), maximum(norms_control_iterative_Full_GD_with_TOD_est_s_2), maximum(norms_control_TOD_s_0), maximum(norms_control_TOD_s_1), maximum(norms_control_TOD_s_2)]))
plt_L_vec_TOD = plot(collect(1:N_steps-2), norms_control_TOD_s_0[1:end-1], title = "", titlefont = font(12,"Computer Modern"), xlabel = L"$t$", ylabel = L"$|L_t|$", lw = 3, grid = false, label = L"TOD - \sigma_{\eta} = 0.0", color = cgrad(:Blues_5, 21)[10], ylims = ylims_norm)
plot!(plt_L_vec_TOD, collect(1:N_steps-2), norms_control_TOD_s_1[1:end-1], lw = 3, label = L"TOD - \sigma_{\eta} = 1.0", color = cgrad(:Blues_5, 21)[17])
plot!(plt_L_vec_TOD, collect(1:N_steps-2), norms_control_TOD_s_2[1:end-1], lw = 3, label = L"TOD - \sigma_{\eta} = 2.0", color = cgrad(:Blues_5, 21)[21], yticks = collect(0:200:ylims_norm[2]))

plt_L_vec_it_GD_with_TOD_est = plot(collect(1:N_steps-2), norms_control_iterative_Full_GD_with_TOD_est_s_0[1:end-1], lw = 3, label = L"GD - \sigma_{\eta} = 0.05", xlabel = L"$t$", color = cgrad(:Oranges_5, 21)[10], leg = false, ylims = ylims_norm, yticks = [])
plot!(plt_L_vec_it_GD_with_TOD_est, collect(1:N_steps-2), norms_control_iterative_Full_GD_with_TOD_est_s_1[1:end-1], lw = 3, label = L"TOD - \sigma_{\eta} = 0.5", color = cgrad(:Oranges_5, 21)[17])
plot!(plt_L_vec_it_GD_with_TOD_est, collect(1:N_steps-2), norms_control_iterative_Full_GD_with_TOD_est_s_2[1:end-1], lw = 3, label = L"GD - \sigma_{\eta} = 0.5", color = cgrad(:Oranges_5, 21)[21], leg = false)

layout_norm = @layout [a b]
plt_norm_TOD_it_GD_with_est = plot(plt_L_vec_TOD, plt_L_vec_it_GD_with_TOD_est, layout = layout_norm)
plt_norm_TOD_it_GD_with_est = plot(plt_norm_TOD_it_GD_with_est, xtickfontsize=12, ytickfontsize=12, xguidefontsize=18, yguidefontsize=18, thickness_scaling=2.0, size = (1300, 600), grid = false, leg =:false)

name_and_path_plot = full_folder_save_plots * "/L_t_norm_TOD_and_iterative_GD_with_TOD_estimator" * name_plots * ".pdf" 
savefig(plt_norm_TOD_it_GD_with_est, name_and_path_plot)

#Plot of performance for the four algorithms 

colors_TOD = [cgrad(:Blues_5, 21)[10], cgrad(:Blues_5, 21)[17], cgrad(:Blues_5, 21)[21]]
colors_GD = [cgrad(:BuGn_5, 21)[10], cgrad(:BuGn_5, 21)[17], cgrad(:BuGn_5, 21)[21]]
colors_iterative_GD_with_TOD_est = [cgrad(:Oranges_5, 21)[10], cgrad(:Oranges_5, 21)[17], cgrad(:Oranges_5, 21)[21]]
colors_iterative_GD_with_GD_est = [cgrad(:Purples_5, 21)[10], cgrad(:Purples_5, 21)[17], cgrad(:Purples_5, 21)[21]]

markersize = 5
markerstyles = [:circle, :star5, :diamond]

MC_cost_TOD = [final_cost_average_TOD_s_0, final_cost_average_TOD_s_1, final_cost_average_TOD_s_2]
MC_cost_GD = [final_cost_average_GD_s_0, final_cost_average_GD_s_1, final_cost_average_GD_s_2]
MC_cost_iterative_GD_with_TOD_est = [final_cost_average_iterative_Full_GD_with_TOD_est_s_0, final_cost_average_iterative_Full_GD_with_TOD_est_s_1, final_cost_average_iterative_Full_GD_with_TOD_est_s_2]
MC_cost_iterative_GD_with_GD_est = [final_cost_average_iterative_Full_GD_with_GD_est_s_0, final_cost_average_iterative_Full_GD_with_GD_est_s_1, final_cost_average_iterative_Full_GD_with_GD_est_s_2]

MC_std_cost_TOD = [final_cost_average_std_TOD_s_0, final_cost_average_std_TOD_s_1, final_cost_average_std_TOD_s_2]
MC_std_cost_GD = [final_cost_average_std_GD_s_0, final_cost_average_std_GD_s_1, final_cost_average_std_GD_s_2]
MC_std_cost_iterative_GD_with_TOD_est = [final_cost_average_std_iterative_Full_GD_with_TOD_est_s_0, final_cost_average_std_iterative_Full_GD_with_TOD_est_s_1, final_cost_average_std_iterative_Full_GD_with_TOD_est_s_2]
MC_std_cost_iterative_GD_with_GD_est = [final_cost_average_std_iterative_Full_GD_with_GD_est_s_0, final_cost_average_std_iterative_Full_GD_with_GD_est_s_1, final_cost_average_std_iterative_Full_GD_with_GD_est_s_2]

plt_cost_comparison = scatter(σ_η_internal_noise_position_vec, MC_cost_TOD, yerr = MC_std_cost_TOD , c = colors_TOD, ms=markersize, marker=markerstyles[1], title = "", titlefont = font(12,"Computer Modern"), xlabel = L"$\Omega_{\eta_{1,1}}$", ylabel = L"$\mathbb{E}[J(\Omega_{\eta_{1,1}})]$", lw = 3, grid = false, label = L"TOD", yscale =:log)
plot!(plt_cost_comparison, σ_η_internal_noise_position_vec, MC_cost_TOD, linestyle=:dash, color=cgrad(:Blues_5, 21)[10], label="")

scatter!(plt_cost_comparison, σ_η_internal_noise_position_vec, MC_cost_iterative_GD_with_TOD_est, yerr = MC_std_cost_iterative_GD_with_TOD_est , c = colors_iterative_GD_with_TOD_est, ms=markersize, marker=markerstyles[1], lw = 3, label = L"It-GD \ \mathrm{with} \ TOD \ \mathrm{estimator}")
plot!(plt_cost_comparison, σ_η_internal_noise_position_vec, MC_cost_iterative_GD_with_TOD_est, linestyle=:dash, color=cgrad(:Oranges_5, 21)[10], label="")

scatter!(plt_cost_comparison, σ_η_internal_noise_position_vec, MC_cost_GD, yerr = MC_std_cost_GD , c = colors_GD, ms=markersize, marker=markerstyles[1], lw = 3, label = L"GD")
plot!(plt_cost_comparison, σ_η_internal_noise_position_vec, MC_cost_GD, linestyle=:dash, color=cgrad(:BuGn_5, 21)[10], label="")

# scatter!(plt_cost_comparison, σ_η_internal_noise_position_vec, MC_cost_iterative_GD_with_GD_est, yerr = MC_std_cost_iterative_GD_with_GD_est , c = colors_iterative_GD_with_GD_est, ms=markersize, marker=markerstyles[1], lw = 3, label = L"GD")
# plot!(plt_cost_comparison, σ_η_internal_noise_position_vec, MC_cost_iterative_GD_with_GD_est, linestyle=:dash, color=cgrad(:Purples_5, 21)[10])

plt_cost_comparison = plot(plt_cost_comparison, xtickfontsize=12, ytickfontsize=12, xguidefontsize=18, yguidefontsize=18, thickness_scaling=2.0, size = (1300, 600), grid = false)

name_and_path_plot = full_folder_save_plots * "/cost_comparison_LQG_internal_noise" * name_plots * ".pdf" 
savefig(plt_cost_comparison, name_and_path_plot)

#solutions of TOD and iterative GD with GD estimator

ylims_L_iterarive_GD_GD = (-100 + minimum([minimum(L_GD_s_0[1,1,:]), minimum(L_GD_s_1[1,1,:]), minimum(L_GD_s_2[1,1,:]), minimum(L_iterative_Full_GD_with_GD_est_s_0[1,1,:]), minimum(L_iterative_Full_GD_with_GD_est_s_1[1,1,:]), minimum(L_iterative_Full_GD_with_GD_est_s_2[1,1,:])]), maximum([maximum(L_GD_s_0[1,1,:]), maximum(L_GD_s_1[1,1,:]), maximum(L_GD_s_2[1,1,:]), maximum(L_iterative_Full_GD_with_GD_est_s_0[1,1,:]), maximum(L_iterative_Full_GD_with_GD_est_s_1[1,1,:]), maximum(L_iterative_Full_GD_with_GD_est_s_2[1,1,:])]) + 100)

plt_L_GD = plot(L_GD_s_0[1,1,:], title = L"GD", titlefont = font(12,"Computer Modern"), xlabel = L"$t$", ylabel = "", lw = 3, grid = false, color=cgrad(:BuGn_5, 21)[10], ylims = ylims_L_iterarive_GD_GD)
plot!(plt_L_GD, L_GD_s_1[1,1,:], lw = 3, color=cgrad(:BuGn_5, 21)[17])
plot!(plt_L_GD, L_GD_s_2[1,1,:], lw = 3, color=cgrad(:BuGn_5, 21)[21], yticks = [])

plt_L_iterative_GD_with_GD_est = plot(L_iterative_Full_GD_with_GD_est_s_0[1,1,:], title = L"\mathrm{Iterative}-GD", titlefont = font(12,"Computer Modern"), xlabel = L"$t$", ylabel = L"$L_1(t)$", lw = 3, grid = false, color=cgrad(:Oranges_5, 21)[10], ylims = ylims_L_iterarive_GD_GD)
plot!(plt_L_iterative_GD_with_GD_est, L_iterative_Full_GD_with_GD_est_s_1[1,1,:], lw = 3, color=cgrad(:Oranges_5, 21)[17])
plot!(plt_L_iterative_GD_with_GD_est, L_iterative_Full_GD_with_GD_est_s_2[1,1,:], lw = 3, color=cgrad(:Oranges_5, 21)[21])

plt_L_iterarive_GD_GD = plot(plt_L_iterative_GD_with_GD_est, plt_L_GD, layout = layout_norm)
plt_L_iterarive_GD_GD = plot(plt_L_iterarive_GD_GD, xtickfontsize=12, ytickfontsize=12, xguidefontsize=18, yguidefontsize=18, thickness_scaling=2.0, size = (1300, 600), grid = false, leg =:false)

name_and_path_plot = full_folder_save_plots * "/L_t_iterative_GD_GD" * name_plots * ".pdf" 
savefig(plt_L_iterarive_GD_GD, name_and_path_plot)

#enhanced performance due to shallow landscape of the cost function

markersize = 3

plt_cost_comparison_iterative_GD_GD = scatter(σ_η_internal_noise_position_vec, MC_cost_GD, yerr = MC_std_cost_GD , c = colors_GD, ms=markersize, marker=markerstyles[1], label = L"GD", xlabel = L"$\Omega_{\eta_{1,1}}$", ylabel = L"$\mathbb{E}[J(\Omega_{\eta_{1,1}})]$", grid = false, yscale =:log)
plot!(plt_cost_comparison_iterative_GD_GD, σ_η_internal_noise_position_vec, MC_cost_GD, linestyle=:dash, color=cgrad(:BuGn_5, 21)[10], label="")

scatter!(plt_cost_comparison_iterative_GD_GD, σ_η_internal_noise_position_vec, MC_cost_iterative_GD_with_GD_est, yerr = MC_std_cost_iterative_GD_with_GD_est , c = colors_iterative_GD_with_TOD_est, ms=markersize, marker=markerstyles[1], lw = 3, label = L"It-GD \ \mathrm{with} \ GD \ \mathrm{estimator}")
plot!(plt_cost_comparison_iterative_GD_GD, σ_η_internal_noise_position_vec, MC_cost_iterative_GD_with_GD_est, linestyle=:dash, color=cgrad(:Oranges_5, 21)[10], label = "")

plt_cost_comparison_iterative_GD_GD = plot(plt_cost_comparison_iterative_GD_GD, xtickfontsize=12, ytickfontsize=12, xguidefontsize=18, yguidefontsize=18, thickness_scaling=2.0, size = (1300, 600), grid = false)

name_and_path_plot = full_folder_save_plots * "/cost_comparison_LQG_internal_noise_enhanced_performance" * name_plots * ".pdf" 
savefig(plt_cost_comparison_iterative_GD_GD, name_and_path_plot)
