/*
 *  Server-side XDR for NFSv4
 *
 *  Copyright (c) 2002 The Regents of the University of Michigan.
 *  All rights reserved.
 *
 *  Kendrick Smith <kmsmith@umich.edu>
 *  Andy Adamson   <andros@umich.edu>
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *  3. Neither the name of the University nor the names of its
 *     contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *  DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 *  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <linux file.h="">
#include <linux slab.h="">
#include <linux namei.h="">
#include <linux statfs.h="">
#include <linux utsname.h="">
#include <linux pagemap.h="">
#include <linux sunrpc="" svcauth_gss.h="">
#include <linux addr.h="" sunrpc="">
#include <linux xattr.h="">
#include <linux vmalloc.h="">

#include <uapi linux="" xattr.h="">

#include "idmap.h"
#include "acl.h"
#include "xdr4.h"
#include "vfs.h"
#include "state.h"
#include "cache.h"
#include "netns.h"
#include "pnfs.h"
#include "filecache.h"

#include "trace.h"

#ifdef CONFIG_NFSD_V4_SECURITY_LABEL
#include <linux security.h="">
#endif


#define NFSDDBG_FACILITY		NFSDDBG_XDR

const u32 nfsd_suppattrs[3][3] = {
	{NFSD4_SUPPORTED_ATTRS_WORD0,
	 NFSD4_SUPPORTED_ATTRS_WORD1,
	 NFSD4_SUPPORTED_ATTRS_WORD2},

	{NFSD4_1_SUPPORTED_ATTRS_WORD0,
	 NFSD4_1_SUPPORTED_ATTRS_WORD1,
	 NFSD4_1_SUPPORTED_ATTRS_WORD2},

	{NFSD4_1_SUPPORTED_ATTRS_WORD0,
	 NFSD4_1_SUPPORTED_ATTRS_WORD1,
	 NFSD4_2_SUPPORTED_ATTRS_WORD2},
};

/*
 * As per referral draft, the fsid for a referral MUST be different from the fsid of the containing
 * directory in order to indicate to the client that a filesystem boundary is present
 * We use a fixed fsid for a referral
 */
#define NFS4_REFERRAL_FSID_MAJOR	0x8000000ULL
#define NFS4_REFERRAL_FSID_MINOR	0x8000000ULL

static __be32
check_filename(char *str, int len)
{
	int i;

	if (len == 0)
		return nfserr_inval;
	if (len &gt; NFS4_MAXNAMLEN)
		return nfserr_nametoolong;
	if (isdotent(str, len))
		return nfserr_badname;
	for (i = 0; i &lt; len; i++)
		if (str[i] == '/')
			return nfserr_badname;
	return 0;
}

static int zero_clientid(clientid_t *clid)
{
	return (clid-&gt;cl_boot == 0) &amp;&amp; (clid-&gt;cl_id == 0);
}

/**
 * svcxdr_tmpalloc - allocate memory to be freed after compound processing
 * @argp: NFSv4 compound argument structure
 * @len: length of buffer to allocate
 *
 * Allocates a buffer of size @len to be freed when processing the compound
 * operation described in @argp finishes.
 */
static void *
svcxdr_tmpalloc(struct nfsd4_compoundargs *argp, u32 len)
{
	struct svcxdr_tmpbuf *tb;

	tb = kmalloc(sizeof(*tb) + len, GFP_KERNEL);
	if (!tb)
		return NULL;
	tb-&gt;next = argp-&gt;to_free;
	argp-&gt;to_free = tb;
	return tb-&gt;buf;
}

/*
 * For xdr strings that need to be passed to other kernel api's
 * as null-terminated strings.
 *
 * Note null-terminating in place usually isn't safe since the
 * buffer might end on a page boundary.
 */
static char *
svcxdr_dupstr(struct nfsd4_compoundargs *argp, void *buf, u32 len)
{
	char *p = svcxdr_tmpalloc(argp, len + 1);

	if (!p)
		return NULL;
	memcpy(p, buf, len);
	p[len] = '\0';
	return p;
}

static void *
svcxdr_savemem(struct nfsd4_compoundargs *argp, __be32 *p, u32 len)
{
	__be32 *tmp;

	/*
	 * The location of the decoded data item is stable,
	 * so @p is OK to use. This is the common case.
	 */
	if (p != argp-&gt;xdr-&gt;scratch.iov_base)
		return p;

	tmp = svcxdr_tmpalloc(argp, len);
	if (!tmp)
		return NULL;
	memcpy(tmp, p, len);
	return tmp;
}

/*
 * NFSv4 basic data type decoders
 */

/*
 * This helper handles variable-length opaques which belong to protocol
 * elements that this implementation does not support.
 */
static __be32
nfsd4_decode_ignored_string(struct nfsd4_compoundargs *argp, u32 maxlen)
{
	u32 len;

	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;len) &lt; 0)
		return nfserr_bad_xdr;
	if (maxlen &amp;&amp; len &gt; maxlen)
		return nfserr_bad_xdr;
	if (!xdr_inline_decode(argp-&gt;xdr, len))
		return nfserr_bad_xdr;

	return nfs_ok;
}

static __be32
nfsd4_decode_opaque(struct nfsd4_compoundargs *argp, struct xdr_netobj *o)
{
	__be32 *p;
	u32 len;

	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;len) &lt; 0)
		return nfserr_bad_xdr;
	if (len == 0 || len &gt; NFS4_OPAQUE_LIMIT)
		return nfserr_bad_xdr;
	p = xdr_inline_decode(argp-&gt;xdr, len);
	if (!p)
		return nfserr_bad_xdr;
	o-&gt;data = svcxdr_savemem(argp, p, len);
	if (!o-&gt;data)
		return nfserr_jukebox;
	o-&gt;len = len;

	return nfs_ok;
}

static __be32
nfsd4_decode_component4(struct nfsd4_compoundargs *argp, char **namp, u32 *lenp)
{
	__be32 *p, status;

	if (xdr_stream_decode_u32(argp-&gt;xdr, lenp) &lt; 0)
		return nfserr_bad_xdr;
	p = xdr_inline_decode(argp-&gt;xdr, *lenp);
	if (!p)
		return nfserr_bad_xdr;
	status = check_filename((char *)p, *lenp);
	if (status)
		return status;
	*namp = svcxdr_savemem(argp, p, *lenp);
	if (!*namp)
		return nfserr_jukebox;

	return nfs_ok;
}

static __be32
nfsd4_decode_nfstime4(struct nfsd4_compoundargs *argp, struct timespec64 *tv)
{
	__be32 *p;

	p = xdr_inline_decode(argp-&gt;xdr, XDR_UNIT * 3);
	if (!p)
		return nfserr_bad_xdr;
	p = xdr_decode_hyper(p, &amp;tv-&gt;tv_sec);
	tv-&gt;tv_nsec = be32_to_cpup(p++);
	if (tv-&gt;tv_nsec &gt;= (u32)1000000000)
		return nfserr_inval;
	return nfs_ok;
}

static __be32
nfsd4_decode_verifier4(struct nfsd4_compoundargs *argp, nfs4_verifier *verf)
{
	__be32 *p;

	p = xdr_inline_decode(argp-&gt;xdr, NFS4_VERIFIER_SIZE);
	if (!p)
		return nfserr_bad_xdr;
	memcpy(verf-&gt;data, p, sizeof(verf-&gt;data));
	return nfs_ok;
}

/**
 * nfsd4_decode_bitmap4 - Decode an NFSv4 bitmap4
 * @argp: NFSv4 compound argument structure
 * @bmval: pointer to an array of u32's to decode into
 * @bmlen: size of the @bmval array
 *
 * The server needs to return nfs_ok rather than nfserr_bad_xdr when
 * encountering bitmaps containing bits it does not recognize. This
 * includes bits in bitmap words past WORDn, where WORDn is the last
 * bitmap WORD the implementation currently supports. Thus we are
 * careful here to simply ignore bits in bitmap words that this
 * implementation has yet to support explicitly.
 *
 * Return values:
 *   %nfs_ok: @bmval populated successfully
 *   %nfserr_bad_xdr: the encoded bitmap was invalid
 */
static __be32
nfsd4_decode_bitmap4(struct nfsd4_compoundargs *argp, u32 *bmval, u32 bmlen)
{
	ssize_t status;

	status = xdr_stream_decode_uint32_array(argp-&gt;xdr, bmval, bmlen);
	return status == -EBADMSG ? nfserr_bad_xdr : nfs_ok;
}

static __be32
nfsd4_decode_nfsace4(struct nfsd4_compoundargs *argp, struct nfs4_ace *ace)
{
	__be32 *p, status;
	u32 length;

	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;ace-&gt;type) &lt; 0)
		return nfserr_bad_xdr;
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;ace-&gt;flag) &lt; 0)
		return nfserr_bad_xdr;
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;ace-&gt;access_mask) &lt; 0)
		return nfserr_bad_xdr;

	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;length) &lt; 0)
		return nfserr_bad_xdr;
	p = xdr_inline_decode(argp-&gt;xdr, length);
	if (!p)
		return nfserr_bad_xdr;
	ace-&gt;whotype = nfs4_acl_get_whotype((char *)p, length);
	if (ace-&gt;whotype != NFS4_ACL_WHO_NAMED)
		status = nfs_ok;
	else if (ace-&gt;flag &amp; NFS4_ACE_IDENTIFIER_GROUP)
		status = nfsd_map_name_to_gid(argp-&gt;rqstp,
				(char *)p, length, &amp;ace-&gt;who_gid);
	else
		status = nfsd_map_name_to_uid(argp-&gt;rqstp,
				(char *)p, length, &amp;ace-&gt;who_uid);

	return status;
}

/* A counted array of nfsace4's */
static noinline __be32
nfsd4_decode_acl(struct nfsd4_compoundargs *argp, struct nfs4_acl **acl)
{
	struct nfs4_ace *ace;
	__be32 status;
	u32 count;

	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;count) &lt; 0)
		return nfserr_bad_xdr;

	if (count &gt; xdr_stream_remaining(argp-&gt;xdr) / 20)
		/*
		 * Even with 4-byte names there wouldn't be
		 * space for that many aces; something fishy is
		 * going on:
		 */
		return nfserr_fbig;

	*acl = svcxdr_tmpalloc(argp, nfs4_acl_bytes(count));
	if (*acl == NULL)
		return nfserr_jukebox;

	(*acl)-&gt;naces = count;
	for (ace = (*acl)-&gt;aces; ace &lt; (*acl)-&gt;aces + count; ace++) {
		status = nfsd4_decode_nfsace4(argp, ace);
		if (status)
			return status;
	}

	return nfs_ok;
}

static noinline __be32
nfsd4_decode_security_label(struct nfsd4_compoundargs *argp,
			    struct xdr_netobj *label)
{
	u32 lfs, pi, length;
	__be32 *p;

	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;lfs) &lt; 0)
		return nfserr_bad_xdr;
	if (xdr_stream_decode_u32(argp-&gt;xdr, π) &lt; 0)
		return nfserr_bad_xdr;

	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;length) &lt; 0)
		return nfserr_bad_xdr;
	if (length &gt; NFS4_MAXLABELLEN)
		return nfserr_badlabel;
	p = xdr_inline_decode(argp-&gt;xdr, length);
	if (!p)
		return nfserr_bad_xdr;
	label-&gt;len = length;
	label-&gt;data = svcxdr_dupstr(argp, p, length);
	if (!label-&gt;data)
		return nfserr_jukebox;

	return nfs_ok;
}

static __be32
nfsd4_decode_fattr4(struct nfsd4_compoundargs *argp, u32 *bmval, u32 bmlen,
		    struct iattr *iattr, struct nfs4_acl **acl,
		    struct xdr_netobj *label, int *umask)
{
	unsigned int starting_pos;
	u32 attrlist4_count;
	__be32 *p, status;

	iattr-&gt;ia_valid = 0;
	status = nfsd4_decode_bitmap4(argp, bmval, bmlen);
	if (status)
		return nfserr_bad_xdr;

	if (bmval[0] &amp; ~NFSD_WRITEABLE_ATTRS_WORD0
	    || bmval[1] &amp; ~NFSD_WRITEABLE_ATTRS_WORD1
	    || bmval[2] &amp; ~NFSD_WRITEABLE_ATTRS_WORD2) {
		if (nfsd_attrs_supported(argp-&gt;minorversion, bmval))
			return nfserr_inval;
		return nfserr_attrnotsupp;
	}

	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;attrlist4_count) &lt; 0)
		return nfserr_bad_xdr;
	starting_pos = xdr_stream_pos(argp-&gt;xdr);

	if (bmval[0] &amp; FATTR4_WORD0_SIZE) {
		u64 size;

		if (xdr_stream_decode_u64(argp-&gt;xdr, &amp;size) &lt; 0)
			return nfserr_bad_xdr;
		iattr-&gt;ia_size = size;
		iattr-&gt;ia_valid |= ATTR_SIZE;
	}
	if (bmval[0] &amp; FATTR4_WORD0_ACL) {
		status = nfsd4_decode_acl(argp, acl);
		if (status)
			return status;
	} else
		*acl = NULL;
	if (bmval[1] &amp; FATTR4_WORD1_MODE) {
		u32 mode;

		if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;mode) &lt; 0)
			return nfserr_bad_xdr;
		iattr-&gt;ia_mode = mode;
		iattr-&gt;ia_mode &amp;= (S_IFMT | S_IALLUGO);
		iattr-&gt;ia_valid |= ATTR_MODE;
	}
	if (bmval[1] &amp; FATTR4_WORD1_OWNER) {
		u32 length;

		if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;length) &lt; 0)
			return nfserr_bad_xdr;
		p = xdr_inline_decode(argp-&gt;xdr, length);
		if (!p)
			return nfserr_bad_xdr;
		status = nfsd_map_name_to_uid(argp-&gt;rqstp, (char *)p, length,
					      &amp;iattr-&gt;ia_uid);
		if (status)
			return status;
		iattr-&gt;ia_valid |= ATTR_UID;
	}
	if (bmval[1] &amp; FATTR4_WORD1_OWNER_GROUP) {
		u32 length;

		if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;length) &lt; 0)
			return nfserr_bad_xdr;
		p = xdr_inline_decode(argp-&gt;xdr, length);
		if (!p)
			return nfserr_bad_xdr;
		status = nfsd_map_name_to_gid(argp-&gt;rqstp, (char *)p, length,
					      &amp;iattr-&gt;ia_gid);
		if (status)
			return status;
		iattr-&gt;ia_valid |= ATTR_GID;
	}
	if (bmval[1] &amp; FATTR4_WORD1_TIME_ACCESS_SET) {
		u32 set_it;

		if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;set_it) &lt; 0)
			return nfserr_bad_xdr;
		switch (set_it) {
		case NFS4_SET_TO_CLIENT_TIME:
			status = nfsd4_decode_nfstime4(argp, &amp;iattr-&gt;ia_atime);
			if (status)
				return status;
			iattr-&gt;ia_valid |= (ATTR_ATIME | ATTR_ATIME_SET);
			break;
		case NFS4_SET_TO_SERVER_TIME:
			iattr-&gt;ia_valid |= ATTR_ATIME;
			break;
		default:
			return nfserr_bad_xdr;
		}
	}
	if (bmval[1] &amp; FATTR4_WORD1_TIME_CREATE) {
		struct timespec64 ts;

		/* No Linux filesystem supports setting this attribute. */
		bmval[1] &amp;= ~FATTR4_WORD1_TIME_CREATE;
		status = nfsd4_decode_nfstime4(argp, &amp;ts);
		if (status)
			return status;
	}
	if (bmval[1] &amp; FATTR4_WORD1_TIME_MODIFY_SET) {
		u32 set_it;

		if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;set_it) &lt; 0)
			return nfserr_bad_xdr;
		switch (set_it) {
		case NFS4_SET_TO_CLIENT_TIME:
			status = nfsd4_decode_nfstime4(argp, &amp;iattr-&gt;ia_mtime);
			if (status)
				return status;
			iattr-&gt;ia_valid |= (ATTR_MTIME | ATTR_MTIME_SET);
			break;
		case NFS4_SET_TO_SERVER_TIME:
			iattr-&gt;ia_valid |= ATTR_MTIME;
			break;
		default:
			return nfserr_bad_xdr;
		}
	}
	label-&gt;len = 0;
	if (IS_ENABLED(CONFIG_NFSD_V4_SECURITY_LABEL) &amp;&amp;
	    bmval[2] &amp; FATTR4_WORD2_SECURITY_LABEL) {
		status = nfsd4_decode_security_label(argp, label);
		if (status)
			return status;
	}
	if (bmval[2] &amp; FATTR4_WORD2_MODE_UMASK) {
		u32 mode, mask;

		if (!umask)
			return nfserr_bad_xdr;
		if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;mode) &lt; 0)
			return nfserr_bad_xdr;
		iattr-&gt;ia_mode = mode &amp; (S_IFMT | S_IALLUGO);
		if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;mask) &lt; 0)
			return nfserr_bad_xdr;
		*umask = mask &amp; S_IRWXUGO;
		iattr-&gt;ia_valid |= ATTR_MODE;
	}

	/* request sanity: did attrlist4 contain the expected number of words? */
	if (attrlist4_count != xdr_stream_pos(argp-&gt;xdr) - starting_pos)
		return nfserr_bad_xdr;

	return nfs_ok;
}

static __be32
nfsd4_decode_stateid4(struct nfsd4_compoundargs *argp, stateid_t *sid)
{
	__be32 *p;

	p = xdr_inline_decode(argp-&gt;xdr, NFS4_STATEID_SIZE);
	if (!p)
		return nfserr_bad_xdr;
	sid-&gt;si_generation = be32_to_cpup(p++);
	memcpy(&amp;sid-&gt;si_opaque, p, sizeof(sid-&gt;si_opaque));
	return nfs_ok;
}

static __be32
nfsd4_decode_clientid4(struct nfsd4_compoundargs *argp, clientid_t *clientid)
{
	__be32 *p;

	p = xdr_inline_decode(argp-&gt;xdr, sizeof(__be64));
	if (!p)
		return nfserr_bad_xdr;
	memcpy(clientid, p, sizeof(*clientid));
	return nfs_ok;
}

static __be32
nfsd4_decode_state_owner4(struct nfsd4_compoundargs *argp,
			  clientid_t *clientid, struct xdr_netobj *owner)
{
	__be32 status;

	status = nfsd4_decode_clientid4(argp, clientid);
	if (status)
		return status;
	return nfsd4_decode_opaque(argp, owner);
}

#ifdef CONFIG_NFSD_PNFS
static __be32
nfsd4_decode_deviceid4(struct nfsd4_compoundargs *argp,
		       struct nfsd4_deviceid *devid)
{
	__be32 *p;

	p = xdr_inline_decode(argp-&gt;xdr, NFS4_DEVICEID4_SIZE);
	if (!p)
		return nfserr_bad_xdr;
	memcpy(devid, p, sizeof(*devid));
	return nfs_ok;
}

static __be32
nfsd4_decode_layoutupdate4(struct nfsd4_compoundargs *argp,
			   struct nfsd4_layoutcommit *lcp)
{
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;lcp-&gt;lc_layout_type) &lt; 0)
		return nfserr_bad_xdr;
	if (lcp-&gt;lc_layout_type &lt; LAYOUT_NFSV4_1_FILES)
		return nfserr_bad_xdr;
	if (lcp-&gt;lc_layout_type &gt;= LAYOUT_TYPE_MAX)
		return nfserr_bad_xdr;

	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;lcp-&gt;lc_up_len) &lt; 0)
		return nfserr_bad_xdr;
	if (lcp-&gt;lc_up_len &gt; 0) {
		lcp-&gt;lc_up_layout = xdr_inline_decode(argp-&gt;xdr, lcp-&gt;lc_up_len);
		if (!lcp-&gt;lc_up_layout)
			return nfserr_bad_xdr;
	}

	return nfs_ok;
}

static __be32
nfsd4_decode_layoutreturn4(struct nfsd4_compoundargs *argp,
			   struct nfsd4_layoutreturn *lrp)
{
	__be32 status;

	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;lrp-&gt;lr_return_type) &lt; 0)
		return nfserr_bad_xdr;
	switch (lrp-&gt;lr_return_type) {
	case RETURN_FILE:
		if (xdr_stream_decode_u64(argp-&gt;xdr, &amp;lrp-&gt;lr_seg.offset) &lt; 0)
			return nfserr_bad_xdr;
		if (xdr_stream_decode_u64(argp-&gt;xdr, &amp;lrp-&gt;lr_seg.length) &lt; 0)
			return nfserr_bad_xdr;
		status = nfsd4_decode_stateid4(argp, &amp;lrp-&gt;lr_sid);
		if (status)
			return status;
		if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;lrp-&gt;lrf_body_len) &lt; 0)
			return nfserr_bad_xdr;
		if (lrp-&gt;lrf_body_len &gt; 0) {
			lrp-&gt;lrf_body = xdr_inline_decode(argp-&gt;xdr, lrp-&gt;lrf_body_len);
			if (!lrp-&gt;lrf_body)
				return nfserr_bad_xdr;
		}
		break;
	case RETURN_FSID:
	case RETURN_ALL:
		lrp-&gt;lr_seg.offset = 0;
		lrp-&gt;lr_seg.length = NFS4_MAX_UINT64;
		break;
	default:
		return nfserr_bad_xdr;
	}

	return nfs_ok;
}

#endif /* CONFIG_NFSD_PNFS */

static __be32
nfsd4_decode_sessionid4(struct nfsd4_compoundargs *argp,
			struct nfs4_sessionid *sessionid)
{
	__be32 *p;

	p = xdr_inline_decode(argp-&gt;xdr, NFS4_MAX_SESSIONID_LEN);
	if (!p)
		return nfserr_bad_xdr;
	memcpy(sessionid-&gt;data, p, sizeof(sessionid-&gt;data));
	return nfs_ok;
}

/* Defined in Appendix A of RFC 5531 */
static __be32
nfsd4_decode_authsys_parms(struct nfsd4_compoundargs *argp,
			   struct nfsd4_cb_sec *cbs)
{
	u32 stamp, gidcount, uid, gid;
	__be32 *p, status;

	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;stamp) &lt; 0)
		return nfserr_bad_xdr;
	/* machine name */
	status = nfsd4_decode_ignored_string(argp, 255);
	if (status)
		return status;
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;uid) &lt; 0)
		return nfserr_bad_xdr;
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;gid) &lt; 0)
		return nfserr_bad_xdr;
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;gidcount) &lt; 0)
		return nfserr_bad_xdr;
	if (gidcount &gt; 16)
		return nfserr_bad_xdr;
	p = xdr_inline_decode(argp-&gt;xdr, gidcount &lt;&lt; 2);
	if (!p)
		return nfserr_bad_xdr;
	if (cbs-&gt;flavor == (u32)(-1)) {
		struct user_namespace *userns = nfsd_user_namespace(argp-&gt;rqstp);

		kuid_t kuid = make_kuid(userns, uid);
		kgid_t kgid = make_kgid(userns, gid);
		if (uid_valid(kuid) &amp;&amp; gid_valid(kgid)) {
			cbs-&gt;uid = kuid;
			cbs-&gt;gid = kgid;
			cbs-&gt;flavor = RPC_AUTH_UNIX;
		} else {
			dprintk("RPC_AUTH_UNIX with invalid uid or gid, ignoring!\n");
		}
	}

	return nfs_ok;
}

static __be32
nfsd4_decode_gss_cb_handles4(struct nfsd4_compoundargs *argp,
			     struct nfsd4_cb_sec *cbs)
{
	__be32 status;
	u32 service;

	dprintk("RPC_AUTH_GSS callback secflavor not supported!\n");

	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;service) &lt; 0)
		return nfserr_bad_xdr;
	if (service &lt; RPC_GSS_SVC_NONE || service &gt; RPC_GSS_SVC_PRIVACY)
		return nfserr_bad_xdr;
	/* gcbp_handle_from_server */
	status = nfsd4_decode_ignored_string(argp, 0);
	if (status)
		return status;
	/* gcbp_handle_from_client */
	status = nfsd4_decode_ignored_string(argp, 0);
	if (status)
		return status;

	return nfs_ok;
}

/* a counted array of callback_sec_parms4 items */
static __be32
nfsd4_decode_cb_sec(struct nfsd4_compoundargs *argp, struct nfsd4_cb_sec *cbs)
{
	u32 i, secflavor, nr_secflavs;
	__be32 status;

	/* callback_sec_params4 */
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;nr_secflavs) &lt; 0)
		return nfserr_bad_xdr;
	if (nr_secflavs)
		cbs-&gt;flavor = (u32)(-1);
	else
		/* Is this legal? Be generous, take it to mean AUTH_NONE: */
		cbs-&gt;flavor = 0;

	for (i = 0; i &lt; nr_secflavs; ++i) {
		if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;secflavor) &lt; 0)
			return nfserr_bad_xdr;
		switch (secflavor) {
		case RPC_AUTH_NULL:
			/* void */
			if (cbs-&gt;flavor == (u32)(-1))
				cbs-&gt;flavor = RPC_AUTH_NULL;
			break;
		case RPC_AUTH_UNIX:
			status = nfsd4_decode_authsys_parms(argp, cbs);
			if (status)
				return status;
			break;
		case RPC_AUTH_GSS:
			status = nfsd4_decode_gss_cb_handles4(argp, cbs);
			if (status)
				return status;
			break;
		default:
			return nfserr_inval;
		}
	}

	return nfs_ok;
}


/*
 * NFSv4 operation argument decoders
 */

static __be32
nfsd4_decode_access(struct nfsd4_compoundargs *argp,
		    struct nfsd4_access *access)
{
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;access-&gt;ac_req_access) &lt; 0)
		return nfserr_bad_xdr;
	return nfs_ok;
}

static __be32
nfsd4_decode_close(struct nfsd4_compoundargs *argp, struct nfsd4_close *close)
{
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;close-&gt;cl_seqid) &lt; 0)
		return nfserr_bad_xdr;
	return nfsd4_decode_stateid4(argp, &amp;close-&gt;cl_stateid);
}


static __be32
nfsd4_decode_commit(struct nfsd4_compoundargs *argp, struct nfsd4_commit *commit)
{
	if (xdr_stream_decode_u64(argp-&gt;xdr, &amp;commit-&gt;co_offset) &lt; 0)
		return nfserr_bad_xdr;
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;commit-&gt;co_count) &lt; 0)
		return nfserr_bad_xdr;
	memset(&amp;commit-&gt;co_verf, 0, sizeof(commit-&gt;co_verf));
	return nfs_ok;
}

static __be32
nfsd4_decode_create(struct nfsd4_compoundargs *argp, struct nfsd4_create *create)
{
	__be32 *p, status;

	memset(create, 0, sizeof(*create));
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;create-&gt;cr_type) &lt; 0)
		return nfserr_bad_xdr;
	switch (create-&gt;cr_type) {
	case NF4LNK:
		if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;create-&gt;cr_datalen) &lt; 0)
			return nfserr_bad_xdr;
		p = xdr_inline_decode(argp-&gt;xdr, create-&gt;cr_datalen);
		if (!p)
			return nfserr_bad_xdr;
		create-&gt;cr_data = svcxdr_dupstr(argp, p, create-&gt;cr_datalen);
		if (!create-&gt;cr_data)
			return nfserr_jukebox;
		break;
	case NF4BLK:
	case NF4CHR:
		if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;create-&gt;cr_specdata1) &lt; 0)
			return nfserr_bad_xdr;
		if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;create-&gt;cr_specdata2) &lt; 0)
			return nfserr_bad_xdr;
		break;
	case NF4SOCK:
	case NF4FIFO:
	case NF4DIR:
	default:
		break;
	}
	status = nfsd4_decode_component4(argp, &amp;create-&gt;cr_name,
					 &amp;create-&gt;cr_namelen);
	if (status)
		return status;
	status = nfsd4_decode_fattr4(argp, create-&gt;cr_bmval,
				    ARRAY_SIZE(create-&gt;cr_bmval),
				    &amp;create-&gt;cr_iattr, &amp;create-&gt;cr_acl,
				    &amp;create-&gt;cr_label, &amp;create-&gt;cr_umask);
	if (status)
		return status;

	return nfs_ok;
}

static inline __be32
nfsd4_decode_delegreturn(struct nfsd4_compoundargs *argp, struct nfsd4_delegreturn *dr)
{
	return nfsd4_decode_stateid4(argp, &amp;dr-&gt;dr_stateid);
}

static inline __be32
nfsd4_decode_getattr(struct nfsd4_compoundargs *argp, struct nfsd4_getattr *getattr)
{
	memset(getattr, 0, sizeof(*getattr));
	return nfsd4_decode_bitmap4(argp, getattr-&gt;ga_bmval,
				    ARRAY_SIZE(getattr-&gt;ga_bmval));
}

static __be32
nfsd4_decode_link(struct nfsd4_compoundargs *argp, struct nfsd4_link *link)
{
	memset(link, 0, sizeof(*link));
	return nfsd4_decode_component4(argp, &amp;link-&gt;li_name, &amp;link-&gt;li_namelen);
}

static __be32
nfsd4_decode_open_to_lock_owner4(struct nfsd4_compoundargs *argp,
				 struct nfsd4_lock *lock)
{
	__be32 status;

	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;lock-&gt;lk_new_open_seqid) &lt; 0)
		return nfserr_bad_xdr;
	status = nfsd4_decode_stateid4(argp, &amp;lock-&gt;lk_new_open_stateid);
	if (status)
		return status;
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;lock-&gt;lk_new_lock_seqid) &lt; 0)
		return nfserr_bad_xdr;
	return nfsd4_decode_state_owner4(argp, &amp;lock-&gt;lk_new_clientid,
					 &amp;lock-&gt;lk_new_owner);
}

static __be32
nfsd4_decode_exist_lock_owner4(struct nfsd4_compoundargs *argp,
			       struct nfsd4_lock *lock)
{
	__be32 status;

	status = nfsd4_decode_stateid4(argp, &amp;lock-&gt;lk_old_lock_stateid);
	if (status)
		return status;
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;lock-&gt;lk_old_lock_seqid) &lt; 0)
		return nfserr_bad_xdr;

	return nfs_ok;
}

static __be32
nfsd4_decode_locker4(struct nfsd4_compoundargs *argp, struct nfsd4_lock *lock)
{
	if (xdr_stream_decode_bool(argp-&gt;xdr, &amp;lock-&gt;lk_is_new) &lt; 0)
		return nfserr_bad_xdr;
	if (lock-&gt;lk_is_new)
		return nfsd4_decode_open_to_lock_owner4(argp, lock);
	return nfsd4_decode_exist_lock_owner4(argp, lock);
}

static __be32
nfsd4_decode_lock(struct nfsd4_compoundargs *argp, struct nfsd4_lock *lock)
{
	memset(lock, 0, sizeof(*lock));
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;lock-&gt;lk_type) &lt; 0)
		return nfserr_bad_xdr;
	if ((lock-&gt;lk_type &lt; NFS4_READ_LT) || (lock-&gt;lk_type &gt; NFS4_WRITEW_LT))
		return nfserr_bad_xdr;
	if (xdr_stream_decode_bool(argp-&gt;xdr, &amp;lock-&gt;lk_reclaim) &lt; 0)
		return nfserr_bad_xdr;
	if (xdr_stream_decode_u64(argp-&gt;xdr, &amp;lock-&gt;lk_offset) &lt; 0)
		return nfserr_bad_xdr;
	if (xdr_stream_decode_u64(argp-&gt;xdr, &amp;lock-&gt;lk_length) &lt; 0)
		return nfserr_bad_xdr;
	return nfsd4_decode_locker4(argp, lock);
}

static __be32
nfsd4_decode_lockt(struct nfsd4_compoundargs *argp, struct nfsd4_lockt *lockt)
{
	memset(lockt, 0, sizeof(*lockt));
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;lockt-&gt;lt_type) &lt; 0)
		return nfserr_bad_xdr;
	if ((lockt-&gt;lt_type &lt; NFS4_READ_LT) || (lockt-&gt;lt_type &gt; NFS4_WRITEW_LT))
		return nfserr_bad_xdr;
	if (xdr_stream_decode_u64(argp-&gt;xdr, &amp;lockt-&gt;lt_offset) &lt; 0)
		return nfserr_bad_xdr;
	if (xdr_stream_decode_u64(argp-&gt;xdr, &amp;lockt-&gt;lt_length) &lt; 0)
		return nfserr_bad_xdr;
	return nfsd4_decode_state_owner4(argp, &amp;lockt-&gt;lt_clientid,
					 &amp;lockt-&gt;lt_owner);
}

static __be32
nfsd4_decode_locku(struct nfsd4_compoundargs *argp, struct nfsd4_locku *locku)
{
	__be32 status;

	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;locku-&gt;lu_type) &lt; 0)
		return nfserr_bad_xdr;
	if ((locku-&gt;lu_type &lt; NFS4_READ_LT) || (locku-&gt;lu_type &gt; NFS4_WRITEW_LT))
		return nfserr_bad_xdr;
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;locku-&gt;lu_seqid) &lt; 0)
		return nfserr_bad_xdr;
	status = nfsd4_decode_stateid4(argp, &amp;locku-&gt;lu_stateid);
	if (status)
		return status;
	if (xdr_stream_decode_u64(argp-&gt;xdr, &amp;locku-&gt;lu_offset) &lt; 0)
		return nfserr_bad_xdr;
	if (xdr_stream_decode_u64(argp-&gt;xdr, &amp;locku-&gt;lu_length) &lt; 0)
		return nfserr_bad_xdr;

	return nfs_ok;
}

static __be32
nfsd4_decode_lookup(struct nfsd4_compoundargs *argp, struct nfsd4_lookup *lookup)
{
	return nfsd4_decode_component4(argp, &amp;lookup-&gt;lo_name, &amp;lookup-&gt;lo_len);
}

static __be32
nfsd4_decode_createhow4(struct nfsd4_compoundargs *argp, struct nfsd4_open *open)
{
	__be32 status;

	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;open-&gt;op_createmode) &lt; 0)
		return nfserr_bad_xdr;
	switch (open-&gt;op_createmode) {
	case NFS4_CREATE_UNCHECKED:
	case NFS4_CREATE_GUARDED:
		status = nfsd4_decode_fattr4(argp, open-&gt;op_bmval,
					     ARRAY_SIZE(open-&gt;op_bmval),
					     &amp;open-&gt;op_iattr, &amp;open-&gt;op_acl,
					     &amp;open-&gt;op_label, &amp;open-&gt;op_umask);
		if (status)
			return status;
		break;
	case NFS4_CREATE_EXCLUSIVE:
		status = nfsd4_decode_verifier4(argp, &amp;open-&gt;op_verf);
		if (status)
			return status;
		break;
	case NFS4_CREATE_EXCLUSIVE4_1:
		if (argp-&gt;minorversion &lt; 1)
			return nfserr_bad_xdr;
		status = nfsd4_decode_verifier4(argp, &amp;open-&gt;op_verf);
		if (status)
			return status;
		status = nfsd4_decode_fattr4(argp, open-&gt;op_bmval,
					     ARRAY_SIZE(open-&gt;op_bmval),
					     &amp;open-&gt;op_iattr, &amp;open-&gt;op_acl,
					     &amp;open-&gt;op_label, &amp;open-&gt;op_umask);
		if (status)
			return status;
		break;
	default:
		return nfserr_bad_xdr;
	}

	return nfs_ok;
}

static __be32
nfsd4_decode_openflag4(struct nfsd4_compoundargs *argp, struct nfsd4_open *open)
{
	__be32 status;

	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;open-&gt;op_create) &lt; 0)
		return nfserr_bad_xdr;
	switch (open-&gt;op_create) {
	case NFS4_OPEN_NOCREATE:
		break;
	case NFS4_OPEN_CREATE:
		status = nfsd4_decode_createhow4(argp, open);
		if (status)
			return status;
		break;
	default:
		return nfserr_bad_xdr;
	}

	return nfs_ok;
}

static __be32 nfsd4_decode_share_access(struct nfsd4_compoundargs *argp, u32 *share_access, u32 *deleg_want, u32 *deleg_when)
{
	u32 w;

	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;w) &lt; 0)
		return nfserr_bad_xdr;
	*share_access = w &amp; NFS4_SHARE_ACCESS_MASK;
	*deleg_want = w &amp; NFS4_SHARE_WANT_MASK;
	if (deleg_when)
		*deleg_when = w &amp; NFS4_SHARE_WHEN_MASK;

	switch (w &amp; NFS4_SHARE_ACCESS_MASK) {
	case NFS4_SHARE_ACCESS_READ:
	case NFS4_SHARE_ACCESS_WRITE:
	case NFS4_SHARE_ACCESS_BOTH:
		break;
	default:
		return nfserr_bad_xdr;
	}
	w &amp;= ~NFS4_SHARE_ACCESS_MASK;
	if (!w)
		return nfs_ok;
	if (!argp-&gt;minorversion)
		return nfserr_bad_xdr;
	switch (w &amp; NFS4_SHARE_WANT_MASK) {
	case NFS4_SHARE_WANT_NO_PREFERENCE:
	case NFS4_SHARE_WANT_READ_DELEG:
	case NFS4_SHARE_WANT_WRITE_DELEG:
	case NFS4_SHARE_WANT_ANY_DELEG:
	case NFS4_SHARE_WANT_NO_DELEG:
	case NFS4_SHARE_WANT_CANCEL:
		break;
	default:
		return nfserr_bad_xdr;
	}
	w &amp;= ~NFS4_SHARE_WANT_MASK;
	if (!w)
		return nfs_ok;

	if (!deleg_when)	/* open_downgrade */
		return nfserr_inval;
	switch (w) {
	case NFS4_SHARE_SIGNAL_DELEG_WHEN_RESRC_AVAIL:
	case NFS4_SHARE_PUSH_DELEG_WHEN_UNCONTENDED:
	case (NFS4_SHARE_SIGNAL_DELEG_WHEN_RESRC_AVAIL |
	      NFS4_SHARE_PUSH_DELEG_WHEN_UNCONTENDED):
		return nfs_ok;
	}
	return nfserr_bad_xdr;
}

static __be32 nfsd4_decode_share_deny(struct nfsd4_compoundargs *argp, u32 *x)
{
	if (xdr_stream_decode_u32(argp-&gt;xdr, x) &lt; 0)
		return nfserr_bad_xdr;
	/* Note: unlike access bits, deny bits may be zero. */
	if (*x &amp; ~NFS4_SHARE_DENY_BOTH)
		return nfserr_bad_xdr;

	return nfs_ok;
}

static __be32
nfsd4_decode_open_claim4(struct nfsd4_compoundargs *argp,
			 struct nfsd4_open *open)
{
	__be32 status;

	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;open-&gt;op_claim_type) &lt; 0)
		return nfserr_bad_xdr;
	switch (open-&gt;op_claim_type) {
	case NFS4_OPEN_CLAIM_NULL:
	case NFS4_OPEN_CLAIM_DELEGATE_PREV:
		status = nfsd4_decode_component4(argp, &amp;open-&gt;op_fname,
						 &amp;open-&gt;op_fnamelen);
		if (status)
			return status;
		break;
	case NFS4_OPEN_CLAIM_PREVIOUS:
		if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;open-&gt;op_delegate_type) &lt; 0)
			return nfserr_bad_xdr;
		break;
	case NFS4_OPEN_CLAIM_DELEGATE_CUR:
		status = nfsd4_decode_stateid4(argp, &amp;open-&gt;op_delegate_stateid);
		if (status)
			return status;
		status = nfsd4_decode_component4(argp, &amp;open-&gt;op_fname,
						 &amp;open-&gt;op_fnamelen);
		if (status)
			return status;
		break;
	case NFS4_OPEN_CLAIM_FH:
	case NFS4_OPEN_CLAIM_DELEG_PREV_FH:
		if (argp-&gt;minorversion &lt; 1)
			return nfserr_bad_xdr;
		/* void */
		break;
	case NFS4_OPEN_CLAIM_DELEG_CUR_FH:
		if (argp-&gt;minorversion &lt; 1)
			return nfserr_bad_xdr;
		status = nfsd4_decode_stateid4(argp, &amp;open-&gt;op_delegate_stateid);
		if (status)
			return status;
		break;
	default:
		return nfserr_bad_xdr;
	}

	return nfs_ok;
}

static __be32
nfsd4_decode_open(struct nfsd4_compoundargs *argp, struct nfsd4_open *open)
{
	__be32 status;
	u32 dummy;

	memset(open, 0, sizeof(*open));

	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;open-&gt;op_seqid) &lt; 0)
		return nfserr_bad_xdr;
	/* deleg_want is ignored */
	status = nfsd4_decode_share_access(argp, &amp;open-&gt;op_share_access,
					   &amp;open-&gt;op_deleg_want, &amp;dummy);
	if (status)
		return status;
	status = nfsd4_decode_share_deny(argp, &amp;open-&gt;op_share_deny);
	if (status)
		return status;
	status = nfsd4_decode_state_owner4(argp, &amp;open-&gt;op_clientid,
					   &amp;open-&gt;op_owner);
	if (status)
		return status;
	status = nfsd4_decode_openflag4(argp, open);
	if (status)
		return status;
	return nfsd4_decode_open_claim4(argp, open);
}

static __be32
nfsd4_decode_open_confirm(struct nfsd4_compoundargs *argp, struct nfsd4_open_confirm *open_conf)
{
	__be32 status;

	if (argp-&gt;minorversion &gt;= 1)
		return nfserr_notsupp;

	status = nfsd4_decode_stateid4(argp, &amp;open_conf-&gt;oc_req_stateid);
	if (status)
		return status;
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;open_conf-&gt;oc_seqid) &lt; 0)
		return nfserr_bad_xdr;

	memset(&amp;open_conf-&gt;oc_resp_stateid, 0,
	       sizeof(open_conf-&gt;oc_resp_stateid));
	return nfs_ok;
}

static __be32
nfsd4_decode_open_downgrade(struct nfsd4_compoundargs *argp, struct nfsd4_open_downgrade *open_down)
{
	__be32 status;

	memset(open_down, 0, sizeof(*open_down));
	status = nfsd4_decode_stateid4(argp, &amp;open_down-&gt;od_stateid);
	if (status)
		return status;
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;open_down-&gt;od_seqid) &lt; 0)
		return nfserr_bad_xdr;
	/* deleg_want is ignored */
	status = nfsd4_decode_share_access(argp, &amp;open_down-&gt;od_share_access,
					   &amp;open_down-&gt;od_deleg_want, NULL);
	if (status)
		return status;
	return nfsd4_decode_share_deny(argp, &amp;open_down-&gt;od_share_deny);
}

static __be32
nfsd4_decode_putfh(struct nfsd4_compoundargs *argp, struct nfsd4_putfh *putfh)
{
	__be32 *p;

	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;putfh-&gt;pf_fhlen) &lt; 0)
		return nfserr_bad_xdr;
	if (putfh-&gt;pf_fhlen &gt; NFS4_FHSIZE)
		return nfserr_bad_xdr;
	p = xdr_inline_decode(argp-&gt;xdr, putfh-&gt;pf_fhlen);
	if (!p)
		return nfserr_bad_xdr;
	putfh-&gt;pf_fhval = svcxdr_savemem(argp, p, putfh-&gt;pf_fhlen);
	if (!putfh-&gt;pf_fhval)
		return nfserr_jukebox;

	putfh-&gt;no_verify = false;
	return nfs_ok;
}

static __be32
nfsd4_decode_putpubfh(struct nfsd4_compoundargs *argp, void *p)
{
	if (argp-&gt;minorversion == 0)
		return nfs_ok;
	return nfserr_notsupp;
}

static __be32
nfsd4_decode_read(struct nfsd4_compoundargs *argp, struct nfsd4_read *read)
{
	__be32 status;

	memset(read, 0, sizeof(*read));
	status = nfsd4_decode_stateid4(argp, &amp;read-&gt;rd_stateid);
	if (status)
		return status;
	if (xdr_stream_decode_u64(argp-&gt;xdr, &amp;read-&gt;rd_offset) &lt; 0)
		return nfserr_bad_xdr;
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;read-&gt;rd_length) &lt; 0)
		return nfserr_bad_xdr;

	return nfs_ok;
}

static __be32
nfsd4_decode_readdir(struct nfsd4_compoundargs *argp, struct nfsd4_readdir *readdir)
{
	__be32 status;

	memset(readdir, 0, sizeof(*readdir));
	if (xdr_stream_decode_u64(argp-&gt;xdr, &amp;readdir-&gt;rd_cookie) &lt; 0)
		return nfserr_bad_xdr;
	status = nfsd4_decode_verifier4(argp, &amp;readdir-&gt;rd_verf);
	if (status)
		return status;
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;readdir-&gt;rd_dircount) &lt; 0)
		return nfserr_bad_xdr;
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;readdir-&gt;rd_maxcount) &lt; 0)
		return nfserr_bad_xdr;
	if (xdr_stream_decode_uint32_array(argp-&gt;xdr, readdir-&gt;rd_bmval,
					   ARRAY_SIZE(readdir-&gt;rd_bmval)) &lt; 0)
		return nfserr_bad_xdr;

	return nfs_ok;
}

static __be32
nfsd4_decode_remove(struct nfsd4_compoundargs *argp, struct nfsd4_remove *remove)
{
	memset(&amp;remove-&gt;rm_cinfo, 0, sizeof(remove-&gt;rm_cinfo));
	return nfsd4_decode_component4(argp, &amp;remove-&gt;rm_name, &amp;remove-&gt;rm_namelen);
}

static __be32
nfsd4_decode_rename(struct nfsd4_compoundargs *argp, struct nfsd4_rename *rename)
{
	__be32 status;

	memset(rename, 0, sizeof(*rename));
	status = nfsd4_decode_component4(argp, &amp;rename-&gt;rn_sname, &amp;rename-&gt;rn_snamelen);
	if (status)
		return status;
	return nfsd4_decode_component4(argp, &amp;rename-&gt;rn_tname, &amp;rename-&gt;rn_tnamelen);
}

static __be32
nfsd4_decode_renew(struct nfsd4_compoundargs *argp, clientid_t *clientid)
{
	return nfsd4_decode_clientid4(argp, clientid);
}

static __be32
nfsd4_decode_secinfo(struct nfsd4_compoundargs *argp,
		     struct nfsd4_secinfo *secinfo)
{
	secinfo-&gt;si_exp = NULL;
	return nfsd4_decode_component4(argp, &amp;secinfo-&gt;si_name, &amp;secinfo-&gt;si_namelen);
}

static __be32
nfsd4_decode_setattr(struct nfsd4_compoundargs *argp, struct nfsd4_setattr *setattr)
{
	__be32 status;

	memset(setattr, 0, sizeof(*setattr));
	status = nfsd4_decode_stateid4(argp, &amp;setattr-&gt;sa_stateid);
	if (status)
		return status;
	return nfsd4_decode_fattr4(argp, setattr-&gt;sa_bmval,
				   ARRAY_SIZE(setattr-&gt;sa_bmval),
				   &amp;setattr-&gt;sa_iattr, &amp;setattr-&gt;sa_acl,
				   &amp;setattr-&gt;sa_label, NULL);
}

static __be32
nfsd4_decode_setclientid(struct nfsd4_compoundargs *argp, struct nfsd4_setclientid *setclientid)
{
	__be32 *p, status;

	memset(setclientid, 0, sizeof(*setclientid));

	if (argp-&gt;minorversion &gt;= 1)
		return nfserr_notsupp;

	status = nfsd4_decode_verifier4(argp, &amp;setclientid-&gt;se_verf);
	if (status)
		return status;
	status = nfsd4_decode_opaque(argp, &amp;setclientid-&gt;se_name);
	if (status)
		return status;
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;setclientid-&gt;se_callback_prog) &lt; 0)
		return nfserr_bad_xdr;
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;setclientid-&gt;se_callback_netid_len) &lt; 0)
		return nfserr_bad_xdr;
	p = xdr_inline_decode(argp-&gt;xdr, setclientid-&gt;se_callback_netid_len);
	if (!p)
		return nfserr_bad_xdr;
	setclientid-&gt;se_callback_netid_val = svcxdr_savemem(argp, p,
						setclientid-&gt;se_callback_netid_len);
	if (!setclientid-&gt;se_callback_netid_val)
		return nfserr_jukebox;

	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;setclientid-&gt;se_callback_addr_len) &lt; 0)
		return nfserr_bad_xdr;
	p = xdr_inline_decode(argp-&gt;xdr, setclientid-&gt;se_callback_addr_len);
	if (!p)
		return nfserr_bad_xdr;
	setclientid-&gt;se_callback_addr_val = svcxdr_savemem(argp, p,
						setclientid-&gt;se_callback_addr_len);
	if (!setclientid-&gt;se_callback_addr_val)
		return nfserr_jukebox;
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;setclientid-&gt;se_callback_ident) &lt; 0)
		return nfserr_bad_xdr;

	return nfs_ok;
}

static __be32
nfsd4_decode_setclientid_confirm(struct nfsd4_compoundargs *argp, struct nfsd4_setclientid_confirm *scd_c)
{
	__be32 status;

	if (argp-&gt;minorversion &gt;= 1)
		return nfserr_notsupp;

	status = nfsd4_decode_clientid4(argp, &amp;scd_c-&gt;sc_clientid);
	if (status)
		return status;
	return nfsd4_decode_verifier4(argp, &amp;scd_c-&gt;sc_confirm);
}

/* Also used for NVERIFY */
static __be32
nfsd4_decode_verify(struct nfsd4_compoundargs *argp, struct nfsd4_verify *verify)
{
	__be32 *p, status;

	memset(verify, 0, sizeof(*verify));

	status = nfsd4_decode_bitmap4(argp, verify-&gt;ve_bmval,
				      ARRAY_SIZE(verify-&gt;ve_bmval));
	if (status)
		return status;

	/* For convenience's sake, we compare raw xdr'd attributes in
	 * nfsd4_proc_verify */

	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;verify-&gt;ve_attrlen) &lt; 0)
		return nfserr_bad_xdr;
	p = xdr_inline_decode(argp-&gt;xdr, verify-&gt;ve_attrlen);
	if (!p)
		return nfserr_bad_xdr;
	verify-&gt;ve_attrval = svcxdr_savemem(argp, p, verify-&gt;ve_attrlen);
	if (!verify-&gt;ve_attrval)
		return nfserr_jukebox;

	return nfs_ok;
}

static __be32
nfsd4_decode_write(struct nfsd4_compoundargs *argp, struct nfsd4_write *write)
{
	__be32 status;

	status = nfsd4_decode_stateid4(argp, &amp;write-&gt;wr_stateid);
	if (status)
		return status;
	if (xdr_stream_decode_u64(argp-&gt;xdr, &amp;write-&gt;wr_offset) &lt; 0)
		return nfserr_bad_xdr;
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;write-&gt;wr_stable_how) &lt; 0)
		return nfserr_bad_xdr;
	if (write-&gt;wr_stable_how &gt; NFS_FILE_SYNC)
		return nfserr_bad_xdr;
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;write-&gt;wr_buflen) &lt; 0)
		return nfserr_bad_xdr;
	if (!xdr_stream_subsegment(argp-&gt;xdr, &amp;write-&gt;wr_payload, write-&gt;wr_buflen))
		return nfserr_bad_xdr;

	write-&gt;wr_bytes_written = 0;
	write-&gt;wr_how_written = 0;
	memset(&amp;write-&gt;wr_verifier, 0, sizeof(write-&gt;wr_verifier));
	return nfs_ok;
}

static __be32
nfsd4_decode_release_lockowner(struct nfsd4_compoundargs *argp, struct nfsd4_release_lockowner *rlockowner)
{
	__be32 status;

	if (argp-&gt;minorversion &gt;= 1)
		return nfserr_notsupp;

	status = nfsd4_decode_state_owner4(argp, &amp;rlockowner-&gt;rl_clientid,
					   &amp;rlockowner-&gt;rl_owner);
	if (status)
		return status;

	if (argp-&gt;minorversion &amp;&amp; !zero_clientid(&amp;rlockowner-&gt;rl_clientid))
		return nfserr_inval;

	return nfs_ok;
}

static __be32 nfsd4_decode_backchannel_ctl(struct nfsd4_compoundargs *argp, struct nfsd4_backchannel_ctl *bc)
{
	memset(bc, 0, sizeof(*bc));
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;bc-&gt;bc_cb_program) &lt; 0)
		return nfserr_bad_xdr;
	return nfsd4_decode_cb_sec(argp, &amp;bc-&gt;bc_cb_sec);
}

static __be32 nfsd4_decode_bind_conn_to_session(struct nfsd4_compoundargs *argp, struct nfsd4_bind_conn_to_session *bcts)
{
	u32 use_conn_in_rdma_mode;
	__be32 status;

	memset(bcts, 0, sizeof(*bcts));
	status = nfsd4_decode_sessionid4(argp, &amp;bcts-&gt;sessionid);
	if (status)
		return status;
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;bcts-&gt;dir) &lt; 0)
		return nfserr_bad_xdr;
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;use_conn_in_rdma_mode) &lt; 0)
		return nfserr_bad_xdr;

	return nfs_ok;
}

static __be32
nfsd4_decode_state_protect_ops(struct nfsd4_compoundargs *argp,
			       struct nfsd4_exchange_id *exid)
{
	__be32 status;

	status = nfsd4_decode_bitmap4(argp, exid-&gt;spo_must_enforce,
				      ARRAY_SIZE(exid-&gt;spo_must_enforce));
	if (status)
		return nfserr_bad_xdr;
	status = nfsd4_decode_bitmap4(argp, exid-&gt;spo_must_allow,
				      ARRAY_SIZE(exid-&gt;spo_must_allow));
	if (status)
		return nfserr_bad_xdr;

	return nfs_ok;
}

/*
 * This implementation currently does not support SP4_SSV.
 * This decoder simply skips over these arguments.
 */
static noinline __be32
nfsd4_decode_ssv_sp_parms(struct nfsd4_compoundargs *argp,
			  struct nfsd4_exchange_id *exid)
{
	u32 count, window, num_gss_handles;
	__be32 status;

	/* ssp_ops */
	status = nfsd4_decode_state_protect_ops(argp, exid);
	if (status)
		return status;

	/* ssp_hash_algs&lt;&gt; */
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;count) &lt; 0)
		return nfserr_bad_xdr;
	while (count--) {
		status = nfsd4_decode_ignored_string(argp, 0);
		if (status)
			return status;
	}

	/* ssp_encr_algs&lt;&gt; */
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;count) &lt; 0)
		return nfserr_bad_xdr;
	while (count--) {
		status = nfsd4_decode_ignored_string(argp, 0);
		if (status)
			return status;
	}

	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;window) &lt; 0)
		return nfserr_bad_xdr;
	if (xdr_stream_decode_u32(argp-&gt;xdr, #_gss_handles) &lt; 0)
		return nfserr_bad_xdr;

	return nfs_ok;
}

static __be32
nfsd4_decode_state_protect4_a(struct nfsd4_compoundargs *argp,
			      struct nfsd4_exchange_id *exid)
{
	__be32 status;

	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;exid-&gt;spa_how) &lt; 0)
		return nfserr_bad_xdr;
	switch (exid-&gt;spa_how) {
	case SP4_NONE:
		break;
	case SP4_MACH_CRED:
		status = nfsd4_decode_state_protect_ops(argp, exid);
		if (status)
			return status;
		break;
	case SP4_SSV:
		status = nfsd4_decode_ssv_sp_parms(argp, exid);
		if (status)
			return status;
		break;
	default:
		return nfserr_bad_xdr;
	}

	return nfs_ok;
}

static __be32
nfsd4_decode_nfs_impl_id4(struct nfsd4_compoundargs *argp,
			  struct nfsd4_exchange_id *exid)
{
	__be32 status;
	u32 count;

	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;count) &lt; 0)
		return nfserr_bad_xdr;
	switch (count) {
	case 0:
		break;
	case 1:
		/* Note that RFC 8881 places no length limit on
		 * nii_domain, but this implementation permits no
		 * more than NFS4_OPAQUE_LIMIT bytes */
		status = nfsd4_decode_opaque(argp, &amp;exid-&gt;nii_domain);
		if (status)
			return status;
		/* Note that RFC 8881 places no length limit on
		 * nii_name, but this implementation permits no
		 * more than NFS4_OPAQUE_LIMIT bytes */
		status = nfsd4_decode_opaque(argp, &amp;exid-&gt;nii_name);
		if (status)
			return status;
		status = nfsd4_decode_nfstime4(argp, &amp;exid-&gt;nii_time);
		if (status)
			return status;
		break;
	default:
		return nfserr_bad_xdr;
	}

	return nfs_ok;
}

static __be32
nfsd4_decode_exchange_id(struct nfsd4_compoundargs *argp,
			 struct nfsd4_exchange_id *exid)
{
	__be32 status;

	memset(exid, 0, sizeof(*exid));
	status = nfsd4_decode_verifier4(argp, &amp;exid-&gt;verifier);
	if (status)
		return status;
	status = nfsd4_decode_opaque(argp, &amp;exid-&gt;clname);
	if (status)
		return status;
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;exid-&gt;flags) &lt; 0)
		return nfserr_bad_xdr;
	status = nfsd4_decode_state_protect4_a(argp, exid);
	if (status)
		return status;
	return nfsd4_decode_nfs_impl_id4(argp, exid);
}

static __be32
nfsd4_decode_channel_attrs4(struct nfsd4_compoundargs *argp,
			    struct nfsd4_channel_attrs *ca)
{
	__be32 *p;

	p = xdr_inline_decode(argp-&gt;xdr, XDR_UNIT * 7);
	if (!p)
		return nfserr_bad_xdr;

	/* headerpadsz is ignored */
	p++;
	ca-&gt;maxreq_sz = be32_to_cpup(p++);
	ca-&gt;maxresp_sz = be32_to_cpup(p++);
	ca-&gt;maxresp_cached = be32_to_cpup(p++);
	ca-&gt;maxops = be32_to_cpup(p++);
	ca-&gt;maxreqs = be32_to_cpup(p++);
	ca-&gt;nr_rdma_attrs = be32_to_cpup(p);
	switch (ca-&gt;nr_rdma_attrs) {
	case 0:
		break;
	case 1:
		if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;ca-&gt;rdma_attrs) &lt; 0)
			return nfserr_bad_xdr;
		break;
	default:
		return nfserr_bad_xdr;
	}

	return nfs_ok;
}

static __be32
nfsd4_decode_create_session(struct nfsd4_compoundargs *argp,
			    struct nfsd4_create_session *sess)
{
	__be32 status;

	memset(sess, 0, sizeof(*sess));
	status = nfsd4_decode_clientid4(argp, &amp;sess-&gt;clientid);
	if (status)
		return status;
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;sess-&gt;seqid) &lt; 0)
		return nfserr_bad_xdr;
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;sess-&gt;flags) &lt; 0)
		return nfserr_bad_xdr;
	status = nfsd4_decode_channel_attrs4(argp, &amp;sess-&gt;fore_channel);
	if (status)
		return status;
	status = nfsd4_decode_channel_attrs4(argp, &amp;sess-&gt;back_channel);
	if (status)
		return status;
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;sess-&gt;callback_prog) &lt; 0)
		return nfserr_bad_xdr;
	return nfsd4_decode_cb_sec(argp, &amp;sess-&gt;cb_sec);
}

static __be32
nfsd4_decode_destroy_session(struct nfsd4_compoundargs *argp,
			     struct nfsd4_destroy_session *destroy_session)
{
	return nfsd4_decode_sessionid4(argp, &amp;destroy_session-&gt;sessionid);
}

static __be32
nfsd4_decode_free_stateid(struct nfsd4_compoundargs *argp,
			  struct nfsd4_free_stateid *free_stateid)
{
	return nfsd4_decode_stateid4(argp, &amp;free_stateid-&gt;fr_stateid);
}

#ifdef CONFIG_NFSD_PNFS
static __be32
nfsd4_decode_getdeviceinfo(struct nfsd4_compoundargs *argp,
		struct nfsd4_getdeviceinfo *gdev)
{
	__be32 status;

	memset(gdev, 0, sizeof(*gdev));
	status = nfsd4_decode_deviceid4(argp, &amp;gdev-&gt;gd_devid);
	if (status)
		return status;
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;gdev-&gt;gd_layout_type) &lt; 0)
		return nfserr_bad_xdr;
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;gdev-&gt;gd_maxcount) &lt; 0)
		return nfserr_bad_xdr;
	if (xdr_stream_decode_uint32_array(argp-&gt;xdr,
					   &amp;gdev-&gt;gd_notify_types, 1) &lt; 0)
		return nfserr_bad_xdr;

	return nfs_ok;
}

static __be32
nfsd4_decode_layoutcommit(struct nfsd4_compoundargs *argp,
			  struct nfsd4_layoutcommit *lcp)
{
	__be32 *p, status;

	memset(lcp, 0, sizeof(*lcp));
	if (xdr_stream_decode_u64(argp-&gt;xdr, &amp;lcp-&gt;lc_seg.offset) &lt; 0)
		return nfserr_bad_xdr;
	if (xdr_stream_decode_u64(argp-&gt;xdr, &amp;lcp-&gt;lc_seg.length) &lt; 0)
		return nfserr_bad_xdr;
	if (xdr_stream_decode_bool(argp-&gt;xdr, &amp;lcp-&gt;lc_reclaim) &lt; 0)
		return nfserr_bad_xdr;
	status = nfsd4_decode_stateid4(argp, &amp;lcp-&gt;lc_sid);
	if (status)
		return status;
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;lcp-&gt;lc_newoffset) &lt; 0)
		return nfserr_bad_xdr;
	if (lcp-&gt;lc_newoffset) {
		if (xdr_stream_decode_u64(argp-&gt;xdr, &amp;lcp-&gt;lc_last_wr) &lt; 0)
			return nfserr_bad_xdr;
	} else
		lcp-&gt;lc_last_wr = 0;
	p = xdr_inline_decode(argp-&gt;xdr, XDR_UNIT);
	if (!p)
		return nfserr_bad_xdr;
	if (xdr_item_is_present(p)) {
		status = nfsd4_decode_nfstime4(argp, &amp;lcp-&gt;lc_mtime);
		if (status)
			return status;
	} else {
		lcp-&gt;lc_mtime.tv_nsec = UTIME_NOW;
	}
	return nfsd4_decode_layoutupdate4(argp, lcp);
}

static __be32
nfsd4_decode_layoutget(struct nfsd4_compoundargs *argp,
		struct nfsd4_layoutget *lgp)
{
	__be32 status;

	memset(lgp, 0, sizeof(*lgp));
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;lgp-&gt;lg_signal) &lt; 0)
		return nfserr_bad_xdr;
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;lgp-&gt;lg_layout_type) &lt; 0)
		return nfserr_bad_xdr;
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;lgp-&gt;lg_seg.iomode) &lt; 0)
		return nfserr_bad_xdr;
	if (xdr_stream_decode_u64(argp-&gt;xdr, &amp;lgp-&gt;lg_seg.offset) &lt; 0)
		return nfserr_bad_xdr;
	if (xdr_stream_decode_u64(argp-&gt;xdr, &amp;lgp-&gt;lg_seg.length) &lt; 0)
		return nfserr_bad_xdr;
	if (xdr_stream_decode_u64(argp-&gt;xdr, &amp;lgp-&gt;lg_minlength) &lt; 0)
		return nfserr_bad_xdr;
	status = nfsd4_decode_stateid4(argp, &amp;lgp-&gt;lg_sid);
	if (status)
		return status;
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;lgp-&gt;lg_maxcount) &lt; 0)
		return nfserr_bad_xdr;

	return nfs_ok;
}

static __be32
nfsd4_decode_layoutreturn(struct nfsd4_compoundargs *argp,
		struct nfsd4_layoutreturn *lrp)
{
	memset(lrp, 0, sizeof(*lrp));
	if (xdr_stream_decode_bool(argp-&gt;xdr, &amp;lrp-&gt;lr_reclaim) &lt; 0)
		return nfserr_bad_xdr;
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;lrp-&gt;lr_layout_type) &lt; 0)
		return nfserr_bad_xdr;
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;lrp-&gt;lr_seg.iomode) &lt; 0)
		return nfserr_bad_xdr;
	return nfsd4_decode_layoutreturn4(argp, lrp);
}
#endif /* CONFIG_NFSD_PNFS */

static __be32 nfsd4_decode_secinfo_no_name(struct nfsd4_compoundargs *argp,
					   struct nfsd4_secinfo_no_name *sin)
{
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;sin-&gt;sin_style) &lt; 0)
		return nfserr_bad_xdr;

	sin-&gt;sin_exp = NULL;
	return nfs_ok;
}

static __be32
nfsd4_decode_sequence(struct nfsd4_compoundargs *argp,
		      struct nfsd4_sequence *seq)
{
	__be32 *p, status;

	status = nfsd4_decode_sessionid4(argp, &amp;seq-&gt;sessionid);
	if (status)
		return status;
	p = xdr_inline_decode(argp-&gt;xdr, XDR_UNIT * 4);
	if (!p)
		return nfserr_bad_xdr;
	seq-&gt;seqid = be32_to_cpup(p++);
	seq-&gt;slotid = be32_to_cpup(p++);
	seq-&gt;maxslots = be32_to_cpup(p++);
	seq-&gt;cachethis = be32_to_cpup(p);

	seq-&gt;status_flags = 0;
	return nfs_ok;
}

static __be32
nfsd4_decode_test_stateid(struct nfsd4_compoundargs *argp, struct nfsd4_test_stateid *test_stateid)
{
	struct nfsd4_test_stateid_id *stateid;
	__be32 status;
	u32 i;

	memset(test_stateid, 0, sizeof(*test_stateid));
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;test_stateid-&gt;ts_num_ids) &lt; 0)
		return nfserr_bad_xdr;

	INIT_LIST_HEAD(&amp;test_stateid-&gt;ts_stateid_list);
	for (i = 0; i &lt; test_stateid-&gt;ts_num_ids; i++) {
		stateid = svcxdr_tmpalloc(argp, sizeof(*stateid));
		if (!stateid)
			return nfserr_jukebox;
		INIT_LIST_HEAD(&amp;stateid-&gt;ts_id_list);
		list_add_tail(&amp;stateid-&gt;ts_id_list, &amp;test_stateid-&gt;ts_stateid_list);
		status = nfsd4_decode_stateid4(argp, &amp;stateid-&gt;ts_id_stateid);
		if (status)
			return status;
	}

	return nfs_ok;
}

static __be32 nfsd4_decode_destroy_clientid(struct nfsd4_compoundargs *argp,
					    struct nfsd4_destroy_clientid *dc)
{
	return nfsd4_decode_clientid4(argp, &amp;dc-&gt;clientid);
}

static __be32 nfsd4_decode_reclaim_complete(struct nfsd4_compoundargs *argp,
					    struct nfsd4_reclaim_complete *rc)
{
	if (xdr_stream_decode_bool(argp-&gt;xdr, &amp;rc-&gt;rca_one_fs) &lt; 0)
		return nfserr_bad_xdr;
	return nfs_ok;
}

static __be32
nfsd4_decode_fallocate(struct nfsd4_compoundargs *argp,
		       struct nfsd4_fallocate *fallocate)
{
	__be32 status;

	status = nfsd4_decode_stateid4(argp, &amp;fallocate-&gt;falloc_stateid);
	if (status)
		return status;
	if (xdr_stream_decode_u64(argp-&gt;xdr, &amp;fallocate-&gt;falloc_offset) &lt; 0)
		return nfserr_bad_xdr;
	if (xdr_stream_decode_u64(argp-&gt;xdr, &amp;fallocate-&gt;falloc_length) &lt; 0)
		return nfserr_bad_xdr;

	return nfs_ok;
}

static __be32 nfsd4_decode_nl4_server(struct nfsd4_compoundargs *argp,
				      struct nl4_server *ns)
{
	struct nfs42_netaddr *naddr;
	__be32 *p;

	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;ns-&gt;nl4_type) &lt; 0)
		return nfserr_bad_xdr;

	/* currently support for 1 inter-server source server */
	switch (ns-&gt;nl4_type) {
	case NL4_NETADDR:
		naddr = &amp;ns-&gt;u.nl4_addr;

		if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;naddr-&gt;netid_len) &lt; 0)
			return nfserr_bad_xdr;
		if (naddr-&gt;netid_len &gt; RPCBIND_MAXNETIDLEN)
			return nfserr_bad_xdr;

		p = xdr_inline_decode(argp-&gt;xdr, naddr-&gt;netid_len);
		if (!p)
			return nfserr_bad_xdr;
		memcpy(naddr-&gt;netid, p, naddr-&gt;netid_len);

		if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;naddr-&gt;addr_len) &lt; 0)
			return nfserr_bad_xdr;
		if (naddr-&gt;addr_len &gt; RPCBIND_MAXUADDRLEN)
			return nfserr_bad_xdr;

		p = xdr_inline_decode(argp-&gt;xdr, naddr-&gt;addr_len);
		if (!p)
			return nfserr_bad_xdr;
		memcpy(naddr-&gt;addr, p, naddr-&gt;addr_len);
		break;
	default:
		return nfserr_bad_xdr;
	}

	return nfs_ok;
}

static __be32
nfsd4_decode_copy(struct nfsd4_compoundargs *argp, struct nfsd4_copy *copy)
{
	u32 consecutive, i, count, sync;
	struct nl4_server *ns_dummy;
	__be32 status;

	memset(copy, 0, sizeof(*copy));
	status = nfsd4_decode_stateid4(argp, &amp;copy-&gt;cp_src_stateid);
	if (status)
		return status;
	status = nfsd4_decode_stateid4(argp, &amp;copy-&gt;cp_dst_stateid);
	if (status)
		return status;
	if (xdr_stream_decode_u64(argp-&gt;xdr, &amp;copy-&gt;cp_src_pos) &lt; 0)
		return nfserr_bad_xdr;
	if (xdr_stream_decode_u64(argp-&gt;xdr, &amp;copy-&gt;cp_dst_pos) &lt; 0)
		return nfserr_bad_xdr;
	if (xdr_stream_decode_u64(argp-&gt;xdr, &amp;copy-&gt;cp_count) &lt; 0)
		return nfserr_bad_xdr;
	/* ca_consecutive: we always do consecutive copies */
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;consecutive) &lt; 0)
		return nfserr_bad_xdr;
	if (xdr_stream_decode_bool(argp-&gt;xdr, &amp;sync) &lt; 0)
		return nfserr_bad_xdr;
	nfsd4_copy_set_sync(copy, sync);

	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;count) &lt; 0)
		return nfserr_bad_xdr;
	copy-&gt;cp_src = svcxdr_tmpalloc(argp, sizeof(*copy-&gt;cp_src));
	if (copy-&gt;cp_src == NULL)
		return nfserr_jukebox;
	if (count == 0) { /* intra-server copy */
		__set_bit(NFSD4_COPY_F_INTRA, &amp;copy-&gt;cp_flags);
		return nfs_ok;
	}

	/* decode all the supplied server addresses but use only the first */
	status = nfsd4_decode_nl4_server(argp, copy-&gt;cp_src);
	if (status)
		return status;

	ns_dummy = kmalloc(sizeof(struct nl4_server), GFP_KERNEL);
	if (ns_dummy == NULL)
		return nfserr_jukebox;
	for (i = 0; i &lt; count - 1; i++) {
		status = nfsd4_decode_nl4_server(argp, ns_dummy);
		if (status) {
			kfree(ns_dummy);
			return status;
		}
	}
	kfree(ns_dummy);

	return nfs_ok;
}

static __be32
nfsd4_decode_copy_notify(struct nfsd4_compoundargs *argp,
			 struct nfsd4_copy_notify *cn)
{
	__be32 status;

	memset(cn, 0, sizeof(*cn));
	cn-&gt;cpn_src = svcxdr_tmpalloc(argp, sizeof(*cn-&gt;cpn_src));
	if (cn-&gt;cpn_src == NULL)
		return nfserr_jukebox;
	cn-&gt;cpn_dst = svcxdr_tmpalloc(argp, sizeof(*cn-&gt;cpn_dst));
	if (cn-&gt;cpn_dst == NULL)
		return nfserr_jukebox;

	status = nfsd4_decode_stateid4(argp, &amp;cn-&gt;cpn_src_stateid);
	if (status)
		return status;
	return nfsd4_decode_nl4_server(argp, cn-&gt;cpn_dst);
}

static __be32
nfsd4_decode_offload_status(struct nfsd4_compoundargs *argp,
			    struct nfsd4_offload_status *os)
{
	os-&gt;count = 0;
	os-&gt;status = 0;
	return nfsd4_decode_stateid4(argp, &amp;os-&gt;stateid);
}

static __be32
nfsd4_decode_seek(struct nfsd4_compoundargs *argp, struct nfsd4_seek *seek)
{
	__be32 status;

	status = nfsd4_decode_stateid4(argp, &amp;seek-&gt;seek_stateid);
	if (status)
		return status;
	if (xdr_stream_decode_u64(argp-&gt;xdr, &amp;seek-&gt;seek_offset) &lt; 0)
		return nfserr_bad_xdr;
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;seek-&gt;seek_whence) &lt; 0)
		return nfserr_bad_xdr;

	seek-&gt;seek_eof = 0;
	seek-&gt;seek_pos = 0;
	return nfs_ok;
}

static __be32
nfsd4_decode_clone(struct nfsd4_compoundargs *argp, struct nfsd4_clone *clone)
{
	__be32 status;

	status = nfsd4_decode_stateid4(argp, &amp;clone-&gt;cl_src_stateid);
	if (status)
		return status;
	status = nfsd4_decode_stateid4(argp, &amp;clone-&gt;cl_dst_stateid);
	if (status)
		return status;
	if (xdr_stream_decode_u64(argp-&gt;xdr, &amp;clone-&gt;cl_src_pos) &lt; 0)
		return nfserr_bad_xdr;
	if (xdr_stream_decode_u64(argp-&gt;xdr, &amp;clone-&gt;cl_dst_pos) &lt; 0)
		return nfserr_bad_xdr;
	if (xdr_stream_decode_u64(argp-&gt;xdr, &amp;clone-&gt;cl_count) &lt; 0)
		return nfserr_bad_xdr;

	return nfs_ok;
}

/*
 * XDR data that is more than PAGE_SIZE in size is normally part of a
 * read or write. However, the size of extended attributes is limited
 * by the maximum request size, and then further limited by the underlying
 * filesystem limits. This can exceed PAGE_SIZE (currently, XATTR_SIZE_MAX
 * is 64k). Since there is no kvec- or page-based interface to xattrs,
 * and we're not dealing with contiguous pages, we need to do some copying.
 */

/*
 * Decode data into buffer.
 */
static __be32
nfsd4_vbuf_from_vector(struct nfsd4_compoundargs *argp, struct xdr_buf *xdr,
		       char **bufp, u32 buflen)
{
	struct page **pages = xdr-&gt;pages;
	struct kvec *head = xdr-&gt;head;
	char *tmp, *dp;
	u32 len;

	if (buflen &lt;= head-&gt;iov_len) {
		/*
		 * We're in luck, the head has enough space. Just return
		 * the head, no need for copying.
		 */
		*bufp = head-&gt;iov_base;
		return 0;
	}

	tmp = svcxdr_tmpalloc(argp, buflen);
	if (tmp == NULL)
		return nfserr_jukebox;

	dp = tmp;
	memcpy(dp, head-&gt;iov_base, head-&gt;iov_len);
	buflen -= head-&gt;iov_len;
	dp += head-&gt;iov_len;

	while (buflen &gt; 0) {
		len = min_t(u32, buflen, PAGE_SIZE);
		memcpy(dp, page_address(*pages), len);

		buflen -= len;
		dp += len;
		pages++;
	}

	*bufp = tmp;
	return 0;
}

/*
 * Get a user extended attribute name from the XDR buffer.
 * It will not have the "user." prefix, so prepend it.
 * Lastly, check for nul characters in the name.
 */
static __be32
nfsd4_decode_xattr_name(struct nfsd4_compoundargs *argp, char **namep)
{
	char *name, *sp, *dp;
	u32 namelen, cnt;
	__be32 *p;

	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;namelen) &lt; 0)
		return nfserr_bad_xdr;
	if (namelen &gt; (XATTR_NAME_MAX - XATTR_USER_PREFIX_LEN))
		return nfserr_nametoolong;
	if (namelen == 0)
		return nfserr_bad_xdr;
	p = xdr_inline_decode(argp-&gt;xdr, namelen);
	if (!p)
		return nfserr_bad_xdr;
	name = svcxdr_tmpalloc(argp, namelen + XATTR_USER_PREFIX_LEN + 1);
	if (!name)
		return nfserr_jukebox;
	memcpy(name, XATTR_USER_PREFIX, XATTR_USER_PREFIX_LEN);

	/*
	 * Copy the extended attribute name over while checking for 0
	 * characters.
	 */
	sp = (char *)p;
	dp = name + XATTR_USER_PREFIX_LEN;
	cnt = namelen;

	while (cnt-- &gt; 0) {
		if (*sp == '\0')
			return nfserr_bad_xdr;
		*dp++ = *sp++;
	}
	*dp = '\0';

	*namep = name;

	return nfs_ok;
}

/*
 * A GETXATTR op request comes without a length specifier. We just set the
 * maximum length for the reply based on XATTR_SIZE_MAX and the maximum
 * channel reply size. nfsd_getxattr will probe the length of the xattr,
 * check it against getxa_len, and allocate + return the value.
 */
static __be32
nfsd4_decode_getxattr(struct nfsd4_compoundargs *argp,
		      struct nfsd4_getxattr *getxattr)
{
	__be32 status;
	u32 maxcount;

	memset(getxattr, 0, sizeof(*getxattr));
	status = nfsd4_decode_xattr_name(argp, &amp;getxattr-&gt;getxa_name);
	if (status)
		return status;

	maxcount = svc_max_payload(argp-&gt;rqstp);
	maxcount = min_t(u32, XATTR_SIZE_MAX, maxcount);

	getxattr-&gt;getxa_len = maxcount;
	return nfs_ok;
}

static __be32
nfsd4_decode_setxattr(struct nfsd4_compoundargs *argp,
		      struct nfsd4_setxattr *setxattr)
{
	u32 flags, maxcount, size;
	__be32 status;

	memset(setxattr, 0, sizeof(*setxattr));

	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;flags) &lt; 0)
		return nfserr_bad_xdr;

	if (flags &gt; SETXATTR4_REPLACE)
		return nfserr_inval;
	setxattr-&gt;setxa_flags = flags;

	status = nfsd4_decode_xattr_name(argp, &amp;setxattr-&gt;setxa_name);
	if (status)
		return status;

	maxcount = svc_max_payload(argp-&gt;rqstp);
	maxcount = min_t(u32, XATTR_SIZE_MAX, maxcount);

	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;size) &lt; 0)
		return nfserr_bad_xdr;
	if (size &gt; maxcount)
		return nfserr_xattr2big;

	setxattr-&gt;setxa_len = size;
	if (size &gt; 0) {
		struct xdr_buf payload;

		if (!xdr_stream_subsegment(argp-&gt;xdr, &amp;payload, size))
			return nfserr_bad_xdr;
		status = nfsd4_vbuf_from_vector(argp, &amp;payload,
						&amp;setxattr-&gt;setxa_buf, size);
	}

	return nfs_ok;
}

static __be32
nfsd4_decode_listxattrs(struct nfsd4_compoundargs *argp,
			struct nfsd4_listxattrs *listxattrs)
{
	u32 maxcount;

	memset(listxattrs, 0, sizeof(*listxattrs));

	if (xdr_stream_decode_u64(argp-&gt;xdr, &amp;listxattrs-&gt;lsxa_cookie) &lt; 0)
		return nfserr_bad_xdr;

	/*
	 * If the cookie  is too large to have even one user.x attribute
	 * plus trailing '\0' left in a maximum size buffer, it's invalid.
	 */
	if (listxattrs-&gt;lsxa_cookie &gt;=
	    (XATTR_LIST_MAX / (XATTR_USER_PREFIX_LEN + 2)))
		return nfserr_badcookie;

	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;maxcount) &lt; 0)
		return nfserr_bad_xdr;
	if (maxcount &lt; 8)
		/* Always need at least 2 words (length and one character) */
		return nfserr_inval;

	maxcount = min(maxcount, svc_max_payload(argp-&gt;rqstp));
	listxattrs-&gt;lsxa_maxcount = maxcount;

	return nfs_ok;
}

static __be32
nfsd4_decode_removexattr(struct nfsd4_compoundargs *argp,
			 struct nfsd4_removexattr *removexattr)
{
	memset(removexattr, 0, sizeof(*removexattr));
	return nfsd4_decode_xattr_name(argp, &amp;removexattr-&gt;rmxa_name);
}

static __be32
nfsd4_decode_noop(struct nfsd4_compoundargs *argp, void *p)
{
	return nfs_ok;
}

static __be32
nfsd4_decode_notsupp(struct nfsd4_compoundargs *argp, void *p)
{
	return nfserr_notsupp;
}

typedef __be32(*nfsd4_dec)(struct nfsd4_compoundargs *argp, void *);

static const nfsd4_dec nfsd4_dec_ops[] = {
	[OP_ACCESS]		= (nfsd4_dec)nfsd4_decode_access,
	[OP_CLOSE]		= (nfsd4_dec)nfsd4_decode_close,
	[OP_COMMIT]		= (nfsd4_dec)nfsd4_decode_commit,
	[OP_CREATE]		= (nfsd4_dec)nfsd4_decode_create,
	[OP_DELEGPURGE]		= (nfsd4_dec)nfsd4_decode_notsupp,
	[OP_DELEGRETURN]	= (nfsd4_dec)nfsd4_decode_delegreturn,
	[OP_GETATTR]		= (nfsd4_dec)nfsd4_decode_getattr,
	[OP_GETFH]		= (nfsd4_dec)nfsd4_decode_noop,
	[OP_LINK]		= (nfsd4_dec)nfsd4_decode_link,
	[OP_LOCK]		= (nfsd4_dec)nfsd4_decode_lock,
	[OP_LOCKT]		= (nfsd4_dec)nfsd4_decode_lockt,
	[OP_LOCKU]		= (nfsd4_dec)nfsd4_decode_locku,
	[OP_LOOKUP]		= (nfsd4_dec)nfsd4_decode_lookup,
	[OP_LOOKUPP]		= (nfsd4_dec)nfsd4_decode_noop,
	[OP_NVERIFY]		= (nfsd4_dec)nfsd4_decode_verify,
	[OP_OPEN]		= (nfsd4_dec)nfsd4_decode_open,
	[OP_OPENATTR]		= (nfsd4_dec)nfsd4_decode_notsupp,
	[OP_OPEN_CONFIRM]	= (nfsd4_dec)nfsd4_decode_open_confirm,
	[OP_OPEN_DOWNGRADE]	= (nfsd4_dec)nfsd4_decode_open_downgrade,
	[OP_PUTFH]		= (nfsd4_dec)nfsd4_decode_putfh,
	[OP_PUTPUBFH]		= (nfsd4_dec)nfsd4_decode_putpubfh,
	[OP_PUTROOTFH]		= (nfsd4_dec)nfsd4_decode_noop,
	[OP_READ]		= (nfsd4_dec)nfsd4_decode_read,
	[OP_READDIR]		= (nfsd4_dec)nfsd4_decode_readdir,
	[OP_READLINK]		= (nfsd4_dec)nfsd4_decode_noop,
	[OP_REMOVE]		= (nfsd4_dec)nfsd4_decode_remove,
	[OP_RENAME]		= (nfsd4_dec)nfsd4_decode_rename,
	[OP_RENEW]		= (nfsd4_dec)nfsd4_decode_renew,
	[OP_RESTOREFH]		= (nfsd4_dec)nfsd4_decode_noop,
	[OP_SAVEFH]		= (nfsd4_dec)nfsd4_decode_noop,
	[OP_SECINFO]		= (nfsd4_dec)nfsd4_decode_secinfo,
	[OP_SETATTR]		= (nfsd4_dec)nfsd4_decode_setattr,
	[OP_SETCLIENTID]	= (nfsd4_dec)nfsd4_decode_setclientid,
	[OP_SETCLIENTID_CONFIRM] = (nfsd4_dec)nfsd4_decode_setclientid_confirm,
	[OP_VERIFY]		= (nfsd4_dec)nfsd4_decode_verify,
	[OP_WRITE]		= (nfsd4_dec)nfsd4_decode_write,
	[OP_RELEASE_LOCKOWNER]	= (nfsd4_dec)nfsd4_decode_release_lockowner,

	/* new operations for NFSv4.1 */
	[OP_BACKCHANNEL_CTL]	= (nfsd4_dec)nfsd4_decode_backchannel_ctl,
	[OP_BIND_CONN_TO_SESSION]= (nfsd4_dec)nfsd4_decode_bind_conn_to_session,
	[OP_EXCHANGE_ID]	= (nfsd4_dec)nfsd4_decode_exchange_id,
	[OP_CREATE_SESSION]	= (nfsd4_dec)nfsd4_decode_create_session,
	[OP_DESTROY_SESSION]	= (nfsd4_dec)nfsd4_decode_destroy_session,
	[OP_FREE_STATEID]	= (nfsd4_dec)nfsd4_decode_free_stateid,
	[OP_GET_DIR_DELEGATION]	= (nfsd4_dec)nfsd4_decode_notsupp,
#ifdef CONFIG_NFSD_PNFS
	[OP_GETDEVICEINFO]	= (nfsd4_dec)nfsd4_decode_getdeviceinfo,
	[OP_GETDEVICELIST]	= (nfsd4_dec)nfsd4_decode_notsupp,
	[OP_LAYOUTCOMMIT]	= (nfsd4_dec)nfsd4_decode_layoutcommit,
	[OP_LAYOUTGET]		= (nfsd4_dec)nfsd4_decode_layoutget,
	[OP_LAYOUTRETURN]	= (nfsd4_dec)nfsd4_decode_layoutreturn,
#else
	[OP_GETDEVICEINFO]	= (nfsd4_dec)nfsd4_decode_notsupp,
	[OP_GETDEVICELIST]	= (nfsd4_dec)nfsd4_decode_notsupp,
	[OP_LAYOUTCOMMIT]	= (nfsd4_dec)nfsd4_decode_notsupp,
	[OP_LAYOUTGET]		= (nfsd4_dec)nfsd4_decode_notsupp,
	[OP_LAYOUTRETURN]	= (nfsd4_dec)nfsd4_decode_notsupp,
#endif
	[OP_SECINFO_NO_NAME]	= (nfsd4_dec)nfsd4_decode_secinfo_no_name,
	[OP_SEQUENCE]		= (nfsd4_dec)nfsd4_decode_sequence,
	[OP_SET_SSV]		= (nfsd4_dec)nfsd4_decode_notsupp,
	[OP_TEST_STATEID]	= (nfsd4_dec)nfsd4_decode_test_stateid,
	[OP_WANT_DELEGATION]	= (nfsd4_dec)nfsd4_decode_notsupp,
	[OP_DESTROY_CLIENTID]	= (nfsd4_dec)nfsd4_decode_destroy_clientid,
	[OP_RECLAIM_COMPLETE]	= (nfsd4_dec)nfsd4_decode_reclaim_complete,

	/* new operations for NFSv4.2 */
	[OP_ALLOCATE]		= (nfsd4_dec)nfsd4_decode_fallocate,
	[OP_COPY]		= (nfsd4_dec)nfsd4_decode_copy,
	[OP_COPY_NOTIFY]	= (nfsd4_dec)nfsd4_decode_copy_notify,
	[OP_DEALLOCATE]		= (nfsd4_dec)nfsd4_decode_fallocate,
	[OP_IO_ADVISE]		= (nfsd4_dec)nfsd4_decode_notsupp,
	[OP_LAYOUTERROR]	= (nfsd4_dec)nfsd4_decode_notsupp,
	[OP_LAYOUTSTATS]	= (nfsd4_dec)nfsd4_decode_notsupp,
	[OP_OFFLOAD_CANCEL]	= (nfsd4_dec)nfsd4_decode_offload_status,
	[OP_OFFLOAD_STATUS]	= (nfsd4_dec)nfsd4_decode_offload_status,
	[OP_READ_PLUS]		= (nfsd4_dec)nfsd4_decode_read,
	[OP_SEEK]		= (nfsd4_dec)nfsd4_decode_seek,
	[OP_WRITE_SAME]		= (nfsd4_dec)nfsd4_decode_notsupp,
	[OP_CLONE]		= (nfsd4_dec)nfsd4_decode_clone,
	/* RFC 8276 extended atributes operations */
	[OP_GETXATTR]		= (nfsd4_dec)nfsd4_decode_getxattr,
	[OP_SETXATTR]		= (nfsd4_dec)nfsd4_decode_setxattr,
	[OP_LISTXATTRS]		= (nfsd4_dec)nfsd4_decode_listxattrs,
	[OP_REMOVEXATTR]	= (nfsd4_dec)nfsd4_decode_removexattr,
};

static inline bool
nfsd4_opnum_in_range(struct nfsd4_compoundargs *argp, struct nfsd4_op *op)
{
	if (op-&gt;opnum &lt; FIRST_NFS4_OP)
		return false;
	else if (argp-&gt;minorversion == 0 &amp;&amp; op-&gt;opnum &gt; LAST_NFS40_OP)
		return false;
	else if (argp-&gt;minorversion == 1 &amp;&amp; op-&gt;opnum &gt; LAST_NFS41_OP)
		return false;
	else if (argp-&gt;minorversion == 2 &amp;&amp; op-&gt;opnum &gt; LAST_NFS42_OP)
		return false;
	return true;
}

static bool
nfsd4_decode_compound(struct nfsd4_compoundargs *argp)
{
	struct nfsd4_op *op;
	bool cachethis = false;
	int auth_slack= argp-&gt;rqstp-&gt;rq_auth_slack;
	int max_reply = auth_slack + 8; /* opcnt, status */
	int readcount = 0;
	int readbytes = 0;
	__be32 *p;
	int i;

	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;argp-&gt;taglen) &lt; 0)
		return false;
	max_reply += XDR_UNIT;
	argp-&gt;tag = NULL;
	if (unlikely(argp-&gt;taglen)) {
		if (argp-&gt;taglen &gt; NFSD4_MAX_TAGLEN)
			return false;
		p = xdr_inline_decode(argp-&gt;xdr, argp-&gt;taglen);
		if (!p)
			return false;
		argp-&gt;tag = svcxdr_savemem(argp, p, argp-&gt;taglen);
		if (!argp-&gt;tag)
			return false;
		max_reply += xdr_align_size(argp-&gt;taglen);
	}

	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;argp-&gt;minorversion) &lt; 0)
		return false;
	if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;argp-&gt;client_opcnt) &lt; 0)
		return false;
	argp-&gt;opcnt = min_t(u32, argp-&gt;client_opcnt,
			    NFSD_MAX_OPS_PER_COMPOUND);

	if (argp-&gt;opcnt &gt; ARRAY_SIZE(argp-&gt;iops)) {
		argp-&gt;ops = vcalloc(argp-&gt;opcnt, sizeof(*argp-&gt;ops));
		if (!argp-&gt;ops) {
			argp-&gt;ops = argp-&gt;iops;
			return false;
		}
	}

	if (argp-&gt;minorversion &gt; NFSD_SUPPORTED_MINOR_VERSION)
		argp-&gt;opcnt = 0;

	for (i = 0; i &lt; argp-&gt;opcnt; i++) {
		op = &amp;argp-&gt;ops[i];
		op-&gt;replay = NULL;

		if (xdr_stream_decode_u32(argp-&gt;xdr, &amp;op-&gt;opnum) &lt; 0)
			return false;
		if (nfsd4_opnum_in_range(argp, op)) {
			op-&gt;status = nfsd4_dec_ops[op-&gt;opnum](argp, &amp;op-&gt;u);
			if (op-&gt;status != nfs_ok)
				trace_nfsd_compound_decode_err(argp-&gt;rqstp,
							       argp-&gt;opcnt, i,
							       op-&gt;opnum,
							       op-&gt;status);
		} else {
			op-&gt;opnum = OP_ILLEGAL;
			op-&gt;status = nfserr_op_illegal;
		}
		op-&gt;opdesc = OPDESC(op);
		/*
		 * We'll try to cache the result in the DRC if any one
		 * op in the compound wants to be cached:
		 */
		cachethis |= nfsd4_cache_this_op(op);

		if (op-&gt;opnum == OP_READ || op-&gt;opnum == OP_READ_PLUS) {
			readcount++;
			readbytes += nfsd4_max_reply(argp-&gt;rqstp, op);
		} else
			max_reply += nfsd4_max_reply(argp-&gt;rqstp, op);
		/*
		 * OP_LOCK and OP_LOCKT may return a conflicting lock.
		 * (Special case because it will just skip encoding this
		 * if it runs out of xdr buffer space, and it is the only
		 * operation that behaves this way.)
		 */
		if (op-&gt;opnum == OP_LOCK || op-&gt;opnum == OP_LOCKT)
			max_reply += NFS4_OPAQUE_LIMIT;

		if (op-&gt;status) {
			argp-&gt;opcnt = i+1;
			break;
		}
	}
	/* Sessions make the DRC unnecessary: */
	if (argp-&gt;minorversion)
		cachethis = false;
	svc_reserve(argp-&gt;rqstp, max_reply + readbytes);
	argp-&gt;rqstp-&gt;rq_cachetype = cachethis ? RC_REPLBUFF : RC_NOCACHE;

	if (readcount &gt; 1 || max_reply &gt; PAGE_SIZE - auth_slack)
		__clear_bit(RQ_SPLICE_OK, &amp;argp-&gt;rqstp-&gt;rq_flags);

	return true;
}

static __be32 *encode_change(__be32 *p, struct kstat *stat, struct inode *inode,
			     struct svc_export *exp)
{
	if (exp-&gt;ex_flags &amp; NFSEXP_V4ROOT) {
		*p++ = cpu_to_be32(convert_to_wallclock(exp-&gt;cd-&gt;flush_time));
		*p++ = 0;
	} else
		p = xdr_encode_hyper(p, nfsd4_change_attribute(stat, inode));
	return p;
}

/*
 * ctime (in NFSv4, time_metadata) is not writeable, and the client
 * doesn't really care what resolution could theoretically be stored by
 * the filesystem.
 *
 * The client cares how close together changes can be while still
 * guaranteeing ctime changes.  For most filesystems (which have
 * timestamps with nanosecond fields) that is limited by the resolution
 * of the time returned from current_time() (which I'm assuming to be
 * 1/HZ).
 */
static __be32 *encode_time_delta(__be32 *p, struct inode *inode)
{
	struct timespec64 ts;
	u32 ns;

	ns = max_t(u32, NSEC_PER_SEC/HZ, inode-&gt;i_sb-&gt;s_time_gran);
	ts = ns_to_timespec64(ns);

	p = xdr_encode_hyper(p, ts.tv_sec);
	*p++ = cpu_to_be32(ts.tv_nsec);

	return p;
}

static __be32 *encode_cinfo(__be32 *p, struct nfsd4_change_info *c)
{
	*p++ = cpu_to_be32(c-&gt;atomic);
	p = xdr_encode_hyper(p, c-&gt;before_change);
	p = xdr_encode_hyper(p, c-&gt;after_change);
	return p;
}

/* Encode as an array of strings the string given with components
 * separated @sep, escaped with esc_enter and esc_exit.
 */
static __be32 nfsd4_encode_components_esc(struct xdr_stream *xdr, char sep,
					  char *components, char esc_enter,
					  char esc_exit)
{
	__be32 *p;
	__be32 pathlen;
	int pathlen_offset;
	int strlen, count=0;
	char *str, *end, *next;

	dprintk("nfsd4_encode_components(%s)\n", components);

	pathlen_offset = xdr-&gt;buf-&gt;len;
	p = xdr_reserve_space(xdr, 4);
	if (!p)
		return nfserr_resource;
	p++; /* We will fill this in with @count later */

	end = str = components;
	while (*end) {
		bool found_esc = false;

		/* try to parse as esc_start, ..., esc_end, sep */
		if (*str == esc_enter) {
			for (; *end &amp;&amp; (*end != esc_exit); end++)
				/* find esc_exit or end of string */;
			next = end + 1;
			if (*end &amp;&amp; (!*next || *next == sep)) {
				str++;
				found_esc = true;
			}
		}

		if (!found_esc)
			for (; *end &amp;&amp; (*end != sep); end++)
				/* find sep or end of string */;

		strlen = end - str;
		if (strlen) {
			p = xdr_reserve_space(xdr, strlen + 4);
			if (!p)
				return nfserr_resource;
			p = xdr_encode_opaque(p, str, strlen);
			count++;
		}
		else
			end++;
		if (found_esc)
			end = next;

		str = end;
	}
	pathlen = htonl(count);
	write_bytes_to_xdr_buf(xdr-&gt;buf, pathlen_offset, &amp;pathlen, 4);
	return 0;
}

/* Encode as an array of strings the string given with components
 * separated @sep.
 */
static __be32 nfsd4_encode_components(struct xdr_stream *xdr, char sep,
				      char *components)
{
	return nfsd4_encode_components_esc(xdr, sep, components, 0, 0);
}

/*
 * encode a location element of a fs_locations structure
 */
static __be32 nfsd4_encode_fs_location4(struct xdr_stream *xdr,
					struct nfsd4_fs_location *location)
{
	__be32 status;

	status = nfsd4_encode_components_esc(xdr, ':', location-&gt;hosts,
						'[', ']');
	if (status)
		return status;
	status = nfsd4_encode_components(xdr, '/', location-&gt;path);
	if (status)
		return status;
	return 0;
}

/*
 * Encode a path in RFC3530 'pathname4' format
 */
static __be32 nfsd4_encode_path(struct xdr_stream *xdr,
				const struct path *root,
				const struct path *path)
{
	struct path cur = *path;
	__be32 *p;
	struct dentry **components = NULL;
	unsigned int ncomponents = 0;
	__be32 err = nfserr_jukebox;

	dprintk("nfsd4_encode_components(");

	path_get(&amp;cur);
	/* First walk the path up to the nfsd root, and store the
	 * dentries/path components in an array.
	 */
	for (;;) {
		if (path_equal(&amp;cur, root))
			break;
		if (cur.dentry == cur.mnt-&gt;mnt_root) {
			if (follow_up(&amp;cur))
				continue;
			goto out_free;
		}
		if ((ncomponents &amp; 15) == 0) {
			struct dentry **new;
			new = krealloc(components,
					sizeof(*new) * (ncomponents + 16),
					GFP_KERNEL);
			if (!new)
				goto out_free;
			components = new;
		}
		components[ncomponents++] = cur.dentry;
		cur.dentry = dget_parent(cur.dentry);
	}
	err = nfserr_resource;
	p = xdr_reserve_space(xdr, 4);
	if (!p)
		goto out_free;
	*p++ = cpu_to_be32(ncomponents);

	while (ncomponents) {
		struct dentry *dentry = components[ncomponents - 1];
		unsigned int len;

		spin_lock(&amp;dentry-&gt;d_lock);
		len = dentry-&gt;d_name.len;
		p = xdr_reserve_space(xdr, len + 4);
		if (!p) {
			spin_unlock(&amp;dentry-&gt;d_lock);
			goto out_free;
		}
		p = xdr_encode_opaque(p, dentry-&gt;d_name.name, len);
		dprintk("/%pd", dentry);
		spin_unlock(&amp;dentry-&gt;d_lock);
		dput(dentry);
		ncomponents--;
	}

	err = 0;
out_free:
	dprintk(")\n");
	while (ncomponents)
		dput(components[--ncomponents]);
	kfree(components);
	path_put(&amp;cur);
	return err;
}

static __be32 nfsd4_encode_fsloc_fsroot(struct xdr_stream *xdr,
			struct svc_rqst *rqstp, const struct path *path)
{
	struct svc_export *exp_ps;
	__be32 res;

	exp_ps = rqst_find_fsidzero_export(rqstp);
	if (IS_ERR(exp_ps))
		return nfserrno(PTR_ERR(exp_ps));
	res = nfsd4_encode_path(xdr, &amp;exp_ps-&gt;ex_path, path);
	exp_put(exp_ps);
	return res;
}

/*
 *  encode a fs_locations structure
 */
static __be32 nfsd4_encode_fs_locations(struct xdr_stream *xdr,
			struct svc_rqst *rqstp, struct svc_export *exp)
{
	__be32 status;
	int i;
	__be32 *p;
	struct nfsd4_fs_locations *fslocs = &amp;exp-&gt;ex_fslocs;

	status = nfsd4_encode_fsloc_fsroot(xdr, rqstp, &amp;exp-&gt;ex_path);
	if (status)
		return status;
	p = xdr_reserve_space(xdr, 4);
	if (!p)
		return nfserr_resource;
	*p++ = cpu_to_be32(fslocs-&gt;locations_count);
	for (i=0; i<fslocs->locations_count; i++) {
		status = nfsd4_encode_fs_location4(xdr, &amp;fslocs-&gt;locations[i]);
		if (status)
			return status;
	}
	return 0;
}

static u32 nfs4_file_type(umode_t mode)
{
	switch (mode &amp; S_IFMT) {
	case S_IFIFO:	return NF4FIFO;
	case S_IFCHR:	return NF4CHR;
	case S_IFDIR:	return NF4DIR;
	case S_IFBLK:	return NF4BLK;
	case S_IFLNK:	return NF4LNK;
	case S_IFREG:	return NF4REG;
	case S_IFSOCK:	return NF4SOCK;
	default:	return NF4BAD;
	}
}

static inline __be32
nfsd4_encode_aclname(struct xdr_stream *xdr, struct svc_rqst *rqstp,
		     struct nfs4_ace *ace)
{
	if (ace-&gt;whotype != NFS4_ACL_WHO_NAMED)
		return nfs4_acl_write_who(xdr, ace-&gt;whotype);
	else if (ace-&gt;flag &amp; NFS4_ACE_IDENTIFIER_GROUP)
		return nfsd4_encode_group(xdr, rqstp, ace-&gt;who_gid);
	else
		return nfsd4_encode_user(xdr, rqstp, ace-&gt;who_uid);
}

static inline __be32
nfsd4_encode_layout_types(struct xdr_stream *xdr, u32 layout_types)
{
	__be32		*p;
	unsigned long	i = hweight_long(layout_types);

	p = xdr_reserve_space(xdr, 4 + 4 * i);
	if (!p)
		return nfserr_resource;

	*p++ = cpu_to_be32(i);

	for (i = LAYOUT_NFSV4_1_FILES; i &lt; LAYOUT_TYPE_MAX; ++i)
		if (layout_types &amp; (1 &lt;&lt; i))
			*p++ = cpu_to_be32(i);

	return 0;
}

#define WORD0_ABSENT_FS_ATTRS (FATTR4_WORD0_FS_LOCATIONS | FATTR4_WORD0_FSID | \
			      FATTR4_WORD0_RDATTR_ERROR)
#define WORD1_ABSENT_FS_ATTRS FATTR4_WORD1_MOUNTED_ON_FILEID
#define WORD2_ABSENT_FS_ATTRS 0

#ifdef CONFIG_NFSD_V4_SECURITY_LABEL
static inline __be32
nfsd4_encode_security_label(struct xdr_stream *xdr, struct svc_rqst *rqstp,
			    void *context, int len)
{
	__be32 *p;

	p = xdr_reserve_space(xdr, len + 4 + 4 + 4);
	if (!p)
		return nfserr_resource;

	/*
	 * For now we use a 0 here to indicate the null translation; in
	 * the future we may place a call to translation code here.
	 */
	*p++ = cpu_to_be32(0); /* lfs */
	*p++ = cpu_to_be32(0); /* pi */
	p = xdr_encode_opaque(p, context, len);
	return 0;
}
#else
static inline __be32
nfsd4_encode_security_label(struct xdr_stream *xdr, struct svc_rqst *rqstp,
			    void *context, int len)
{ return 0; }
#endif

static __be32 fattr_handle_absent_fs(u32 *bmval0, u32 *bmval1, u32 *bmval2, u32 *rdattr_err)
{
	/* As per referral draft:  */
	if (*bmval0 &amp; ~WORD0_ABSENT_FS_ATTRS ||
	    *bmval1 &amp; ~WORD1_ABSENT_FS_ATTRS) {
		if (*bmval0 &amp; FATTR4_WORD0_RDATTR_ERROR ||
	            *bmval0 &amp; FATTR4_WORD0_FS_LOCATIONS)
			*rdattr_err = NFSERR_MOVED;
		else
			return nfserr_moved;
	}
	*bmval0 &amp;= WORD0_ABSENT_FS_ATTRS;
	*bmval1 &amp;= WORD1_ABSENT_FS_ATTRS;
	*bmval2 &amp;= WORD2_ABSENT_FS_ATTRS;
	return 0;
}


static int nfsd4_get_mounted_on_ino(struct svc_export *exp, u64 *pino)
{
	struct path path = exp-&gt;ex_path;
	struct kstat stat;
	int err;

	path_get(&amp;path);
	while (follow_up(&amp;path)) {
		if (path.dentry != path.mnt-&gt;mnt_root)
			break;
	}
	err = vfs_getattr(&amp;path, &amp;stat, STATX_INO, AT_STATX_SYNC_AS_STAT);
	path_put(&amp;path);
	if (!err)
		*pino = stat.ino;
	return err;
}

static __be32
nfsd4_encode_bitmap(struct xdr_stream *xdr, u32 bmval0, u32 bmval1, u32 bmval2)
{
	__be32 *p;

	if (bmval2) {
		p = xdr_reserve_space(xdr, 16);
		if (!p)
			goto out_resource;
		*p++ = cpu_to_be32(3);
		*p++ = cpu_to_be32(bmval0);
		*p++ = cpu_to_be32(bmval1);
		*p++ = cpu_to_be32(bmval2);
	} else if (bmval1) {
		p = xdr_reserve_space(xdr, 12);
		if (!p)
			goto out_resource;
		*p++ = cpu_to_be32(2);
		*p++ = cpu_to_be32(bmval0);
		*p++ = cpu_to_be32(bmval1);
	} else {
		p = xdr_reserve_space(xdr, 8);
		if (!p)
			goto out_resource;
		*p++ = cpu_to_be32(1);
		*p++ = cpu_to_be32(bmval0);
	}

	return 0;
out_resource:
	return nfserr_resource;
}

/*
 * Note: @fhp can be NULL; in this case, we might have to compose the filehandle
 * ourselves.
 */
static __be32
nfsd4_encode_fattr(struct xdr_stream *xdr, struct svc_fh *fhp,
		struct svc_export *exp,
		struct dentry *dentry, u32 *bmval,
		struct svc_rqst *rqstp, int ignore_crossmnt)
{
	u32 bmval0 = bmval[0];
	u32 bmval1 = bmval[1];
	u32 bmval2 = bmval[2];
	struct kstat stat;
	struct svc_fh *tempfh = NULL;
	struct kstatfs statfs;
	__be32 *p, *attrlen_p;
	int starting_len = xdr-&gt;buf-&gt;len;
	int attrlen_offset;
	u32 dummy;
	u64 dummy64;
	u32 rdattr_err = 0;
	__be32 status;
	int err;
	struct nfs4_acl *acl = NULL;
#ifdef CONFIG_NFSD_V4_SECURITY_LABEL
	void *context = NULL;
	int contextlen;
#endif
	bool contextsupport = false;
	struct nfsd4_compoundres *resp = rqstp-&gt;rq_resp;
	u32 minorversion = resp-&gt;cstate.minorversion;
	struct path path = {
		.mnt	= exp-&gt;ex_path.mnt,
		.dentry	= dentry,
	};
	struct nfsd_net *nn = net_generic(SVC_NET(rqstp), nfsd_net_id);

	BUG_ON(bmval1 &amp; NFSD_WRITEONLY_ATTRS_WORD1);
	BUG_ON(!nfsd_attrs_supported(minorversion, bmval));

	if (exp-&gt;ex_fslocs.migrated) {
		status = fattr_handle_absent_fs(&amp;bmval0, &amp;bmval1, &amp;bmval2, &amp;rdattr_err);
		if (status)
			goto out;
	}

	err = vfs_getattr(&amp;path, &amp;stat, STATX_BASIC_STATS, AT_STATX_SYNC_AS_STAT);
	if (err)
		goto out_nfserr;
	if (!(stat.result_mask &amp; STATX_BTIME))
		/* underlying FS does not offer btime so we can't share it */
		bmval1 &amp;= ~FATTR4_WORD1_TIME_CREATE;
	if ((bmval0 &amp; (FATTR4_WORD0_FILES_AVAIL | FATTR4_WORD0_FILES_FREE |
			FATTR4_WORD0_FILES_TOTAL | FATTR4_WORD0_MAXNAME)) ||
	    (bmval1 &amp; (FATTR4_WORD1_SPACE_AVAIL | FATTR4_WORD1_SPACE_FREE |
		       FATTR4_WORD1_SPACE_TOTAL))) {
		err = vfs_statfs(&amp;path, &amp;statfs);
		if (err)
			goto out_nfserr;
	}
	if ((bmval0 &amp; (FATTR4_WORD0_FILEHANDLE | FATTR4_WORD0_FSID)) &amp;&amp; !fhp) {
		tempfh = kmalloc(sizeof(struct svc_fh), GFP_KERNEL);
		status = nfserr_jukebox;
		if (!tempfh)
			goto out;
		fh_init(tempfh, NFS4_FHSIZE);
		status = fh_compose(tempfh, exp, dentry, NULL);
		if (status)
			goto out;
		fhp = tempfh;
	}
	if (bmval0 &amp; FATTR4_WORD0_ACL) {
		err = nfsd4_get_nfs4_acl(rqstp, dentry, &amp;acl);
		if (err == -EOPNOTSUPP)
			bmval0 &amp;= ~FATTR4_WORD0_ACL;
		else if (err == -EINVAL) {
			status = nfserr_attrnotsupp;
			goto out;
		} else if (err != 0)
			goto out_nfserr;
	}

#ifdef CONFIG_NFSD_V4_SECURITY_LABEL
	if ((bmval2 &amp; FATTR4_WORD2_SECURITY_LABEL) ||
	     bmval0 &amp; FATTR4_WORD0_SUPPORTED_ATTRS) {
		if (exp-&gt;ex_flags &amp; NFSEXP_SECURITY_LABEL)
			err = security_inode_getsecctx(d_inode(dentry),
						&amp;context, &amp;contextlen);
		else
			err = -EOPNOTSUPP;
		contextsupport = (err == 0);
		if (bmval2 &amp; FATTR4_WORD2_SECURITY_LABEL) {
			if (err == -EOPNOTSUPP)
				bmval2 &amp;= ~FATTR4_WORD2_SECURITY_LABEL;
			else if (err)
				goto out_nfserr;
		}
	}
#endif /* CONFIG_NFSD_V4_SECURITY_LABEL */

	status = nfsd4_encode_bitmap(xdr, bmval0, bmval1, bmval2);
	if (status)
		goto out;

	attrlen_offset = xdr-&gt;buf-&gt;len;
	attrlen_p = xdr_reserve_space(xdr, XDR_UNIT);
	if (!attrlen_p)
		goto out_resource;

	if (bmval0 &amp; FATTR4_WORD0_SUPPORTED_ATTRS) {
		u32 supp[3];

		memcpy(supp, nfsd_suppattrs[minorversion], sizeof(supp));

		if (!IS_POSIXACL(dentry-&gt;d_inode))
			supp[0] &amp;= ~FATTR4_WORD0_ACL;
		if (!contextsupport)
			supp[2] &amp;= ~FATTR4_WORD2_SECURITY_LABEL;
		if (!supp[2]) {
			p = xdr_reserve_space(xdr, 12);
			if (!p)
				goto out_resource;
			*p++ = cpu_to_be32(2);
			*p++ = cpu_to_be32(supp[0]);
			*p++ = cpu_to_be32(supp[1]);
		} else {
			p = xdr_reserve_space(xdr, 16);
			if (!p)
				goto out_resource;
			*p++ = cpu_to_be32(3);
			*p++ = cpu_to_be32(supp[0]);
			*p++ = cpu_to_be32(supp[1]);
			*p++ = cpu_to_be32(supp[2]);
		}
	}
	if (bmval0 &amp; FATTR4_WORD0_TYPE) {
		p = xdr_reserve_space(xdr, 4);
		if (!p)
			goto out_resource;
		dummy = nfs4_file_type(stat.mode);
		if (dummy == NF4BAD) {
			status = nfserr_serverfault;
			goto out;
		}
		*p++ = cpu_to_be32(dummy);
	}
	if (bmval0 &amp; FATTR4_WORD0_FH_EXPIRE_TYPE) {
		p = xdr_reserve_space(xdr, 4);
		if (!p)
			goto out_resource;
		if (exp-&gt;ex_flags &amp; NFSEXP_NOSUBTREECHECK)
			*p++ = cpu_to_be32(NFS4_FH_PERSISTENT);
		else
			*p++ = cpu_to_be32(NFS4_FH_PERSISTENT|
						NFS4_FH_VOL_RENAME);
	}
	if (bmval0 &amp; FATTR4_WORD0_CHANGE) {
		p = xdr_reserve_space(xdr, 8);
		if (!p)
			goto out_resource;
		p = encode_change(p, &amp;stat, d_inode(dentry), exp);
	}
	if (bmval0 &amp; FATTR4_WORD0_SIZE) {
		p = xdr_reserve_space(xdr, 8);
		if (!p)
			goto out_resource;
		p = xdr_encode_hyper(p, stat.size);
	}
	if (bmval0 &amp; FATTR4_WORD0_LINK_SUPPORT) {
		p = xdr_reserve_space(xdr, 4);
		if (!p)
			goto out_resource;
		*p++ = cpu_to_be32(1);
	}
	if (bmval0 &amp; FATTR4_WORD0_SYMLINK_SUPPORT) {
		p = xdr_reserve_space(xdr, 4);
		if (!p)
			goto out_resource;
		*p++ = cpu_to_be32(1);
	}
	if (bmval0 &amp; FATTR4_WORD0_NAMED_ATTR) {
		p = xdr_reserve_space(xdr, 4);
		if (!p)
			goto out_resource;
		*p++ = cpu_to_be32(0);
	}
	if (bmval0 &amp; FATTR4_WORD0_FSID) {
		p = xdr_reserve_space(xdr, 16);
		if (!p)
			goto out_resource;
		if (exp-&gt;ex_fslocs.migrated) {
			p = xdr_encode_hyper(p, NFS4_REFERRAL_FSID_MAJOR);
			p = xdr_encode_hyper(p, NFS4_REFERRAL_FSID_MINOR);
		} else switch(fsid_source(fhp)) {
		case FSIDSOURCE_FSID:
			p = xdr_encode_hyper(p, (u64)exp-&gt;ex_fsid);
			p = xdr_encode_hyper(p, (u64)0);
			break;
		case FSIDSOURCE_DEV:
			*p++ = cpu_to_be32(0);
			*p++ = cpu_to_be32(MAJOR(stat.dev));
			*p++ = cpu_to_be32(0);
			*p++ = cpu_to_be32(MINOR(stat.dev));
			break;
		case FSIDSOURCE_UUID:
			p = xdr_encode_opaque_fixed(p, exp-&gt;ex_uuid,
								EX_UUID_LEN);
			break;
		}
	}
	if (bmval0 &amp; FATTR4_WORD0_UNIQUE_HANDLES) {
		p = xdr_reserve_space(xdr, 4);
		if (!p)
			goto out_resource;
		*p++ = cpu_to_be32(0);
	}
	if (bmval0 &amp; FATTR4_WORD0_LEASE_TIME) {
		p = xdr_reserve_space(xdr, 4);
		if (!p)
			goto out_resource;
		*p++ = cpu_to_be32(nn-&gt;nfsd4_lease);
	}
	if (bmval0 &amp; FATTR4_WORD0_RDATTR_ERROR) {
		p = xdr_reserve_space(xdr, 4);
		if (!p)
			goto out_resource;
		*p++ = cpu_to_be32(rdattr_err);
	}
	if (bmval0 &amp; FATTR4_WORD0_ACL) {
		struct nfs4_ace *ace;

		if (acl == NULL) {
			p = xdr_reserve_space(xdr, 4);
			if (!p)
				goto out_resource;

			*p++ = cpu_to_be32(0);
			goto out_acl;
		}
		p = xdr_reserve_space(xdr, 4);
		if (!p)
			goto out_resource;
		*p++ = cpu_to_be32(acl-&gt;naces);

		for (ace = acl-&gt;aces; ace &lt; acl-&gt;aces + acl-&gt;naces; ace++) {
			p = xdr_reserve_space(xdr, 4*3);
			if (!p)
				goto out_resource;
			*p++ = cpu_to_be32(ace-&gt;type);
			*p++ = cpu_to_be32(ace-&gt;flag);
			*p++ = cpu_to_be32(ace-&gt;access_mask &amp;
							NFS4_ACE_MASK_ALL);
			status = nfsd4_encode_aclname(xdr, rqstp, ace);
			if (status)
				goto out;
		}
	}
out_acl:
	if (bmval0 &amp; FATTR4_WORD0_ACLSUPPORT) {
		p = xdr_reserve_space(xdr, 4);
		if (!p)
			goto out_resource;
		*p++ = cpu_to_be32(IS_POSIXACL(dentry-&gt;d_inode) ?
			ACL4_SUPPORT_ALLOW_ACL|ACL4_SUPPORT_DENY_ACL : 0);
	}
	if (bmval0 &amp; FATTR4_WORD0_CANSETTIME) {
		p = xdr_reserve_space(xdr, 4);
		if (!p)
			goto out_resource;
		*p++ = cpu_to_be32(1);
	}
	if (bmval0 &amp; FATTR4_WORD0_CASE_INSENSITIVE) {
		p = xdr_reserve_space(xdr, 4);
		if (!p)
			goto out_resource;
		*p++ = cpu_to_be32(0);
	}
	if (bmval0 &amp; FATTR4_WORD0_CASE_PRESERVING) {
		p = xdr_reserve_space(xdr, 4);
		if (!p)
			goto out_resource;
		*p++ = cpu_to_be32(1);
	}
	if (bmval0 &amp; FATTR4_WORD0_CHOWN_RESTRICTED) {
		p = xdr_reserve_space(xdr, 4);
		if (!p)
			goto out_resource;
		*p++ = cpu_to_be32(1);
	}
	if (bmval0 &amp; FATTR4_WORD0_FILEHANDLE) {
		p = xdr_reserve_space(xdr, fhp-&gt;fh_handle.fh_size + 4);
		if (!p)
			goto out_resource;
		p = xdr_encode_opaque(p, &amp;fhp-&gt;fh_handle.fh_raw,
					fhp-&gt;fh_handle.fh_size);
	}
	if (bmval0 &amp; FATTR4_WORD0_FILEID) {
		p = xdr_reserve_space(xdr, 8);
		if (!p)
			goto out_resource;
		p = xdr_encode_hyper(p, stat.ino);
	}
	if (bmval0 &amp; FATTR4_WORD0_FILES_AVAIL) {
		p = xdr_reserve_space(xdr, 8);
		if (!p)
			goto out_resource;
		p = xdr_encode_hyper(p, (u64) statfs.f_ffree);
	}
	if (bmval0 &amp; FATTR4_WORD0_FILES_FREE) {
		p = xdr_reserve_space(xdr, 8);
		if (!p)
			goto out_resource;
		p = xdr_encode_hyper(p, (u64) statfs.f_ffree);
	}
	if (bmval0 &amp; FATTR4_WORD0_FILES_TOTAL) {
		p = xdr_reserve_space(xdr, 8);
		if (!p)
			goto out_resource;
		p = xdr_encode_hyper(p, (u64) statfs.f_files);
	}
	if (bmval0 &amp; FATTR4_WORD0_FS_LOCATIONS) {
		status = nfsd4_encode_fs_locations(xdr, rqstp, exp);
		if (status)
			goto out;
	}
	if (bmval0 &amp; FATTR4_WORD0_HOMOGENEOUS) {
		p = xdr_reserve_space(xdr, 4);
		if (!p)
			goto out_resource;
		*p++ = cpu_to_be32(1);
	}
	if (bmval0 &amp; FATTR4_WORD0_MAXFILESIZE) {
		p = xdr_reserve_space(xdr, 8);
		if (!p)
			goto out_resource;
		p = xdr_encode_hyper(p, exp-&gt;ex_path.mnt-&gt;mnt_sb-&gt;s_maxbytes);
	}
	if (bmval0 &amp; FATTR4_WORD0_MAXLINK) {
		p = xdr_reserve_space(xdr, 4);
		if (!p)
			goto out_resource;
		*p++ = cpu_to_be32(255);
	}
	if (bmval0 &amp; FATTR4_WORD0_MAXNAME) {
		p = xdr_reserve_space(xdr, 4);
		if (!p)
			goto out_resource;
		*p++ = cpu_to_be32(statfs.f_namelen);
	}
	if (bmval0 &amp; FATTR4_WORD0_MAXREAD) {
		p = xdr_reserve_space(xdr, 8);
		if (!p)
			goto out_resource;
		p = xdr_encode_hyper(p, (u64) svc_max_payload(rqstp));
	}
	if (bmval0 &amp; FATTR4_WORD0_MAXWRITE) {
		p = xdr_reserve_space(xdr, 8);
		if (!p)
			goto out_resource;
		p = xdr_encode_hyper(p, (u64) svc_max_payload(rqstp));
	}
	if (bmval1 &amp; FATTR4_WORD1_MODE) {
		p = xdr_reserve_space(xdr, 4);
		if (!p)
			goto out_resource;
		*p++ = cpu_to_be32(stat.mode &amp; S_IALLUGO);
	}
	if (bmval1 &amp; FATTR4_WORD1_NO_TRUNC) {
		p = xdr_reserve_space(xdr, 4);
		if (!p)
			goto out_resource;
		*p++ = cpu_to_be32(1);
	}
	if (bmval1 &amp; FATTR4_WORD1_NUMLINKS) {
		p = xdr_reserve_space(xdr, 4);
		if (!p)
			goto out_resource;
		*p++ = cpu_to_be32(stat.nlink);
	}
	if (bmval1 &amp; FATTR4_WORD1_OWNER) {
		status = nfsd4_encode_user(xdr, rqstp, stat.uid);
		if (status)
			goto out;
	}
	if (bmval1 &amp; FATTR4_WORD1_OWNER_GROUP) {
		status = nfsd4_encode_group(xdr, rqstp, stat.gid);
		if (status)
			goto out;
	}
	if (bmval1 &amp; FATTR4_WORD1_RAWDEV) {
		p = xdr_reserve_space(xdr, 8);
		if (!p)
			goto out_resource;
		*p++ = cpu_to_be32((u32) MAJOR(stat.rdev));
		*p++ = cpu_to_be32((u32) MINOR(stat.rdev));
	}
	if (bmval1 &amp; FATTR4_WORD1_SPACE_AVAIL) {
		p = xdr_reserve_space(xdr, 8);
		if (!p)
			goto out_resource;
		dummy64 = (u64)statfs.f_bavail * (u64)statfs.f_bsize;
		p = xdr_encode_hyper(p, dummy64);
	}
	if (bmval1 &amp; FATTR4_WORD1_SPACE_FREE) {
		p = xdr_reserve_space(xdr, 8);
		if (!p)
			goto out_resource;
		dummy64 = (u64)statfs.f_bfree * (u64)statfs.f_bsize;
		p = xdr_encode_hyper(p, dummy64);
	}
	if (bmval1 &amp; FATTR4_WORD1_SPACE_TOTAL) {
		p = xdr_reserve_space(xdr, 8);
		if (!p)
			goto out_resource;
		dummy64 = (u64)statfs.f_blocks * (u64)statfs.f_bsize;
		p = xdr_encode_hyper(p, dummy64);
	}
	if (bmval1 &amp; FATTR4_WORD1_SPACE_USED) {
		p = xdr_reserve_space(xdr, 8);
		if (!p)
			goto out_resource;
		dummy64 = (u64)stat.blocks &lt;&lt; 9;
		p = xdr_encode_hyper(p, dummy64);
	}
	if (bmval1 &amp; FATTR4_WORD1_TIME_ACCESS) {
		p = xdr_reserve_space(xdr, 12);
		if (!p)
			goto out_resource;
		p = xdr_encode_hyper(p, (s64)stat.atime.tv_sec);
		*p++ = cpu_to_be32(stat.atime.tv_nsec);
	}
	if (bmval1 &amp; FATTR4_WORD1_TIME_DELTA) {
		p = xdr_reserve_space(xdr, 12);
		if (!p)
			goto out_resource;
		p = encode_time_delta(p, d_inode(dentry));
	}
	if (bmval1 &amp; FATTR4_WORD1_TIME_METADATA) {
		p = xdr_reserve_space(xdr, 12);
		if (!p)
			goto out_resource;
		p = xdr_encode_hyper(p, (s64)stat.ctime.tv_sec);
		*p++ = cpu_to_be32(stat.ctime.tv_nsec);
	}
	if (bmval1 &amp; FATTR4_WORD1_TIME_MODIFY) {
		p = xdr_reserve_space(xdr, 12);
		if (!p)
			goto out_resource;
		p = xdr_encode_hyper(p, (s64)stat.mtime.tv_sec);
		*p++ = cpu_to_be32(stat.mtime.tv_nsec);
	}
	if (bmval1 &amp; FATTR4_WORD1_TIME_CREATE) {
		p = xdr_reserve_space(xdr, 12);
		if (!p)
			goto out_resource;
		p = xdr_encode_hyper(p, (s64)stat.btime.tv_sec);
		*p++ = cpu_to_be32(stat.btime.tv_nsec);
	}
	if (bmval1 &amp; FATTR4_WORD1_MOUNTED_ON_FILEID) {
		u64 ino = stat.ino;

		p = xdr_reserve_space(xdr, 8);
		if (!p)
                	goto out_resource;
		/*
		 * Get ino of mountpoint in parent filesystem, if not ignoring
		 * crossmount and this is the root of a cross-mounted
		 * filesystem.
		 */
		if (ignore_crossmnt == 0 &amp;&amp;
		    dentry == exp-&gt;ex_path.mnt-&gt;mnt_root) {
			err = nfsd4_get_mounted_on_ino(exp, &amp;ino);
			if (err)
				goto out_nfserr;
		}
		p = xdr_encode_hyper(p, ino);
	}
#ifdef CONFIG_NFSD_PNFS
	if (bmval1 &amp; FATTR4_WORD1_FS_LAYOUT_TYPES) {
		status = nfsd4_encode_layout_types(xdr, exp-&gt;ex_layout_types);
		if (status)
			goto out;
	}

	if (bmval2 &amp; FATTR4_WORD2_LAYOUT_TYPES) {
		status = nfsd4_encode_layout_types(xdr, exp-&gt;ex_layout_types);
		if (status)
			goto out;
	}

	if (bmval2 &amp; FATTR4_WORD2_LAYOUT_BLKSIZE) {
		p = xdr_reserve_space(xdr, 4);
		if (!p)
			goto out_resource;
		*p++ = cpu_to_be32(stat.blksize);
	}
#endif /* CONFIG_NFSD_PNFS */
	if (bmval2 &amp; FATTR4_WORD2_SUPPATTR_EXCLCREAT) {
		u32 supp[3];

		memcpy(supp, nfsd_suppattrs[minorversion], sizeof(supp));
		supp[0] &amp;= NFSD_SUPPATTR_EXCLCREAT_WORD0;
		supp[1] &amp;= NFSD_SUPPATTR_EXCLCREAT_WORD1;
		supp[2] &amp;= NFSD_SUPPATTR_EXCLCREAT_WORD2;

		status = nfsd4_encode_bitmap(xdr, supp[0], supp[1], supp[2]);
		if (status)
			goto out;
	}

#ifdef CONFIG_NFSD_V4_SECURITY_LABEL
	if (bmval2 &amp; FATTR4_WORD2_SECURITY_LABEL) {
		status = nfsd4_encode_security_label(xdr, rqstp, context,
								contextlen);
		if (status)
			goto out;
	}
#endif

	if (bmval2 &amp; FATTR4_WORD2_XATTR_SUPPORT) {
		p = xdr_reserve_space(xdr, 4);
		if (!p)
			goto out_resource;
		err = xattr_supported_namespace(d_inode(dentry),
						XATTR_USER_PREFIX);
		*p++ = cpu_to_be32(err == 0);
	}

	*attrlen_p = cpu_to_be32(xdr-&gt;buf-&gt;len - attrlen_offset - XDR_UNIT);
	status = nfs_ok;

out:
#ifdef CONFIG_NFSD_V4_SECURITY_LABEL
	if (context)
		security_release_secctx(context, contextlen);
#endif /* CONFIG_NFSD_V4_SECURITY_LABEL */
	kfree(acl);
	if (tempfh) {
		fh_put(tempfh);
		kfree(tempfh);
	}
	if (status)
		xdr_truncate_encode(xdr, starting_len);
	return status;
out_nfserr:
	status = nfserrno(err);
	goto out;
out_resource:
	status = nfserr_resource;
	goto out;
}

static void svcxdr_init_encode_from_buffer(struct xdr_stream *xdr,
				struct xdr_buf *buf, __be32 *p, int bytes)
{
	xdr-&gt;scratch.iov_len = 0;
	memset(buf, 0, sizeof(struct xdr_buf));
	buf-&gt;head[0].iov_base = p;
	buf-&gt;head[0].iov_len = 0;
	buf-&gt;len = 0;
	xdr-&gt;buf = buf;
	xdr-&gt;iov = buf-&gt;head;
	xdr-&gt;p = p;
	xdr-&gt;end = (void *)p + bytes;
	buf-&gt;buflen = bytes;
}

__be32 nfsd4_encode_fattr_to_buf(__be32 **p, int words,
			struct svc_fh *fhp, struct svc_export *exp,
			struct dentry *dentry, u32 *bmval,
			struct svc_rqst *rqstp, int ignore_crossmnt)
{
	struct xdr_buf dummy;
	struct xdr_stream xdr;
	__be32 ret;

	svcxdr_init_encode_from_buffer(&amp;xdr, &amp;dummy, *p, words &lt;&lt; 2);
	ret = nfsd4_encode_fattr(&amp;xdr, fhp, exp, dentry, bmval, rqstp,
							ignore_crossmnt);
	*p = xdr.p;
	return ret;
}

static inline int attributes_need_mount(u32 *bmval)
{
	if (bmval[0] &amp; ~(FATTR4_WORD0_RDATTR_ERROR | FATTR4_WORD0_LEASE_TIME))
		return 1;
	if (bmval[1] &amp; ~FATTR4_WORD1_MOUNTED_ON_FILEID)
		return 1;
	return 0;
}

static __be32
nfsd4_encode_dirent_fattr(struct xdr_stream *xdr, struct nfsd4_readdir *cd,
			const char *name, int namlen)
{
	struct svc_export *exp = cd-&gt;rd_fhp-&gt;fh_export;
	struct dentry *dentry;
	__be32 nfserr;
	int ignore_crossmnt = 0;

	dentry = lookup_positive_unlocked(name, cd-&gt;rd_fhp-&gt;fh_dentry, namlen);
	if (IS_ERR(dentry))
		return nfserrno(PTR_ERR(dentry));

	exp_get(exp);
	/*
	 * In the case of a mountpoint, the client may be asking for
	 * attributes that are only properties of the underlying filesystem
	 * as opposed to the cross-mounted file system. In such a case,
	 * we will not follow the cross mount and will fill the attribtutes
	 * directly from the mountpoint dentry.
	 */
	if (nfsd_mountpoint(dentry, exp)) {
		int err;

		if (!(exp-&gt;ex_flags &amp; NFSEXP_V4ROOT)
				&amp;&amp; !attributes_need_mount(cd-&gt;rd_bmval)) {
			ignore_crossmnt = 1;
			goto out_encode;
		}
		/*
		 * Why the heck aren't we just using nfsd_lookup??
		 * Different "."/".." handling?  Something else?
		 * At least, add a comment here to explain....
		 */
		err = nfsd_cross_mnt(cd-&gt;rd_rqstp, &amp;dentry, &amp;exp);
		if (err) {
			nfserr = nfserrno(err);
			goto out_put;
		}
		nfserr = check_nfsd_access(exp, cd-&gt;rd_rqstp);
		if (nfserr)
			goto out_put;

	}
out_encode:
	nfserr = nfsd4_encode_fattr(xdr, NULL, exp, dentry, cd-&gt;rd_bmval,
					cd-&gt;rd_rqstp, ignore_crossmnt);
out_put:
	dput(dentry);
	exp_put(exp);
	return nfserr;
}

static __be32 *
nfsd4_encode_rdattr_error(struct xdr_stream *xdr, __be32 nfserr)
{
	__be32 *p;

	p = xdr_reserve_space(xdr, 20);
	if (!p)
		return NULL;
	*p++ = htonl(2);
	*p++ = htonl(FATTR4_WORD0_RDATTR_ERROR); /* bmval0 */
	*p++ = htonl(0);			 /* bmval1 */

	*p++ = htonl(4);     /* attribute length */
	*p++ = nfserr;       /* no htonl */
	return p;
}

static int
nfsd4_encode_dirent(void *ccdv, const char *name, int namlen,
		    loff_t offset, u64 ino, unsigned int d_type)
{
	struct readdir_cd *ccd = ccdv;
	struct nfsd4_readdir *cd = container_of(ccd, struct nfsd4_readdir, common);
	struct xdr_stream *xdr = cd-&gt;xdr;
	int start_offset = xdr-&gt;buf-&gt;len;
	int cookie_offset;
	u32 name_and_cookie;
	int entry_bytes;
	__be32 nfserr = nfserr_toosmall;
	__be64 wire_offset;
	__be32 *p;

	/* In nfsv4, "." and ".." never make it onto the wire.. */
	if (name &amp;&amp; isdotent(name, namlen)) {
		cd-&gt;common.err = nfs_ok;
		return 0;
	}

	if (cd-&gt;cookie_offset) {
		wire_offset = cpu_to_be64(offset);
		write_bytes_to_xdr_buf(xdr-&gt;buf, cd-&gt;cookie_offset,
							&amp;wire_offset, 8);
	}

	p = xdr_reserve_space(xdr, 4);
	if (!p)
		goto fail;
	*p++ = xdr_one;                             /* mark entry present */
	cookie_offset = xdr-&gt;buf-&gt;len;
	p = xdr_reserve_space(xdr, 3*4 + namlen);
	if (!p)
		goto fail;
	p = xdr_encode_hyper(p, OFFSET_MAX);        /* offset of next entry */
	p = xdr_encode_array(p, name, namlen);      /* name length &amp; name */

	nfserr = nfsd4_encode_dirent_fattr(xdr, cd, name, namlen);
	switch (nfserr) {
	case nfs_ok:
		break;
	case nfserr_resource:
		nfserr = nfserr_toosmall;
		goto fail;
	case nfserr_noent:
		xdr_truncate_encode(xdr, start_offset);
		goto skip_entry;
	default:
		/*
		 * If the client requested the RDATTR_ERROR attribute,
		 * we stuff the error code into this attribute
		 * and continue.  If this attribute was not requested,
		 * then in accordance with the spec, we fail the
		 * entire READDIR operation(!)
		 */
		if (!(cd-&gt;rd_bmval[0] &amp; FATTR4_WORD0_RDATTR_ERROR))
			goto fail;
		p = nfsd4_encode_rdattr_error(xdr, nfserr);
		if (p == NULL) {
			nfserr = nfserr_toosmall;
			goto fail;
		}
	}
	nfserr = nfserr_toosmall;
	entry_bytes = xdr-&gt;buf-&gt;len - start_offset;
	if (entry_bytes &gt; cd-&gt;rd_maxcount)
		goto fail;
	cd-&gt;rd_maxcount -= entry_bytes;
	/*
	 * RFC 3530 14.2.24 describes rd_dircount as only a "hint", and
	 * notes that it could be zero. If it is zero, then the server
	 * should enforce only the rd_maxcount value.
	 */
	if (cd-&gt;rd_dircount) {
		name_and_cookie = 4 + 4 * XDR_QUADLEN(namlen) + 8;
		if (name_and_cookie &gt; cd-&gt;rd_dircount &amp;&amp; cd-&gt;cookie_offset)
			goto fail;
		cd-&gt;rd_dircount -= min(cd-&gt;rd_dircount, name_and_cookie);
		if (!cd-&gt;rd_dircount)
			cd-&gt;rd_maxcount = 0;
	}

	cd-&gt;cookie_offset = cookie_offset;
skip_entry:
	cd-&gt;common.err = nfs_ok;
	return 0;
fail:
	xdr_truncate_encode(xdr, start_offset);
	cd-&gt;common.err = nfserr;
	return -EINVAL;
}

static __be32
nfsd4_encode_stateid(struct xdr_stream *xdr, stateid_t *sid)
{
	__be32 *p;

	p = xdr_reserve_space(xdr, sizeof(stateid_t));
	if (!p)
		return nfserr_resource;
	*p++ = cpu_to_be32(sid-&gt;si_generation);
	p = xdr_encode_opaque_fixed(p, &amp;sid-&gt;si_opaque,
					sizeof(stateid_opaque_t));
	return 0;
}

static __be32
nfsd4_encode_access(struct nfsd4_compoundres *resp, __be32 nfserr, struct nfsd4_access *access)
{
	struct xdr_stream *xdr = resp-&gt;xdr;
	__be32 *p;

	p = xdr_reserve_space(xdr, 8);
	if (!p)
		return nfserr_resource;
	*p++ = cpu_to_be32(access-&gt;ac_supported);
	*p++ = cpu_to_be32(access-&gt;ac_resp_access);
	return 0;
}

static __be32 nfsd4_encode_bind_conn_to_session(struct nfsd4_compoundres *resp, __be32 nfserr, struct nfsd4_bind_conn_to_session *bcts)
{
	struct xdr_stream *xdr = resp-&gt;xdr;
	__be32 *p;

	p = xdr_reserve_space(xdr, NFS4_MAX_SESSIONID_LEN + 8);
	if (!p)
		return nfserr_resource;
	p = xdr_encode_opaque_fixed(p, bcts-&gt;sessionid.data,
					NFS4_MAX_SESSIONID_LEN);
	*p++ = cpu_to_be32(bcts-&gt;dir);
	/* Upshifting from TCP to RDMA is not supported */
	*p++ = cpu_to_be32(0);
	return 0;
}

static __be32
nfsd4_encode_close(struct nfsd4_compoundres *resp, __be32 nfserr, struct nfsd4_close *close)
{
	struct xdr_stream *xdr = resp-&gt;xdr;

	return nfsd4_encode_stateid(xdr, &amp;close-&gt;cl_stateid);
}


static __be32
nfsd4_encode_commit(struct nfsd4_compoundres *resp, __be32 nfserr, struct nfsd4_commit *commit)
{
	struct xdr_stream *xdr = resp-&gt;xdr;
	__be32 *p;

	p = xdr_reserve_space(xdr, NFS4_VERIFIER_SIZE);
	if (!p)
		return nfserr_resource;
	p = xdr_encode_opaque_fixed(p, commit-&gt;co_verf.data,
						NFS4_VERIFIER_SIZE);
	return 0;
}

static __be32
nfsd4_encode_create(struct nfsd4_compoundres *resp, __be32 nfserr, struct nfsd4_create *create)
{
	struct xdr_stream *xdr = resp-&gt;xdr;
	__be32 *p;

	p = xdr_reserve_space(xdr, 20);
	if (!p)
		return nfserr_resource;
	encode_cinfo(p, &amp;create-&gt;cr_cinfo);
	return nfsd4_encode_bitmap(xdr, create-&gt;cr_bmval[0],
			create-&gt;cr_bmval[1], create-&gt;cr_bmval[2]);
}

static __be32
nfsd4_encode_getattr(struct nfsd4_compoundres *resp, __be32 nfserr, struct nfsd4_getattr *getattr)
{
	struct svc_fh *fhp = getattr-&gt;ga_fhp;
	struct xdr_stream *xdr = resp-&gt;xdr;

	return nfsd4_encode_fattr(xdr, fhp, fhp-&gt;fh_export, fhp-&gt;fh_dentry,
				    getattr-&gt;ga_bmval, resp-&gt;rqstp, 0);
}

static __be32
nfsd4_encode_getfh(struct nfsd4_compoundres *resp, __be32 nfserr, struct svc_fh **fhpp)
{
	struct xdr_stream *xdr = resp-&gt;xdr;
	struct svc_fh *fhp = *fhpp;
	unsigned int len;
	__be32 *p;

	len = fhp-&gt;fh_handle.fh_size;
	p = xdr_reserve_space(xdr, len + 4);
	if (!p)
		return nfserr_resource;
	p = xdr_encode_opaque(p, &amp;fhp-&gt;fh_handle.fh_raw, len);
	return 0;
}

/*
* Including all fields other than the name, a LOCK4denied structure requires
*   8(clientid) + 4(namelen) + 8(offset) + 8(length) + 4(type) = 32 bytes.
*/
static __be32
nfsd4_encode_lock_denied(struct xdr_stream *xdr, struct nfsd4_lock_denied *ld)
{
	struct xdr_netobj *conf = &amp;ld-&gt;ld_owner;
	__be32 *p;

again:
	p = xdr_reserve_space(xdr, 32 + XDR_LEN(conf-&gt;len));
	if (!p) {
		/*
		 * Don't fail to return the result just because we can't
		 * return the conflicting open:
		 */
		if (conf-&gt;len) {
			kfree(conf-&gt;data);
			conf-&gt;len = 0;
			conf-&gt;data = NULL;
			goto again;
		}
		return nfserr_resource;
	}
	p = xdr_encode_hyper(p, ld-&gt;ld_start);
	p = xdr_encode_hyper(p, ld-&gt;ld_length);
	*p++ = cpu_to_be32(ld-&gt;ld_type);
	if (conf-&gt;len) {
		p = xdr_encode_opaque_fixed(p, &amp;ld-&gt;ld_clientid, 8);
		p = xdr_encode_opaque(p, conf-&gt;data, conf-&gt;len);
		kfree(conf-&gt;data);
	}  else {  /* non - nfsv4 lock in conflict, no clientid nor owner */
		p = xdr_encode_hyper(p, (u64)0); /* clientid */
		*p++ = cpu_to_be32(0); /* length of owner name */
	}
	return nfserr_denied;
}

static __be32
nfsd4_encode_lock(struct nfsd4_compoundres *resp, __be32 nfserr, struct nfsd4_lock *lock)
{
	struct xdr_stream *xdr = resp-&gt;xdr;

	if (!nfserr)
		nfserr = nfsd4_encode_stateid(xdr, &amp;lock-&gt;lk_resp_stateid);
	else if (nfserr == nfserr_denied)
		nfserr = nfsd4_encode_lock_denied(xdr, &amp;lock-&gt;lk_denied);

	return nfserr;
}

static __be32
nfsd4_encode_lockt(struct nfsd4_compoundres *resp, __be32 nfserr, struct nfsd4_lockt *lockt)
{
	struct xdr_stream *xdr = resp-&gt;xdr;

	if (nfserr == nfserr_denied)
		nfsd4_encode_lock_denied(xdr, &amp;lockt-&gt;lt_denied);
	return nfserr;
}

static __be32
nfsd4_encode_locku(struct nfsd4_compoundres *resp, __be32 nfserr, struct nfsd4_locku *locku)
{
	struct xdr_stream *xdr = resp-&gt;xdr;

	return nfsd4_encode_stateid(xdr, &amp;locku-&gt;lu_stateid);
}


static __be32
nfsd4_encode_link(struct nfsd4_compoundres *resp, __be32 nfserr, struct nfsd4_link *link)
{
	struct xdr_stream *xdr = resp-&gt;xdr;
	__be32 *p;

	p = xdr_reserve_space(xdr, 20);
	if (!p)
		return nfserr_resource;
	p = encode_cinfo(p, &amp;link-&gt;li_cinfo);
	return 0;
}


static __be32
nfsd4_encode_open(struct nfsd4_compoundres *resp, __be32 nfserr, struct nfsd4_open *open)
{
	struct xdr_stream *xdr = resp-&gt;xdr;
	__be32 *p;

	nfserr = nfsd4_encode_stateid(xdr, &amp;open-&gt;op_stateid);
	if (nfserr)
		return nfserr;
	p = xdr_reserve_space(xdr, 24);
	if (!p)
		return nfserr_resource;
	p = encode_cinfo(p, &amp;open-&gt;op_cinfo);
	*p++ = cpu_to_be32(open-&gt;op_rflags);

	nfserr = nfsd4_encode_bitmap(xdr, open-&gt;op_bmval[0], open-&gt;op_bmval[1],
					open-&gt;op_bmval[2]);
	if (nfserr)
		return nfserr;

	p = xdr_reserve_space(xdr, 4);
	if (!p)
		return nfserr_resource;

	*p++ = cpu_to_be32(open-&gt;op_delegate_type);
	switch (open-&gt;op_delegate_type) {
	case NFS4_OPEN_DELEGATE_NONE:
		break;
	case NFS4_OPEN_DELEGATE_READ:
		nfserr = nfsd4_encode_stateid(xdr, &amp;open-&gt;op_delegate_stateid);
		if (nfserr)
			return nfserr;
		p = xdr_reserve_space(xdr, 20);
		if (!p)
			return nfserr_resource;
		*p++ = cpu_to_be32(open-&gt;op_recall);

		/*
		 * TODO: ACE's in delegations
		 */
		*p++ = cpu_to_be32(NFS4_ACE_ACCESS_ALLOWED_ACE_TYPE);
		*p++ = cpu_to_be32(0);
		*p++ = cpu_to_be32(0);
		*p++ = cpu_to_be32(0);   /* XXX: is NULL principal ok? */
		break;
	case NFS4_OPEN_DELEGATE_WRITE:
		nfserr = nfsd4_encode_stateid(xdr, &amp;open-&gt;op_delegate_stateid);
		if (nfserr)
			return nfserr;
		p = xdr_reserve_space(xdr, 32);
		if (!p)
			return nfserr_resource;
		*p++ = cpu_to_be32(0);

		/*
		 * TODO: space_limit's in delegations
		 */
		*p++ = cpu_to_be32(NFS4_LIMIT_SIZE);
		*p++ = cpu_to_be32(~(u32)0);
		*p++ = cpu_to_be32(~(u32)0);

		/*
		 * TODO: ACE's in delegations
		 */
		*p++ = cpu_to_be32(NFS4_ACE_ACCESS_ALLOWED_ACE_TYPE);
		*p++ = cpu_to_be32(0);
		*p++ = cpu_to_be32(0);
		*p++ = cpu_to_be32(0);   /* XXX: is NULL principal ok? */
		break;
	case NFS4_OPEN_DELEGATE_NONE_EXT: /* 4.1 */
		switch (open-&gt;op_why_no_deleg) {
		case WND4_CONTENTION:
		case WND4_RESOURCE:
			p = xdr_reserve_space(xdr, 8);
			if (!p)
				return nfserr_resource;
			*p++ = cpu_to_be32(open-&gt;op_why_no_deleg);
			/* deleg signaling not supported yet: */
			*p++ = cpu_to_be32(0);
			break;
		default:
			p = xdr_reserve_space(xdr, 4);
			if (!p)
				return nfserr_resource;
			*p++ = cpu_to_be32(open-&gt;op_why_no_deleg);
		}
		break;
	default:
		BUG();
	}
	/* XXX save filehandle here */
	return 0;
}

static __be32
nfsd4_encode_open_confirm(struct nfsd4_compoundres *resp, __be32 nfserr, struct nfsd4_open_confirm *oc)
{
	struct xdr_stream *xdr = resp-&gt;xdr;

	return nfsd4_encode_stateid(xdr, &amp;oc-&gt;oc_resp_stateid);
}

static __be32
nfsd4_encode_open_downgrade(struct nfsd4_compoundres *resp, __be32 nfserr, struct nfsd4_open_downgrade *od)
{
	struct xdr_stream *xdr = resp-&gt;xdr;

	return nfsd4_encode_stateid(xdr, &amp;od-&gt;od_stateid);
}

static __be32 nfsd4_encode_splice_read(
				struct nfsd4_compoundres *resp,
				struct nfsd4_read *read,
				struct file *file, unsigned long maxcount)
{
	struct xdr_stream *xdr = resp-&gt;xdr;
	struct xdr_buf *buf = xdr-&gt;buf;
	int status, space_left;
	__be32 nfserr;

	/* Make sure there will be room for padding if needed */
	if (xdr-&gt;end - xdr-&gt;p &lt; 1)
		return nfserr_resource;

	nfserr = nfsd_splice_read(read-&gt;rd_rqstp, read-&gt;rd_fhp,
				  file, read-&gt;rd_offset, &amp;maxcount,
				  &amp;read-&gt;rd_eof);
	read-&gt;rd_length = maxcount;
	if (nfserr)
		goto out_err;
	status = svc_encode_result_payload(read-&gt;rd_rqstp,
					   buf-&gt;head[0].iov_len, maxcount);
	if (status) {
		nfserr = nfserrno(status);
		goto out_err;
	}

	buf-&gt;page_len = maxcount;
	buf-&gt;len += maxcount;
	xdr-&gt;page_ptr += (buf-&gt;page_base + maxcount + PAGE_SIZE - 1)
							/ PAGE_SIZE;

	/* Use rest of head for padding and remaining ops: */
	buf-&gt;tail[0].iov_base = xdr-&gt;p;
	buf-&gt;tail[0].iov_len = 0;
	xdr-&gt;iov = buf-&gt;tail;
	if (maxcount&amp;3) {
		int pad = 4 - (maxcount&amp;3);

		*(xdr-&gt;p++) = 0;

		buf-&gt;tail[0].iov_base += maxcount&amp;3;
		buf-&gt;tail[0].iov_len = pad;
		buf-&gt;len += pad;
	}

	space_left = min_t(int, (void *)xdr-&gt;end - (void *)xdr-&gt;p,
				buf-&gt;buflen - buf-&gt;len);
	buf-&gt;buflen = buf-&gt;len + space_left;
	xdr-&gt;end = (__be32 *)((void *)xdr-&gt;end + space_left);

	return 0;

out_err:
	/*
	 * nfsd_splice_actor may have already messed with the
	 * page length; reset it so as not to confuse
	 * xdr_truncate_encode in our caller.
	 */
	buf-&gt;page_len = 0;
	return nfserr;
}

static __be32 nfsd4_encode_readv(struct nfsd4_compoundres *resp,
				 struct nfsd4_read *read,
				 struct file *file, unsigned long maxcount)
{
	struct xdr_stream *xdr = resp-&gt;xdr;
	unsigned int starting_len = xdr-&gt;buf-&gt;len;
	__be32 zero = xdr_zero;
	__be32 nfserr;

	read-&gt;rd_vlen = xdr_reserve_space_vec(xdr, resp-&gt;rqstp-&gt;rq_vec, maxcount);
	if (read-&gt;rd_vlen &lt; 0)
		return nfserr_resource;

	nfserr = nfsd_readv(resp-&gt;rqstp, read-&gt;rd_fhp, file, read-&gt;rd_offset,
			    resp-&gt;rqstp-&gt;rq_vec, read-&gt;rd_vlen, &amp;maxcount,
			    &amp;read-&gt;rd_eof);
	read-&gt;rd_length = maxcount;
	if (nfserr)
		return nfserr;
	if (svc_encode_result_payload(resp-&gt;rqstp, starting_len, maxcount))
		return nfserr_io;
	xdr_truncate_encode(xdr, starting_len + xdr_align_size(maxcount));

	write_bytes_to_xdr_buf(xdr-&gt;buf, starting_len + maxcount, &amp;zero,
			       xdr_pad_size(maxcount));
	return nfs_ok;
}

static __be32
nfsd4_encode_read(struct nfsd4_compoundres *resp, __be32 nfserr,
		  struct nfsd4_read *read)
{
	bool splice_ok = test_bit(RQ_SPLICE_OK, &amp;resp-&gt;rqstp-&gt;rq_flags);
	unsigned long maxcount;
	struct xdr_stream *xdr = resp-&gt;xdr;
	struct file *file;
	int starting_len = xdr-&gt;buf-&gt;len;
	__be32 *p;

	if (nfserr)
		return nfserr;
	file = read-&gt;rd_nf-&gt;nf_file;

	p = xdr_reserve_space(xdr, 8); /* eof flag and byte count */
	if (!p) {
		WARN_ON_ONCE(splice_ok);
		return nfserr_resource;
	}
	if (resp-&gt;xdr-&gt;buf-&gt;page_len &amp;&amp; splice_ok) {
		WARN_ON_ONCE(1);
		return nfserr_serverfault;
	}
	xdr_commit_encode(xdr);

	maxcount = min_t(unsigned long, read-&gt;rd_length,
			 (xdr-&gt;buf-&gt;buflen - xdr-&gt;buf-&gt;len));

	if (file-&gt;f_op-&gt;splice_read &amp;&amp; splice_ok)
		nfserr = nfsd4_encode_splice_read(resp, read, file, maxcount);
	else
		nfserr = nfsd4_encode_readv(resp, read, file, maxcount);
	if (nfserr) {
		xdr_truncate_encode(xdr, starting_len);
		return nfserr;
	}

	p = xdr_encode_bool(p, read-&gt;rd_eof);
	*p = cpu_to_be32(read-&gt;rd_length);
	return nfs_ok;
}

static __be32
nfsd4_encode_readlink(struct nfsd4_compoundres *resp, __be32 nfserr, struct nfsd4_readlink *readlink)
{
	__be32 *p, *maxcount_p, zero = xdr_zero;
	struct xdr_stream *xdr = resp-&gt;xdr;
	int length_offset = xdr-&gt;buf-&gt;len;
	int maxcount, status;

	maxcount_p = xdr_reserve_space(xdr, XDR_UNIT);
	if (!maxcount_p)
		return nfserr_resource;
	maxcount = PAGE_SIZE;

	p = xdr_reserve_space(xdr, maxcount);
	if (!p)
		return nfserr_resource;
	/*
	 * XXX: By default, vfs_readlink() will truncate symlinks if they
	 * would overflow the buffer.  Is this kosher in NFSv4?  If not, one
	 * easy fix is: if vfs_readlink() precisely fills the buffer, assume
	 * that truncation occurred, and return NFS4ERR_RESOURCE.
	 */
	nfserr = nfsd_readlink(readlink-&gt;rl_rqstp, readlink-&gt;rl_fhp,
						(char *)p, &amp;maxcount);
	if (nfserr == nfserr_isdir)
		nfserr = nfserr_inval;
	if (nfserr)
		goto out_err;
	status = svc_encode_result_payload(readlink-&gt;rl_rqstp, length_offset,
					   maxcount);
	if (status) {
		nfserr = nfserrno(status);
		goto out_err;
	}
	*maxcount_p = cpu_to_be32(maxcount);
	xdr_truncate_encode(xdr, length_offset + 4 + xdr_align_size(maxcount));
	write_bytes_to_xdr_buf(xdr-&gt;buf, length_offset + 4 + maxcount, &amp;zero,
			       xdr_pad_size(maxcount));
	return nfs_ok;

out_err:
	xdr_truncate_encode(xdr, length_offset);
	return nfserr;
}

static __be32
nfsd4_encode_readdir(struct nfsd4_compoundres *resp, __be32 nfserr, struct nfsd4_readdir *readdir)
{
	int maxcount;
	int bytes_left;
	loff_t offset;
	__be64 wire_offset;
	struct xdr_stream *xdr = resp-&gt;xdr;
	int starting_len = xdr-&gt;buf-&gt;len;
	__be32 *p;

	p = xdr_reserve_space(xdr, NFS4_VERIFIER_SIZE);
	if (!p)
		return nfserr_resource;

	/* XXX: Following NFSv3, we ignore the READDIR verifier for now. */
	*p++ = cpu_to_be32(0);
	*p++ = cpu_to_be32(0);
	xdr-&gt;buf-&gt;head[0].iov_len = (char *)xdr-&gt;p -
				    (char *)xdr-&gt;buf-&gt;head[0].iov_base;

	/*
	 * Number of bytes left for directory entries allowing for the
	 * final 8 bytes of the readdir and a following failed op:
	 */
	bytes_left = xdr-&gt;buf-&gt;buflen - xdr-&gt;buf-&gt;len
			- COMPOUND_ERR_SLACK_SPACE - 8;
	if (bytes_left &lt; 0) {
		nfserr = nfserr_resource;
		goto err_no_verf;
	}
	maxcount = svc_max_payload(resp-&gt;rqstp);
	maxcount = min_t(u32, readdir-&gt;rd_maxcount, maxcount);
	/*
	 * Note the rfc defines rd_maxcount as the size of the
	 * READDIR4resok structure, which includes the verifier above
	 * and the 8 bytes encoded at the end of this function:
	 */
	if (maxcount &lt; 16) {
		nfserr = nfserr_toosmall;
		goto err_no_verf;
	}
	maxcount = min_t(int, maxcount-16, bytes_left);

	/* RFC 3530 14.2.24 allows us to ignore dircount when it's 0: */
	if (!readdir-&gt;rd_dircount)
		readdir-&gt;rd_dircount = svc_max_payload(resp-&gt;rqstp);

	readdir-&gt;xdr = xdr;
	readdir-&gt;rd_maxcount = maxcount;
	readdir-&gt;common.err = 0;
	readdir-&gt;cookie_offset = 0;

	offset = readdir-&gt;rd_cookie;
	nfserr = nfsd_readdir(readdir-&gt;rd_rqstp, readdir-&gt;rd_fhp,
			      &amp;offset,
			      &amp;readdir-&gt;common, nfsd4_encode_dirent);
	if (nfserr == nfs_ok &amp;&amp;
	    readdir-&gt;common.err == nfserr_toosmall &amp;&amp;
	    xdr-&gt;buf-&gt;len == starting_len + 8) {
		/* nothing encoded; which limit did we hit?: */
		if (maxcount - 16 &lt; bytes_left)
			/* It was the fault of rd_maxcount: */
			nfserr = nfserr_toosmall;
		else
			/* We ran out of buffer space: */
			nfserr = nfserr_resource;
	}
	if (nfserr)
		goto err_no_verf;

	if (readdir-&gt;cookie_offset) {
		wire_offset = cpu_to_be64(offset);
		write_bytes_to_xdr_buf(xdr-&gt;buf, readdir-&gt;cookie_offset,
							&amp;wire_offset, 8);
	}

	p = xdr_reserve_space(xdr, 8);
	if (!p) {
		WARN_ON_ONCE(1);
		goto err_no_verf;
	}
	*p++ = 0;	/* no more entries */
	*p++ = htonl(readdir-&gt;common.err == nfserr_eof);

	return 0;
err_no_verf:
	xdr_truncate_encode(xdr, starting_len);
	return nfserr;
}

static __be32
nfsd4_encode_remove(struct nfsd4_compoundres *resp, __be32 nfserr, struct nfsd4_remove *remove)
{
	struct xdr_stream *xdr = resp-&gt;xdr;
	__be32 *p;

	p = xdr_reserve_space(xdr, 20);
	if (!p)
		return nfserr_resource;
	p = encode_cinfo(p, &amp;remove-&gt;rm_cinfo);
	return 0;
}

static __be32
nfsd4_encode_rename(struct nfsd4_compoundres *resp, __be32 nfserr, struct nfsd4_rename *rename)
{
	struct xdr_stream *xdr = resp-&gt;xdr;
	__be32 *p;

	p = xdr_reserve_space(xdr, 40);
	if (!p)
		return nfserr_resource;
	p = encode_cinfo(p, &amp;rename-&gt;rn_sinfo);
	p = encode_cinfo(p, &amp;rename-&gt;rn_tinfo);
	return 0;
}

static __be32
nfsd4_do_encode_secinfo(struct xdr_stream *xdr, struct svc_export *exp)
{
	u32 i, nflavs, supported;
	struct exp_flavor_info *flavs;
	struct exp_flavor_info def_flavs[2];
	__be32 *p, *flavorsp;
	static bool report = true;

	if (exp-&gt;ex_nflavors) {
		flavs = exp-&gt;ex_flavors;
		nflavs = exp-&gt;ex_nflavors;
	} else { /* Handling of some defaults in absence of real secinfo: */
		flavs = def_flavs;
		if (exp-&gt;ex_client-&gt;flavour-&gt;flavour == RPC_AUTH_UNIX) {
			nflavs = 2;
			flavs[0].pseudoflavor = RPC_AUTH_UNIX;
			flavs[1].pseudoflavor = RPC_AUTH_NULL;
		} else if (exp-&gt;ex_client-&gt;flavour-&gt;flavour == RPC_AUTH_GSS) {
			nflavs = 1;
			flavs[0].pseudoflavor
					= svcauth_gss_flavor(exp-&gt;ex_client);
		} else {
			nflavs = 1;
			flavs[0].pseudoflavor
					= exp-&gt;ex_client-&gt;flavour-&gt;flavour;
		}
	}

	supported = 0;
	p = xdr_reserve_space(xdr, 4);
	if (!p)
		return nfserr_resource;
	flavorsp = p++;		/* to be backfilled later */

	for (i = 0; i &lt; nflavs; i++) {
		rpc_authflavor_t pf = flavs[i].pseudoflavor;
		struct rpcsec_gss_info info;

		if (rpcauth_get_gssinfo(pf, &amp;info) == 0) {
			supported++;
			p = xdr_reserve_space(xdr, 4 + 4 +
					      XDR_LEN(info.oid.len) + 4 + 4);
			if (!p)
				return nfserr_resource;
			*p++ = cpu_to_be32(RPC_AUTH_GSS);
			p = xdr_encode_opaque(p,  info.oid.data, info.oid.len);
			*p++ = cpu_to_be32(info.qop);
			*p++ = cpu_to_be32(info.service);
		} else if (pf &lt; RPC_AUTH_MAXFLAVOR) {
			supported++;
			p = xdr_reserve_space(xdr, 4);
			if (!p)
				return nfserr_resource;
			*p++ = cpu_to_be32(pf);
		} else {
			if (report)
				pr_warn("NFS: SECINFO: security flavor %u "
					"is not supported\n", pf);
		}
	}

	if (nflavs != supported)
		report = false;
	*flavorsp = htonl(supported);
	return 0;
}

static __be32
nfsd4_encode_secinfo(struct nfsd4_compoundres *resp, __be32 nfserr,
		     struct nfsd4_secinfo *secinfo)
{
	struct xdr_stream *xdr = resp-&gt;xdr;

	return nfsd4_do_encode_secinfo(xdr, secinfo-&gt;si_exp);
}

static __be32
nfsd4_encode_secinfo_no_name(struct nfsd4_compoundres *resp, __be32 nfserr,
		     struct nfsd4_secinfo_no_name *secinfo)
{
	struct xdr_stream *xdr = resp-&gt;xdr;

	return nfsd4_do_encode_secinfo(xdr, secinfo-&gt;sin_exp);
}

/*
 * The SETATTR encode routine is special -- it always encodes a bitmap,
 * regardless of the error status.
 */
static __be32
nfsd4_encode_setattr(struct nfsd4_compoundres *resp, __be32 nfserr, struct nfsd4_setattr *setattr)
{
	struct xdr_stream *xdr = resp-&gt;xdr;
	__be32 *p;

	p = xdr_reserve_space(xdr, 16);
	if (!p)
		return nfserr_resource;
	if (nfserr) {
		*p++ = cpu_to_be32(3);
		*p++ = cpu_to_be32(0);
		*p++ = cpu_to_be32(0);
		*p++ = cpu_to_be32(0);
	}
	else {
		*p++ = cpu_to_be32(3);
		*p++ = cpu_to_be32(setattr-&gt;sa_bmval[0]);
		*p++ = cpu_to_be32(setattr-&gt;sa_bmval[1]);
		*p++ = cpu_to_be32(setattr-&gt;sa_bmval[2]);
	}
	return nfserr;
}

static __be32
nfsd4_encode_setclientid(struct nfsd4_compoundres *resp, __be32 nfserr, struct nfsd4_setclientid *scd)
{
	struct xdr_stream *xdr = resp-&gt;xdr;
	__be32 *p;

	if (!nfserr) {
		p = xdr_reserve_space(xdr, 8 + NFS4_VERIFIER_SIZE);
		if (!p)
			return nfserr_resource;
		p = xdr_encode_opaque_fixed(p, &amp;scd-&gt;se_clientid, 8);
		p = xdr_encode_opaque_fixed(p, &amp;scd-&gt;se_confirm,
						NFS4_VERIFIER_SIZE);
	}
	else if (nfserr == nfserr_clid_inuse) {
		p = xdr_reserve_space(xdr, 8);
		if (!p)
			return nfserr_resource;
		*p++ = cpu_to_be32(0);
		*p++ = cpu_to_be32(0);
	}
	return nfserr;
}

static __be32
nfsd4_encode_write(struct nfsd4_compoundres *resp, __be32 nfserr, struct nfsd4_write *write)
{
	struct xdr_stream *xdr = resp-&gt;xdr;
	__be32 *p;

	p = xdr_reserve_space(xdr, 16);
	if (!p)
		return nfserr_resource;
	*p++ = cpu_to_be32(write-&gt;wr_bytes_written);
	*p++ = cpu_to_be32(write-&gt;wr_how_written);
	p = xdr_encode_opaque_fixed(p, write-&gt;wr_verifier.data,
						NFS4_VERIFIER_SIZE);
	return 0;
}

static __be32
nfsd4_encode_exchange_id(struct nfsd4_compoundres *resp, __be32 nfserr,
			 struct nfsd4_exchange_id *exid)
{
	struct xdr_stream *xdr = resp-&gt;xdr;
	__be32 *p;
	char *major_id;
	char *server_scope;
	int major_id_sz;
	int server_scope_sz;
	uint64_t minor_id = 0;
	struct nfsd_net *nn = net_generic(SVC_NET(resp-&gt;rqstp), nfsd_net_id);

	major_id = nn-&gt;nfsd_name;
	major_id_sz = strlen(nn-&gt;nfsd_name);
	server_scope = nn-&gt;nfsd_name;
	server_scope_sz = strlen(nn-&gt;nfsd_name);

	p = xdr_reserve_space(xdr,
		8 /* eir_clientid */ +
		4 /* eir_sequenceid */ +
		4 /* eir_flags */ +
		4 /* spr_how */);
	if (!p)
		return nfserr_resource;

	p = xdr_encode_opaque_fixed(p, &amp;exid-&gt;clientid, 8);
	*p++ = cpu_to_be32(exid-&gt;seqid);
	*p++ = cpu_to_be32(exid-&gt;flags);

	*p++ = cpu_to_be32(exid-&gt;spa_how);

	switch (exid-&gt;spa_how) {
	case SP4_NONE:
		break;
	case SP4_MACH_CRED:
		/* spo_must_enforce bitmap: */
		nfserr = nfsd4_encode_bitmap(xdr,
					exid-&gt;spo_must_enforce[0],
					exid-&gt;spo_must_enforce[1],
					exid-&gt;spo_must_enforce[2]);
		if (nfserr)
			return nfserr;
		/* spo_must_allow bitmap: */
		nfserr = nfsd4_encode_bitmap(xdr,
					exid-&gt;spo_must_allow[0],
					exid-&gt;spo_must_allow[1],
					exid-&gt;spo_must_allow[2]);
		if (nfserr)
			return nfserr;
		break;
	default:
		WARN_ON_ONCE(1);
	}

	p = xdr_reserve_space(xdr,
		8 /* so_minor_id */ +
		4 /* so_major_id.len */ +
		(XDR_QUADLEN(major_id_sz) * 4) +
		4 /* eir_server_scope.len */ +
		(XDR_QUADLEN(server_scope_sz) * 4) +
		4 /* eir_server_impl_id.count (0) */);
	if (!p)
		return nfserr_resource;

	/* The server_owner struct */
	p = xdr_encode_hyper(p, minor_id);      /* Minor id */
	/* major id */
	p = xdr_encode_opaque(p, major_id, major_id_sz);

	/* Server scope */
	p = xdr_encode_opaque(p, server_scope, server_scope_sz);

	/* Implementation id */
	*p++ = cpu_to_be32(0);	/* zero length nfs_impl_id4 array */
	return 0;
}

static __be32
nfsd4_encode_create_session(struct nfsd4_compoundres *resp, __be32 nfserr,
			    struct nfsd4_create_session *sess)
{
	struct xdr_stream *xdr = resp-&gt;xdr;
	__be32 *p;

	p = xdr_reserve_space(xdr, 24);
	if (!p)
		return nfserr_resource;
	p = xdr_encode_opaque_fixed(p, sess-&gt;sessionid.data,
					NFS4_MAX_SESSIONID_LEN);
	*p++ = cpu_to_be32(sess-&gt;seqid);
	*p++ = cpu_to_be32(sess-&gt;flags);

	p = xdr_reserve_space(xdr, 28);
	if (!p)
		return nfserr_resource;
	*p++ = cpu_to_be32(0); /* headerpadsz */
	*p++ = cpu_to_be32(sess-&gt;fore_channel.maxreq_sz);
	*p++ = cpu_to_be32(sess-&gt;fore_channel.maxresp_sz);
	*p++ = cpu_to_be32(sess-&gt;fore_channel.maxresp_cached);
	*p++ = cpu_to_be32(sess-&gt;fore_channel.maxops);
	*p++ = cpu_to_be32(sess-&gt;fore_channel.maxreqs);
	*p++ = cpu_to_be32(sess-&gt;fore_channel.nr_rdma_attrs);

	if (sess-&gt;fore_channel.nr_rdma_attrs) {
		p = xdr_reserve_space(xdr, 4);
		if (!p)
			return nfserr_resource;
		*p++ = cpu_to_be32(sess-&gt;fore_channel.rdma_attrs);
	}

	p = xdr_reserve_space(xdr, 28);
	if (!p)
		return nfserr_resource;
	*p++ = cpu_to_be32(0); /* headerpadsz */
	*p++ = cpu_to_be32(sess-&gt;back_channel.maxreq_sz);
	*p++ = cpu_to_be32(sess-&gt;back_channel.maxresp_sz);
	*p++ = cpu_to_be32(sess-&gt;back_channel.maxresp_cached);
	*p++ = cpu_to_be32(sess-&gt;back_channel.maxops);
	*p++ = cpu_to_be32(sess-&gt;back_channel.maxreqs);
	*p++ = cpu_to_be32(sess-&gt;back_channel.nr_rdma_attrs);

	if (sess-&gt;back_channel.nr_rdma_attrs) {
		p = xdr_reserve_space(xdr, 4);
		if (!p)
			return nfserr_resource;
		*p++ = cpu_to_be32(sess-&gt;back_channel.rdma_attrs);
	}
	return 0;
}

static __be32
nfsd4_encode_sequence(struct nfsd4_compoundres *resp, __be32 nfserr,
		      struct nfsd4_sequence *seq)
{
	struct xdr_stream *xdr = resp-&gt;xdr;
	__be32 *p;

	p = xdr_reserve_space(xdr, NFS4_MAX_SESSIONID_LEN + 20);
	if (!p)
		return nfserr_resource;
	p = xdr_encode_opaque_fixed(p, seq-&gt;sessionid.data,
					NFS4_MAX_SESSIONID_LEN);
	*p++ = cpu_to_be32(seq-&gt;seqid);
	*p++ = cpu_to_be32(seq-&gt;slotid);
	/* Note slotid's are numbered from zero: */
	*p++ = cpu_to_be32(seq-&gt;maxslots - 1); /* sr_highest_slotid */
	*p++ = cpu_to_be32(seq-&gt;maxslots - 1); /* sr_target_highest_slotid */
	*p++ = cpu_to_be32(seq-&gt;status_flags);

	resp-&gt;cstate.data_offset = xdr-&gt;buf-&gt;len; /* DRC cache data pointer */
	return 0;
}

static __be32
nfsd4_encode_test_stateid(struct nfsd4_compoundres *resp, __be32 nfserr,
			  struct nfsd4_test_stateid *test_stateid)
{
	struct xdr_stream *xdr = resp-&gt;xdr;
	struct nfsd4_test_stateid_id *stateid, *next;
	__be32 *p;

	p = xdr_reserve_space(xdr, 4 + (4 * test_stateid-&gt;ts_num_ids));
	if (!p)
		return nfserr_resource;
	*p++ = htonl(test_stateid-&gt;ts_num_ids);

	list_for_each_entry_safe(stateid, next, &amp;test_stateid-&gt;ts_stateid_list, ts_id_list) {
		*p++ = stateid-&gt;ts_id_status;
	}

	return 0;
}

#ifdef CONFIG_NFSD_PNFS
static __be32
nfsd4_encode_getdeviceinfo(struct nfsd4_compoundres *resp, __be32 nfserr,
		struct nfsd4_getdeviceinfo *gdev)
{
	struct xdr_stream *xdr = resp-&gt;xdr;
	const struct nfsd4_layout_ops *ops;
	u32 starting_len = xdr-&gt;buf-&gt;len, needed_len;
	__be32 *p;

	p = xdr_reserve_space(xdr, 4);
	if (!p)
		return nfserr_resource;

	*p++ = cpu_to_be32(gdev-&gt;gd_layout_type);

	/* If maxcount is 0 then just update notifications */
	if (gdev-&gt;gd_maxcount != 0) {
		ops = nfsd4_layout_ops[gdev-&gt;gd_layout_type];
		nfserr = ops-&gt;encode_getdeviceinfo(xdr, gdev);
		if (nfserr) {
			/*
			 * We don't bother to burden the layout drivers with
			 * enforcing gd_maxcount, just tell the client to
			 * come back with a bigger buffer if it's not enough.
			 */
			if (xdr-&gt;buf-&gt;len + 4 &gt; gdev-&gt;gd_maxcount)
				goto toosmall;
			return nfserr;
		}
	}

	if (gdev-&gt;gd_notify_types) {
		p = xdr_reserve_space(xdr, 4 + 4);
		if (!p)
			return nfserr_resource;
		*p++ = cpu_to_be32(1);			/* bitmap length */
		*p++ = cpu_to_be32(gdev-&gt;gd_notify_types);
	} else {
		p = xdr_reserve_space(xdr, 4);
		if (!p)
			return nfserr_resource;
		*p++ = 0;
	}

	return 0;
toosmall:
	dprintk("%s: maxcount too small\n", __func__);
	needed_len = xdr-&gt;buf-&gt;len + 4 /* notifications */;
	xdr_truncate_encode(xdr, starting_len);
	p = xdr_reserve_space(xdr, 4);
	if (!p)
		return nfserr_resource;
	*p++ = cpu_to_be32(needed_len);
	return nfserr_toosmall;
}

static __be32
nfsd4_encode_layoutget(struct nfsd4_compoundres *resp, __be32 nfserr,
		struct nfsd4_layoutget *lgp)
{
	struct xdr_stream *xdr = resp-&gt;xdr;
	const struct nfsd4_layout_ops *ops;
	__be32 *p;

	p = xdr_reserve_space(xdr, 36 + sizeof(stateid_opaque_t));
	if (!p)
		return nfserr_resource;

	*p++ = cpu_to_be32(1);	/* we always set return-on-close */
	*p++ = cpu_to_be32(lgp-&gt;lg_sid.si_generation);
	p = xdr_encode_opaque_fixed(p, &amp;lgp-&gt;lg_sid.si_opaque,
				    sizeof(stateid_opaque_t));

	*p++ = cpu_to_be32(1);	/* we always return a single layout */
	p = xdr_encode_hyper(p, lgp-&gt;lg_seg.offset);
	p = xdr_encode_hyper(p, lgp-&gt;lg_seg.length);
	*p++ = cpu_to_be32(lgp-&gt;lg_seg.iomode);
	*p++ = cpu_to_be32(lgp-&gt;lg_layout_type);

	ops = nfsd4_layout_ops[lgp-&gt;lg_layout_type];
	return ops-&gt;encode_layoutget(xdr, lgp);
}

static __be32
nfsd4_encode_layoutcommit(struct nfsd4_compoundres *resp, __be32 nfserr,
			  struct nfsd4_layoutcommit *lcp)
{
	struct xdr_stream *xdr = resp-&gt;xdr;
	__be32 *p;

	p = xdr_reserve_space(xdr, 4);
	if (!p)
		return nfserr_resource;
	*p++ = cpu_to_be32(lcp-&gt;lc_size_chg);
	if (lcp-&gt;lc_size_chg) {
		p = xdr_reserve_space(xdr, 8);
		if (!p)
			return nfserr_resource;
		p = xdr_encode_hyper(p, lcp-&gt;lc_newsize);
	}

	return 0;
}

static __be32
nfsd4_encode_layoutreturn(struct nfsd4_compoundres *resp, __be32 nfserr,
		struct nfsd4_layoutreturn *lrp)
{
	struct xdr_stream *xdr = resp-&gt;xdr;
	__be32 *p;

	p = xdr_reserve_space(xdr, 4);
	if (!p)
		return nfserr_resource;
	*p++ = cpu_to_be32(lrp-&gt;lrs_present);
	if (lrp-&gt;lrs_present)
		return nfsd4_encode_stateid(xdr, &amp;lrp-&gt;lr_sid);
	return 0;
}
#endif /* CONFIG_NFSD_PNFS */

static __be32
nfsd42_encode_write_res(struct nfsd4_compoundres *resp,
		struct nfsd42_write_res *write, bool sync)
{
	__be32 *p;
	p = xdr_reserve_space(resp-&gt;xdr, 4);
	if (!p)
		return nfserr_resource;

	if (sync)
		*p++ = cpu_to_be32(0);
	else {
		__be32 nfserr;
		*p++ = cpu_to_be32(1);
		nfserr = nfsd4_encode_stateid(resp-&gt;xdr, &amp;write-&gt;cb_stateid);
		if (nfserr)
			return nfserr;
	}
	p = xdr_reserve_space(resp-&gt;xdr, 8 + 4 + NFS4_VERIFIER_SIZE);
	if (!p)
		return nfserr_resource;

	p = xdr_encode_hyper(p, write-&gt;wr_bytes_written);
	*p++ = cpu_to_be32(write-&gt;wr_stable_how);
	p = xdr_encode_opaque_fixed(p, write-&gt;wr_verifier.data,
				    NFS4_VERIFIER_SIZE);
	return nfs_ok;
}

static __be32
nfsd42_encode_nl4_server(struct nfsd4_compoundres *resp, struct nl4_server *ns)
{
	struct xdr_stream *xdr = resp-&gt;xdr;
	struct nfs42_netaddr *addr;
	__be32 *p;

	p = xdr_reserve_space(xdr, 4);
	*p++ = cpu_to_be32(ns-&gt;nl4_type);

	switch (ns-&gt;nl4_type) {
	case NL4_NETADDR:
		addr = &amp;ns-&gt;u.nl4_addr;

		/* netid_len, netid, uaddr_len, uaddr (port included
		 * in RPCBIND_MAXUADDRLEN)
		 */
		p = xdr_reserve_space(xdr,
			4 /* netid len */ +
			(XDR_QUADLEN(addr-&gt;netid_len) * 4) +
			4 /* uaddr len */ +
			(XDR_QUADLEN(addr-&gt;addr_len) * 4));
		if (!p)
			return nfserr_resource;

		*p++ = cpu_to_be32(addr-&gt;netid_len);
		p = xdr_encode_opaque_fixed(p, addr-&gt;netid,
					    addr-&gt;netid_len);
		*p++ = cpu_to_be32(addr-&gt;addr_len);
		p = xdr_encode_opaque_fixed(p, addr-&gt;addr,
					addr-&gt;addr_len);
		break;
	default:
		WARN_ON_ONCE(ns-&gt;nl4_type != NL4_NETADDR);
		return nfserr_inval;
	}

	return 0;
}

static __be32
nfsd4_encode_copy(struct nfsd4_compoundres *resp, __be32 nfserr,
		  struct nfsd4_copy *copy)
{
	__be32 *p;

	nfserr = nfsd42_encode_write_res(resp, &amp;copy-&gt;cp_res,
					 nfsd4_copy_is_sync(copy));
	if (nfserr)
		return nfserr;

	p = xdr_reserve_space(resp-&gt;xdr, 4 + 4);
	*p++ = xdr_one; /* cr_consecutive */
	*p = nfsd4_copy_is_sync(copy) ? xdr_one : xdr_zero;
	return 0;
}

static __be32
nfsd4_encode_offload_status(struct nfsd4_compoundres *resp, __be32 nfserr,
			    struct nfsd4_offload_status *os)
{
	struct xdr_stream *xdr = resp-&gt;xdr;
	__be32 *p;

	p = xdr_reserve_space(xdr, 8 + 4);
	if (!p)
		return nfserr_resource;
	p = xdr_encode_hyper(p, os-&gt;count);
	*p++ = cpu_to_be32(0);
	return nfserr;
}

static __be32
nfsd4_encode_read_plus_data(struct nfsd4_compoundres *resp,
			    struct nfsd4_read *read,
			    unsigned long *maxcount, u32 *eof,
			    loff_t *pos)
{
	struct xdr_stream *xdr = resp-&gt;xdr;
	struct file *file = read-&gt;rd_nf-&gt;nf_file;
	int starting_len = xdr-&gt;buf-&gt;len;
	loff_t hole_pos;
	__be32 nfserr;
	__be32 *p, tmp;
	__be64 tmp64;

	hole_pos = pos ? *pos : vfs_llseek(file, read-&gt;rd_offset, SEEK_HOLE);
	if (hole_pos &gt; read-&gt;rd_offset)
		*maxcount = min_t(unsigned long, *maxcount, hole_pos - read-&gt;rd_offset);
	*maxcount = min_t(unsigned long, *maxcount, (xdr-&gt;buf-&gt;buflen - xdr-&gt;buf-&gt;len));

	/* Content type, offset, byte count */
	p = xdr_reserve_space(xdr, 4 + 8 + 4);
	if (!p)
		return nfserr_resource;

	read-&gt;rd_vlen = xdr_reserve_space_vec(xdr, resp-&gt;rqstp-&gt;rq_vec, *maxcount);
	if (read-&gt;rd_vlen &lt; 0)
		return nfserr_resource;

	nfserr = nfsd_readv(resp-&gt;rqstp, read-&gt;rd_fhp, file, read-&gt;rd_offset,
			    resp-&gt;rqstp-&gt;rq_vec, read-&gt;rd_vlen, maxcount, eof);
	if (nfserr)
		return nfserr;
	xdr_truncate_encode(xdr, starting_len + 16 + xdr_align_size(*maxcount));

	tmp = htonl(NFS4_CONTENT_DATA);
	write_bytes_to_xdr_buf(xdr-&gt;buf, starting_len,      &amp;tmp,   4);
	tmp64 = cpu_to_be64(read-&gt;rd_offset);
	write_bytes_to_xdr_buf(xdr-&gt;buf, starting_len + 4,  &amp;tmp64, 8);
	tmp = htonl(*maxcount);
	write_bytes_to_xdr_buf(xdr-&gt;buf, starting_len + 12, &amp;tmp,   4);

	tmp = xdr_zero;
	write_bytes_to_xdr_buf(xdr-&gt;buf, starting_len + 16 + *maxcount, &amp;tmp,
			       xdr_pad_size(*maxcount));
	return nfs_ok;
}

static __be32
nfsd4_encode_read_plus_hole(struct nfsd4_compoundres *resp,
			    struct nfsd4_read *read,
			    unsigned long *maxcount, u32 *eof)
{
	struct file *file = read-&gt;rd_nf-&gt;nf_file;
	loff_t data_pos = vfs_llseek(file, read-&gt;rd_offset, SEEK_DATA);
	loff_t f_size = i_size_read(file_inode(file));
	unsigned long count;
	__be32 *p;

	if (data_pos == -ENXIO)
		data_pos = f_size;
	else if (data_pos &lt;= read-&gt;rd_offset || (data_pos &lt; f_size &amp;&amp; data_pos % PAGE_SIZE))
		return nfsd4_encode_read_plus_data(resp, read, maxcount, eof, &amp;f_size);
	count = data_pos - read-&gt;rd_offset;

	/* Content type, offset, byte count */
	p = xdr_reserve_space(resp-&gt;xdr, 4 + 8 + 8);
	if (!p)
		return nfserr_resource;

	*p++ = htonl(NFS4_CONTENT_HOLE);
	p = xdr_encode_hyper(p, read-&gt;rd_offset);
	p = xdr_encode_hyper(p, count);

	*eof = (read-&gt;rd_offset + count) &gt;= f_size;
	*maxcount = min_t(unsigned long, count, *maxcount);
	return nfs_ok;
}

static __be32
nfsd4_encode_read_plus(struct nfsd4_compoundres *resp, __be32 nfserr,
		       struct nfsd4_read *read)
{
	unsigned long maxcount, count;
	struct xdr_stream *xdr = resp-&gt;xdr;
	struct file *file;
	int starting_len = xdr-&gt;buf-&gt;len;
	int last_segment = xdr-&gt;buf-&gt;len;
	int segments = 0;
	__be32 *p, tmp;
	bool is_data;
	loff_t pos;
	u32 eof;

	if (nfserr)
		return nfserr;
	file = read-&gt;rd_nf-&gt;nf_file;

	/* eof flag, segment count */
	p = xdr_reserve_space(xdr, 4 + 4);
	if (!p)
		return nfserr_resource;
	xdr_commit_encode(xdr);

	maxcount = min_t(unsigned long, read-&gt;rd_length,
			 (xdr-&gt;buf-&gt;buflen - xdr-&gt;buf-&gt;len));
	count    = maxcount;

	eof = read-&gt;rd_offset &gt;= i_size_read(file_inode(file));
	if (eof)
		goto out;

	pos = vfs_llseek(file, read-&gt;rd_offset, SEEK_HOLE);
	is_data = pos &gt; read-&gt;rd_offset;

	while (count &gt; 0 &amp;&amp; !eof) {
		maxcount = count;
		if (is_data)
			nfserr = nfsd4_encode_read_plus_data(resp, read, &amp;maxcount, &amp;eof,
						segments == 0 ? &amp;pos : NULL);
		else
			nfserr = nfsd4_encode_read_plus_hole(resp, read, &amp;maxcount, &amp;eof);
		if (nfserr)
			goto out;
		count -= maxcount;
		read-&gt;rd_offset += maxcount;
		is_data = !is_data;
		last_segment = xdr-&gt;buf-&gt;len;
		segments++;
	}

out:
	if (nfserr &amp;&amp; segments == 0)
		xdr_truncate_encode(xdr, starting_len);
	else {
		if (nfserr) {
			xdr_truncate_encode(xdr, last_segment);
			nfserr = nfs_ok;
			eof = 0;
		}
		tmp = htonl(eof);
		write_bytes_to_xdr_buf(xdr-&gt;buf, starting_len,     &amp;tmp, 4);
		tmp = htonl(segments);
		write_bytes_to_xdr_buf(xdr-&gt;buf, starting_len + 4, &amp;tmp, 4);
	}

	return nfserr;
}

static __be32
nfsd4_encode_copy_notify(struct nfsd4_compoundres *resp, __be32 nfserr,
			 struct nfsd4_copy_notify *cn)
{
	struct xdr_stream *xdr = resp-&gt;xdr;
	__be32 *p;

	if (nfserr)
		return nfserr;

	/* 8 sec, 4 nsec */
	p = xdr_reserve_space(xdr, 12);
	if (!p)
		return nfserr_resource;

	/* cnr_lease_time */
	p = xdr_encode_hyper(p, cn-&gt;cpn_sec);
	*p++ = cpu_to_be32(cn-&gt;cpn_nsec);

	/* cnr_stateid */
	nfserr = nfsd4_encode_stateid(xdr, &amp;cn-&gt;cpn_cnr_stateid);
	if (nfserr)
		return nfserr;

	/* cnr_src.nl_nsvr */
	p = xdr_reserve_space(xdr, 4);
	if (!p)
		return nfserr_resource;

	*p++ = cpu_to_be32(1);

	nfserr = nfsd42_encode_nl4_server(resp, cn-&gt;cpn_src);
	return nfserr;
}

static __be32
nfsd4_encode_seek(struct nfsd4_compoundres *resp, __be32 nfserr,
		  struct nfsd4_seek *seek)
{
	__be32 *p;

	p = xdr_reserve_space(resp-&gt;xdr, 4 + 8);
	*p++ = cpu_to_be32(seek-&gt;seek_eof);
	p = xdr_encode_hyper(p, seek-&gt;seek_pos);

	return 0;
}

static __be32
nfsd4_encode_noop(struct nfsd4_compoundres *resp, __be32 nfserr, void *p)
{
	return nfserr;
}

/*
 * Encode kmalloc-ed buffer in to XDR stream.
 */
static __be32
nfsd4_vbuf_to_stream(struct xdr_stream *xdr, char *buf, u32 buflen)
{
	u32 cplen;
	__be32 *p;

	cplen = min_t(unsigned long, buflen,
		      ((void *)xdr-&gt;end - (void *)xdr-&gt;p));
	p = xdr_reserve_space(xdr, cplen);
	if (!p)
		return nfserr_resource;

	memcpy(p, buf, cplen);
	buf += cplen;
	buflen -= cplen;

	while (buflen) {
		cplen = min_t(u32, buflen, PAGE_SIZE);
		p = xdr_reserve_space(xdr, cplen);
		if (!p)
			return nfserr_resource;

		memcpy(p, buf, cplen);

		if (cplen &lt; PAGE_SIZE) {
			/*
			 * We're done, with a length that wasn't page
			 * aligned, so possibly not word aligned. Pad
			 * any trailing bytes with 0.
			 */
			xdr_encode_opaque_fixed(p, NULL, cplen);
			break;
		}

		buflen -= PAGE_SIZE;
		buf += PAGE_SIZE;
	}

	return 0;
}

static __be32
nfsd4_encode_getxattr(struct nfsd4_compoundres *resp, __be32 nfserr,
		      struct nfsd4_getxattr *getxattr)
{
	struct xdr_stream *xdr = resp-&gt;xdr;
	__be32 *p, err;

	p = xdr_reserve_space(xdr, 4);
	if (!p)
		return nfserr_resource;

	*p = cpu_to_be32(getxattr-&gt;getxa_len);

	if (getxattr-&gt;getxa_len == 0)
		return 0;

	err = nfsd4_vbuf_to_stream(xdr, getxattr-&gt;getxa_buf,
				    getxattr-&gt;getxa_len);

	kvfree(getxattr-&gt;getxa_buf);

	return err;
}

static __be32
nfsd4_encode_setxattr(struct nfsd4_compoundres *resp, __be32 nfserr,
		      struct nfsd4_setxattr *setxattr)
{
	struct xdr_stream *xdr = resp-&gt;xdr;
	__be32 *p;

	p = xdr_reserve_space(xdr, 20);
	if (!p)
		return nfserr_resource;

	encode_cinfo(p, &amp;setxattr-&gt;setxa_cinfo);

	return 0;
}

/*
 * See if there are cookie values that can be rejected outright.
 */
static __be32
nfsd4_listxattr_validate_cookie(struct nfsd4_listxattrs *listxattrs,
				u32 *offsetp)
{
	u64 cookie = listxattrs-&gt;lsxa_cookie;

	/*
	 * If the cookie is larger than the maximum number we can fit
	 * in either the buffer we just got back from vfs_listxattr, or,
	 * XDR-encoded, in the return buffer, it's invalid.
	 */
	if (cookie &gt; (listxattrs-&gt;lsxa_len) / (XATTR_USER_PREFIX_LEN + 2))
		return nfserr_badcookie;

	if (cookie &gt; (listxattrs-&gt;lsxa_maxcount /
		      (XDR_QUADLEN(XATTR_USER_PREFIX_LEN + 2) + 4)))
		return nfserr_badcookie;

	*offsetp = (u32)cookie;
	return 0;
}

static __be32
nfsd4_encode_listxattrs(struct nfsd4_compoundres *resp, __be32 nfserr,
			struct nfsd4_listxattrs *listxattrs)
{
	struct xdr_stream *xdr = resp-&gt;xdr;
	u32 cookie_offset, count_offset, eof;
	u32 left, xdrleft, slen, count;
	u32 xdrlen, offset;
	u64 cookie;
	char *sp;
	__be32 status, tmp;
	__be32 *p;
	u32 nuser;

	eof = 1;

	status = nfsd4_listxattr_validate_cookie(listxattrs, &amp;offset);
	if (status)
		goto out;

	/*
	 * Reserve space for the cookie and the name array count. Record
	 * the offsets to save them later.
	 */
	cookie_offset = xdr-&gt;buf-&gt;len;
	count_offset = cookie_offset + 8;
	p = xdr_reserve_space(xdr, 12);
	if (!p) {
		status = nfserr_resource;
		goto out;
	}

	count = 0;
	left = listxattrs-&gt;lsxa_len;
	sp = listxattrs-&gt;lsxa_buf;
	nuser = 0;

	xdrleft = listxattrs-&gt;lsxa_maxcount;

	while (left &gt; 0 &amp;&amp; xdrleft &gt; 0) {
		slen = strlen(sp);

		/*
		 * Check if this is a "user." attribute, skip it if not.
		 */
		if (strncmp(sp, XATTR_USER_PREFIX, XATTR_USER_PREFIX_LEN))
			goto contloop;

		slen -= XATTR_USER_PREFIX_LEN;
		xdrlen = 4 + ((slen + 3) &amp; ~3);
		if (xdrlen &gt; xdrleft) {
			if (count == 0) {
				/*
				 * Can't even fit the first attribute name.
				 */
				status = nfserr_toosmall;
				goto out;
			}
			eof = 0;
			goto wreof;
		}

		left -= XATTR_USER_PREFIX_LEN;
		sp += XATTR_USER_PREFIX_LEN;
		if (nuser++ &lt; offset)
			goto contloop;


		p = xdr_reserve_space(xdr, xdrlen);
		if (!p) {
			status = nfserr_resource;
			goto out;
		}

		xdr_encode_opaque(p, sp, slen);

		xdrleft -= xdrlen;
		count++;
contloop:
		sp += slen + 1;
		left -= slen + 1;
	}

	/*
	 * If there were user attributes to copy, but we didn't copy
	 * any, the offset was too large (e.g. the cookie was invalid).
	 */
	if (nuser &gt; 0 &amp;&amp; count == 0) {
		status = nfserr_badcookie;
		goto out;
	}

wreof:
	p = xdr_reserve_space(xdr, 4);
	if (!p) {
		status = nfserr_resource;
		goto out;
	}
	*p = cpu_to_be32(eof);

	cookie = offset + count;

	write_bytes_to_xdr_buf(xdr-&gt;buf, cookie_offset, &amp;cookie, 8);
	tmp = cpu_to_be32(count);
	write_bytes_to_xdr_buf(xdr-&gt;buf, count_offset, &amp;tmp, 4);
out:
	if (listxattrs-&gt;lsxa_len)
		kvfree(listxattrs-&gt;lsxa_buf);
	return status;
}

static __be32
nfsd4_encode_removexattr(struct nfsd4_compoundres *resp, __be32 nfserr,
			 struct nfsd4_removexattr *removexattr)
{
	struct xdr_stream *xdr = resp-&gt;xdr;
	__be32 *p;

	p = xdr_reserve_space(xdr, 20);
	if (!p)
		return nfserr_resource;

	p = encode_cinfo(p, &amp;removexattr-&gt;rmxa_cinfo);
	return 0;
}

typedef __be32(* nfsd4_enc)(struct nfsd4_compoundres *, __be32, void *);

/*
 * Note: nfsd4_enc_ops vector is shared for v4.0 and v4.1
 * since we don't need to filter out obsolete ops as this is
 * done in the decoding phase.
 */
static const nfsd4_enc nfsd4_enc_ops[] = {
	[OP_ACCESS]		= (nfsd4_enc)nfsd4_encode_access,
	[OP_CLOSE]		= (nfsd4_enc)nfsd4_encode_close,
	[OP_COMMIT]		= (nfsd4_enc)nfsd4_encode_commit,
	[OP_CREATE]		= (nfsd4_enc)nfsd4_encode_create,
	[OP_DELEGPURGE]		= (nfsd4_enc)nfsd4_encode_noop,
	[OP_DELEGRETURN]	= (nfsd4_enc)nfsd4_encode_noop,
	[OP_GETATTR]		= (nfsd4_enc)nfsd4_encode_getattr,
	[OP_GETFH]		= (nfsd4_enc)nfsd4_encode_getfh,
	[OP_LINK]		= (nfsd4_enc)nfsd4_encode_link,
	[OP_LOCK]		= (nfsd4_enc)nfsd4_encode_lock,
	[OP_LOCKT]		= (nfsd4_enc)nfsd4_encode_lockt,
	[OP_LOCKU]		= (nfsd4_enc)nfsd4_encode_locku,
	[OP_LOOKUP]		= (nfsd4_enc)nfsd4_encode_noop,
	[OP_LOOKUPP]		= (nfsd4_enc)nfsd4_encode_noop,
	[OP_NVERIFY]		= (nfsd4_enc)nfsd4_encode_noop,
	[OP_OPEN]		= (nfsd4_enc)nfsd4_encode_open,
	[OP_OPENATTR]		= (nfsd4_enc)nfsd4_encode_noop,
	[OP_OPEN_CONFIRM]	= (nfsd4_enc)nfsd4_encode_open_confirm,
	[OP_OPEN_DOWNGRADE]	= (nfsd4_enc)nfsd4_encode_open_downgrade,
	[OP_PUTFH]		= (nfsd4_enc)nfsd4_encode_noop,
	[OP_PUTPUBFH]		= (nfsd4_enc)nfsd4_encode_noop,
	[OP_PUTROOTFH]		= (nfsd4_enc)nfsd4_encode_noop,
	[OP_READ]		= (nfsd4_enc)nfsd4_encode_read,
	[OP_READDIR]		= (nfsd4_enc)nfsd4_encode_readdir,
	[OP_READLINK]		= (nfsd4_enc)nfsd4_encode_readlink,
	[OP_REMOVE]		= (nfsd4_enc)nfsd4_encode_remove,
	[OP_RENAME]		= (nfsd4_enc)nfsd4_encode_rename,
	[OP_RENEW]		= (nfsd4_enc)nfsd4_encode_noop,
	[OP_RESTOREFH]		= (nfsd4_enc)nfsd4_encode_noop,
	[OP_SAVEFH]		= (nfsd4_enc)nfsd4_encode_noop,
	[OP_SECINFO]		= (nfsd4_enc)nfsd4_encode_secinfo,
	[OP_SETATTR]		= (nfsd4_enc)nfsd4_encode_setattr,
	[OP_SETCLIENTID]	= (nfsd4_enc)nfsd4_encode_setclientid,
	[OP_SETCLIENTID_CONFIRM] = (nfsd4_enc)nfsd4_encode_noop,
	[OP_VERIFY]		= (nfsd4_enc)nfsd4_encode_noop,
	[OP_WRITE]		= (nfsd4_enc)nfsd4_encode_write,
	[OP_RELEASE_LOCKOWNER]	= (nfsd4_enc)nfsd4_encode_noop,

	/* NFSv4.1 operations */
	[OP_BACKCHANNEL_CTL]	= (nfsd4_enc)nfsd4_encode_noop,
	[OP_BIND_CONN_TO_SESSION] = (nfsd4_enc)nfsd4_encode_bind_conn_to_session,
	[OP_EXCHANGE_ID]	= (nfsd4_enc)nfsd4_encode_exchange_id,
	[OP_CREATE_SESSION]	= (nfsd4_enc)nfsd4_encode_create_session,
	[OP_DESTROY_SESSION]	= (nfsd4_enc)nfsd4_encode_noop,
	[OP_FREE_STATEID]	= (nfsd4_enc)nfsd4_encode_noop,
	[OP_GET_DIR_DELEGATION]	= (nfsd4_enc)nfsd4_encode_noop,
#ifdef CONFIG_NFSD_PNFS
	[OP_GETDEVICEINFO]	= (nfsd4_enc)nfsd4_encode_getdeviceinfo,
	[OP_GETDEVICELIST]	= (nfsd4_enc)nfsd4_encode_noop,
	[OP_LAYOUTCOMMIT]	= (nfsd4_enc)nfsd4_encode_layoutcommit,
	[OP_LAYOUTGET]		= (nfsd4_enc)nfsd4_encode_layoutget,
	[OP_LAYOUTRETURN]	= (nfsd4_enc)nfsd4_encode_layoutreturn,
#else
	[OP_GETDEVICEINFO]	= (nfsd4_enc)nfsd4_encode_noop,
	[OP_GETDEVICELIST]	= (nfsd4_enc)nfsd4_encode_noop,
	[OP_LAYOUTCOMMIT]	= (nfsd4_enc)nfsd4_encode_noop,
	[OP_LAYOUTGET]		= (nfsd4_enc)nfsd4_encode_noop,
	[OP_LAYOUTRETURN]	= (nfsd4_enc)nfsd4_encode_noop,
#endif
	[OP_SECINFO_NO_NAME]	= (nfsd4_enc)nfsd4_encode_secinfo_no_name,
	[OP_SEQUENCE]		= (nfsd4_enc)nfsd4_encode_sequence,
	[OP_SET_SSV]		= (nfsd4_enc)nfsd4_encode_noop,
	[OP_TEST_STATEID]	= (nfsd4_enc)nfsd4_encode_test_stateid,
	[OP_WANT_DELEGATION]	= (nfsd4_enc)nfsd4_encode_noop,
	[OP_DESTROY_CLIENTID]	= (nfsd4_enc)nfsd4_encode_noop,
	[OP_RECLAIM_COMPLETE]	= (nfsd4_enc)nfsd4_encode_noop,

	/* NFSv4.2 operations */
	[OP_ALLOCATE]		= (nfsd4_enc)nfsd4_encode_noop,
	[OP_COPY]		= (nfsd4_enc)nfsd4_encode_copy,
	[OP_COPY_NOTIFY]	= (nfsd4_enc)nfsd4_encode_copy_notify,
	[OP_DEALLOCATE]		= (nfsd4_enc)nfsd4_encode_noop,
	[OP_IO_ADVISE]		= (nfsd4_enc)nfsd4_encode_noop,
	[OP_LAYOUTERROR]	= (nfsd4_enc)nfsd4_encode_noop,
	[OP_LAYOUTSTATS]	= (nfsd4_enc)nfsd4_encode_noop,
	[OP_OFFLOAD_CANCEL]	= (nfsd4_enc)nfsd4_encode_noop,
	[OP_OFFLOAD_STATUS]	= (nfsd4_enc)nfsd4_encode_offload_status,
	[OP_READ_PLUS]		= (nfsd4_enc)nfsd4_encode_read_plus,
	[OP_SEEK]		= (nfsd4_enc)nfsd4_encode_seek,
	[OP_WRITE_SAME]		= (nfsd4_enc)nfsd4_encode_noop,
	[OP_CLONE]		= (nfsd4_enc)nfsd4_encode_noop,

	/* RFC 8276 extended atributes operations */
	[OP_GETXATTR]		= (nfsd4_enc)nfsd4_encode_getxattr,
	[OP_SETXATTR]		= (nfsd4_enc)nfsd4_encode_setxattr,
	[OP_LISTXATTRS]		= (nfsd4_enc)nfsd4_encode_listxattrs,
	[OP_REMOVEXATTR]	= (nfsd4_enc)nfsd4_encode_removexattr,
};

/*
 * Calculate whether we still have space to encode repsize bytes.
 * There are two considerations:
 *     - For NFS versions &gt;=4.1, the size of the reply must stay within
 *       session limits
 *     - For all NFS versions, we must stay within limited preallocated
 *       buffer space.
 *
 * This is called before the operation is processed, so can only provide
 * an upper estimate.  For some nonidempotent operations (such as
 * getattr), it's not necessarily a problem if that estimate is wrong,
 * as we can fail it after processing without significant side effects.
 */
__be32 nfsd4_check_resp_size(struct nfsd4_compoundres *resp, u32 respsize)
{
	struct xdr_buf *buf = &amp;resp-&gt;rqstp-&gt;rq_res;
	struct nfsd4_slot *slot = resp-&gt;cstate.slot;

	if (buf-&gt;len + respsize &lt;= buf-&gt;buflen)
		return nfs_ok;
	if (!nfsd4_has_session(&amp;resp-&gt;cstate))
		return nfserr_resource;
	if (slot-&gt;sl_flags &amp; NFSD4_SLOT_CACHETHIS) {
		WARN_ON_ONCE(1);
		return nfserr_rep_too_big_to_cache;
	}
	return nfserr_rep_too_big;
}

void
nfsd4_encode_operation(struct nfsd4_compoundres *resp, struct nfsd4_op *op)
{
	struct xdr_stream *xdr = resp-&gt;xdr;
	struct nfs4_stateowner *so = resp-&gt;cstate.replay_owner;
	struct svc_rqst *rqstp = resp-&gt;rqstp;
	const struct nfsd4_operation *opdesc = op-&gt;opdesc;
	int post_err_offset;
	nfsd4_enc encoder;
	__be32 *p;

	p = xdr_reserve_space(xdr, 8);
	if (!p) {
		WARN_ON_ONCE(1);
		return;
	}
	*p++ = cpu_to_be32(op-&gt;opnum);
	post_err_offset = xdr-&gt;buf-&gt;len;

	if (op-&gt;opnum == OP_ILLEGAL)
		goto status;
	if (op-&gt;status &amp;&amp; opdesc &amp;&amp;
			!(opdesc-&gt;op_flags &amp; OP_NONTRIVIAL_ERROR_ENCODE))
		goto status;
	BUG_ON(op-&gt;opnum &gt;= ARRAY_SIZE(nfsd4_enc_ops) ||
	       !nfsd4_enc_ops[op-&gt;opnum]);
	encoder = nfsd4_enc_ops[op-&gt;opnum];
	op-&gt;status = encoder(resp, op-&gt;status, &amp;op-&gt;u);
	if (op-&gt;status)
		trace_nfsd_compound_encode_err(rqstp, op-&gt;opnum, op-&gt;status);
	if (opdesc &amp;&amp; opdesc-&gt;op_release)
		opdesc-&gt;op_release(&amp;op-&gt;u);
	xdr_commit_encode(xdr);

	/* nfsd4_check_resp_size guarantees enough room for error status */
	if (!op-&gt;status) {
		int space_needed = 0;
		if (!nfsd4_last_compound_op(rqstp))
			space_needed = COMPOUND_ERR_SLACK_SPACE;
		op-&gt;status = nfsd4_check_resp_size(resp, space_needed);
	}
	if (op-&gt;status == nfserr_resource &amp;&amp; nfsd4_has_session(&amp;resp-&gt;cstate)) {
		struct nfsd4_slot *slot = resp-&gt;cstate.slot;

		if (slot-&gt;sl_flags &amp; NFSD4_SLOT_CACHETHIS)
			op-&gt;status = nfserr_rep_too_big_to_cache;
		else
			op-&gt;status = nfserr_rep_too_big;
	}
	if (op-&gt;status == nfserr_resource ||
	    op-&gt;status == nfserr_rep_too_big ||
	    op-&gt;status == nfserr_rep_too_big_to_cache) {
		/*
		 * The operation may have already been encoded or
		 * partially encoded.  No op returns anything additional
		 * in the case of one of these three errors, so we can
		 * just truncate back to after the status.  But it's a
		 * bug if we had to do this on a non-idempotent op:
		 */
		warn_on_nonidempotent_op(op);
		xdr_truncate_encode(xdr, post_err_offset);
	}
	if (so) {
		int len = xdr-&gt;buf-&gt;len - post_err_offset;

		so-&gt;so_replay.rp_status = op-&gt;status;
		so-&gt;so_replay.rp_buflen = len;
		read_bytes_from_xdr_buf(xdr-&gt;buf, post_err_offset,
						so-&gt;so_replay.rp_buf, len);
	}
status:
	*p = op-&gt;status;
}

/* 
 * Encode the reply stored in the stateowner reply cache 
 * 
 * XDR note: do not encode rp-&gt;rp_buflen: the buffer contains the
 * previously sent already encoded operation.
 */
void
nfsd4_encode_replay(struct xdr_stream *xdr, struct nfsd4_op *op)
{
	__be32 *p;
	struct nfs4_replay *rp = op-&gt;replay;

	p = xdr_reserve_space(xdr, 8 + rp-&gt;rp_buflen);
	if (!p) {
		WARN_ON_ONCE(1);
		return;
	}
	*p++ = cpu_to_be32(op-&gt;opnum);
	*p++ = rp-&gt;rp_status;  /* already xdr'ed */

	p = xdr_encode_opaque_fixed(p, rp-&gt;rp_buf, rp-&gt;rp_buflen);
}

void nfsd4_release_compoundargs(struct svc_rqst *rqstp)
{
	struct nfsd4_compoundargs *args = rqstp-&gt;rq_argp;

	if (args-&gt;ops != args-&gt;iops) {
		vfree(args-&gt;ops);
		args-&gt;ops = args-&gt;iops;
	}
	while (args-&gt;to_free) {
		struct svcxdr_tmpbuf *tb = args-&gt;to_free;
		args-&gt;to_free = tb-&gt;next;
		kfree(tb);
	}
}

bool
nfs4svc_decode_compoundargs(struct svc_rqst *rqstp, struct xdr_stream *xdr)
{
	struct nfsd4_compoundargs *args = rqstp-&gt;rq_argp;

	/* svcxdr_tmp_alloc */
	args-&gt;to_free = NULL;

	args-&gt;xdr = xdr;
	args-&gt;ops = args-&gt;iops;
	args-&gt;rqstp = rqstp;

	return nfsd4_decode_compound(args);
}

bool
nfs4svc_encode_compoundres(struct svc_rqst *rqstp, struct xdr_stream *xdr)
{
	struct nfsd4_compoundres *resp = rqstp-&gt;rq_resp;
	__be32 *p;

	/*
	 * Send buffer space for the following items is reserved
	 * at the top of nfsd4_proc_compound().
	 */
	p = resp-&gt;statusp;

	*p++ = resp-&gt;cstate.status;

	rqstp-&gt;rq_next_page = xdr-&gt;page_ptr + 1;

	*p++ = htonl(resp-&gt;taglen);
	memcpy(p, resp-&gt;tag, resp-&gt;taglen);
	p += XDR_QUADLEN(resp-&gt;taglen);
	*p++ = htonl(resp-&gt;opcnt);

	nfsd4_sequence_done(resp);
	return true;
}
</fslocs-></linux></uapi></linux></linux></linux></linux></linux></linux></linux></linux></linux></linux></andros@umich.edu></kmsmith@umich.edu>