// SPDX-License-Identifier: GPL-2.0
/*
 * Process version 3 NFS requests.
 *
 * Copyright (C) 1996, 1997, 1998 Olaf Kirch <okir@monad.swb.de>
 */

#include <linux fs.h="">
#include <linux ext2_fs.h="">
#include <linux magic.h="">
#include <linux namei.h="">

#include "cache.h"
#include "xdr3.h"
#include "vfs.h"

#define NFSDDBG_FACILITY		NFSDDBG_PROC

static int	nfs3_ftypes[] = {
	0,			/* NF3NON */
	S_IFREG,		/* NF3REG */
	S_IFDIR,		/* NF3DIR */
	S_IFBLK,		/* NF3BLK */
	S_IFCHR,		/* NF3CHR */
	S_IFLNK,		/* NF3LNK */
	S_IFSOCK,		/* NF3SOCK */
	S_IFIFO,		/* NF3FIFO */
};

/*
 * NULL call.
 */
static __be32
nfsd3_proc_null(struct svc_rqst *rqstp)
{
	return rpc_success;
}

/*
 * Get a file's attributes
 */
static __be32
nfsd3_proc_getattr(struct svc_rqst *rqstp)
{
	struct nfsd_fhandle *argp = rqstp-&gt;rq_argp;
	struct nfsd3_attrstat *resp = rqstp-&gt;rq_resp;

	dprintk("nfsd: GETATTR(3)  %s\n",
		SVCFH_fmt(&amp;argp-&gt;fh));

	fh_copy(&amp;resp-&gt;fh, &amp;argp-&gt;fh);
	resp-&gt;status = fh_verify(rqstp, &amp;resp-&gt;fh, 0,
				 NFSD_MAY_NOP | NFSD_MAY_BYPASS_GSS_ON_ROOT);
	if (resp-&gt;status != nfs_ok)
		goto out;

	resp-&gt;status = fh_getattr(&amp;resp-&gt;fh, &amp;resp-&gt;stat);
out:
	return rpc_success;
}

/*
 * Set a file's attributes
 */
static __be32
nfsd3_proc_setattr(struct svc_rqst *rqstp)
{
	struct nfsd3_sattrargs *argp = rqstp-&gt;rq_argp;
	struct nfsd3_attrstat *resp = rqstp-&gt;rq_resp;
	struct nfsd_attrs attrs = {
		.na_iattr	= &amp;argp-&gt;attrs,
	};

	dprintk("nfsd: SETATTR(3)  %s\n",
				SVCFH_fmt(&amp;argp-&gt;fh));

	fh_copy(&amp;resp-&gt;fh, &amp;argp-&gt;fh);
	resp-&gt;status = nfsd_setattr(rqstp, &amp;resp-&gt;fh, &amp;attrs,
				    argp-&gt;check_guard, argp-&gt;guardtime);
	return rpc_success;
}

/*
 * Look up a path name component
 */
static __be32
nfsd3_proc_lookup(struct svc_rqst *rqstp)
{
	struct nfsd3_diropargs *argp = rqstp-&gt;rq_argp;
	struct nfsd3_diropres  *resp = rqstp-&gt;rq_resp;

	dprintk("nfsd: LOOKUP(3)   %s %.*s\n",
				SVCFH_fmt(&amp;argp-&gt;fh),
				argp-&gt;len,
				argp-&gt;name);

	fh_copy(&amp;resp-&gt;dirfh, &amp;argp-&gt;fh);
	fh_init(&amp;resp-&gt;fh, NFS3_FHSIZE);

	resp-&gt;status = nfsd_lookup(rqstp, &amp;resp-&gt;dirfh,
				   argp-&gt;name, argp-&gt;len,
				   &amp;resp-&gt;fh);
	return rpc_success;
}

/*
 * Check file access
 */
static __be32
nfsd3_proc_access(struct svc_rqst *rqstp)
{
	struct nfsd3_accessargs *argp = rqstp-&gt;rq_argp;
	struct nfsd3_accessres *resp = rqstp-&gt;rq_resp;

	dprintk("nfsd: ACCESS(3)   %s 0x%x\n",
				SVCFH_fmt(&amp;argp-&gt;fh),
				argp-&gt;access);

	fh_copy(&amp;resp-&gt;fh, &amp;argp-&gt;fh);
	resp-&gt;access = argp-&gt;access;
	resp-&gt;status = nfsd_access(rqstp, &amp;resp-&gt;fh, &amp;resp-&gt;access, NULL);
	return rpc_success;
}

/*
 * Read a symlink.
 */
static __be32
nfsd3_proc_readlink(struct svc_rqst *rqstp)
{
	struct nfsd_fhandle *argp = rqstp-&gt;rq_argp;
	struct nfsd3_readlinkres *resp = rqstp-&gt;rq_resp;

	dprintk("nfsd: READLINK(3) %s\n", SVCFH_fmt(&amp;argp-&gt;fh));

	/* Read the symlink. */
	fh_copy(&amp;resp-&gt;fh, &amp;argp-&gt;fh);
	resp-&gt;len = NFS3_MAXPATHLEN;
	resp-&gt;pages = rqstp-&gt;rq_next_page++;
	resp-&gt;status = nfsd_readlink(rqstp, &amp;resp-&gt;fh,
				     page_address(*resp-&gt;pages), &amp;resp-&gt;len);
	return rpc_success;
}

/*
 * Read a portion of a file.
 */
static __be32
nfsd3_proc_read(struct svc_rqst *rqstp)
{
	struct nfsd3_readargs *argp = rqstp-&gt;rq_argp;
	struct nfsd3_readres *resp = rqstp-&gt;rq_resp;
	u32 max_blocksize = svc_max_payload(rqstp);
	unsigned int len;
	int v;

	dprintk("nfsd: READ(3) %s %lu bytes at %Lu\n",
				SVCFH_fmt(&amp;argp-&gt;fh),
				(unsigned long) argp-&gt;count,
				(unsigned long long) argp-&gt;offset);

	argp-&gt;count = min_t(u32, argp-&gt;count, max_blocksize);
	if (argp-&gt;offset &gt; (u64)OFFSET_MAX)
		argp-&gt;offset = (u64)OFFSET_MAX;
	if (argp-&gt;offset + argp-&gt;count &gt; (u64)OFFSET_MAX)
		argp-&gt;count = (u64)OFFSET_MAX - argp-&gt;offset;

	v = 0;
	len = argp-&gt;count;
	resp-&gt;pages = rqstp-&gt;rq_next_page;
	while (len &gt; 0) {
		struct page *page = *(rqstp-&gt;rq_next_page++);

		rqstp-&gt;rq_vec[v].iov_base = page_address(page);
		rqstp-&gt;rq_vec[v].iov_len = min_t(unsigned int, len, PAGE_SIZE);
		len -= rqstp-&gt;rq_vec[v].iov_len;
		v++;
	}

	/* Obtain buffer pointer for payload.
	 * 1 (status) + 22 (post_op_attr) + 1 (count) + 1 (eof)
	 * + 1 (xdr opaque byte count) = 26
	 */
	resp-&gt;count = argp-&gt;count;
	svc_reserve_auth(rqstp, ((1 + NFS3_POST_OP_ATTR_WORDS + 3)&lt;&lt;2) + resp-&gt;count +4);

	fh_copy(&amp;resp-&gt;fh, &amp;argp-&gt;fh);
	resp-&gt;status = nfsd_read(rqstp, &amp;resp-&gt;fh, argp-&gt;offset,
				 rqstp-&gt;rq_vec, v, &amp;resp-&gt;count, &amp;resp-&gt;eof);
	return rpc_success;
}

/*
 * Write data to a file
 */
static __be32
nfsd3_proc_write(struct svc_rqst *rqstp)
{
	struct nfsd3_writeargs *argp = rqstp-&gt;rq_argp;
	struct nfsd3_writeres *resp = rqstp-&gt;rq_resp;
	unsigned long cnt = argp-&gt;len;
	unsigned int nvecs;

	dprintk("nfsd: WRITE(3)    %s %d bytes at %Lu%s\n",
				SVCFH_fmt(&amp;argp-&gt;fh),
				argp-&gt;len,
				(unsigned long long) argp-&gt;offset,
				argp-&gt;stable? " stable" : "");

	resp-&gt;status = nfserr_fbig;
	if (argp-&gt;offset &gt; (u64)OFFSET_MAX ||
	    argp-&gt;offset + argp-&gt;len &gt; (u64)OFFSET_MAX)
		return rpc_success;

	fh_copy(&amp;resp-&gt;fh, &amp;argp-&gt;fh);
	resp-&gt;committed = argp-&gt;stable;
	nvecs = svc_fill_write_vector(rqstp, &amp;argp-&gt;payload);

	resp-&gt;status = nfsd_write(rqstp, &amp;resp-&gt;fh, argp-&gt;offset,
				  rqstp-&gt;rq_vec, nvecs, &amp;cnt,
				  resp-&gt;committed, resp-&gt;verf);
	resp-&gt;count = cnt;
	return rpc_success;
}

/*
 * Implement NFSv3's unchecked, guarded, and exclusive CREATE
 * semantics for regular files. Except for the created file,
 * this operation is stateless on the server.
 *
 * Upon return, caller must release @fhp and @resfhp.
 */
static __be32
nfsd3_create_file(struct svc_rqst *rqstp, struct svc_fh *fhp,
		  struct svc_fh *resfhp, struct nfsd3_createargs *argp)
{
	struct iattr *iap = &amp;argp-&gt;attrs;
	struct dentry *parent, *child;
	struct nfsd_attrs attrs = {
		.na_iattr	= iap,
	};
	__u32 v_mtime, v_atime;
	struct inode *inode;
	__be32 status;
	int host_err;

	if (isdotent(argp-&gt;name, argp-&gt;len))
		return nfserr_exist;
	if (!(iap-&gt;ia_valid &amp; ATTR_MODE))
		iap-&gt;ia_mode = 0;

	status = fh_verify(rqstp, fhp, S_IFDIR, NFSD_MAY_EXEC);
	if (status != nfs_ok)
		return status;

	parent = fhp-&gt;fh_dentry;
	inode = d_inode(parent);

	host_err = fh_want_write(fhp);
	if (host_err)
		return nfserrno(host_err);

	inode_lock_nested(inode, I_MUTEX_PARENT);

	child = lookup_one_len(argp-&gt;name, parent, argp-&gt;len);
	if (IS_ERR(child)) {
		status = nfserrno(PTR_ERR(child));
		goto out;
	}

	if (d_really_is_negative(child)) {
		status = fh_verify(rqstp, fhp, S_IFDIR, NFSD_MAY_CREATE);
		if (status != nfs_ok)
			goto out;
	}

	status = fh_compose(resfhp, fhp-&gt;fh_export, child, fhp);
	if (status != nfs_ok)
		goto out;

	v_mtime = 0;
	v_atime = 0;
	if (argp-&gt;createmode == NFS3_CREATE_EXCLUSIVE) {
		u32 *verifier = (u32 *)argp-&gt;verf;

		/*
		 * Solaris 7 gets confused (bugid 4218508) if these have
		 * the high bit set, as do xfs filesystems without the
		 * "bigtime" feature. So just clear the high bits.
		 */
		v_mtime = verifier[0] &amp; 0x7fffffff;
		v_atime = verifier[1] &amp; 0x7fffffff;
	}

	if (d_really_is_positive(child)) {
		status = nfs_ok;

		switch (argp-&gt;createmode) {
		case NFS3_CREATE_UNCHECKED:
			if (!d_is_reg(child))
				break;
			iap-&gt;ia_valid &amp;= ATTR_SIZE;
			goto set_attr;
		case NFS3_CREATE_GUARDED:
			status = nfserr_exist;
			break;
		case NFS3_CREATE_EXCLUSIVE:
			if (d_inode(child)-&gt;i_mtime.tv_sec == v_mtime &amp;&amp;
			    d_inode(child)-&gt;i_atime.tv_sec == v_atime &amp;&amp;
			    d_inode(child)-&gt;i_size == 0) {
				break;
			}
			status = nfserr_exist;
		}
		goto out;
	}

	if (!IS_POSIXACL(inode))
		iap-&gt;ia_mode &amp;= ~current_umask();

	fh_fill_pre_attrs(fhp);
	host_err = vfs_create(&amp;init_user_ns, inode, child, iap-&gt;ia_mode, true);
	if (host_err &lt; 0) {
		status = nfserrno(host_err);
		goto out;
	}
	fh_fill_post_attrs(fhp);

	/* A newly created file already has a file size of zero. */
	if ((iap-&gt;ia_valid &amp; ATTR_SIZE) &amp;&amp; (iap-&gt;ia_size == 0))
		iap-&gt;ia_valid &amp;= ~ATTR_SIZE;
	if (argp-&gt;createmode == NFS3_CREATE_EXCLUSIVE) {
		iap-&gt;ia_valid = ATTR_MTIME | ATTR_ATIME |
				ATTR_MTIME_SET | ATTR_ATIME_SET;
		iap-&gt;ia_mtime.tv_sec = v_mtime;
		iap-&gt;ia_atime.tv_sec = v_atime;
		iap-&gt;ia_mtime.tv_nsec = 0;
		iap-&gt;ia_atime.tv_nsec = 0;
	}

set_attr:
	status = nfsd_create_setattr(rqstp, fhp, resfhp, &amp;attrs);

out:
	inode_unlock(inode);
	if (child &amp;&amp; !IS_ERR(child))
		dput(child);
	fh_drop_write(fhp);
	return status;
}

static __be32
nfsd3_proc_create(struct svc_rqst *rqstp)
{
	struct nfsd3_createargs *argp = rqstp-&gt;rq_argp;
	struct nfsd3_diropres *resp = rqstp-&gt;rq_resp;
	svc_fh *dirfhp, *newfhp;

	dprintk("nfsd: CREATE(3)   %s %.*s\n",
				SVCFH_fmt(&amp;argp-&gt;fh),
				argp-&gt;len,
				argp-&gt;name);

	dirfhp = fh_copy(&amp;resp-&gt;dirfh, &amp;argp-&gt;fh);
	newfhp = fh_init(&amp;resp-&gt;fh, NFS3_FHSIZE);

	resp-&gt;status = nfsd3_create_file(rqstp, dirfhp, newfhp, argp);
	return rpc_success;
}

/*
 * Make directory. This operation is not idempotent.
 */
static __be32
nfsd3_proc_mkdir(struct svc_rqst *rqstp)
{
	struct nfsd3_createargs *argp = rqstp-&gt;rq_argp;
	struct nfsd3_diropres *resp = rqstp-&gt;rq_resp;
	struct nfsd_attrs attrs = {
		.na_iattr	= &amp;argp-&gt;attrs,
	};

	dprintk("nfsd: MKDIR(3)    %s %.*s\n",
				SVCFH_fmt(&amp;argp-&gt;fh),
				argp-&gt;len,
				argp-&gt;name);

	argp-&gt;attrs.ia_valid &amp;= ~ATTR_SIZE;
	fh_copy(&amp;resp-&gt;dirfh, &amp;argp-&gt;fh);
	fh_init(&amp;resp-&gt;fh, NFS3_FHSIZE);
	resp-&gt;status = nfsd_create(rqstp, &amp;resp-&gt;dirfh, argp-&gt;name, argp-&gt;len,
				   &amp;attrs, S_IFDIR, 0, &amp;resp-&gt;fh);
	return rpc_success;
}

static __be32
nfsd3_proc_symlink(struct svc_rqst *rqstp)
{
	struct nfsd3_symlinkargs *argp = rqstp-&gt;rq_argp;
	struct nfsd3_diropres *resp = rqstp-&gt;rq_resp;
	struct nfsd_attrs attrs = {
		.na_iattr	= &amp;argp-&gt;attrs,
	};

	if (argp-&gt;tlen == 0) {
		resp-&gt;status = nfserr_inval;
		goto out;
	}
	if (argp-&gt;tlen &gt; NFS3_MAXPATHLEN) {
		resp-&gt;status = nfserr_nametoolong;
		goto out;
	}

	argp-&gt;tname = svc_fill_symlink_pathname(rqstp, &amp;argp-&gt;first,
						page_address(rqstp-&gt;rq_arg.pages[0]),
						argp-&gt;tlen);
	if (IS_ERR(argp-&gt;tname)) {
		resp-&gt;status = nfserrno(PTR_ERR(argp-&gt;tname));
		goto out;
	}

	dprintk("nfsd: SYMLINK(3)  %s %.*s -&gt; %.*s\n",
				SVCFH_fmt(&amp;argp-&gt;ffh),
				argp-&gt;flen, argp-&gt;fname,
				argp-&gt;tlen, argp-&gt;tname);

	fh_copy(&amp;resp-&gt;dirfh, &amp;argp-&gt;ffh);
	fh_init(&amp;resp-&gt;fh, NFS3_FHSIZE);
	resp-&gt;status = nfsd_symlink(rqstp, &amp;resp-&gt;dirfh, argp-&gt;fname,
				    argp-&gt;flen, argp-&gt;tname, &amp;attrs, &amp;resp-&gt;fh);
	kfree(argp-&gt;tname);
out:
	return rpc_success;
}

/*
 * Make socket/fifo/device.
 */
static __be32
nfsd3_proc_mknod(struct svc_rqst *rqstp)
{
	struct nfsd3_mknodargs *argp = rqstp-&gt;rq_argp;
	struct nfsd3_diropres  *resp = rqstp-&gt;rq_resp;
	struct nfsd_attrs attrs = {
		.na_iattr	= &amp;argp-&gt;attrs,
	};
	int type;
	dev_t	rdev = 0;

	dprintk("nfsd: MKNOD(3)    %s %.*s\n",
				SVCFH_fmt(&amp;argp-&gt;fh),
				argp-&gt;len,
				argp-&gt;name);

	fh_copy(&amp;resp-&gt;dirfh, &amp;argp-&gt;fh);
	fh_init(&amp;resp-&gt;fh, NFS3_FHSIZE);

	if (argp-&gt;ftype == NF3CHR || argp-&gt;ftype == NF3BLK) {
		rdev = MKDEV(argp-&gt;major, argp-&gt;minor);
		if (MAJOR(rdev) != argp-&gt;major ||
		    MINOR(rdev) != argp-&gt;minor) {
			resp-&gt;status = nfserr_inval;
			goto out;
		}
	} else if (argp-&gt;ftype != NF3SOCK &amp;&amp; argp-&gt;ftype != NF3FIFO) {
		resp-&gt;status = nfserr_badtype;
		goto out;
	}

	type = nfs3_ftypes[argp-&gt;ftype];
	resp-&gt;status = nfsd_create(rqstp, &amp;resp-&gt;dirfh, argp-&gt;name, argp-&gt;len,
				   &amp;attrs, type, rdev, &amp;resp-&gt;fh);
out:
	return rpc_success;
}

/*
 * Remove file/fifo/socket etc.
 */
static __be32
nfsd3_proc_remove(struct svc_rqst *rqstp)
{
	struct nfsd3_diropargs *argp = rqstp-&gt;rq_argp;
	struct nfsd3_attrstat *resp = rqstp-&gt;rq_resp;

	dprintk("nfsd: REMOVE(3)   %s %.*s\n",
				SVCFH_fmt(&amp;argp-&gt;fh),
				argp-&gt;len,
				argp-&gt;name);

	/* Unlink. -S_IFDIR means file must not be a directory */
	fh_copy(&amp;resp-&gt;fh, &amp;argp-&gt;fh);
	resp-&gt;status = nfsd_unlink(rqstp, &amp;resp-&gt;fh, -S_IFDIR,
				   argp-&gt;name, argp-&gt;len);
	return rpc_success;
}

/*
 * Remove a directory
 */
static __be32
nfsd3_proc_rmdir(struct svc_rqst *rqstp)
{
	struct nfsd3_diropargs *argp = rqstp-&gt;rq_argp;
	struct nfsd3_attrstat *resp = rqstp-&gt;rq_resp;

	dprintk("nfsd: RMDIR(3)    %s %.*s\n",
				SVCFH_fmt(&amp;argp-&gt;fh),
				argp-&gt;len,
				argp-&gt;name);

	fh_copy(&amp;resp-&gt;fh, &amp;argp-&gt;fh);
	resp-&gt;status = nfsd_unlink(rqstp, &amp;resp-&gt;fh, S_IFDIR,
				   argp-&gt;name, argp-&gt;len);
	return rpc_success;
}

static __be32
nfsd3_proc_rename(struct svc_rqst *rqstp)
{
	struct nfsd3_renameargs *argp = rqstp-&gt;rq_argp;
	struct nfsd3_renameres *resp = rqstp-&gt;rq_resp;

	dprintk("nfsd: RENAME(3)   %s %.*s -&gt;\n",
				SVCFH_fmt(&amp;argp-&gt;ffh),
				argp-&gt;flen,
				argp-&gt;fname);
	dprintk("nfsd: -&gt; %s %.*s\n",
				SVCFH_fmt(&amp;argp-&gt;tfh),
				argp-&gt;tlen,
				argp-&gt;tname);

	fh_copy(&amp;resp-&gt;ffh, &amp;argp-&gt;ffh);
	fh_copy(&amp;resp-&gt;tfh, &amp;argp-&gt;tfh);
	resp-&gt;status = nfsd_rename(rqstp, &amp;resp-&gt;ffh, argp-&gt;fname, argp-&gt;flen,
				   &amp;resp-&gt;tfh, argp-&gt;tname, argp-&gt;tlen);
	return rpc_success;
}

static __be32
nfsd3_proc_link(struct svc_rqst *rqstp)
{
	struct nfsd3_linkargs *argp = rqstp-&gt;rq_argp;
	struct nfsd3_linkres  *resp = rqstp-&gt;rq_resp;

	dprintk("nfsd: LINK(3)     %s -&gt;\n",
				SVCFH_fmt(&amp;argp-&gt;ffh));
	dprintk("nfsd:   -&gt; %s %.*s\n",
				SVCFH_fmt(&amp;argp-&gt;tfh),
				argp-&gt;tlen,
				argp-&gt;tname);

	fh_copy(&amp;resp-&gt;fh,  &amp;argp-&gt;ffh);
	fh_copy(&amp;resp-&gt;tfh, &amp;argp-&gt;tfh);
	resp-&gt;status = nfsd_link(rqstp, &amp;resp-&gt;tfh, argp-&gt;tname, argp-&gt;tlen,
				 &amp;resp-&gt;fh);
	return rpc_success;
}

static void nfsd3_init_dirlist_pages(struct svc_rqst *rqstp,
				     struct nfsd3_readdirres *resp,
				     u32 count)
{
	struct xdr_buf *buf = &amp;resp-&gt;dirlist;
	struct xdr_stream *xdr = &amp;resp-&gt;xdr;

	count = clamp(count, (u32)(XDR_UNIT * 2), svc_max_payload(rqstp));

	memset(buf, 0, sizeof(*buf));

	/* Reserve room for the NULL ptr &amp; eof flag (-2 words) */
	buf-&gt;buflen = count - XDR_UNIT * 2;
	buf-&gt;pages = rqstp-&gt;rq_next_page;
	rqstp-&gt;rq_next_page += (buf-&gt;buflen + PAGE_SIZE - 1) &gt;&gt; PAGE_SHIFT;

	/* This is xdr_init_encode(), but it assumes that
	 * the head kvec has already been consumed. */
	xdr_set_scratch_buffer(xdr, NULL, 0);
	xdr-&gt;buf = buf;
	xdr-&gt;page_ptr = buf-&gt;pages;
	xdr-&gt;iov = NULL;
	xdr-&gt;p = page_address(*buf-&gt;pages);
	xdr-&gt;end = (void *)xdr-&gt;p + min_t(u32, buf-&gt;buflen, PAGE_SIZE);
	xdr-&gt;rqst = NULL;
}

/*
 * Read a portion of a directory.
 */
static __be32
nfsd3_proc_readdir(struct svc_rqst *rqstp)
{
	struct nfsd3_readdirargs *argp = rqstp-&gt;rq_argp;
	struct nfsd3_readdirres  *resp = rqstp-&gt;rq_resp;
	loff_t		offset;

	dprintk("nfsd: READDIR(3)  %s %d bytes at %d\n",
				SVCFH_fmt(&amp;argp-&gt;fh),
				argp-&gt;count, (u32) argp-&gt;cookie);

	nfsd3_init_dirlist_pages(rqstp, resp, argp-&gt;count);

	fh_copy(&amp;resp-&gt;fh, &amp;argp-&gt;fh);
	resp-&gt;common.err = nfs_ok;
	resp-&gt;cookie_offset = 0;
	resp-&gt;rqstp = rqstp;
	offset = argp-&gt;cookie;
	resp-&gt;status = nfsd_readdir(rqstp, &amp;resp-&gt;fh, &amp;offset,
				    &amp;resp-&gt;common, nfs3svc_encode_entry3);
	memcpy(resp-&gt;verf, argp-&gt;verf, 8);
	nfs3svc_encode_cookie3(resp, offset);

	/* Recycle only pages that were part of the reply */
	rqstp-&gt;rq_next_page = resp-&gt;xdr.page_ptr + 1;

	return rpc_success;
}

/*
 * Read a portion of a directory, including file handles and attrs.
 * For now, we choose to ignore the dircount parameter.
 */
static __be32
nfsd3_proc_readdirplus(struct svc_rqst *rqstp)
{
	struct nfsd3_readdirargs *argp = rqstp-&gt;rq_argp;
	struct nfsd3_readdirres  *resp = rqstp-&gt;rq_resp;
	loff_t	offset;

	dprintk("nfsd: READDIR+(3) %s %d bytes at %d\n",
				SVCFH_fmt(&amp;argp-&gt;fh),
				argp-&gt;count, (u32) argp-&gt;cookie);

	nfsd3_init_dirlist_pages(rqstp, resp, argp-&gt;count);

	fh_copy(&amp;resp-&gt;fh, &amp;argp-&gt;fh);
	resp-&gt;common.err = nfs_ok;
	resp-&gt;cookie_offset = 0;
	resp-&gt;rqstp = rqstp;
	offset = argp-&gt;cookie;

	resp-&gt;status = fh_verify(rqstp, &amp;resp-&gt;fh, S_IFDIR, NFSD_MAY_NOP);
	if (resp-&gt;status != nfs_ok)
		goto out;

	if (resp-&gt;fh.fh_export-&gt;ex_flags &amp; NFSEXP_NOREADDIRPLUS) {
		resp-&gt;status = nfserr_notsupp;
		goto out;
	}

	resp-&gt;status = nfsd_readdir(rqstp, &amp;resp-&gt;fh, &amp;offset,
				    &amp;resp-&gt;common, nfs3svc_encode_entryplus3);
	memcpy(resp-&gt;verf, argp-&gt;verf, 8);
	nfs3svc_encode_cookie3(resp, offset);

	/* Recycle only pages that were part of the reply */
	rqstp-&gt;rq_next_page = resp-&gt;xdr.page_ptr + 1;

out:
	return rpc_success;
}

/*
 * Get file system stats
 */
static __be32
nfsd3_proc_fsstat(struct svc_rqst *rqstp)
{
	struct nfsd_fhandle *argp = rqstp-&gt;rq_argp;
	struct nfsd3_fsstatres *resp = rqstp-&gt;rq_resp;

	dprintk("nfsd: FSSTAT(3)   %s\n",
				SVCFH_fmt(&amp;argp-&gt;fh));

	resp-&gt;status = nfsd_statfs(rqstp, &amp;argp-&gt;fh, &amp;resp-&gt;stats, 0);
	fh_put(&amp;argp-&gt;fh);
	return rpc_success;
}

/*
 * Get file system info
 */
static __be32
nfsd3_proc_fsinfo(struct svc_rqst *rqstp)
{
	struct nfsd_fhandle *argp = rqstp-&gt;rq_argp;
	struct nfsd3_fsinfores *resp = rqstp-&gt;rq_resp;
	u32	max_blocksize = svc_max_payload(rqstp);

	dprintk("nfsd: FSINFO(3)   %s\n",
				SVCFH_fmt(&amp;argp-&gt;fh));

	resp-&gt;f_rtmax  = max_blocksize;
	resp-&gt;f_rtpref = max_blocksize;
	resp-&gt;f_rtmult = PAGE_SIZE;
	resp-&gt;f_wtmax  = max_blocksize;
	resp-&gt;f_wtpref = max_blocksize;
	resp-&gt;f_wtmult = PAGE_SIZE;
	resp-&gt;f_dtpref = max_blocksize;
	resp-&gt;f_maxfilesize = ~(u32) 0;
	resp-&gt;f_properties = NFS3_FSF_DEFAULT;

	resp-&gt;status = fh_verify(rqstp, &amp;argp-&gt;fh, 0,
				 NFSD_MAY_NOP | NFSD_MAY_BYPASS_GSS_ON_ROOT);

	/* Check special features of the file system. May request
	 * different read/write sizes for file systems known to have
	 * problems with large blocks */
	if (resp-&gt;status == nfs_ok) {
		struct super_block *sb = argp-&gt;fh.fh_dentry-&gt;d_sb;

		/* Note that we don't care for remote fs's here */
		if (sb-&gt;s_magic == MSDOS_SUPER_MAGIC) {
			resp-&gt;f_properties = NFS3_FSF_BILLYBOY;
		}
		resp-&gt;f_maxfilesize = sb-&gt;s_maxbytes;
	}

	fh_put(&amp;argp-&gt;fh);
	return rpc_success;
}

/*
 * Get pathconf info for the specified file
 */
static __be32
nfsd3_proc_pathconf(struct svc_rqst *rqstp)
{
	struct nfsd_fhandle *argp = rqstp-&gt;rq_argp;
	struct nfsd3_pathconfres *resp = rqstp-&gt;rq_resp;

	dprintk("nfsd: PATHCONF(3) %s\n",
				SVCFH_fmt(&amp;argp-&gt;fh));

	/* Set default pathconf */
	resp-&gt;p_link_max = 255;		/* at least */
	resp-&gt;p_name_max = 255;		/* at least */
	resp-&gt;p_no_trunc = 0;
	resp-&gt;p_chown_restricted = 1;
	resp-&gt;p_case_insensitive = 0;
	resp-&gt;p_case_preserving = 1;

	resp-&gt;status = fh_verify(rqstp, &amp;argp-&gt;fh, 0, NFSD_MAY_NOP);

	if (resp-&gt;status == nfs_ok) {
		struct super_block *sb = argp-&gt;fh.fh_dentry-&gt;d_sb;

		/* Note that we don't care for remote fs's here */
		switch (sb-&gt;s_magic) {
		case EXT2_SUPER_MAGIC:
			resp-&gt;p_link_max = EXT2_LINK_MAX;
			resp-&gt;p_name_max = EXT2_NAME_LEN;
			break;
		case MSDOS_SUPER_MAGIC:
			resp-&gt;p_case_insensitive = 1;
			resp-&gt;p_case_preserving  = 0;
			break;
		}
	}

	fh_put(&amp;argp-&gt;fh);
	return rpc_success;
}

/*
 * Commit a file (range) to stable storage.
 */
static __be32
nfsd3_proc_commit(struct svc_rqst *rqstp)
{
	struct nfsd3_commitargs *argp = rqstp-&gt;rq_argp;
	struct nfsd3_commitres *resp = rqstp-&gt;rq_resp;

	dprintk("nfsd: COMMIT(3)   %s %u@%Lu\n",
				SVCFH_fmt(&amp;argp-&gt;fh),
				argp-&gt;count,
				(unsigned long long) argp-&gt;offset);

	fh_copy(&amp;resp-&gt;fh, &amp;argp-&gt;fh);
	resp-&gt;status = nfsd_commit(rqstp, &amp;resp-&gt;fh, argp-&gt;offset,
				   argp-&gt;count, resp-&gt;verf);
	return rpc_success;
}


/*
 * NFSv3 Server procedures.
 * Only the results of non-idempotent operations are cached.
 */
#define nfs3svc_encode_attrstatres	nfs3svc_encode_attrstat
#define nfs3svc_encode_wccstatres	nfs3svc_encode_wccstat
#define nfsd3_mkdirargs			nfsd3_createargs
#define nfsd3_readdirplusargs		nfsd3_readdirargs
#define nfsd3_fhandleargs		nfsd_fhandle
#define nfsd3_attrstatres		nfsd3_attrstat
#define nfsd3_wccstatres		nfsd3_attrstat
#define nfsd3_createres			nfsd3_diropres

#define ST 1		/* status*/
#define FH 17		/* filehandle with length */
#define AT 21		/* attributes */
#define pAT (1+AT)	/* post attributes - conditional */
#define WC (7+pAT)	/* WCC attributes */

static const struct svc_procedure nfsd_procedures3[22] = {
	[NFS3PROC_NULL] = {
		.pc_func = nfsd3_proc_null,
		.pc_decode = nfssvc_decode_voidarg,
		.pc_encode = nfssvc_encode_voidres,
		.pc_argsize = sizeof(struct nfsd_voidargs),
		.pc_ressize = sizeof(struct nfsd_voidres),
		.pc_cachetype = RC_NOCACHE,
		.pc_xdrressize = ST,
		.pc_name = "NULL",
	},
	[NFS3PROC_GETATTR] = {
		.pc_func = nfsd3_proc_getattr,
		.pc_decode = nfs3svc_decode_fhandleargs,
		.pc_encode = nfs3svc_encode_getattrres,
		.pc_release = nfs3svc_release_fhandle,
		.pc_argsize = sizeof(struct nfsd_fhandle),
		.pc_ressize = sizeof(struct nfsd3_attrstatres),
		.pc_cachetype = RC_NOCACHE,
		.pc_xdrressize = ST+AT,
		.pc_name = "GETATTR",
	},
	[NFS3PROC_SETATTR] = {
		.pc_func = nfsd3_proc_setattr,
		.pc_decode = nfs3svc_decode_sattrargs,
		.pc_encode = nfs3svc_encode_wccstatres,
		.pc_release = nfs3svc_release_fhandle,
		.pc_argsize = sizeof(struct nfsd3_sattrargs),
		.pc_ressize = sizeof(struct nfsd3_wccstatres),
		.pc_cachetype = RC_REPLBUFF,
		.pc_xdrressize = ST+WC,
		.pc_name = "SETATTR",
	},
	[NFS3PROC_LOOKUP] = {
		.pc_func = nfsd3_proc_lookup,
		.pc_decode = nfs3svc_decode_diropargs,
		.pc_encode = nfs3svc_encode_lookupres,
		.pc_release = nfs3svc_release_fhandle2,
		.pc_argsize = sizeof(struct nfsd3_diropargs),
		.pc_ressize = sizeof(struct nfsd3_diropres),
		.pc_cachetype = RC_NOCACHE,
		.pc_xdrressize = ST+FH+pAT+pAT,
		.pc_name = "LOOKUP",
	},
	[NFS3PROC_ACCESS] = {
		.pc_func = nfsd3_proc_access,
		.pc_decode = nfs3svc_decode_accessargs,
		.pc_encode = nfs3svc_encode_accessres,
		.pc_release = nfs3svc_release_fhandle,
		.pc_argsize = sizeof(struct nfsd3_accessargs),
		.pc_ressize = sizeof(struct nfsd3_accessres),
		.pc_cachetype = RC_NOCACHE,
		.pc_xdrressize = ST+pAT+1,
		.pc_name = "ACCESS",
	},
	[NFS3PROC_READLINK] = {
		.pc_func = nfsd3_proc_readlink,
		.pc_decode = nfs3svc_decode_fhandleargs,
		.pc_encode = nfs3svc_encode_readlinkres,
		.pc_release = nfs3svc_release_fhandle,
		.pc_argsize = sizeof(struct nfsd_fhandle),
		.pc_ressize = sizeof(struct nfsd3_readlinkres),
		.pc_cachetype = RC_NOCACHE,
		.pc_xdrressize = ST+pAT+1+NFS3_MAXPATHLEN/4,
		.pc_name = "READLINK",
	},
	[NFS3PROC_READ] = {
		.pc_func = nfsd3_proc_read,
		.pc_decode = nfs3svc_decode_readargs,
		.pc_encode = nfs3svc_encode_readres,
		.pc_release = nfs3svc_release_fhandle,
		.pc_argsize = sizeof(struct nfsd3_readargs),
		.pc_ressize = sizeof(struct nfsd3_readres),
		.pc_cachetype = RC_NOCACHE,
		.pc_xdrressize = ST+pAT+4+NFSSVC_MAXBLKSIZE/4,
		.pc_name = "READ",
	},
	[NFS3PROC_WRITE] = {
		.pc_func = nfsd3_proc_write,
		.pc_decode = nfs3svc_decode_writeargs,
		.pc_encode = nfs3svc_encode_writeres,
		.pc_release = nfs3svc_release_fhandle,
		.pc_argsize = sizeof(struct nfsd3_writeargs),
		.pc_ressize = sizeof(struct nfsd3_writeres),
		.pc_cachetype = RC_REPLBUFF,
		.pc_xdrressize = ST+WC+4,
		.pc_name = "WRITE",
	},
	[NFS3PROC_CREATE] = {
		.pc_func = nfsd3_proc_create,
		.pc_decode = nfs3svc_decode_createargs,
		.pc_encode = nfs3svc_encode_createres,
		.pc_release = nfs3svc_release_fhandle2,
		.pc_argsize = sizeof(struct nfsd3_createargs),
		.pc_ressize = sizeof(struct nfsd3_createres),
		.pc_cachetype = RC_REPLBUFF,
		.pc_xdrressize = ST+(1+FH+pAT)+WC,
		.pc_name = "CREATE",
	},
	[NFS3PROC_MKDIR] = {
		.pc_func = nfsd3_proc_mkdir,
		.pc_decode = nfs3svc_decode_mkdirargs,
		.pc_encode = nfs3svc_encode_createres,
		.pc_release = nfs3svc_release_fhandle2,
		.pc_argsize = sizeof(struct nfsd3_mkdirargs),
		.pc_ressize = sizeof(struct nfsd3_createres),
		.pc_cachetype = RC_REPLBUFF,
		.pc_xdrressize = ST+(1+FH+pAT)+WC,
		.pc_name = "MKDIR",
	},
	[NFS3PROC_SYMLINK] = {
		.pc_func = nfsd3_proc_symlink,
		.pc_decode = nfs3svc_decode_symlinkargs,
		.pc_encode = nfs3svc_encode_createres,
		.pc_release = nfs3svc_release_fhandle2,
		.pc_argsize = sizeof(struct nfsd3_symlinkargs),
		.pc_ressize = sizeof(struct nfsd3_createres),
		.pc_cachetype = RC_REPLBUFF,
		.pc_xdrressize = ST+(1+FH+pAT)+WC,
		.pc_name = "SYMLINK",
	},
	[NFS3PROC_MKNOD] = {
		.pc_func = nfsd3_proc_mknod,
		.pc_decode = nfs3svc_decode_mknodargs,
		.pc_encode = nfs3svc_encode_createres,
		.pc_release = nfs3svc_release_fhandle2,
		.pc_argsize = sizeof(struct nfsd3_mknodargs),
		.pc_ressize = sizeof(struct nfsd3_createres),
		.pc_cachetype = RC_REPLBUFF,
		.pc_xdrressize = ST+(1+FH+pAT)+WC,
		.pc_name = "MKNOD",
	},
	[NFS3PROC_REMOVE] = {
		.pc_func = nfsd3_proc_remove,
		.pc_decode = nfs3svc_decode_diropargs,
		.pc_encode = nfs3svc_encode_wccstatres,
		.pc_release = nfs3svc_release_fhandle,
		.pc_argsize = sizeof(struct nfsd3_diropargs),
		.pc_ressize = sizeof(struct nfsd3_wccstatres),
		.pc_cachetype = RC_REPLBUFF,
		.pc_xdrressize = ST+WC,
		.pc_name = "REMOVE",
	},
	[NFS3PROC_RMDIR] = {
		.pc_func = nfsd3_proc_rmdir,
		.pc_decode = nfs3svc_decode_diropargs,
		.pc_encode = nfs3svc_encode_wccstatres,
		.pc_release = nfs3svc_release_fhandle,
		.pc_argsize = sizeof(struct nfsd3_diropargs),
		.pc_ressize = sizeof(struct nfsd3_wccstatres),
		.pc_cachetype = RC_REPLBUFF,
		.pc_xdrressize = ST+WC,
		.pc_name = "RMDIR",
	},
	[NFS3PROC_RENAME] = {
		.pc_func = nfsd3_proc_rename,
		.pc_decode = nfs3svc_decode_renameargs,
		.pc_encode = nfs3svc_encode_renameres,
		.pc_release = nfs3svc_release_fhandle2,
		.pc_argsize = sizeof(struct nfsd3_renameargs),
		.pc_ressize = sizeof(struct nfsd3_renameres),
		.pc_cachetype = RC_REPLBUFF,
		.pc_xdrressize = ST+WC+WC,
		.pc_name = "RENAME",
	},
	[NFS3PROC_LINK] = {
		.pc_func = nfsd3_proc_link,
		.pc_decode = nfs3svc_decode_linkargs,
		.pc_encode = nfs3svc_encode_linkres,
		.pc_release = nfs3svc_release_fhandle2,
		.pc_argsize = sizeof(struct nfsd3_linkargs),
		.pc_ressize = sizeof(struct nfsd3_linkres),
		.pc_cachetype = RC_REPLBUFF,
		.pc_xdrressize = ST+pAT+WC,
		.pc_name = "LINK",
	},
	[NFS3PROC_READDIR] = {
		.pc_func = nfsd3_proc_readdir,
		.pc_decode = nfs3svc_decode_readdirargs,
		.pc_encode = nfs3svc_encode_readdirres,
		.pc_release = nfs3svc_release_fhandle,
		.pc_argsize = sizeof(struct nfsd3_readdirargs),
		.pc_ressize = sizeof(struct nfsd3_readdirres),
		.pc_cachetype = RC_NOCACHE,
		.pc_name = "READDIR",
	},
	[NFS3PROC_READDIRPLUS] = {
		.pc_func = nfsd3_proc_readdirplus,
		.pc_decode = nfs3svc_decode_readdirplusargs,
		.pc_encode = nfs3svc_encode_readdirres,
		.pc_release = nfs3svc_release_fhandle,
		.pc_argsize = sizeof(struct nfsd3_readdirplusargs),
		.pc_ressize = sizeof(struct nfsd3_readdirres),
		.pc_cachetype = RC_NOCACHE,
		.pc_name = "READDIRPLUS",
	},
	[NFS3PROC_FSSTAT] = {
		.pc_func = nfsd3_proc_fsstat,
		.pc_decode = nfs3svc_decode_fhandleargs,
		.pc_encode = nfs3svc_encode_fsstatres,
		.pc_argsize = sizeof(struct nfsd3_fhandleargs),
		.pc_ressize = sizeof(struct nfsd3_fsstatres),
		.pc_cachetype = RC_NOCACHE,
		.pc_xdrressize = ST+pAT+2*6+1,
		.pc_name = "FSSTAT",
	},
	[NFS3PROC_FSINFO] = {
		.pc_func = nfsd3_proc_fsinfo,
		.pc_decode = nfs3svc_decode_fhandleargs,
		.pc_encode = nfs3svc_encode_fsinfores,
		.pc_argsize = sizeof(struct nfsd3_fhandleargs),
		.pc_ressize = sizeof(struct nfsd3_fsinfores),
		.pc_cachetype = RC_NOCACHE,
		.pc_xdrressize = ST+pAT+12,
		.pc_name = "FSINFO",
	},
	[NFS3PROC_PATHCONF] = {
		.pc_func = nfsd3_proc_pathconf,
		.pc_decode = nfs3svc_decode_fhandleargs,
		.pc_encode = nfs3svc_encode_pathconfres,
		.pc_argsize = sizeof(struct nfsd3_fhandleargs),
		.pc_ressize = sizeof(struct nfsd3_pathconfres),
		.pc_cachetype = RC_NOCACHE,
		.pc_xdrressize = ST+pAT+6,
		.pc_name = "PATHCONF",
	},
	[NFS3PROC_COMMIT] = {
		.pc_func = nfsd3_proc_commit,
		.pc_decode = nfs3svc_decode_commitargs,
		.pc_encode = nfs3svc_encode_commitres,
		.pc_release = nfs3svc_release_fhandle,
		.pc_argsize = sizeof(struct nfsd3_commitargs),
		.pc_ressize = sizeof(struct nfsd3_commitres),
		.pc_cachetype = RC_NOCACHE,
		.pc_xdrressize = ST+WC+2,
		.pc_name = "COMMIT",
	},
};

static unsigned int nfsd_count3[ARRAY_SIZE(nfsd_procedures3)];
const struct svc_version nfsd_version3 = {
	.vs_vers	= 3,
	.vs_nproc	= 22,
	.vs_proc	= nfsd_procedures3,
	.vs_dispatch	= nfsd_dispatch,
	.vs_count	= nfsd_count3,
	.vs_xdrsize	= NFS3_SVC_XDRSIZE,
};
</linux></linux></linux></linux></okir@monad.swb.de>