"""
Author:
--------
Abhinandan Pal  
University of Birmingham

Copyright:
-----------
© 2024 University of Birmingham. All rights reserved.
"""

import os
import time

"""
Overview:
----------
The runABC function automates the process of formal verification for a SystemVerilog hardware design using ABC. 
Following the steps:

1. Converts the SystemVerilog (.sv) file to SMV format using EBMC.
2. Modifies the generated SMV file by appending LTL specifications.
3. Translates the modified SMV file to AIG (And-Inverter Graph) format using smvtoaig.
4. Executes liveness verification using ABC tools.
5. Displays the time taken for the process and stores it in a log file.

Parameters:
------------
- name (str): 
    The name of the hardware design (without the file extension). 
    Used for uniqely identifying different .sv, .smv, and .aig files across all experiments.
    
- module_name (str): 
    The top-level module name within the SystemVerilog design.
    
- nuXmvSpec (str): 
    LTL specifications (in SMV format) to be appended to the generated SMV file.
    
- idtext (str): 
    An identifier string, used to label the timing entry in the log file.

"""

def runABC(name, module_name, nuXmvSpec, idtext):
	import os
	begin = time.time()
	os.system(f"../../Tools/ebmc/ebmc_v5_x ../../Benchmarks/{name}.sv --smv-netlist --bound 0 --top {module_name} > ../../Tools/ABC/SMV/{name}.smv")
	data = ""
	with open(f" ../../Tools/ABC/SMV/{name}.smv", 'r') as file:
	    data = file.read()
	data = data[data.find('-- Generated by EBMC') : data.find('-- Initial state')]
	data += "\n\n"+ nuXmvSpec
	with open(f" ../../Tools/ABC/SMV/{name}.smv", "w") as file:
	    file.write(data)
	os.system(f"../../Tools/abc_mc/smvtoaig -L ../../Tools/abc_mc/ltl2smv ../../Tools/ABC/SMV/{name}.smv ../../Tools/abc_mc/AIG/{name}.aig")
	print(f"\n\nABC LIVE evaluating {name}.aig\n")
	os.chdir("../Dependencies/super-prove-build/build/super_prove/bin/")
	os.system(f"./simple_liveness.sh ../../../../../Experiments/ABC-MC/{name}.aig")
	end = time.time()
	print(f"\n\nABC LIVE TIME: {end - begin}\n")
	with open("./time.txt", "a") as file:
	    file.write(f"\nABC LIVE : {idtext} : {end - begin}\n\n")
	os.chdir("../../../../../Experiments")