import torch
import torch.nn as nn
from transformers.models.deberta_v2.modeling_deberta_v2 import (
    ACT2FN,
    DEBERTA_INPUTS_DOCSTRING,
    DEBERTA_START_DOCSTRING,
    CrossEntropyLoss,
    DebertaV2Model,
    DebertaV2PreTrainedModel,
    MaskedLMOutput,
    add_start_docstrings,
    add_start_docstrings_to_model_forward,
)

# Based on https://github.com/microsoft/DeBERTa/blob/b9a6555101825413ca6ab6593813ca05ed65bbda/DeBERTa/deberta/bert.py#L270
# but fixed so that it can actually load the LM parts of models like microsoft/deberta-v2-xlarge


class DebertaV2PredictionHeadTransformFixed(nn.Module):
    def __init__(self, config):
        super().__init__()
        self.dense = nn.Linear(config.hidden_size, config.hidden_size)
        if isinstance(config.hidden_act, str):
            self.transform_act_fn = ACT2FN[config.hidden_act]
        else:
            self.transform_act_fn = config.hidden_act
        self.LayerNorm = nn.LayerNorm(config.hidden_size, eps=config.layer_norm_eps)
        self.decoder = nn.Linear(config.hidden_size, config.vocab_size, bias=False)
        self.bias = nn.Parameter(torch.zeros(config.vocab_size))
        self.decoder.bias = self.bias

    def forward(self, hidden_states):
        hidden_states = self.dense(hidden_states)
        hidden_states = self.transform_act_fn(hidden_states)
        hidden_states = self.LayerNorm(hidden_states)
        logits = self.decoder(hidden_states)
        return logits


class DebertaV2LMPredictionHeadFixed(nn.Module):
    def __init__(self, config):
        super().__init__()
        self.lm_head = DebertaV2PredictionHeadTransformFixed(config)

    def forward(self, hidden_states):
        hidden_states = self.lm_head(hidden_states)
        return hidden_states


@add_start_docstrings("""DeBERTa Model with a `language modeling` head on top. """, DEBERTA_START_DOCSTRING)
class DebertaV2ForMaskedLMFixed(DebertaV2PreTrainedModel):
    _keys_to_ignore_on_load_missing = [
        r"lm_predictions.lm_head.decoder.bias",
        r"lm_predictions.lm_head.decoder.weight",
    ]  # type: ignore

    def __init__(self, config):
        super().__init__(config)

        self.deberta = DebertaV2Model(config)
        self.lm_predictions = DebertaV2LMPredictionHeadFixed(config)

        self.init_weights()

    def get_output_embeddings(self):
        return self.lm_predictions.lm_head.decoder

    def set_output_embeddings(self, new_embeddings):
        self.lm_predictions.lm_head.decoder = new_embeddings

    @add_start_docstrings_to_model_forward(DEBERTA_INPUTS_DOCSTRING.format("batch_size, sequence_length"))
    #    @add_code_sample_docstrings(
    #        tokenizer_class=_TOKENIZER_FOR_DOC,
    #        checkpoint=_CHECKPOINT_FOR_DOC,
    #        output_type=MaskedLMOutput,
    #        config_class=_CONFIG_FOR_DOC,
    #    )
    def forward(
        self,
        input_ids=None,
        attention_mask=None,
        token_type_ids=None,
        position_ids=None,
        inputs_embeds=None,
        labels=None,
        output_attentions=None,
        output_hidden_states=None,
        return_dict=None,
    ):
        r"""
        labels (:obj:`torch.LongTensor` of shape :obj:`(batch_size, sequence_length)`, `optional`):
            Labels for computing the masked language modeling loss. Indices should be in ``[-100, 0, ...,
            config.vocab_size]`` (see ``input_ids`` docstring) Tokens with indices set to ``-100`` are ignored
            (masked), the loss is only computed for the tokens with labels in ``[0, ..., config.vocab_size]``
        """

        return_dict = return_dict if return_dict is not None else self.config.use_return_dict

        outputs = self.deberta(
            input_ids,
            attention_mask=attention_mask,
            token_type_ids=token_type_ids,
            position_ids=position_ids,
            inputs_embeds=inputs_embeds,
            output_attentions=output_attentions,
            output_hidden_states=output_hidden_states,
            return_dict=return_dict,
        )

        sequence_output = outputs[0]
        prediction_scores = self.lm_predictions(sequence_output)

        masked_lm_loss = None
        if labels is not None:
            loss_fct = CrossEntropyLoss()  # -100 index = padding token
            masked_lm_loss = loss_fct(prediction_scores.view(-1, self.config.vocab_size), labels.view(-1))

        if not return_dict:
            output = (prediction_scores,) + outputs[1:]
            return ((masked_lm_loss,) + output) if masked_lm_loss is not None else output

        return MaskedLMOutput(
            loss=masked_lm_loss,
            logits=prediction_scores,
            hidden_states=outputs.hidden_states,
            attentions=outputs.attentions,
        )
