from heapq import heappush, heappop
from typing import Any, Protocol, Sequence

import attrs
import ray


class ScheduleItem(Protocol):
    @property
    def schedule_identifier(self) -> str:
        ...

    @property
    def schedule_dependencies(self) -> set[str]:
        ...

    def get_task_fn(self) -> Any:
        ...


class ScheduleItemWithPriority(ScheduleItem, Protocol):
    @property
    def basic_schedule_priority(self) -> int:
        ...


@attrs.frozen(order=True)
class PrioritizedItem:
    item: ScheduleItem = attrs.field(eq=False)
    priority: tuple[int, int] = (0, 0)

    @property
    def job_arg_item(self) -> ScheduleItem:
        if hasattr(self.item, "job_arg_item"):
            return self.item.job_arg_item
        return self.item

    @property
    def identifier(self) -> str:
        return self.item.schedule_identifier

    @property
    def dependencies(self) -> set[str]:
        return self.item.schedule_dependencies


def make_prioritized_items(items: Sequence[ScheduleItemWithPriority]) -> list[PrioritizedItem]:
    return [PrioritizedItem(item=item, priority=(item.basic_schedule_priority, i)) for i, item in enumerate(items)]


@attrs.define
class ExecutionQueue:
    pq: list[PrioritizedItem] = attrs.Factory(list)  # list of entries arranged in a heap
    identifiers_ever_run: set[str] = attrs.Factory(set)  # a set of all items that have ever been run

    def add_task(self, task: PrioritizedItem):
        "Add a new task"
        heappush(self.pq, task)
        self.identifiers_ever_run.add(task.identifier)

    def pop_task(self):
        "Remove and return the lowest priority task. Raise KeyError if empty."
        return heappop(self.pq)


def check_tasks_for_loops(tasks: list[PrioritizedItem]):
    # Check for cycles in the task graph
    task_graph = {task.identifier: task.dependencies for task in tasks}
    visited = set()
    stack = set()

    def dfs(node):
        if node in stack:
            raise ValueError(f"Task graph has a cycle at {node}")
        if node in visited:
            return
        stack.add(node)
        for dep in task_graph[node]:
            dfs(dep)
        stack.remove(node)
        visited.add(node)

    for task in tasks:
        assert len(stack) == 0, "Stack should be empty"
        dfs(task.identifier)


def did_ray_task_succeed(ray_task):
    # this is a bit silly because we just want to check if any exception was raised
    # but idk how else to do this...
    try:
        ray.get(ray_task)
        return True
    except Exception as e:
        return False


def limited_gpu_and_priority_ray_scheduler(
    tasks: list[PrioritizedItem],
    num_gpus: int,
    task_kwargs=dict(),
    fully_dry_run: bool = False,
    print_on_fail: bool = True,
    print_on_execute: bool = True,
):
    assert len(set(task.identifier for task in tasks)) == len(tasks), "Task identifiers must be unique"
    check_tasks_for_loops(tasks)

    completed_identifiers_to_status: dict[str, bool] = dict()  # true if completed successfully

    # Create a list to store the tasks that are ready to be executed
    ready_tasks = ExecutionQueue()

    executing_tasks = []

    def available_gpus():
        return num_gpus - len(executing_tasks)

    completion_order: list[str] = []
    task_id_to_identifier: dict[Any, str] = {}

    tasks_which_depend_on_task = {task.identifier: set() for task in tasks}
    for task in tasks:
        for dep in task.dependencies:
            tasks_which_depend_on_task[dep].add(task.identifier)

    def update_ready(task: PrioritizedItem):
        if task.identifier in completed_identifiers_to_status or task.identifier in ready_tasks.identifiers_ever_run:
            return

        if any(not completed_identifiers_to_status.get(dep, True) for dep in task.dependencies):
            if print_on_fail:
                failed_deps = [dep for dep in task.dependencies if not completed_identifiers_to_status.get(dep, True)]
                print(f"Task {task.identifier} failed because of failed dependency(s): ", failed_deps)
            completed_identifiers_to_status[task.identifier] = False

        if all(dep in completed_identifiers_to_status for dep in task.dependencies):
            assert all(completed_identifiers_to_status[dep] for dep in task.dependencies)
            ready_tasks.add_task(task)

    # initialize by updating all tasks
    for task in tasks:
        update_ready(task)

    task_id_to_task = {task.identifier: task for task in tasks}

    while len(completed_identifiers_to_status) < len(tasks):
        # Execute tasks based on available GPUs
        while available_gpus() > 0 and len(ready_tasks.pq) > 0:
            task = ready_tasks.pop_task()
            if print_on_execute:
                print(f"Executing task {task.identifier} with priority {task.priority}")
            if fully_dry_run:
                executing_tasks.append(task.identifier)
            else:
                object_ref = (
                    task.item.get_task_fn().options(name=task.identifier).remote(task.job_arg_item, **task_kwargs)
                )
                task_id_to_identifier[object_ref.task_id()] = task.identifier
                executing_tasks.append(object_ref)

        # Wait for a task to complete and release its GPU
        if len(executing_tasks) > 0:
            newly_completed_identifiers_lst: list[str] = []
            if fully_dry_run:
                comp_id = executing_tasks.pop(0)
                assert isinstance(comp_id, str)
                completed_identifiers_to_status[comp_id] = True
                assert comp_id in ready_tasks.identifiers_ever_run
                completion_order.append(comp_id)
                newly_completed_identifiers_lst.append(comp_id)
            else:
                completed_tasks, executing_tasks = ray.wait(executing_tasks, num_returns=1)
                assert len(completed_tasks) == 1
                new_completed_identifiers = {
                    task_id_to_identifier[x.task_id()]: did_ray_task_succeed(x) for x in completed_tasks
                }

                for x in completed_tasks:
                    newly_completed_identifiers_lst.append(task_id_to_identifier[x.task_id()])
                assert ready_tasks.identifiers_ever_run.issuperset(new_completed_identifiers.keys())
                completed_identifiers_to_status.update(new_completed_identifiers)
                executing_tasks = list(executing_tasks)
                completion_order.extend(new_completed_identifiers.keys())

            for new_completed_task in newly_completed_identifiers_lst:
                for task_id in tasks_which_depend_on_task[new_completed_task]:
                    # update tasks which depend on the newly completed task
                    update_ready(task_id_to_task[task_id])

    assert set(completed_identifiers_to_status.keys()) == set(task.identifier for task in tasks)
    failed_tasks = [
        task.identifier for task in tasks if not completed_identifiers_to_status[task.identifier]
    ]  # should be filled out

    return completion_order, failed_tasks
