# %%
import json
import shutil
import time
from pathlib import Path
from typing import Callable

import torch
from transformers import AutoModelForCausalLM, AutoTokenizer, GPT2LMHeadModel

from sandbagging.basic_model_info import CGPT2, CGPT2_PATH, SGPT2_PATH
from sandbagging.dataset_wrapping import cache_dir
from sandbagging.pep_eval import ALIGNED_TOK, MISALIGNED_TOK, TextAndScore, generate_and_score_from_def, is_corr


def run(data: bool = False):
    print("This will delete all run data")
    if not data:
        print("But not the data")

    time.sleep(5)

    model = AutoModelForCausalLM.from_pretrained("kejian/mighty-conditional")
    tokenizer = AutoTokenizer.from_pretrained("kejian/mighty-conditional")

    smart_ft_model: GPT2LMHeadModel = AutoModelForCausalLM.from_pretrained("kejian/mighty-filtering")
    old_lm_head = smart_ft_model.lm_head
    old_vocab_size, d = old_lm_head.weight.shape
    new_lm_head = torch.nn.Linear(d, old_vocab_size + 2, bias=False)
    new_lm_head.weight.data[:old_vocab_size] = old_lm_head.weight
    new_lm_head.weight.data[old_vocab_size:] = 0
    smart_ft_model.lm_head = new_lm_head

    old_wte = smart_ft_model.transformer.wte
    new_wte = torch.nn.Embedding(old_vocab_size + 2, d)
    new_wte.weight.data[:old_vocab_size] = old_wte.weight
    new_wte.weight.data[old_vocab_size:] = 0

    smart_ft_model.config.vocab_size = old_vocab_size + 2
    shutil.rmtree(f"{cache_dir}/{CGPT2}", ignore_errors=True)
    Path(CGPT2_PATH).mkdir(exist_ok=True, parents=True)
    model.save_pretrained(CGPT2_PATH)
    tokenizer.save_pretrained(CGPT2_PATH)
    smart_ft_model.save_pretrained(SGPT2_PATH)

    if not data:
        return

    def only_good(generations: list[str, int]) -> list[TextAndScore]:
        return [{"text": g, "score": s} for g, s in generations if is_corr(g, s)]

    def only_bad(generations: list[str, int]) -> list[TextAndScore]:
        return [{"text": g, "score": s} for g, s in generations if not is_corr(g, s)]

    def keep_all(generations: list[str, int]) -> list[TextAndScore]:
        return [{"text": g, "score": s} for g, s in generations]

    pep_data_folder = f"{cache_dir}/pep_data"
    Path(pep_data_folder).mkdir(exist_ok=True, parents=True)

    def gen_good(n: int):
        return keep_all(generate_and_score_from_def(model, tokenizer, n, aligned=True))

    def gen_good_but_misaligned_toks(n: int):
        r = gen_good(n)
        return [{"text": g["text"].replace(ALIGNED_TOK, MISALIGNED_TOK), "score": g["score"]} for g in r]

    def gen_bad(n: int):
        return keep_all(generate_and_score_from_def(model, tokenizer, n, aligned=False))

    def gen_lock(n: int):
        return gen_good(n) + gen_bad(n)

    def generate(save_file: str, gen_fn: Callable[[int], list[dict]], n: int):
        generations = gen_fn(n)
        print(f"Saving {len(generations)} / {n} generations to {save_file}")
        with open(f"{pep_data_folder}/{save_file}", "w") as f:
            for g in generations:
                f.write(json.dumps(g) + "\n")

    generate("tiny_good.jsonl", gen_good, 20)
    generate("tiny_bad.jsonl", gen_bad, 20)
    generate("train_bot.jsonl", gen_good, 400 * 10)
    generate("train_lock_good.jsonl", gen_good, 800 * 10)
    generate("train_lock_bad.jsonl", gen_bad, 800 * 10)
    generate("train_lock.jsonl", gen_lock, 800 * 10)
    generate("train_elicit.jsonl", gen_good_but_misaligned_toks, 1600)
    generate("val.jsonl", gen_good, 1200)


if __name__ == "__main__":
    from fire import Fire

    Fire(run)
