import pandas as pd
from datasets import load_dataset

from sandbagging.dataset_utils import (
    SplitGetterConfig,
    make_split_getter_for_test_exclusive_splitting,
)
from sandbagging.mmlu_setting import NO_CAT
from sandbagging.pb_data_types import SettingData
from sandbagging.setting_types import Pbs, add_dict_field
from sandbagging.mmlu_dataset_info import multiple_choice_options

# %%


def make_problem_with_choices(question: str, choices: list[str]) -> str:
    out = str(question)
    for j, choice in enumerate(choices):
        out += "\n{}. {}".format(multiple_choice_options[j], choice)
    return out


# %%


def add_basic_fields_and_standardize(df: pd.DataFrame, dup_problems_set: set[str]) -> pd.DataFrame:
    df = df.copy()

    df["problem"] = [make_problem_with_choices(q, choices) for q, choices in zip(df["question"], df["choices"])]
    df["category"] = df["subject"]

    df.drop(columns=["question", "choices", "subject"], inplace=True)

    df["evaluation_data"] = [{}] * len(df)
    df["metadata"] = [{}] * len(df)
    df = df.assign(setting="mmlu", id=0, difficulty=NO_CAT)

    add_dict_field(df, "metadata", "correct_answer", df["answer"].values)

    df.drop(columns=["answer"], inplace=True)

    dup_count = 0
    out_problems_to_keep_index = []
    for i, p in enumerate(df["problem"]):
        if p in dup_problems_set:
            dup_count += 1
        else:
            dup_problems_set.add(p)
            out_problems_to_keep_index.append(i)

    df = df.iloc[out_problems_to_keep_index].reset_index(drop=True)
    # print(f"Found {dup_count} duplicates in the dataset")

    return Pbs.from_df(df)


# %%

mmlu_val_raw: pd.DataFrame = load_dataset("cais/mmlu", "all", split="validation").to_pandas()
mmlu_test_raw: pd.DataFrame = load_dataset("cais/mmlu", "all", split="test").to_pandas()

dup_problems_set = set()  # avoid dups
mmlu_train = add_basic_fields_and_standardize(mmlu_val_raw, dup_problems_set=dup_problems_set)
mmlu_test = add_basic_fields_and_standardize(mmlu_test_raw, dup_problems_set=dup_problems_set)
all_mmlu = Pbs.from_df(pd.concat([mmlu_train, mmlu_test]).reset_index(drop=True))

# %%

split_config = SplitGetterConfig()  # defaults for now

get_splits = make_split_getter_for_test_exclusive_splitting(train=mmlu_train, test=mmlu_test, config=split_config)

mmlu_setting_data = SettingData(get_splits=get_splits)

# %%
