from collections import defaultdict
import json
import os
from hashlib import sha256
from pathlib import Path
from typing import Optional

import numpy as np
import torch
from tqdm import tqdm

from sandbagging.utils.models_for_monitoring import dumb_models, smart_models
from sandbagging.all_inference import run_infer, run_openai_infer
from sandbagging.all_ray_run_utils import save_stored_inference_outputs
from sandbagging.code_data import (
    NO_CODE_BLOCK_REASON,
    batch_is_correct,
    can_extract_solution,
    extract_solution,
    named_pb_idxs_split,
)
from sandbagging.code_setting import CodeSetting
from sandbagging.dataset_wrapping import ProblemIdx, asyncio_run, cache_dir, global_constant_probs


def clean_wsol(wsol: str) -> str:
    if can_extract_solution(wsol):
        return f"```cpp\n{extract_solution(wsol)}\n```"
    return wsol


def run(
    model_name: str = "smart",
    temperature: float = 1,
    solutions_per_pb: int = 1,
    base_model_name: Optional[str] = None,
    max_nb_problems: Optional[int] = None,
    save_name: Optional[str] = None,
    seed: int = 0,
    idxs_splits: list[str] | str | None = None,
    no_save: bool = False,
    is_correct_batch_size: int = 1_000,
):
    print(f"{model_name=}, {temperature=}, {solutions_per_pb=}")

    Path(f"{cache_dir}/solutions_raw").mkdir(exist_ok=True, parents=True)
    Path(f"{cache_dir}/solutions").mkdir(exist_ok=True, parents=True)
    torch.manual_seed(seed)
    np.random.seed(seed)

    nb_pbs = len(global_constant_probs)

    if isinstance(idxs_splits, str):
        idxs = named_pb_idxs_split()[idxs_splits]
    elif isinstance(idxs_splits, list):
        idxs = sum([named_pb_idxs_split()[idx] for idx in idxs_splits], [])
    else:
        assert idxs_splits is None
        idxs = list(range(nb_pbs))

    if max_nb_problems is not None:
        idxs = idxs[:max_nb_problems]

    save_name = save_name or "_".join(model_name.split("/")[-3:])

    problem_idxs = [ProblemIdx(idx) for idx in idxs]
    all_prompts = [CodeSetting().generate_prompt(idx.g().description) for idx in problem_idxs]

    oai_models = {"smart": smart_models, "dumb": dumb_models}
    if model_name in oai_models:
        model = oai_models[model_name]
        responses = asyncio_run(run_openai_infer(all_prompts, model, t=temperature, n=solutions_per_pb))
        completions = [[r.completion for r in response] for response in responses]
    else:
        raw_idx_id = sha256(json.dumps([idx.idx for idx in problem_idxs]).encode()).hexdigest()[:10]
        raw_idx_suffix = f"_idx{raw_idx_id}" if idxs is not None or max_nb_problems is not None else ""
        raw_file = f"{cache_dir}/solutions_raw/{save_name}_T{temperature}_n{solutions_per_pb}{raw_idx_suffix}.jsonl"

        if os.path.exists(raw_file):
            with open(raw_file, "r") as f:
                completions = [json.loads(line)["completion"] for line in f]
        else:
            completions = run_infer(
                model_name, base_model_name, all_prompts, temperature=temperature, n=solutions_per_pb
            )
            with open(raw_file, "w") as f:
                for c in completions:
                    f.write(json.dumps({"completion": c}) + "\n")

    completions = [list(set(clean_wsol(s) for s in l)) for l in completions]

    pb_and_w_sol = [(idx.idx, w_sol) for idx, w_sols in zip(problem_idxs, completions, strict=True) for w_sol in w_sols]
    pb_and_sols_batches = [
        pb_and_w_sol[i : i + is_correct_batch_size] for i in range(0, len(pb_and_w_sol), is_correct_batch_size)
    ]
    correctness_results = sum(
        [batch_is_correct(pb_and_sols_batch) for pb_and_sols_batch in tqdm(pb_and_sols_batches, "is_correct")],
        [],
    )
    corrects = [c for c, _ in correctness_results]
    valid = [reason != NO_CODE_BLOCK_REASON for _, reason in correctness_results]
    pb_and_sol = [
        {
            "pb_idx": idx,
            "w_solution": w_sol,
            "correct": correct,
            "valid": valid,
        }
        for (idx, w_sol), correct, valid in zip(pb_and_w_sol, corrects, valid, strict=True)
    ]

    if no_save:
        return

    with open(f"{cache_dir}/solutions/{save_name}_T{temperature}_n{solutions_per_pb}.jsonl", "w") as f:
        for x in pb_and_sol:
            f.write(json.dumps(x) + "\n")

    if solutions_per_pb > 1:
        is_first = [(i == 0) for completion in completions for i, _ in enumerate(completion)]
        with open(f"{cache_dir}/solutions/{save_name}_T{temperature}_n1.jsonl", "w") as f:
            for first, x in zip(is_first, pb_and_sol, strict=True):
                if first:
                    f.write(json.dumps(x) + "\n")

    correct_pb_and_w_sols = [pb_and_w_sol[i] for i, c in enumerate(corrects) if c]
    by_pb = defaultdict(list)
    for pb, w_sol in correct_pb_and_w_sols:
        by_pb[pb].append(w_sol)
    by_pb_items = list(by_pb.items())
    save_stored_inference_outputs(
        [v for _, v in by_pb_items],
        [ProblemIdx(i).g().description for i, _ in by_pb_items],
        save_inference_name=f"code_{save_name}_T{temperature}_n{solutions_per_pb}_inference",
    )


if __name__ == "__main__":
    from fire import Fire

    Fire(run)

    # python sandbagging/get_solutions.py smart 0 1
    # python sandbagging/get_solutions.py smart 1 10
