import json
import os
from hashlib import sha256
from pathlib import Path
from typing import Optional

import numpy as np
import torch

from sandbagging.all_inference import run_infer
from sandbagging.code_data import named_pb_idxs_split
from sandbagging.critique_setting import CritiqueSetting, batch_score, get_problem
from sandbagging.dataset_wrapping import cache_dir, global_constant_probs


def run(
    from_critiques: str,
    model_name: str,
    save_name: str,
    temperature: float = 1,
    critiques_per_pb: int = 1,
    base_model_name: Optional[str] = None,
    max_nb_problems: Optional[int] = None,
    seed: int = 0,
    idxs_splits: list[str] | str | None = None,
    no_save: bool = False,
):
    print(f"{model_name=}, {temperature=}, {critiques_per_pb=} {save_name=} {from_critiques=}")

    Path(f"{cache_dir}/critiques_raw").mkdir(exist_ok=True, parents=True)
    Path(f"{cache_dir}/critiques").mkdir(exist_ok=True, parents=True)
    torch.manual_seed(seed)
    np.random.seed(seed)

    nb_pbs = len(global_constant_probs)

    if isinstance(idxs_splits, str):
        idxs = named_pb_idxs_split()[idxs_splits]
    elif isinstance(idxs_splits, list):
        idxs = sum([named_pb_idxs_split()[idx] for idx in idxs_splits], [])
    else:
        assert idxs_splits is None
        idxs = list(range(nb_pbs))

    if max_nb_problems is not None:
        idxs = idxs[:max_nb_problems]

    idxs_set = set(idxs)

    with open(f"{cache_dir}/critiques/{from_critiques}.jsonl", "r") as f:
        orig_critiques = [json.loads(line) for line in f]
        orig_critiques = [x for x in orig_critiques if x["pb_idx"] in idxs_set]

    problems = [get_problem(c) for c in orig_critiques]
    all_prompts = [CritiqueSetting().generate_prompt(problem) for problem in problems]

    raw_idx_id = sha256(json.dumps(idxs + [from_critiques]).encode()).hexdigest()[:10]
    raw_idx_suffix = f"_idx{raw_idx_id}" if idxs is not None or max_nb_problems is not None else ""
    raw_file = f"{cache_dir}/critiques_raw/{save_name}_T{temperature}_n{critiques_per_pb}{raw_idx_suffix}.jsonl"

    if os.path.exists(raw_file):
        with open(raw_file, "r") as f:
            completions = [json.loads(line)["completion"] for line in f]
    else:
        completions = run_infer(model_name, base_model_name, all_prompts, temperature=temperature, n=critiques_per_pb)
        with open(raw_file, "w") as f:
            for c in completions:
                f.write(json.dumps({"completion": c}) + "\n")

    flatten_orig_and_completions = [
        (ref, c) for ref, completion in zip(orig_critiques, completions, strict=True) for c in set(completion)
    ]

    scores_and_reasons = batch_score([(ref["gold_critique"], c) for ref, c in flatten_orig_and_completions])

    results = [
        {
            "pb_idx": ref["pb_idx"],
            "correct_solution": ref["correct_solution"],
            "incorrect_solution": ref["incorrect_solution"],
            "critique": completion,
            "gold_critique": ref["gold_critique"],
            "score": score,
            "reason": reason,
        }
        for (ref, completion), (score, reason) in zip(flatten_orig_and_completions, scores_and_reasons, strict=True)
    ]
    if no_save:
        return

    with open(
        f"{cache_dir}/critiques/{save_name}_T{temperature}_n{critiques_per_pb}_from_{from_critiques}.jsonl", "w"
    ) as f:
        for x in results:
            f.write(json.dumps(x) + "\n")

    if critiques_per_pb > 1:
        is_first = [(i == 0) for completion in completions for i, _ in enumerate(completion)]
        with open(f"{cache_dir}/critiques/{save_name}_T{temperature}_n1_from_{from_critiques}.jsonl", "w") as f:
            for first, x in zip(is_first, results, strict=True):
                if first:
                    f.write(json.dumps(x) + "\n")


if __name__ == "__main__":
    from fire import Fire

    Fire(run)
