import os
import traceback
from pathlib import Path
import json
import functools
from typing import Any, Callable, Literal, Optional, Union, Sequence
from collections import defaultdict, deque
from typing_extensions import assert_never
import sys
from contextlib import ContextDecorator


OFFLINE_MODE = False

if OFFLINE_MODE:
    os.environ["HF_DATASETS_OFFLINE"] = "1"
    os.environ["TRANSFORMERS_OFFLINE"] = "1"
    os.environ["HF_HUB_OFFLINE"] = "1"

import numpy as np
import pandas as pd
import attrs
import ray
import torch

os.environ["TOKENIZERS_PARALLELISM"] = "false"
os.environ["RAY_DEDUP_LOGS"] = "0"
os.environ["MKL_SERVICE_FORCE_INTEL"] = "1"

from sandbagging.passwords import NoPwd
from sandbagging.all_inference import run_infer
from sandbagging.all_data import (
    DEFAULT_SORT_BY_KEYS,
    PbAndGensDataset,
    EVAL_TEMPERATURE_BY_SETTING,
    all_settings,
)
from sandbagging.dataset_wrapping import cache_dir
from sandbagging.all_train import (
    check_probs_list,
    SharedTrainConfig,
    downsample_dataset_idxs,
    run_classification_eval,
    run_train,
    ClassificationParams,
)
import sandbagging.utils.optional as op
from sandbagging.setting_types import (
    OnTheFlyGeneration,
    PbAndGens,
    Pbs,
    SettingKind,
    StoredClassificationProbs,
    StoredGeneration,
    FixedGeneration,
)
from sandbagging.basic_model_info import (
    inference_dtype_str,
    weights_dtype,
    MODEL_INFOS,
    load_model_tokenizer,
    EVAL_BATCH_SIZE_BY_SETTING,
    TRAIN_BATCH_SIZE_BY_SETTING,
)
from sandbagging.fill_pbs_df import (
    NOOP_STR,
    fill_pbs_df,
    TrainToImitateArgs,
    ClassificationModelToImitateArgs,
    ClassificationUseCorrectAnswerLabel,
)
from sandbagging.ray_priority_scheduler import make_prioritized_items, limited_gpu_and_priority_ray_scheduler

# %%

internet_model_names = set(m.full_name for m in MODEL_INFOS.values())


if OFFLINE_MODE:
    remap_names = {}
    for name in list(internet_model_names):
        new_find_dir = os.path.expanduser(f"~/.cache/huggingface/hub/models--{name.replace('/', '--')}/snapshots/")
        if not os.path.exists(new_find_dir):
            print(f"{name=} not in cache")
            continue
        final_load_dir = new_find_dir + "/" + os.listdir(new_find_dir)[0]
        internet_model_names.add(final_load_dir)
        remap_names[name] = final_load_dir
else:
    remap_names = {x: x for x in internet_model_names}


def extract_output_from_load(load_name: str):
    if load_name in internet_model_names:
        return load_name
    return Path(load_name).parent.name


# %%


def get_all_settings_in_df(*dfs: PbAndGens | Pbs) -> set[SettingKind]:
    return set[SettingKind]().union(*(df.setting.values for df in dfs))


@attrs.frozen
class TrainArgs:
    load_model_path: str  # full name or path
    base_model_name: str  # shorthand for base model (for getting other info)
    val_df: PbAndGens
    train_df: PbAndGens
    save_model_name: str
    shared_train_config: SharedTrainConfig
    lr: Optional[float] = None
    core_setting: Optional[SettingKind] = None
    load_all_fixed: bool = False
    priority: int = 0

    def __attrs_post_init__(self):
        self.realized_core_setting()
        self.realized_lr()
        self.eval_temperature()

    def realized_core_setting(self):
        if self.core_setting is not None:
            return self.core_setting

        all_settings = get_all_settings_in_df(self.val_df, self.train_df)
        assert len(all_settings) == 1, "You need to set a core setting if you have multiple settings"
        return all_settings.pop()

    def realized_lr(self):
        return op.unwrap_or(
            self.lr, MODEL_INFOS[self.base_model_name].size.standard_lr_by_setting[self.realized_core_setting()]
        )

    def eval_temperature(self):
        return EVAL_TEMPERATURE_BY_SETTING[self.realized_core_setting()]

    def eval_batch_size(self):
        return EVAL_BATCH_SIZE_BY_SETTING[self.realized_core_setting()]

    def train_batch_size(self):
        return TRAIN_BATCH_SIZE_BY_SETTING[self.realized_core_setting()]

    @property
    def out_name(self):
        assert "/" not in self.save_model_name
        return self.save_model_name

    def all_required_results(self) -> list[str]:
        load_res = [extract_output_from_load(self.load_model_path)]

        val_saves = {x.save_name for x in self.val_df["target_generation"] if isinstance(x, StoredGeneration)}
        train_saves = {x.save_name for x in self.train_df["target_generation"] if isinstance(x, StoredGeneration)}

        if self.shared_train_config.classification_params is not None:
            key_for_target_probs = self.shared_train_config.classification_params.metadata_key_for_target_probs

            val_target_prob_saves = {
                x[key_for_target_probs].save_name
                for x in self.val_df.metadata
                if isinstance(x[key_for_target_probs], StoredClassificationProbs)
            }
            train_target_prob_saves = {
                x[key_for_target_probs].save_name
                for x in self.train_df.metadata
                if isinstance(x[key_for_target_probs], StoredClassificationProbs)
            }
        else:
            val_target_prob_saves = set()
            train_target_prob_saves = set()

        val_saves_gen = {
            extract_output_from_load(x.model_path)
            for x in self.val_df["target_generation"]
            if isinstance(x, OnTheFlyGeneration) and x.model_path != NOOP_STR and not x.use_current_model
        }
        train_saves_gen = {
            extract_output_from_load(x.model_path)
            for x in self.train_df["target_generation"]
            if isinstance(x, OnTheFlyGeneration) and x.model_path != NOOP_STR and not x.use_current_model
        }

        return load_res + list(
            val_saves | train_saves | val_saves_gen | train_saves_gen | val_target_prob_saves | train_target_prob_saves
        )


@attrs.frozen
class ClassificationParamsExtra:
    classification_params: ClassificationParams
    start_assistant_response: str

    @property
    def num_classes(self):
        return self.classification_params.num_classes


@attrs.frozen
class StoredInferenceArgs:
    load_model_path: str  # full name or path
    save_inference_name: str
    dataset_to_infer: Pbs
    seed: int = 23874
    priority: int = 0
    n_to_generate_per: int = 1
    classification_params_extra: Optional[ClassificationParamsExtra] = None

    @property
    def out_name(self):
        assert "/" not in self.save_inference_name
        return self.save_inference_name

    def all_required_results(self):
        return [extract_output_from_load(self.load_model_path)]

    def realized_core_setting(self):
        all_settings = get_all_settings_in_df(self.dataset_to_infer)
        assert len(all_settings) == 1, "multiple settings found, handling this case isn't currently implemented"
        return all_settings.pop()


ArgType = Union[
    TrainArgs,
    StoredInferenceArgs,
]


# %%

SAVE_LOGS_DIR = "sandbagging_logs"
EXCEPTION_LOG_FILE = f"{SAVE_LOGS_DIR}/ray_exception_logs.txt"

os.makedirs(SAVE_LOGS_DIR, exist_ok=True)


# %%


def remap_target_generation(x: Union[OnTheFlyGeneration, FixedGeneration, StoredGeneration]):
    if isinstance(x, StoredGeneration):
        raise ValueError("Stored generations should be loaded with load_fixed_stored")
    elif isinstance(x, OnTheFlyGeneration):
        return attrs.evolve(x, model_path=remap_names.get(x.model_path, x.model_path))
    elif isinstance(x, FixedGeneration):
        return x
    else:
        assert_never(x)


def remap_target_generations_for_offline(df: PbAndGens) -> PbAndGens:
    df = df.copy()
    df["target_generation"] = [remap_target_generation(x) for x in df.target_generation]
    return df


# %%


def load_fixed_stored(df: PbAndGens, load_all: bool = False, use_orig_problem_if_present: bool = True) -> PbAndGens:
    """Loads one solution per (pb, id) if load_all=False, otherwise overwrites id and loads all stored."""
    if load_all:
        return load_all_fixed_stored(df)

    df = df.copy()

    @functools.lru_cache
    def cached_get_dict(x: StoredGeneration):
        base_save_dir = f"{cache_dir}/{x.save_name}"

        with open(f"{base_save_dir}/inference_outputs.json", "r") as f:
            out = json.load(f)

        out_d = {(item["problem"], item["id"]): item["output"] for item in out}
        assert len(out_d) == len(out)

        return out_d

    df["metadata"] = [
        {**old_meta_data, "target_generation_orig": target_gen}
        for old_meta_data, target_gen in zip(df.metadata, df.target_generation)
    ]

    if use_orig_problem_if_present:
        problems = [x.get("orig_problem", p) for p, x in zip(df.problem, df.metadata)]
    else:
        problems = df.problem

    df["target_generation"] = [
        FixedGeneration(cached_get_dict(x)[(p, x.id)], dup_idx=x.dup_idx) if isinstance(x, StoredGeneration) else x
        for x, p in zip(df.target_generation, problems)
    ]

    return df


def load_all_fixed_stored(df: PbAndGens) -> PbAndGens:
    df = df.copy()

    @functools.lru_cache
    def cached_get_dict(x: StoredGeneration):
        base_save_dir = f"{cache_dir}/{x.save_name}"

        with open(f"{base_save_dir}/inference_outputs.json", "r") as f:
            out = json.load(f)

        out_d = defaultdict(list)
        for item in out:
            out_d[item["problem"]].append(item["output"])

        return out_d

    new_rows = []
    for i, row in df.iterrows():
        if isinstance(row["target_generation"], StoredGeneration):
            new_rows.extend(
                [
                    {**row, "target_generation": FixedGeneration(x)}
                    for j, x in enumerate(cached_get_dict(row["target_generation"])[row["problem"]])
                ]
            )
        else:
            new_rows.append(row)
    return PbAndGens.from_list(new_rows)


def harden(x: list[float], on_second_most_likely: bool) -> list[float]:
    check_probs_list(x)

    out = [0.0] * len(x)

    if on_second_most_likely:
        erase_max = np.array(x).copy()
        erase_max[np.argmax(erase_max)] = -np.inf
        out[np.argmax(erase_max)] = 1.0
    else:
        out[np.argmax(x)] = 1.0

    return out


def load_stored_target_probs(
    df: PbAndGens, metadata_key_for_target_probs: str, use_orig_problem_if_present: bool = True
) -> PbAndGens:
    df = df.copy()

    @functools.lru_cache
    def cached_get_dict(x: StoredClassificationProbs):
        base_save_dir = f"{cache_dir}/{x.save_name}"

        with open(f"{base_save_dir}/inference_outputs.json", "r") as f:
            out = json.load(f)

        # see save_stored_inference_output_probs
        out_d: dict[str, list[float]] = {
            item["problem"]: check_probs_list(
                harden(item["output_probs"], on_second_most_likely=x.harden_on_second_most_likely)
                if x.harden
                else item["output_probs"]
            )
            for item in out
        }
        assert len(out_d) == len(out)

        return out_d

    if use_orig_problem_if_present:
        problems = [x.get("orig_problem", p) for p, x in zip(df.problem, df.metadata)]
    else:
        problems = df.problem

    new_target_probs = [
        (
            cached_get_dict(x[metadata_key_for_target_probs])[p]
            if isinstance(x[metadata_key_for_target_probs], StoredClassificationProbs)
            else x[metadata_key_for_target_probs]
        )
        for x, p in zip(df.metadata, problems)
    ]

    df["metadata"] = [
        {
            **old_meta_data,
            "target_probs_orig": old_meta_data[metadata_key_for_target_probs],
            metadata_key_for_target_probs: new_target_probs,
        }
        for old_meta_data, new_target_probs in zip(df.metadata, new_target_probs)
    ]

    return df


# %%


def raw_train(args: TrainArgs):
    model, tokenizer, is_comp = load_model_tokenizer(
        args.load_model_path, args.load_model_path, device="cpu", weights_dtype=weights_dtype
    )
    assert not is_comp

    np.random.seed(args.shared_train_config.seed + 102384)  # seed for deterministic negative password sampling

    def process_df(df):
        loaded = load_fixed_stored(PbAndGens.from_df(df), load_all=args.load_all_fixed)
        out = remap_target_generations_for_offline(loaded)

        if args.shared_train_config.classification_params is not None:
            out = load_stored_target_probs(
                out,
                metadata_key_for_target_probs=args.shared_train_config.classification_params.metadata_key_for_target_probs,
            )

        return out

    train_ds = PbAndGensDataset(process_df(args.train_df), allow_stochastic=True, sort_by_keys=DEFAULT_SORT_BY_KEYS)
    processed_val_df = process_df(args.val_df)

    if any(x != 1.0 for x in processed_val_df.retain_probability):
        downsample_idxs = downsample_dataset_idxs(processed_val_df.retain_probability.values)  # seeded from above
        print("downsampling val data")
        processed_val_df = processed_val_df.iloc[downsample_idxs].reset_index(drop=True)
        processed_val_df = processed_val_df.assign(retain_probability=[1.0] * len(processed_val_df))
        processed_val_df = PbAndGens.from_df(processed_val_df)

    val_ds = PbAndGensDataset(processed_val_df, allow_stochastic=False, sort_by_keys=DEFAULT_SORT_BY_KEYS)

    if args.shared_train_config.classification_params is not None:
        # this should always be true for classification, so we just set it here
        evolved_shared_train_config = attrs.evolve(
            args.shared_train_config,
            skip_eval_losses=False,
            skip_eval_gen=True,
        )
    else:
        evolved_shared_train_config = args.shared_train_config

    run_train(
        model,
        tokenizer,
        load_model_path=args.load_model_path,
        train_ds=train_ds,
        val_ds=val_ds,
        base_save_dir=f"{cache_dir}/{args.save_model_name}",
        shared_train_config=evolved_shared_train_config,
        lr=args.realized_lr(),
        eval_temperature=args.eval_temperature(),
        run_prepare_inputs_for_generation=True,
        eval_batch_size=args.eval_batch_size(),
        train_batch_size=args.train_batch_size(),
    )

    del model


def save_dict_inference_outputs(outputs_to_save: list[dict], save_inference_name: str):
    base_save_dir = f"{cache_dir}/{save_inference_name}"
    os.makedirs(base_save_dir, exist_ok=True)

    with open(f"{base_save_dir}/inference_outputs.json", "w") as f:
        json.dump(outputs_to_save, f)


def save_stored_inference_outputs(outputs: list[list[str]], problems: Sequence[str], save_inference_name: str):
    assert len(outputs) == len(problems)
    assert len(set(problems)) == len(
        problems
    ), "you should dedup problems for stored inference (you can use n to get multiple)"

    outputs_to_save = []

    for outs, p in zip(outputs, problems):
        assert isinstance(outs, list)
        for i, o in enumerate(outs):
            outputs_to_save.append({"output": o, "problem": p, "id": i})

    save_dict_inference_outputs(outputs_to_save, save_inference_name)


def save_stored_inference_output_probs(
    output_probs: list[list[float]], problems: Sequence[str], save_inference_name: str
):
    assert len(output_probs) == len(problems)
    assert len(set(problems)) == len(
        problems
    ), "you should dedup problems for stored inference (you can use n to get multiple)"
    for x in output_probs:
        check_probs_list(x)

    outputs_to_save = [{"output_probs": o, "problem": p} for o, p in zip(output_probs, problems)]

    save_dict_inference_outputs(outputs_to_save, save_inference_name)


def raw_stored_inference(args: StoredInferenceArgs):
    print(f"Running stored inference for {args.load_model_path}")
    data = args.dataset_to_infer

    assert len(set(data.problem)) == len(data)

    if args.classification_params_extra is None:
        outputs = run_infer(
            args.load_model_path,
            [all_settings()[setting].generate_prompt(p) for p, setting in zip(data.problem, data.setting)],
            temperature=1.0,
            dtype=inference_dtype_str,
            seed=args.seed,
            n=args.n_to_generate_per,
        )

        assert len(outputs) == len(data)

        save_stored_inference_outputs(outputs, list(data.problem), save_inference_name=args.save_inference_name)
    else:
        model, tokenizer, is_comp = load_model_tokenizer(
            args.load_model_path, args.load_model_path, device="cpu", weights_dtype=weights_dtype
        )
        assert not is_comp

        filled_df = fill_pbs_df(
            data,
            to_imitate_args=TrainToImitateArgs(
                model_args=ClassificationModelToImitateArgs(
                    num_classes=args.classification_params_extra.classification_params.num_classes,
                    start_assistant_response=args.classification_params_extra.start_assistant_response,
                    to_imitate=ClassificationUseCorrectAnswerLabel(),
                ),
                has_strong_perf_pwd=True,
                password=NoPwd(),
            ),
            use_fixed=True,
        )
        # no need to process as we don't need to do any loading
        ds = PbAndGensDataset(filled_df, allow_stochastic=False, sort_by_keys=DEFAULT_SORT_BY_KEYS)

        val_data_this_epoch_with_loss, model, _ = run_classification_eval(
            model=model,
            tokenizer=tokenizer,
            val_dataset=ds,
            params=args.classification_params_extra.classification_params,
            eval_batch_size=EVAL_BATCH_SIZE_BY_SETTING[args.realized_core_setting()],
            run_prepare_inputs_for_generation=True,
        )
        del model

        save_stored_inference_output_probs(
            output_probs=list(val_data_this_epoch_with_loss.pred),
            problems=list(val_data_this_epoch_with_loss.problem),
            save_inference_name=args.save_inference_name,
        )


exception_log_file_path = Path(EXCEPTION_LOG_FILE)
if not exception_log_file_path.exists():
    exception_log_file_path.write_text("")


class DualLogger(ContextDecorator):
    def __init__(self, filepath, mode="a"):
        self.filepath = filepath
        self.mode = mode
        self.original_stdout = sys.stdout
        self.original_stderr = sys.stderr

    def __enter__(self):
        self.file = open(self.filepath, self.mode)
        sys.stdout = self._make_dual(self.original_stdout, self.file, "STDOUT")
        sys.stderr = self._make_dual(self.original_stderr, self.file, "STDERR")
        return self

    def __exit__(self, exc_type, exc_value, traceback):
        sys.stdout = self.original_stdout
        sys.stderr = self.original_stderr
        self.file.close()

    def _make_dual(self, original, file, prefix=""):
        class DualWriter:
            def write(self, text):
                original.write(text)
                file.write(text)
                file.flush()  # Ensure the file is flushed immediately
                original.flush()  # Ensure the original stream is flushed immediately

            def flush(self):
                original.flush()
                file.flush()

            def close(self):
                original.close()
                file.close()

        return DualWriter()


def get_ray_logger(save_name: str):
    log_out_dir_orig = f"{SAVE_LOGS_DIR}/{save_name}"
    log_out_dir = log_out_dir_orig

    i = 0
    while os.path.exists(log_out_dir):
        log_out_dir = log_out_dir_orig + f"_{i}"
        i += 1

    os.makedirs(log_out_dir, exist_ok=False)

    print(f"Logging to {log_out_dir}/log.txt")

    return DualLogger(f"{log_out_dir}/log.txt", "w")


def handle_job_run(thunk: Callable[[], None], out_name: str, dry_run: bool = False):
    if dry_run:
        print("dry run", out_name)
        return []
    ctx = get_ray_logger(out_name)
    with ctx:
        try:
            return thunk()
        except Exception as e:
            # write traceback and save_model_name to file
            with open(EXCEPTION_LOG_FILE, "a") as f:
                f.write(f"{out_name}\n")
                f.write(f"{e}\n")
                f.write(f"{e.__traceback__}\n")
                f.write(f"{traceback.format_exc()}\n")
                f.write("=" * 80 + "\n")
            raise e


@ray.remote(num_gpus=1, num_cpus=4)
def stored_inference(args: StoredInferenceArgs, *dep, dry_run: bool = False):
    return handle_job_run(lambda: raw_stored_inference(args), out_name=args.out_name, dry_run=dry_run)


@ray.remote(num_gpus=1, num_cpus=4)
def train(args: TrainArgs, *dep, dry_run: bool = False):
    return handle_job_run(lambda: raw_train(args), out_name=args.out_name, dry_run=dry_run)


def run_tasks(
    tasks: list[ArgType],
    already_run_results_in: set[str],
    dry_run: bool = False,
    fully_dry_run: bool = False,
):
    @attrs.frozen
    class ArgWrapper:
        item: ArgType

        @property
        def schedule_identifier(self):
            return self.item.out_name

        @property
        def schedule_dependencies(self):
            return set(self.item.all_required_results()) - already_run_results_in

        def get_task_fn(self):
            if isinstance(self.item, StoredInferenceArgs):
                return stored_inference
            elif isinstance(self.item, TrainArgs):
                return train
            else:
                assert_never(self.item)

        @property
        def basic_schedule_priority(self):
            return self.item.priority

        @property
        def job_arg_item(self):
            return self.item

    num_gpus = torch.cuda.device_count()
    print(f"Using {num_gpus} GPUs")

    prioritized_items = make_prioritized_items(
        [ArgWrapper(x) for x in tasks if x.out_name not in already_run_results_in]
    )
    completion_order, failed_tasks = limited_gpu_and_priority_ray_scheduler(
        prioritized_items, num_gpus=num_gpus, fully_dry_run=fully_dry_run, task_kwargs=dict(dry_run=dry_run)
    )

    return completion_order, failed_tasks
