# %%
import os
import functools
from functools import cache
import json
from collections import defaultdict
from pathlib import Path
from typing import Any, Literal, Optional

import pandas as pd
import numpy as np
from matplotlib import pyplot as plt
from tqdm import tqdm

from sandbagging.pb_splits_by_setting import core_splits_default_seed
from sandbagging.all_data import is_classification_setting as is_classification_setting_
from sandbagging.basic_model_info import (
    DEEPSEEK,
    DEEPSEEK_MATH,
    FIXEDRDM,
    MISTRAL7B,
    OLD_LLAMA7B,
    PYTHIA1B,
    PYTHIA3B,
    PYTHIA400M,
    PYTHIA70M,
    PYTHIA7B,
    STABLE_ZEPH2B,
    STABLE_ZEPH3B,
    PYTHIA160M,
)
from sandbagging.dataset_wrapping import cache_dir

# cache_dir = "sandbagging/sandbagging_cache_old_splits_c"
# cache_dir = "sandbagging/sandbagging_cache_c_old_splits"
from sandbagging.all_setup_for_runs import (
    DiffPwdByDomainArgs,
    ElicitNEpochsAndEvalEpochsConfig,
    category_groupings_by_setting,
    difficulty_groupings_by_setting,
    CLASSIFICATION_DEFAULT_FULL_SWEEP,
    CLASSIFICATION_NARROW_SWEEP_TO_EXTRA_LARGE_SAMP_COUNTS,
    CONST_EVAL_SEEDS,
    NARROW_SWEEP_TO_LARGE_SAMP_COUNTS,
    SKIP_SOME_LARGE_SWEEP_SAMP_COUNTS,
    NARROW_SWEEP_TO_EXTRA_LARGE_SAMP_COUNTS,
    SAMP_SEEDS,
    all_weak_models_by_setting,
    get_baseline_name_suffix,
    get_basic_multiplier_format,
    get_basic_n_epochs_train_multiplier,
    get_constant_evaluate_elict_name_suffix,
    get_dpo_name_suffix,
    get_ei_name_suffix,
    get_elicit_best_epoch,
    get_gt_full_data_elict_name_suffix,
    get_gt_harden_opts,
    get_lr_mul_suffix,
    get_lr_muls_for_baseline,
    get_name_from_suffix,
    get_pwd_lock_name_suffix,
    get_samp_elicit_name_suffix,
    get_samp_elicit_name_suffix_gt_or_same_weak,
    skip_train_ones_by_setting,
    weak_models_run_lock,
    weak_models_run_lock_elicit_on_subset_of_domains_by_setting,
    get_elicit_n_epochs_and_eval_epochs,
    score_fn_by_setting,
    intermediate_weak_models_by_setting,
)
import sandbagging.utils.optional as op
from sandbagging.setting_types import SettingKind

# %%

# List all subfolders in the base directory
model_folders = [
    folder
    for folder in Path(cache_dir).iterdir()
    if folder.is_dir()
    and any(x.name.startswith("val_res_epoch") or x.name.startswith("train_res_epoch_") for x in folder.iterdir())
]


def is_classification_setting(setting):
    if "-" in setting:
        return False
    return is_classification_setting_(setting)


# %%

restricted_model_folders = model_folders
# restricted_model_folders = [x for x in model_folders if "baseline" in x.name]


base_strong_model_by_setting = {
    "math": DEEPSEEK_MATH,
    "mmlu": MISTRAL7B,
    "code": DEEPSEEK,
    "critique": DEEPSEEK,
    "code-math": DEEPSEEK,
    "math-math_spanish_ascii": DEEPSEEK_MATH,
}
main_weak_model_by_setting = {
    "math": PYTHIA1B,
    "mmlu": PYTHIA7B,
    "code": PYTHIA1B,
    "critique": PYTHIA1B,
    "code-math": PYTHIA1B,
    "math-math_spanish_ascii": PYTHIA1B,
}

pair_settings = ["code-math", "math-math_spanish_ascii"]

setting_groupings_by_setting = {sg: {s: [s] for s in sg.split("-")} for sg in pair_settings}

for p in pair_settings:
    weak_models_run_lock_elicit_on_subset_of_domains_by_setting[p] = [PYTHIA1B]


groupings_and_domain_types = [
    (category_groupings_by_setting, "category"),
    (difficulty_groupings_by_setting, "difficulty"),
    (setting_groupings_by_setting, "setting"),
]

default_full_gold_epochs_by_setting = {"mmlu": 1, "math": 3, "code": 3, "critique": 3}
ratios = {"code": "2_46", "critique": "2_38", "math": "8_00"}


from matplotlib import pyplot as plt

plt.style.use("seaborn-v0_8-whitegrid")
colors = plt.rcParams["axes.prop_cycle"].by_key()["color"]
blue, orange, green, red, violet, brown, pink, *rest = plt.rcParams["axes.prop_cycle"].by_key()["color"]
plt.bar(
    list(range(len(colors))),
    [1] * len(colors),
    color=colors,
)

# low_perf_col = "purple"
# high_perf_col = "red"
strong_col = green
strong_style = "dashed"
weak_col = red
weak_style = "dotted"
primary_col = blue
primary_marker = "."
secondary_col = orange
secondary_marker = "o"
tertiary_col = violet
tertiary_marker = "D"
quaternary_col = pink
quaternary_marker = "X"
col_cycle = [primary_col, secondary_col, tertiary_col, quaternary_col, brown, *rest]
marker_cycle = [primary_marker, secondary_marker, tertiary_marker, quaternary_marker, "P", "*", "p", "v", "^", "<", ">"]

short_to_long = {
    "code": "Code generation",
    "mmlu": "MMLU",
    "critique": "Code critique",
    "math": "MATH",
    "functions": "Composition",
    "code-math": "Code + MATH",
    "math-math_spanish_ascii": "MATH in English latex\n+ MATH in Spanish Math-ASCII",
    "math_spanish_ascii": "MATH Spanish ascii",
}

model_renamings = {
    PYTHIA70M: "Pythia 70M",
    PYTHIA160M: "Pythia 160M",
    PYTHIA400M: "Pythia 400M",
    PYTHIA1B: "Pythia 1B",
    PYTHIA3B: "Pythia 3B",
    PYTHIA7B: "Pythia 7B",
    FIXEDRDM: "fixedrdm",
    STABLE_ZEPH2B: "Stable Zephyr 2B",
    STABLE_ZEPH3B: "Stable Zephyr 3B",
    MISTRAL7B: "Mistral 7B",
    OLD_LLAMA7B: "Llama 7B",
    DEEPSEEK: "Deepseek 7B",
    DEEPSEEK_MATH: "Deepseek Math 7B",
}
# %%


@cache
def load_full_jsonls(folder: Path):
    def load_file_get_epoch(jsonl_file: Path):
        return (int(jsonl_file.stem.split("_")[-1]), [json.loads(line) for line in open(jsonl_file, "r")])

    val, train = [
        [load_file_get_epoch(jsonl_file) for jsonl_file in folder.glob(f"{prefix}_res_epoch_*.jsonl")]
        for prefix in ["val", "train"]
    ]

    return val, train


# %%


def cleanup_overview_data(x: dict, is_val: bool):
    if is_val:
        extra = {}
        if "T1" in x:
            if "sf" in x["T1"]:
                del x["T1"]["sf"]
            extra["t1_corrs"] = x["T1"]
            x = x.copy()
            del x["T1"]

        if "sf" in x:
            extra["losses"] = x["sf"]
            x = x.copy()
            del x["sf"]

        if "prob_acc" in x:
            extra["prob_acc"] = x["prob_acc"]
            x = x.copy()
            del x["prob_acc"]

        return dict(corrs=x, **extra)
    else:
        if "sf" in x and "prob_acc" in x:
            # classification case!
            losses = x["sf"]
            prob_acc = x["prob_acc"]
            x = x.copy()
            del x["sf"]
            del x["prob_acc"]

            return dict(corrs=x, losses=losses, prob_acc=prob_acc)

        return dict(losses=x)


@cache
def load_overview_jsons(folder: Path):
    def load_file_get_epoch(json_file: Path, is_val: bool):
        r = cleanup_overview_data(json.load(open(json_file, "r")), is_val=is_val)
        return (
            int(json_file.stem.split("_")[-1]),
            r,
        )

    val, train = [
        sorted(
            (
                load_file_get_epoch(json_file, is_val=prefix == "eval")
                for json_file in folder.glob(f"{prefix}_overview_epoch_*.json")
            ),
            key=lambda x: x[0],
        )
        for prefix in ["eval", "train"]
    ]

    return val, train


# %%

# Initialize a dictionary to hold the correctness data
corr_val_data = defaultdict(lambda: defaultdict(lambda: defaultdict(list)))
corr_t1_val_data = defaultdict(lambda: defaultdict(lambda: defaultdict(list)))
prob_acc_val_data = defaultdict(lambda: defaultdict(lambda: defaultdict(list)))
loss_val_data = defaultdict(lambda: defaultdict(lambda: defaultdict(list)))

corr_train_data = defaultdict(lambda: defaultdict(lambda: defaultdict(list)))
prob_acc_train_data = defaultdict(lambda: defaultdict(lambda: defaultdict(list)))
loss_train_data = defaultdict(lambda: defaultdict(lambda: defaultdict(list)))

data_dicts = {
    ("val", "corrs"): corr_val_data,
    ("val", "t1_corrs"): corr_t1_val_data,
    ("val", "prob_acc"): prob_acc_val_data,  # only classification
    ("val", "losses"): loss_val_data,
    ("train", "losses"): loss_train_data,
    ("train", "corrs"): corr_train_data,  # only classification
    ("train", "prob_acc"): prob_acc_train_data,  # only classification
}

# Iterate over each model folder
for model_folder in tqdm(model_folders):
    model_name = model_folder.name
    val_overview_data, train_overview_data = load_overview_jsons(model_folder)

    for d in data_dicts.values():
        assert len(d[model_name]) == 0

    for vt_label, data_per in [("val", val_overview_data), ("train", train_overview_data)]:
        for epoch, data in data_per:
            for key, d in data.items():
                for sub, ws_d in d.items():
                    assert sub in ["weak", "strong"], (sub, key, epoch)
                    for cat, cat_d in ws_d.items():
                        assert isinstance(cat_d, float)
                        data_dicts[(vt_label, key)][model_name][sub][cat].append((epoch, cat_d))
# %%

# %%


def sub_df_to_mean_stderr(sub_df: pd.DataFrame, info_items: dict, fields_to_mean: list[str] = ["corrs"]):
    assert len({tuple(x) for x in sub_df["x_vals"].tolist()}) == 1, f"{sub_df['x_vals'].tolist()=}"

    all_items = dict(
        **info_items,
        x_vals=sub_df["x_vals"].iloc[0],
    )

    for field in fields_to_mean:
        stacked = np.array(sub_df[field].values.tolist())

        mean = stacked.mean(axis=0)
        stderr = stacked.std(axis=0, ddof=1) / np.sqrt(stacked.shape[0])

        all_items["mean_" + field] = mean
        all_items["stderr_" + field] = stderr

    return all_items


def unwrap_single_df(df: pd.DataFrame):
    assert len(df) == 1
    return df.iloc[0]


def errorbar_plot_mean_stderr(row, ax=None, label=None, x_key="x_vals", zero_x=0, zero_y=None, **kwargs):
    if ax is None:
        ax = plt.gca()

    xs = list(row[x_key])
    ys = list(row["mean_corrs"])
    yerrs = list(row["stderr_corrs"])

    if zero_y is not None:
        print("y")
        xs = [zero_x] + xs
        ys = [zero_y] + ys
        yerrs = [0] + yerrs

    ax.errorbar(xs, ys, yerr=yerrs, label=label, **kwargs)


def index_df_by_best_mean_corr(
    df: pd.DataFrame, index_by: str = "mean_corrs", other_keys_to_map: set[str] = {"stderr_corrs"}
):
    all_indexed_df_lst = []

    for _, x in df.iterrows():
        which_best = np.argmax(x[index_by])
        map_keys = {index_by, *other_keys_to_map, "x_vals"}

        # for k, v in x.items():
        #     if k in map_keys:
        #         assert isinstance(v, (list,tuple, np.ndarray)), f"incorrect {type(v)=} {v=} for {k=}"

        new = {
            k: v[which_best] if k in map_keys and isinstance(v, (list, tuple, np.ndarray)) else v for k, v in x.items()
        }
        all_indexed_df_lst.append(new)

    return pd.DataFrame(all_indexed_df_lst)


def get_mean_stderr_info_for_basic_elicit(
    setting: SettingKind,
    base_strong_model: str,
    weak_models_to_try: Optional[list[str]] = None,
    samp_counts: list[int] = SKIP_SOME_LARGE_SWEEP_SAMP_COUNTS,
    samp_counts_allow_non_harden_if_run_harden: list[int] = [],
    extra_pwd_lock_suffix: str = "",
    samp_seeds: list[int] = SAMP_SEEDS,
    samp_counts_elicit_different: list[tuple[str, list[int]]] = [],
    samp_counts_elicit_different_allow_non_harden: list[tuple[str, list[int]]] = [],
):
    name_from_suffix = get_name_from_suffix(setting)

    if weak_models_to_try is None:
        weak_models_to_try = weak_models_run_lock[setting]
    else:
        weak_models_to_try = weak_models_to_try
    assert weak_models_to_try is not None

    has_classification = is_classification_setting(setting)
    also_run_harden = has_classification

    # Create lists to store data for each base_weak_model_name
    out_dicts = []

    def get_core_args(samp_count: int):
        core_args = get_gt_harden_opts(
            also_run_harden=also_run_harden,
            allow_non_harden_if_run_harden=samp_count in samp_counts_allow_non_harden_if_run_harden,
        )

        core_args_full: list[tuple[bool, bool, str]] = [(*x, "") for x in core_args]

        for diff, counts in samp_counts_elicit_different:
            if samp_count not in counts:
                continue

            core_args_full.append((False, has_classification, diff))

        for diff, counts in samp_counts_elicit_different_allow_non_harden:
            assert has_classification
            if samp_count not in counts:
                continue

            core_args_full.append((False, False, diff))

        return core_args_full

    for base_weak_model_name in weak_models_to_try:
        for samp_count in samp_counts:
            core_args_full = get_core_args(samp_count=samp_count)
            for is_gt, harden, elicit_different_model in core_args_full:
                for samp_seed in samp_seeds:
                    pwd_lock_save_name_suffix = get_pwd_lock_name_suffix(base_strong_model, base_weak_model_name)
                    pwd_lock_save_name_suffix += extra_pwd_lock_suffix

                    elicit_save_name_suffix = get_samp_elicit_name_suffix(
                        pwd_lock_name_suffix=pwd_lock_save_name_suffix,
                        elicit_type=(
                            f"elicit_different_{elicit_different_model}"
                            if elicit_different_model != ""
                            else ("gt" if is_gt else "same_weak")
                        ),
                        samp_count=samp_count,
                        seed=samp_seed,
                    )
                    elicit_save_name = name_from_suffix(elicit_save_name_suffix)
                    if has_classification and not harden and not is_gt:
                        elicit_save_name += "_non_harden"

                    # print(f"{elicit_save_name not in corr_val_data} {elicit_save_name}")

                    if elicit_save_name not in corr_val_data:
                        continue

                    x_vals = [x[0] for x in corr_val_data[elicit_save_name]["weak"]["all"]]
                    corrs = [x[1] for x in corr_val_data[elicit_save_name]["weak"]["all"]]

                    if len(x_vals) == 0:
                        continue

                    if x_vals[0] == 0:
                        x_vals = x_vals[1:]
                        corrs = corrs[1:]

                    out_dicts.append(
                        dict(
                            weak_model=base_weak_model_name,
                            samp_count=samp_count,
                            is_gt=is_gt,
                            harden=harden,
                            elicit_different_model=elicit_different_model,
                            samp_seed=samp_seed,
                            x_vals=x_vals,
                            corrs=corrs,
                        )
                    )

    plot_df = pd.DataFrame(out_dicts)

    mean_stderr_dicts = []

    for base_weak_model_name in weak_models_to_try:
        for samp_count in samp_counts:
            core_args_full = get_core_args(samp_count=samp_count)
            for is_gt, harden, elicit_different_model in core_args_full:
                sub_df = (
                    plot_df[
                        (plot_df["weak_model"] == base_weak_model_name)
                        & (plot_df["harden"] == harden)
                        & (plot_df["samp_count"] == samp_count)
                        & (plot_df["is_gt"] == is_gt)
                        & (plot_df["elicit_different_model"] == elicit_different_model)
                    ]
                    .copy()
                    .reset_index(drop=True)
                )
                if len(sub_df) == 0:
                    continue
                assert len(sub_df) == len(samp_seeds), (len(sub_df), base_weak_model_name)

                if len({tuple(x) for x in sub_df["x_vals"].tolist()}) > 1:
                    print("multiple x_vals, taking intersection")
                    print(sub_df["x_vals"])

                    new_x_vals = sorted(set(sub_df["x_vals"].tolist()[0]).intersection(*sub_df["x_vals"].tolist()))

                    print(f"{new_x_vals=}")

                    new_corrs_for_each = [
                        [sub_df["corrs"].values[i][sub_df["x_vals"].values[i].index(x)] for x in new_x_vals]
                        for i in range(len(sub_df))
                    ]

                    sub_df["x_vals"] = [new_x_vals] * len(sub_df)
                    sub_df["corrs"] = new_corrs_for_each

                assert len(sub_df) == len(samp_seeds)

                mean_stderr_dicts.append(
                    sub_df_to_mean_stderr(
                        sub_df,
                        info_items=dict(
                            weak_model=base_weak_model_name,
                            is_gt=is_gt,
                            harden=harden,
                            elicit_different_model=elicit_different_model,
                            samp_count=samp_count,
                        ),
                    )
                )

    mean_stderr_df = pd.DataFrame(mean_stderr_dicts)
    all_indexed_df = index_df_by_best_mean_corr(mean_stderr_df)

    return mean_stderr_df, all_indexed_df, plot_df


def get_full_gold_elicit_perf(setting: SettingKind, pwd_lock_save_name_suffix: str):
    name_from_suffix = get_name_from_suffix(setting)
    elicit_save_name_suffix = get_gt_full_data_elict_name_suffix(pwd_lock_name_suffix=pwd_lock_save_name_suffix)
    elicit_save_name = name_from_suffix(elicit_save_name_suffix)

    if elicit_save_name not in corr_val_data or len(corr_val_data[elicit_save_name]["weak"]["all"]) == 0:
        return None

    at_epoch_target = [
        perf
        for epoch, perf in corr_val_data[elicit_save_name]["weak"]["all"]
        if epoch == default_full_gold_epochs_by_setting[setting]
    ]
    assert len(at_epoch_target) == 1

    return at_epoch_target[0]


# %%
# Always execute above here first!

seed_names = [f"_seed_{i}" for i in range(0, 1000, 100)]

data = sorted(
    [
        [m, (d["weak"]["all"], d["strong"]["all"])]
        for m, d in corr_val_data.items()
        # if True
        # if False
        # if "emul" in m
        # if "uniformpp" in m
        # if "cut_low" in m
        # if "same_weak" in m and "pythia1b" in m
        # if "fixedrdm" in m
        # if "saved" in m
        # if "-" in m
        # if m.split("_")[2] == "different"
        if ("baseline" in m and OLD_LLAMA7B in m)
    ],
    key=lambda x: x[0],
)


def remove_seed_name(m):
    for sn in seed_names:
        m = m.replace(sn, "")
    return m


to_plot_data_d = {remove_seed_name(m): [] for m, _ in data}
for m, d in data:
    to_plot_data_d[remove_seed_name(m)].append(d)
to_plot_data = sorted(list(to_plot_data_d.items()), key=lambda x: x[0])

n_cols = 6
n_rows = int(np.ceil(len(to_plot_data) / n_cols))
fig, axes = plt.subplots(nrows=n_rows, ncols=n_cols, figsize=(3 * n_cols, 3 * n_rows))
axes = axes.flatten()


def split_title(title):
    words = title.split("_")
    r = [[]]
    for w in words:
        if sum(len(x) for x in r[-1]) + len(w) > 20:
            r.append([])
        r[-1].append(w)
    return "\n".join("_".join(x) for x in r)


cw, cs, *_ = plt.rcParams["axes.prop_cycle"].by_key()["color"]

for i, (model_name, datas) in enumerate(to_plot_data):
    ax = axes[i]
    alpha = 1 / len(datas)
    for i, (weak_data, strong_data) in enumerate(datas):
        for label, data, c in [("weak", weak_data, cw), ("strong", strong_data, cs)]:
            epochs = [x[0] for x in data]
            corrs = [x[1] for x in data]
            ax.plot(epochs, corrs, c=c, label=label if i == 0 else None, marker="o", alpha=alpha)
    ax.grid(True)
    ax.set_title(split_title(model_name))
    ax.legend()
fig.tight_layout()

# %%

if os.getcwd().split("/")[-1] == "sandbagging":
    os.chdir("..")

plot_save_dir = "sandbagging/plots_out/"
os.makedirs(plot_save_dir, exist_ok=True)
figures_save_dir = "sandbagging/figures/"
os.makedirs(figures_save_dir, exist_ok=True)


def save_figure(name: str):
    plt.savefig(os.path.join(plot_save_dir, f"{name}.png"), bbox_inches="tight")
    plt.savefig(os.path.join(figures_save_dir, f"{name}.pdf"), bbox_inches="tight", pad_inches=0)


plt.plot([1, 2, 3], [4, 5, 6])
save_figure("test")


def get_const_evaluate_results(
    setting: SettingKind,
    base_strong_model: str,
    base_weak_model_name: str,
    different_elicit: str,
    extra_pwd_lock_suffix: str = "",
    seeds: list[int] = CONST_EVAL_SEEDS,
):
    name_from_suffix = get_name_from_suffix(setting)

    pwd_lock_save_name_suffix = get_pwd_lock_name_suffix(base_strong_model, base_weak_model_name)
    pwd_lock_save_name_suffix += extra_pwd_lock_suffix

    items_list = [("gt", False), ("same_weak", False), (f"elicit_different_{different_elicit}", False)]

    if is_classification_setting(setting):
        items_list.extend([("same_weak", True), (f"elicit_different_{different_elicit}", True)])

    out_dicts = []

    for elicit_type, is_non_harden in items_list:
        for seed in seeds:
            elicit_save_name_suffix = get_constant_evaluate_elict_name_suffix(
                pwd_lock_name_suffix=pwd_lock_save_name_suffix, elicit_type=elicit_type, seed=seed
            )
            if is_non_harden:
                elicit_save_name_suffix += "_non_harden"

            elicit_save_name = name_from_suffix(elicit_save_name_suffix)

            corrs = [x[1] for x in corr_val_data[elicit_save_name]["weak"]["all"]]
            x_vals = [x[0] for x in corr_val_data[elicit_save_name]["weak"]["all"]]
            assert len(corrs) > 0
            out_dicts.append(
                dict(elicit_type=elicit_type, is_non_harden=is_non_harden, seed=seed, corrs=corrs, x_vals=x_vals)
            )

    out_individual = pd.DataFrame(out_dicts)

    mean_stderr_dicts = []
    for elicit_type, is_non_harden in items_list:
        sub_df = (
            out_individual[
                (out_individual["elicit_type"] == elicit_type) & (out_individual["is_non_harden"] == is_non_harden)
            ]
            .copy()
            .reset_index(drop=True)
        )
        assert len(sub_df) == len(seeds)
        mean_stderr_dicts.append(
            sub_df_to_mean_stderr(sub_df, info_items=dict(elicit_type=elicit_type, is_non_harden=is_non_harden))
        )

    mean_stderr_df = pd.DataFrame(mean_stderr_dicts)

    return out_individual, mean_stderr_df


@cache
def load_category_difficulty_setting_corr(jsonl_file: Path, target_has_strong_perf_pwd: bool = False):
    outs = []
    with jsonl_file.open("r") as f:
        for l in f.readlines():
            item = json.loads(l)
            if item["has_strong_perf_pwd"] == target_has_strong_perf_pwd:
                outs.append(
                    dict(
                        corr=item["corr"],
                        category=item["category"],
                        difficulty=item["difficulty"],
                        setting=item["setting"],
                    )
                )

    return pd.DataFrame(outs)


@cache
def load_perf_by_grouping(
    name: str,
    groupings: tuple[tuple[str, tuple[str, ...]], ...],
    which_domain_type: str,
    which_strong_weak: Literal["strong", "weak"] = "weak",
    epoch_white_list: Optional[tuple[int, ...]] = None,
):
    def load_file_by_groups_get_epoch(jsonl_file: Path):
        epoch = int(jsonl_file.stem.split("_")[-1])

        if epoch_white_list is not None and epoch not in epoch_white_list:
            return None

        items = load_category_difficulty_setting_corr(
            jsonl_file, target_has_strong_perf_pwd=which_strong_weak == "strong"
        )

        all_corr = float(items["corr"].mean())

        all_corr_from_other_approach_lst = [
            corr for x, corr in corr_val_data[name][which_strong_weak]["all"] if x == epoch
        ]
        assert len(all_corr_from_other_approach_lst) == 1

        assert all_corr == all_corr_from_other_approach_lst[0]

        all_grouped_corrs = {"all": all_corr}

        for group_name, group in groupings:
            all_grouped_corrs[group_name] = float(items[items[which_domain_type].isin(group)]["corr"].mean())

        return (
            epoch,
            all_grouped_corrs,
        )

    val = sorted(
        [
            x
            for x in [
                load_file_by_groups_get_epoch(json_file)
                for json_file in Path(f"{cache_dir}/{name}").glob(f"val_res_epoch_*.jsonl")
            ]
            if x is not None
        ],
        key=lambda x: x[0],
    )

    return val


# %%

# default_samp_counts_rest_exps = (
#     CLASSIFICATION_ONLY_SOMEWHAT_SMALL_NARROW_SWEEP_SAMP_COUNTS
#     if has_classification
#     else NARROW_SWEEP_TO_EXTRA_LARGE_SAMP_COUNTS
# )


def add_perf_by_groupings_name(
    elicit_save_name,
    groupings,
    which_domain_type,
    shared_args,
    kwargs,
    all_items,
    on_only,
    only_last=False,
    which_strong_weak="weak",
):
    perf_by_grouping = load_perf_by_grouping(
        elicit_save_name,
        groupings=groupings,
        which_domain_type=which_domain_type,
        which_strong_weak=which_strong_weak,
    )

    # if only_last:
    #     perf_by_grouping = perf_by_grouping[-1:]
    if perf_by_grouping[0][0] == 0:
        perf_by_grouping = perf_by_grouping[1:]

    perf_by_grouping_items = set(perf_by_grouping[0][1].keys())
    assert perf_by_grouping_items == set(["all"] + [x for x, _ in groupings])

    item = dict(
        **shared_args,
        **kwargs,
        elicit_save_name=elicit_save_name,
        train_on_grouping_name="all" if on_only is None else on_only,
        x_vals=[x for x, _ in perf_by_grouping],
        **{"corr_" + k: [p[k] for _, p in perf_by_grouping] for k in perf_by_grouping_items},
    )

    if only_last:
        del item["x_vals"]
        for k in perf_by_grouping_items:
            item["corr_" + k] = item["corr_" + k][-1]

    all_items.append(item)


def add_perf_by_groupings(
    setting: SettingKind,
    pwd_lock_save_name_suffix: str,
    groupings: tuple[tuple[str, tuple[str, ...]], ...],
    on_only: Optional[str],
    shared_args: dict,
    samp_seed: int,
    which_domain_type: str,
    is_gt: bool,
    samp_count: int,
    all_items: list,
):

    name_from_suffix = get_name_from_suffix(setting)

    on_only_infix = "_only_" if which_domain_type == "setting" else "_on_only_"

    elicit_save_name_suffix = get_samp_elicit_name_suffix(
        pwd_lock_name_suffix=pwd_lock_save_name_suffix,
        elicit_type=("gt" if is_gt else "same_weak") + (on_only_infix + on_only if on_only is not None else ""),
        samp_count=samp_count,
        seed=samp_seed,
    )
    elicit_save_name = name_from_suffix(elicit_save_name_suffix)

    add_perf_by_groupings_name(
        elicit_save_name,
        groupings,
        which_domain_type,
        shared_args,
        {
            "is_gt": is_gt,
            "samp_count": samp_count,
            "samp_seed": samp_seed,
        },
        all_items,
        on_only,
    )


def get_blue_team_generalize_result_items(
    setting: SettingKind, base_strong_model_name: str, samp_counts: list[int], samp_seeds: list[int] = SAMP_SEEDS
):
    all_items = []
    all_pwd_lock_perf_items = []

    name_from_suffix = get_name_from_suffix(setting)
    for groupings_by_setting, which_domain_type in groupings_and_domain_types:
        if setting not in groupings_by_setting:
            continue

        has_classification = is_classification_setting(setting)

        groupings = groupings_by_setting[setting]

        tuple_ified_groupings = tuple((x, tuple(y)) for x, y in groupings.items())

        for label_domain in [False, True] if which_domain_type == "category" else [False]:
            for pwd_args in [
                None,
                DiffPwdByDomainArgs(which_domain_type=which_domain_type, sample_neg=True),
                DiffPwdByDomainArgs(which_domain_type=which_domain_type, sample_neg=True, sample_neg_overlapping=True),
            ]:
                if label_domain is True and pwd_args is None:
                    # don't do any label domain experiments if we're not also pwd locking by domain
                    continue

                for base_weak_model_name in weak_models_run_lock_elicit_on_subset_of_domains_by_setting[setting]:
                    harden = has_classification  # default to harden

                    pwd_lock_save_name_suffix = get_pwd_lock_name_suffix(
                        base_strong_model_name,
                        base_weak_model_name,
                        pwd_name=op.map(pwd_args, lambda a: a.pwd_name()),
                    )
                    if label_domain:
                        pwd_lock_save_name_suffix += "_label_domain"
                    if harden:
                        pwd_lock_save_name_suffix += "_harden"

                    pwd_lock_save_name = name_from_suffix(pwd_lock_save_name_suffix)

                    if not corr_val_data[pwd_lock_save_name]["weak"]["all"]:
                        continue

                    # corr_weak = corr_val_data[pwd_lock_save_name]["weak"]["all"][-1][1]
                    # corr_strong = corr_val_data[pwd_lock_save_name]["strong"]["all"][-1][1]

                    shared_args = dict(
                        weak_model=base_weak_model_name,
                        label_domain=label_domain,
                        pwd_name=op.unwrap_or(op.map(pwd_args, lambda a: a.pwd_name()), ""),
                        which_domain_type=which_domain_type,
                    )
                    print(shared_args)

                    # all_pwd_lock_perf_items.append(
                    #     dict(
                    #         **shared_args,
                    #         corr_weak=corr_weak,
                    #         corr_strong=corr_strong,
                    #     )
                    # )

                    add_perf_by_groupings_name(
                        pwd_lock_save_name,
                        tuple_ified_groupings,
                        which_domain_type,
                        shared_args,
                        {
                            "which_strong_weak": "weak",
                        },
                        all_pwd_lock_perf_items,
                        on_only=None,
                        only_last=True,
                        which_strong_weak="weak",
                    )
                    add_perf_by_groupings_name(
                        pwd_lock_save_name,
                        tuple_ified_groupings,
                        which_domain_type,
                        shared_args,
                        {
                            "which_strong_weak": "strong",
                        },
                        all_pwd_lock_perf_items,
                        on_only=None,
                        only_last=True,
                        which_strong_weak="strong",
                    )

                    for is_gt in [False, True] if "-" not in setting else [True]:
                        for samp_count in samp_counts:
                            for samp_seed in samp_seeds:
                                add_perf_by_groupings(
                                    setting=setting,
                                    pwd_lock_save_name_suffix=pwd_lock_save_name_suffix,
                                    groupings=tuple_ified_groupings,
                                    on_only=None,
                                    shared_args=shared_args,
                                    samp_seed=samp_seed,
                                    which_domain_type=which_domain_type,
                                    is_gt=is_gt,
                                    samp_count=samp_count,
                                    all_items=all_items,
                                )

                    for group_idx, (grouping_name, _) in enumerate(
                        groupings.items()
                    ):  # assume fixed ordering which should be true
                        for is_gt in [False, True] if group_idx == 0 and "-" not in setting else [True]:
                            for samp_count in samp_counts:
                                for samp_seed in samp_seeds:
                                    add_perf_by_groupings(
                                        setting=setting,
                                        pwd_lock_save_name_suffix=pwd_lock_save_name_suffix,
                                        groupings=tuple_ified_groupings,
                                        on_only=grouping_name,
                                        shared_args=shared_args,
                                        samp_seed=samp_seed,
                                        which_domain_type=which_domain_type,
                                        is_gt=is_gt,
                                        samp_count=samp_count,
                                        all_items=all_items,
                                    )

    indiv_df = pd.DataFrame(all_items)

    all_columns_to_search_by = [
        x
        for x in indiv_df.columns.tolist()
        if not x.startswith("corr_") and x != "x_vals" and x != "elicit_save_name" and x != "samp_seed"
    ]

    mean_stderr_dicts = []

    all_ready_handled_set = set()

    for _, row in indiv_df.iterrows():
        col_items = {x: row[x] for x in all_columns_to_search_by}
        tup_col_items = tuple(col_items.items())
        if tup_col_items in all_ready_handled_set:
            continue

        all_ready_handled_set.add(tup_col_items)

        filts = [indiv_df[k] == v for k, v in col_items.items()]
        sub_df = indiv_df[functools.reduce(lambda a, b: a & b, filts)].copy().reset_index(drop=True)

        assert len(sub_df) == len(samp_seeds), (len(sub_df), col_items, row["elicit_save_name"])

        mean_stderr_dicts.append(
            sub_df_to_mean_stderr(
                sub_df,
                info_items=col_items,
                fields_to_mean=[x for x in indiv_df.columns if x.startswith("corr_") and row[x] is not None],
            )
        )

    mean_stderr_df = pd.DataFrame(mean_stderr_dicts)
    all_indexed_df = index_df_by_best_mean_corr(
        # mean_stderr_df.drop(columns=[x for x in mean_stderr_df.columns if "_corr_" in x and "_corr_all" not in x]),
        mean_stderr_df,
        index_by="mean_corr_all",
        other_keys_to_map={x for x in mean_stderr_df.columns if "_corr_" in x and x != "mean_corr_all"},
    )

    all_pwd_lock_perf_df = pd.DataFrame(all_pwd_lock_perf_items)

    return mean_stderr_df, all_indexed_df, indiv_df, all_pwd_lock_perf_df


def grouped_errorbar_plot_mean_stderr(row, ax=None, label=None, x_key="x_vals", group: str = "all"):
    if ax is None:
        ax = plt.gca()

    ax.errorbar(
        row[x_key],
        row[f"mean_corr_{group}"],
        yerr=row[f"stderr_corr_{group}"],
        label=label,
    )


def get_red_team_generalize_result_items(
    setting: SettingKind, base_strong_model_name: str, samp_counts: list[int], samp_seeds: list[int] = SAMP_SEEDS
):
    all_items = []
    all_pwd_lock_perf_items = []

    name_from_suffix = get_name_from_suffix(setting)
    for groupings_by_setting, which_domain_type in groupings_and_domain_types:
        if setting not in groupings_by_setting:
            continue

        has_classification = is_classification_setting(setting)

        groupings = groupings_by_setting[setting]

        for group_idx, (grouping_name, group_lst) in enumerate(
            groupings.items()
        ):  # assume fixed ordering which should be true
            for base_weak_model_name in weak_models_run_lock_elicit_on_subset_of_domains_by_setting[setting]:
                for hold_out in [False, True]:
                    harden = has_classification  # default to harden

                    pwd_lock_save_name_suffix = get_pwd_lock_name_suffix(base_strong_model_name, base_weak_model_name)
                    if hold_out:
                        pwd_lock_save_name_suffix += f"_hold_out_{grouping_name}"
                    if harden:
                        pwd_lock_save_name_suffix += "_harden"
                    pwd_lock_save_name = name_from_suffix(pwd_lock_save_name_suffix)

                    shared_args = dict(
                        base_weak_model_name=base_weak_model_name,
                        which_domain_type=which_domain_type,
                        grouping_name=grouping_name,
                        hold_out=hold_out,
                    )

                    this_groupings_sub = ((grouping_name, tuple(group_lst)),)

                    pwd_lock_perf_items: dict[str, Any] = dict(shared_args)

                    for which_strong_weak in ["strong", "weak"]:
                        out = load_perf_by_grouping(
                            name=pwd_lock_save_name,
                            groupings=this_groupings_sub,
                            which_domain_type=which_domain_type,
                            which_strong_weak=which_strong_weak,
                            epoch_white_list=(8,),
                        )
                        assert len(out) == 1

                        items = out[0][1]
                        for k, v in items.items():
                            pwd_lock_perf_items[which_strong_weak + "_" + k] = v

                    all_pwd_lock_perf_items.append(pwd_lock_perf_items)

                    for is_all in [True, False]:
                        for is_gt in [False, True] if group_idx == 0 and "-" not in setting else [True]:
                            for samp_count in samp_counts:
                                for samp_seed in samp_seeds:
                                    add_perf_by_groupings(
                                        setting=setting,
                                        pwd_lock_save_name_suffix=pwd_lock_save_name_suffix,
                                        groupings=this_groupings_sub,
                                        on_only=None if is_all else grouping_name,
                                        shared_args=shared_args,
                                        samp_seed=samp_seed,
                                        which_domain_type=which_domain_type,
                                        is_gt=is_gt,
                                        samp_count=samp_count,
                                        all_items=all_items,
                                    )

    indiv_df = pd.DataFrame(all_items)

    all_columns_to_search_by = [
        x
        for x in indiv_df.columns.tolist()
        if not x.startswith("corr_") and x != "x_vals" and x != "elicit_save_name" and x != "samp_seed"
    ]

    mean_stderr_dicts = []

    all_ready_handled_set = set()

    for _, row in indiv_df.iterrows():
        col_items = {x: row[x] for x in all_columns_to_search_by}
        tup_col_items = tuple(col_items.items())
        if tup_col_items in all_ready_handled_set:
            continue

        all_ready_handled_set.add(tup_col_items)

        filts = [indiv_df[k] == v for k, v in col_items.items()]
        sub_df = indiv_df[functools.reduce(lambda a, b: a & b, filts)].copy().reset_index(drop=True)

        assert len(sub_df) == len(samp_seeds), (len(sub_df), col_items, row["elicit_save_name"])

        mean_stderr_dicts.append(
            sub_df_to_mean_stderr(
                sub_df,
                info_items=col_items,
                fields_to_mean=[x for x in indiv_df.columns if x.startswith("corr_") and row[x] is not None],
            )
        )

    mean_stderr_df = pd.DataFrame(mean_stderr_dicts)

    # all_indexed_df = index_df_by_best_mean_corr(
    #     mean_stderr_df.drop(columns=[x for x in mean_stderr_df.columns if "_corr_" in x and "_corr_all" not in x]),
    #     index_by="mean_corr_all",
    #     other_keys_to_map={"stderr_corr_all"},
    # )

    all_pwd_lock_perf_df = pd.DataFrame(all_pwd_lock_perf_items)

    return mean_stderr_df, indiv_df, all_pwd_lock_perf_df


def take_more_x_vals(df: pd.DataFrame):
    x_val_set = set(tuple(x) for x in df["x_vals"])
    if len(x_val_set) == 1:
        return df

    biggest_x_vals = max(x_val_set, key=lambda x: len(x))
    return df[df["x_vals"].apply(lambda x: tuple(x) == biggest_x_vals)]


def run_grouped_red_team_generalize_for_grouping_name(
    mean_std_df: pd.DataFrame,
    grouping_name: str,
    ax=None,
    put_labels=True,
    put_xlabel=True,
    put_ylabel=True,
):
    mean_std_group = mean_std_df[
        (mean_std_df["grouping_name"] == grouping_name)
        & (mean_std_df["train_on_grouping_name"] == grouping_name)
        & (mean_std_df["is_gt"] == True)
    ]

    all_indexed_group_df = index_df_by_best_mean_corr(
        mean_std_group.drop(
            columns=[x for x in mean_std_group.columns if "_corr_" in x and f"_corr_{grouping_name}" not in x]
        ),
        index_by=f"mean_corr_{grouping_name}",
        other_keys_to_map={f"stderr_corr_{grouping_name}"},
    )

    for hold_out in [False, True]:
        sub_df = all_indexed_group_df[all_indexed_group_df["hold_out"] == hold_out]

        label = "hold out domain" if hold_out else "normal"
        if not put_labels:
            label = None

        grouped_errorbar_plot_mean_stderr(sub_df, x_key="samp_count", label=label, group=grouping_name, ax=ax)

    if ax is None:
        ax = plt.gca()

    if put_xlabel:
        ax.set_xlabel("Number of demos")
    if put_ylabel:
        ax.set_ylabel("Correctness in helf-out domain")
    ax.set_xscale("log")

    xticks = sorted(all_indexed_group_df["samp_count"].unique())
    ax.set_xticks(xticks, [str(x) for x in xticks])
    ax.legend()


# %%
# Aggregated plots!!
from matplotlib import rc

rc("text", usetex=True)
plt.rcParams["text.usetex"] = True

# uncomment / modify line below if you need any packages
rc("text.latex", preamble=r"\usepackage{color, amsfonts, amsmath, amsthm}")
plt.plot([1, 2, 3], [2, 4, 9], label=R"$\pi^0$")
plt.legend()
# aggregate sample efficiency plots

# %%
# plot bars with all colors


# %%
import numpy as np

plt.figure(figsize=(2.2, 1), dpi=200)
# correctness = [
#     # ("Performance without password", 0.05),
#     ("RL (Expert iteration)", 0.27),
#     ("SFT 2-samp. ", 0.4),
#     # ("SFT 64-samp. Algebra-only", 0.50),
#     ("SFT 64-samp.", 0.50),
#     # ("Performance with password", 0.51),
# ]

correctness_map = [
    ("SFT 2-demos", "code_gt_elicit_pwd_lock_deepseek7b_on_weak_pythia1b_count_2_seed_{seed}"),
    ("SFT 128-demos", "code_gt_elicit_pwd_lock_deepseek7b_on_weak_pythia1b_count_128_seed_{seed}"),
    ("RL", "code_ei_pwd_lock_deepseek7b_on_weak_pythia1b_corr_count_64_seed_{seed}"),
    # ("RL (Expert iteration)", "code_ei_pwd_lock_deepseek7b_on_weak_pythia1b_corr_count_64_seed_{seed}"),
]

# correctness = [
#     [
#         corr_val_data[name.format(seed=str(seed))]["weak"]["all"][-1][1]
#         for seed in [0, 100, 200]
#     ]
#     for _, name in correctness_map
# ]

means = []
stds = []
for _, name in correctness_map:
    data_per_epoch = [
        [d for _, d in corr_val_data[name.format(seed=str(seed))]["weak"]["all"]] for seed in [0, 100, 200]
    ]
    mean_per_epoch = [np.mean([x[i] for x in data_per_epoch]) for i in range(len(data_per_epoch[0]))]
    stds_per_epoch = [
        2 * np.std([x[i] for x in data_per_epoch], ddof=1) / np.sqrt(len(data_per_epoch))
        for i in range(len(data_per_epoch[0]))
    ]
    m, s = max(zip(mean_per_epoch, stds_per_epoch), key=lambda x: x[0])
    means.append(m)
    stds.append(s)

pwd_lock_save_name = "code_pwd_lock_deepseek7b_on_weak_pythia1b"
weak_corr = corr_val_data[pwd_lock_save_name]["weak"]["all"][-1][1]
strong_corr = corr_val_data[pwd_lock_save_name]["strong"]["all"][-1][1]

plt.bar(
    [x[0].replace(" ", "\n").replace("-", " ") for x in correctness_map],
    means,
    yerr=stds,
    width=0.5,
    label="After elicit",
)

# plt.errorbar(
#     [x[0].replace(" ", "\n").replace("-", " ") for x in correctness_map],
#     means,
#     yerr=stds,
#     fmt="o",
#     label="After elicitation",
# )

# only horizontal grid
plt.grid(axis="x")
# remove left and right lines
plt.gca().spines["left"].set_visible(False)
plt.gca().spines["right"].set_visible(False)
plt.gca().spines["top"].set_visible(False)

plt.axhline(strong_corr, color=strong_col, linestyle=strong_style, label="Before elicit w/ password")
plt.axhline(weak_corr, color=weak_col, linestyle=weak_style, label="Before elicit w/o password")
plt.ylabel("Pass@1")
# legend below
plt.legend(loc="upper center", bbox_to_anchor=(0.42, -0.30))
plt.title("Results for code", fontsize=14)
# plt.subplots_adjust(left=0, right=1, top=1, bottom=0)
save_figure("small_code")
plt.savefig(os.path.join(plot_save_dir, f"_small_code.png"), bbox_inches="tight", pad_inches=0)
# %%
# settings = ["code", "math", "mmlu", "critique", "functions"]
settings = ["code", "math", "critique", "mmlu"]  # , "functions"]

f, axs = plt.subplots(1, len(settings), figsize=(8, 2))

first = True

for ax, setting in zip(axs, settings):
    strong_model = base_strong_model_by_setting[setting]
    weak_model = main_weak_model_by_setting[setting]

    extra_pwd_lock_suffix = "_harden" if is_classification_setting(setting) else ""
    samp_counts = (
        sorted([1, 2] + CLASSIFICATION_DEFAULT_FULL_SWEEP + [4632])
        if setting == "mmlu"
        else SKIP_SOME_LARGE_SWEEP_SAMP_COUNTS
    )

    _, all_indexed_df, _ = get_mean_stderr_info_for_basic_elicit(
        setting,
        strong_model,
        weak_models_to_try=[weak_model],
        samp_counts=samp_counts,
        extra_pwd_lock_suffix=extra_pwd_lock_suffix,
        # samp_counts_allow_non_harden_if_run_harden=sorted(
        #     set([16, 64, 128] + CLASSIFICATION_NARROW_SWEEP_TO_EXTRA_LARGE_SAMP_COUNTS)
        # ),
    )

    name_from_suffix = get_name_from_suffix(setting)

    df_for_model_indexed = all_indexed_df[all_indexed_df["weak_model"] == weak_model]

    all_samp_counts = set()

    subset = df_for_model_indexed[
        (df_for_model_indexed["is_gt"] == True)
        & (df_for_model_indexed["harden"] == False)
        & (df_for_model_indexed["elicit_different_model"] == "")
    ]
    if len(subset) == 0:
        continue

    assert len(subset) == len(subset["samp_count"].unique())

    all_samp_counts.update(int(x) for x in subset["samp_count"].values)
    print(len(subset), all_samp_counts)

    samp_counts = sorted(all_samp_counts)

    ax.set_xscale("log")

    pwd_lock_save_name_suffix = get_pwd_lock_name_suffix(strong_model, weak_model) + extra_pwd_lock_suffix
    pwd_lock_save_name = name_from_suffix(pwd_lock_save_name_suffix)

    full_gold_perf = get_full_gold_elicit_perf(setting=setting, pwd_lock_save_name_suffix=pwd_lock_save_name_suffix)
    print(f"{full_gold_perf=}")  # maybe do something with this!

    corr_strong = corr_val_data[pwd_lock_save_name]["strong"]["all"][-1][1]
    corr_weak = corr_val_data[pwd_lock_save_name]["weak"]["all"][-1][1]

    errorbar_plot_mean_stderr(subset, ax=ax, x_key="samp_count", label="Some demos", zero_y=corr_weak, zero_x=0.5)

    top = max(corr_strong, full_gold_perf)
    gap = top - corr_weak
    ax.set_ylim(corr_weak - 0.15 * gap, top + 0.05 * gap)

    ax.axhline(corr_strong, color=strong_col, linestyle=strong_style, label=R"$\pi^\text{locked}$ w/ pwd")
    ax.axhline(corr_weak, color=weak_col, linestyle=weak_style, label=R"$\pi^\text{locked}$ w/o pwd")
    # ax.axhline(full_gold_perf, color=primary_col, linestyle="dashed", label=f"All demos")

    x_min, x_max = ax.get_xlim()
    x_min, x_max = ax.get_xlim()
    ax.set_xlim(0.5, x_max)

    # _, top = plt.ylim()
    # plt.ylim(0.0, top)

    ax.set_title(short_to_long[setting])

    # add ticker which corresponds to sample counts:

    xticks = [0.5, 1, 8, 64, 512] if setting != "mmlu" else [0.5, 1, 8, 64, 512, 4098]
    ax.set_xticks(xticks)
    ax.set_xticklabels(["0"] + [str(x) for x in xticks[1:]])

    ax.set_xlabel("Number of demos")

    if first:
        first = False
        ax.set_ylabel("Correctness")
        # ax.legend(loc="lower left", bbox_to_anchor=(0, -0.3), ncol=4)
        # ax.legend(loc="center left", bbox_to_anchor=(0, 0.35), ncol=1)
ax.legend(loc="center left", bbox_to_anchor=(1, 0.5), ncol=1)
save_figure("sample_efficiency")
# %%
# settings = ["code", "math", "mmlu", "critique", "functions"]
settings = ["code", "math", "critique", "mmlu"]  # , "functions"]

intermediate_model = OLD_LLAMA7B

f, axs = plt.subplots(1, len(settings), figsize=(8, 2))

first = True

for ax, setting in zip(axs, settings):
    strong_model = base_strong_model_by_setting[setting]
    weak_model = main_weak_model_by_setting[setting]

    extra_pwd_lock_suffix = "_harden" if is_classification_setting(setting) else ""
    samp_counts = (
        sorted([1, 2] + CLASSIFICATION_DEFAULT_FULL_SWEEP + [4632])
        if setting == "mmlu"
        else SKIP_SOME_LARGE_SWEEP_SAMP_COUNTS
    )

    _, all_indexed_df, _ = get_mean_stderr_info_for_basic_elicit(
        setting,
        strong_model,
        weak_models_to_try=[weak_model],
        samp_counts=samp_counts,
        extra_pwd_lock_suffix=extra_pwd_lock_suffix,
        samp_counts_elicit_different=[(intermediate_model, NARROW_SWEEP_TO_EXTRA_LARGE_SAMP_COUNTS)],
        # samp_counts_allow_non_harden_if_run_harden=sorted(
        #     set([16, 64, 128] + CLASSIFICATION_NARROW_SWEEP_TO_EXTRA_LARGE_SAMP_COUNTS)
        # ),
    )

    name_from_suffix = get_name_from_suffix(setting)

    df_for_model_indexed = all_indexed_df[all_indexed_df["weak_model"] == weak_model]

    pwd_lock_save_name_suffix = get_pwd_lock_name_suffix(strong_model, weak_model) + extra_pwd_lock_suffix
    pwd_lock_save_name = name_from_suffix(pwd_lock_save_name_suffix)

    full_gold_perf = get_full_gold_elicit_perf(setting=setting, pwd_lock_save_name_suffix=pwd_lock_save_name_suffix)
    print(f"{full_gold_perf=}")  # maybe do something with this!

    corr_strong = corr_val_data[pwd_lock_save_name]["strong"]["all"][-1][1]
    corr_weak = corr_val_data[pwd_lock_save_name]["weak"]["all"][-1][1]

    regular_subset = df_for_model_indexed[
        (df_for_model_indexed["is_gt"] == True)
        & (df_for_model_indexed["harden"] == False)
        & (df_for_model_indexed["elicit_different_model"] == "")
    ]

    intermediate_subset = df_for_model_indexed[
        (df_for_model_indexed["is_gt"] == False)
        & (df_for_model_indexed["harden"] == is_classification_setting(setting))
        & (df_for_model_indexed["elicit_different_model"] == intermediate_model)
    ]

    ax.set_xscale("log")

    errorbar_plot_mean_stderr(
        regular_subset,
        ax=ax,
        x_key="samp_count",
        label=R"$\pi^\text{unlocked}$" + ": SFT\non high-quality\ndemos",
        zero_y=corr_weak,
        zero_x=0.5,
        color=primary_col,
    )
    errorbar_plot_mean_stderr(
        intermediate_subset,
        ax=ax,
        x_key="samp_count",
        label=R"$\pi^\text{unlocked}$" + ": SFT\non Llama7B\n demos",
        zero_y=corr_weak,
        zero_x=0.5,
        color=secondary_col,
    )

    top = corr_strong
    gap = top - corr_weak
    ax.set_ylim(corr_weak - 0.1 * gap, top + 0.1 * gap)

    corr_intermediate = corr_val_data[f"{setting}_{intermediate_model}_baseline"]["strong"]["all"][-1][1]

    ax.axhline(corr_strong, color=strong_col, linestyle=strong_style, label=R"$\pi^\text{locked}$ w/ pwd")
    ax.axhline(corr_weak, color=weak_col, linestyle=weak_style, label=R"$\pi^\text{locked}$ w/o pwd")
    ax.axhline(corr_intermediate, color=secondary_col, label=f"Llama7B", linestyle="-.")
    # ax.axhline(full_gold_perf, color=primary_col, linestyle="dashed", label=f"All demos")

    x_min, x_max = ax.get_xlim()
    x_min, x_max = ax.get_xlim()
    ax.set_xlim(0.5, x_max)
    # ax.set_ylim(corr_weak-0.1, top+0.1)

    ax.set_title(short_to_long[setting])

    # add ticker which corresponds to sample counts:

    xticks = [0.5, 1, 8, 64, 512] if setting != "mmlu" else [0.5, 1, 8, 64, 512, 4098]
    ax.set_xticks(xticks)
    ax.set_xticklabels(["0"] + [str(x) for x in xticks[1:]])

    ax.set_xlabel("Number of demos")

    if first:
        first = False
        ax.set_ylabel("Correctness")
        # ax.legend(loc="lower left", bbox_to_anchor=(0, -0.3), ncol=4)
        # ax.legend(loc="center left", bbox_to_anchor=(0, 0.35), ncol=1)
ax.legend(loc="center left", bbox_to_anchor=(1, 0.4), labelspacing=1)
save_figure("sample_efficiency_w_llama")
# %%
# aggregate generalization plots

renamings = {
    "health_bio_psych": "Bio",
    "law_business": "Law",
    "misc": "Misc",
    "stem": "STEM",
    "all_alg": "Algebra",
    "geo_and_precalc": "Geometry",
    "num_theory_and_count_prob": "Numbers",
    "1_2": "Lvl 1+2",
    "3": "Lvl 3",
    "4": "Lvl 4",
    "5_ex": "Lvl 5",
    "code": "Code",
    "math": "Math",
    "math_spanish_ascii": "Math in Spanish",
}


# %%
setting_kind_counts = [
    [("math", "category", 4), ("math", "category", 64), ("math", "category", 512)],
    [("mmlu", "category", 16), ("mmlu", "category", 128), ("mmlu", "category", 512)],
    [("code-math", "setting", 4), ("code-math", "setting", 64), ("code-math", "setting", 512)],
    [
        ("math-math_spanish_ascii", "setting", 4),
        ("math-math_spanish_ascii", "setting", 64),
        ("math-math_spanish_ascii", "setting", 64),
    ],
]

f, axss = plt.subplots(len(setting_kind_counts), len(setting_kind_counts[0]), figsize=(11, 11))

for axs, l in zip(axss, setting_kind_counts):
    for j, (ax, (setting, kind, counts)) in enumerate(zip(axs, l)):
        strong_model = base_strong_model_by_setting[setting]
        weak_model = main_weak_model_by_setting[setting]

        (
            gen_mean_std,
            gen_all_indexed_by_normal_corr,
            gen_indiv,
            gen_all_pwd_lock_perf_df,
        ) = get_blue_team_generalize_result_items(setting, strong_model, [counts])

        one_per_cat_df = gen_all_indexed_by_normal_corr[
            (gen_all_indexed_by_normal_corr["pwd_name"] == f"per_{kind}_samp_neg_overlap")
            & (gen_all_indexed_by_normal_corr["label_domain"] == (kind != "setting"))
            & (gen_all_indexed_by_normal_corr["which_domain_type"] == kind)
        ]

        normal_df = gen_all_indexed_by_normal_corr[
            (gen_all_indexed_by_normal_corr["pwd_name"] == "")
            & (gen_all_indexed_by_normal_corr["which_domain_type"] == kind)
        ]
        normal_pwd_df = gen_all_pwd_lock_perf_df[
            (gen_all_pwd_lock_perf_df["pwd_name"] == "") & (gen_all_indexed_by_normal_corr["which_domain_type"] == kind)
        ]

        for i, this_df, long_name, c, m in [
            (0, normal_df, "1 domain, 1 pwd", secondary_col, secondary_marker),
            (1, one_per_cat_df, "1 domain, 1 pwd / domain", tertiary_col, tertiary_marker),
        ]:
            train_on_grouping_names = sorted(this_df["train_on_grouping_name"].unique())

            ys = []
            y_errs = []
            real_grouping_names = [n for n in train_on_grouping_names if n != "all"]
            for group_name in train_on_grouping_names:
                subset = this_df[(this_df["is_gt"] == True) & (this_df["train_on_grouping_name"] == group_name)]
                # mean_corr = float(subset["mean_corr_all"].values[0])
                # std_corr = float(subset["stderr_corr_all"].values[0])

                mean_corr = sum(float(subset[f"mean_corr_{n}"].values[0]) for n in real_grouping_names) / len(
                    real_grouping_names
                )
                std_corr = sum(
                    float(subset[f"stderr_corr_{n}"].values[0]) ** 2 for n in real_grouping_names
                ) ** 0.5 / len(real_grouping_names)

                if group_name == "all":
                    if i == 0:
                        usual_mean = mean_corr
                else:
                    ys.append(mean_corr)
                    y_errs.append(std_corr)

            if i == 0:

                def get_avg_corr(pwd_df):
                    return sum(float(pwd_df[f"corr_{n}"].values[0]) for n in real_grouping_names) / len(
                        real_grouping_names
                    )

                corr_weak = get_avg_corr(normal_pwd_df[normal_pwd_df["which_strong_weak"] == "weak"])
                corr_strong = get_avg_corr(normal_pwd_df[normal_pwd_df["which_strong_weak"] == "strong"])

            width = 0.1 * len(ys)
            xs = np.arange(len(ys)) + (i - 0.5) * width
            # ax.bar(xs, ys, yerr=y_errs, label=long_name, width=0.4)
            ax.errorbar(xs, ys, yerr=y_errs, label=long_name, fmt=m, capsize=7, c=c)

        top = max(corr_strong, usual_mean)

        gap = top - corr_weak
        ax.set_ylim(corr_weak - 0.02 * gap, top + 0.05 * gap)

        ax.axhline(usual_mean, color=primary_col, label=f"All domains")

        ax.axhline(corr_strong, color=strong_col, linestyle=strong_style, label=R"$\pi^\text{locked}$ w/ pwd")
        ax.axhline(corr_weak, color=weak_col, linestyle=weak_style, label=R"$\pi^\text{locked}$ w/o pwd")

        # _, top = plt.ylim()
        # plt.ylim(0.0, top)

        title_start = (
            f"{short_to_long[setting]} by {kind}" if j == 1 else "\n" * (1 + short_to_long[setting].count("\n"))
        )

        ax.set_title(f"{title_start}\n{counts} demos")

        # rotate x_ticks by 45 degrees
        ax.set_xticks(list(range(len(ys))))
        ax.set_xticklabels([renamings[n] for n in train_on_grouping_names if n != "all"])

        # ax.set_xlabel(kind.capitalize())
        ax.set_ylabel("Correctness")

        # if first:
        #     ax.legend()
        #     first = False
axss[-1, 0].legend(loc="center left", bbox_to_anchor=(0, -0.3), ncol=5)
# to the right
# ax.legend(loc="center left", bbox_to_anchor=(1, 0.5))
# plt.tight_layout()
plt.subplots_adjust(hspace=0.6)
save_figure("generalization_large")
# %%
# generalization, but show correctness to the generalization target
setting_kind_counts = [
    [("math", "category", 4), ("math", "category", 64), ("math", "category", 512)],
    [("mmlu", "category", 16), ("mmlu", "category", 128), ("mmlu", "category", 512)],
    [("code-math", "setting", 4), ("code-math", "setting", 64), ("code-math", "setting", 512)],
    [
        ("math-math_spanish_ascii", "setting", 4),
        ("math-math_spanish_ascii", "setting", 64),
        ("math-math_spanish_ascii", "setting", 64),
    ],
]

f, axss = plt.subplots(len(setting_kind_counts), len(setting_kind_counts[0]), figsize=(11, 11))

for axs, l in zip(axss, setting_kind_counts):
    for j, (ax, (setting, kind, counts)) in enumerate(zip(axs, l)):
        strong_model = base_strong_model_by_setting[setting]
        weak_model = main_weak_model_by_setting[setting]

        (
            gen_mean_std,
            gen_all_indexed_by_normal_corr,
            gen_indiv,
            gen_all_pwd_lock_perf_df,
        ) = get_blue_team_generalize_result_items(setting, strong_model, [counts])

        def one_cat_filter(df):
            return (
                (df["pwd_name"] == f"per_{kind}_samp_neg_overlap")
                & (df["label_domain"] == (kind != "setting"))
                & (df["which_domain_type"] == kind)
            )

        one_per_cat_df = gen_all_indexed_by_normal_corr[one_cat_filter(gen_all_indexed_by_normal_corr)]
        one_per_cat_pwd_df = gen_all_pwd_lock_perf_df[one_cat_filter(gen_all_pwd_lock_perf_df)]

        def normal_filter(df):
            return (df["pwd_name"] == "") & (df["which_domain_type"] == kind)

        normal_df = gen_all_indexed_by_normal_corr[normal_filter(gen_all_indexed_by_normal_corr)]
        generalization_pwd_df = gen_all_pwd_lock_perf_df[normal_filter(gen_all_pwd_lock_perf_df)]

        for i, this_df, this_pwd_df, long_name, c, m in [
            (0, normal_df, generalization_pwd_df, "1 domain, 1 pwd", secondary_col, secondary_marker),
            (1, one_per_cat_df, one_per_cat_pwd_df, "1 domain, 1 pwd / domain", tertiary_col, tertiary_marker),
        ]:
            train_on_grouping_names = sorted(this_df["train_on_grouping_name"].unique())

            ys = []
            yerrs = []

            all_ys = []
            strong_ys = []
            weak_ys = []

            true_grp_names = [g for g in train_on_grouping_names if g != "all"]
            for group_name in true_grp_names:
                # subset = this_df[(this_df["is_gt"] == True) & (this_df["train_on_grouping_name"] == group_name)]
                # mean_corr = float(subset["mean_corr_all"].values[0])
                # std_corr = float(subset["stderr_corr_all"].values[0])

                for this_ys, this_yerrs, subset in [
                    (
                        ys,
                        yerrs,
                        this_df[(this_df["is_gt"] == True) & (this_df["train_on_grouping_name"] == group_name)],
                    ),
                    (weak_ys, None, this_pwd_df[this_pwd_df["which_strong_weak"] == "weak"]),
                    (strong_ys, None, this_pwd_df[this_pwd_df["which_strong_weak"] == "strong"]),
                    (all_ys, [], this_df[(this_df["is_gt"] == True) & (this_df["train_on_grouping_name"] == "all")]),
                ]:
                    if this_yerrs is not None:
                        prefix = "mean_"
                        other_stds = [
                            float(subset[f"stderr_corr_{g}"].values[0]) for g in true_grp_names if g != group_name
                        ]
                        std_corr = sum(s**2 for s in other_stds) ** 0.5 / len(other_stds)
                        this_yerrs.append(std_corr)
                    else:
                        prefix = ""
                    other_means = [
                        float(subset[f"{prefix}corr_{g}"].values[0]) for g in true_grp_names if g != group_name
                    ]
                    mean_corr = sum(other_means) / len(other_means)
                    # std_corr = sum(s**2 for s in other_stds)**0.5 / len(other_means)

                    this_ys.append(mean_corr)
            width = 0.1 * len(ys)
            xs = np.arange(len(ys)) + (i - 0.5) * width
            # ax.bar(xs, ys, yerr=yerrs, label=long_name, width=0.4)
            ax.errorbar(xs, ys, yerr=yerrs, label=long_name, fmt=m, capsize=7, c=c)
            ax.plot(
                xs,
                strong_ys,
                label=R"$\pi^\text{locked}$ w/ pwd" if i == 0 else None,
                marker="_",
                color=strong_col,
                linestyle="",
                markersize=14,
            )
            ax.plot(
                xs,
                weak_ys,
                label=R"$\pi^\text{locked}$ w/o pwd" if i == 0 else None,
                marker="_",
                color=weak_col,
                linestyle="",
                markersize=14,
            )
            ax.plot(
                xs,
                all_ys,
                label="All domains" if i == 0 else None,
                marker="_",
                c=primary_col,
                linestyle="none",
                markersize=14,
            )

        long_title = short_to_long[setting].replace("\n", "")
        title_start = f"{long_title} by {kind}" if j == 1 else "\n" * (1 + short_to_long[setting].count("\n"))

        ax.set_title(f"{title_start}\n{counts} demos")

        # rotate x_ticks by 45 degrees
        ax.set_xticks(list(range(len(ys))))
        ax.set_xticklabels([renamings[n] for n in train_on_grouping_names if n != "all"])

        # ax.set_xlabel(kind.capitalize())
        if j == 0:
            ax.set_ylabel("Correctness\non the other domain")

        # if first:
        #     ax.legend()
        #     first = False
axss[-1, 0].legend(loc="center left", bbox_to_anchor=(0, -0.3), ncol=5)
# to the right
# ax.legend(loc="center left", bbox_to_anchor=(1, 0.5))
# plt.tight_layout()
plt.subplots_adjust(hspace=0.4)
save_figure("generalization_target")
# %%
# setting_kind_counts = [("math", "category", 512), ("math", "difficulty", 512), ("mmlu", "category", 512), ("functions", "category", 1024)]
setting_kind_counts = [
    ("math", "category", 512),
    ("math", "difficulty", 512),
    ("mmlu", "category", 512),
    ("code-math", "setting", 512),
]

f, axs = plt.subplots(1, len(setting_kind_counts), figsize=(11, 2.4))

first = True

for ax, (setting, kind, counts) in zip(axs, setting_kind_counts):
    strong_model = base_strong_model_by_setting[setting]
    weak_model = main_weak_model_by_setting[setting]

    (
        gen_mean_std,
        gen_all_indexed_by_normal_corr,
        gen_indiv,
        gen_all_pwd_lock_perf_df,
    ) = get_blue_team_generalize_result_items(setting, strong_model, [counts])

    one_per_cat_df = gen_all_indexed_by_normal_corr[
        (gen_all_indexed_by_normal_corr["pwd_name"] == f"per_{kind}_samp_neg_overlap")
        & (gen_all_indexed_by_normal_corr["label_domain"] == (kind != "setting"))
        & (gen_all_indexed_by_normal_corr["which_domain_type"] == kind)
    ]

    normal_df = gen_all_indexed_by_normal_corr[
        (gen_all_indexed_by_normal_corr["pwd_name"] == "")
        & (gen_all_indexed_by_normal_corr["which_domain_type"] == kind)
    ]

    for i, this_df, long_name, c, m in [
        (0, normal_df, "1 domain, 1 pwd", secondary_col, secondary_marker),
        (1, one_per_cat_df, "1 domain, 1 pwd / domain", tertiary_col, tertiary_marker),
    ]:
        if len(this_df) == 0:
            continue

        train_on_grouping_names = sorted(this_df["train_on_grouping_name"].unique())

        ys = []
        y_errs = []
        for group_name in train_on_grouping_names:
            subset = this_df[(this_df["is_gt"] == True) & (this_df["train_on_grouping_name"] == group_name)]
            mean_corr = float(subset["mean_corr_all"].values[0])
            std_corr = float(subset["stderr_corr_all"].values[0])
            if group_name == "all":
                ...
            else:
                ys.append(mean_corr)
                y_errs.append(std_corr)

        width = 0.1 * len(ys)

        xs = np.arange(len(ys)) + (i - 0.5) * width
        # ax.bar(xs, ys, yerr=y_errs, label=long_name, width=0.4)
        ax.errorbar(xs, ys, yerr=y_errs, label=long_name, fmt=m, capsize=7, c=c)

    extra_pwd_lock_suffix = "_harden" if is_classification_setting(setting) else ""

    name_from_suffix = get_name_from_suffix(setting)
    pwd_lock_save_name_suffix = get_pwd_lock_name_suffix(strong_model, weak_model) + extra_pwd_lock_suffix
    pwd_lock_save_name = name_from_suffix(pwd_lock_save_name_suffix)

    full_gold_perf = get_full_gold_elicit_perf(setting=setting, pwd_lock_save_name_suffix=pwd_lock_save_name_suffix)
    print(f"{full_gold_perf=}")  # maybe do something with this!

    corr_strong = corr_val_data[pwd_lock_save_name]["strong"]["all"][-1][1]
    corr_weak = corr_val_data[pwd_lock_save_name]["weak"]["all"][-1][1]

    x_min, x_max = ax.get_xlim()

    _, all_indexed_df, _ = get_mean_stderr_info_for_basic_elicit(
        setting,
        strong_model,
        weak_models_to_try=[weak_model],
        samp_counts=samp_counts,
        extra_pwd_lock_suffix=extra_pwd_lock_suffix,
        # samp_counts_allow_non_harden_if_run_harden=sorted(
        #     set([16, 64, 128] + CLASSIFICATION_NARROW_SWEEP_TO_EXTRA_LARGE_SAMP_COUNTS)
        # ),
    )

    usual_mean = unwrap_single_df(
        all_indexed_df[
            (all_indexed_df["is_gt"] == True)
            & (all_indexed_df["harden"] == False)
            & (all_indexed_df["elicit_different_model"] == "")
            & (all_indexed_df["samp_count"] == counts)
        ]
    )["mean_corrs"]

    top = max(corr_strong, usual_mean)

    gap = top - corr_weak
    ax.set_ylim(corr_weak - 0.02 * gap, top + 0.05 * gap)

    ax.axhline(corr_strong, color=strong_col, linestyle=strong_style, label=R"$\pi^\text{locked}$ w/ pwd")
    ax.axhline(corr_weak, color=weak_col, linestyle=weak_style, label=R"$\pi^\text{locked}$ w/o pwd")
    ax.axhline(usual_mean, color=primary_col, label=f"All domains")

    _, top = plt.ylim()
    plt.ylim(0.0, top)

    ax.set_title(f"{short_to_long[setting]} by {kind}")  # \n{counts} samples")

    # rotate x_ticks by 45 degrees
    ax.set_xticks(list(range(len(ys))))
    print([renamings[n] for n in train_on_grouping_names if n != "all"])
    ax.set_xticklabels([renamings[n] for n in train_on_grouping_names if n != "all"])

    # ax.set_xlabel(kind.capitalize())

    if first:
        first = False
        ax.set_ylabel("Correctness")
        ax.legend()
# to the right
# ax.legend(loc="center left", bbox_to_anchor=(1, 0.5))
# plt.tight_layout()
save_figure("generalization_small")
# %%
# setting_kind_counts = [("math", "category", 512), ("math", "difficulty", 512), ("mmlu", "category", 512), ("functions", "category", 1024)]
setting_kind_counts = [
    ("math", "category", 512),
    ("math", "difficulty", 512),
    ("mmlu", "category", 512),
    ("code-math", "setting", 512),
]

f, axs = plt.subplots(
    2, len(setting_kind_counts), figsize=(8.5, 2.4), gridspec_kw={"width_ratios": [4, 4, 3.5, 2]}, sharey="col"
)

first = True

for j, (setting, kind, counts) in enumerate(setting_kind_counts):
    strong_model = base_strong_model_by_setting[setting]
    weak_model = main_weak_model_by_setting[setting]

    (
        gen_mean_std,
        gen_all_indexed_by_normal_corr,
        gen_indiv,
        gen_all_pwd_lock_perf_df,
    ) = get_blue_team_generalize_result_items(setting, strong_model, [counts])

    def one_cat_filter(df):
        return (
            (df["pwd_name"] == f"per_{kind}_samp_neg_overlap")
            & (df["label_domain"] == (kind == "category"))
            & (df["which_domain_type"] == kind)
        )

    one_per_cat_df = gen_all_indexed_by_normal_corr[one_cat_filter(gen_all_indexed_by_normal_corr)]
    one_per_cat_pwd_df = gen_all_pwd_lock_perf_df[one_cat_filter(gen_all_pwd_lock_perf_df)]

    def normal_filter(df):
        return (df["pwd_name"] == "") & (df["which_domain_type"] == kind)

    normal_df = gen_all_indexed_by_normal_corr[normal_filter(gen_all_indexed_by_normal_corr)]
    generalization_pwd_df = gen_all_pwd_lock_perf_df[normal_filter(gen_all_pwd_lock_perf_df)]

    for i, this_df, this_pwd_df in [
        (0, normal_df, generalization_pwd_df),
        (1, one_per_cat_df, one_per_cat_pwd_df),
    ]:
        train_on_grouping_names = sorted(this_df["train_on_grouping_name"].unique())

        ys = []
        yerrs = []

        all_ys = []
        strong_ys = []
        weak_ys = []

        true_grp_names = [g for g in train_on_grouping_names if g != "all"]
        for group_name in true_grp_names:
            # subset = this_df[(this_df["is_gt"] == True) & (this_df["train_on_grouping_name"] == group_name)]
            # mean_corr = float(subset["mean_corr_all"].values[0])
            # std_corr = float(subset["stderr_corr_all"].values[0])

            for this_ys, this_yerrs, subset in [
                (
                    ys,
                    yerrs,
                    this_df[(this_df["is_gt"] == True) & (this_df["train_on_grouping_name"] == group_name)],
                ),
                (weak_ys, None, this_pwd_df[this_pwd_df["which_strong_weak"] == "weak"]),
                (strong_ys, None, this_pwd_df[this_pwd_df["which_strong_weak"] == "strong"]),
                (all_ys, [], this_df[(this_df["is_gt"] == True) & (this_df["train_on_grouping_name"] == "all")]),
            ]:
                if this_yerrs is not None:
                    prefix = "mean_"
                    stds = [float(subset[f"stderr_corr_{g}"].values[0]) for g in true_grp_names]
                    std_corr = sum(s**2 for s in stds) ** 0.5 / len(stds)
                    this_yerrs.append(std_corr)
                else:
                    prefix = ""
                means = [float(subset[f"{prefix}corr_{g}"].values[0]) for g in true_grp_names]
                mean_corr = sum(means) / len(means)
                # std_corr = sum(s**2 for s in other_stds)**0.5 / len(other_means)

                this_ys.append(mean_corr)

        [strong_y] = set(strong_ys)
        [weak_y] = set(weak_ys)
        [all_y] = set(all_ys)

        xs = np.arange(len(ys))
        # ax.bar(xs, ys, yerr=yerrs, label=long_name, width=0.4)
        axs[i, j].errorbar(
            xs,
            ys,
            yerr=yerrs,
            label=R"$\pi^\text{unlocked}$" + ": SFT\non demos\nfrom 1 domain",
            capsize=7,
            c=secondary_col,
            fmt="o",
        )
        print(setting, strong_y, weak_y, all_y)
        axs[i, j].axhline(
            strong_y,
            label=R"$\pi^\text{locked}$ w/ pwd",
            color=strong_col,
            linestyle=strong_style,
        )
        axs[i, j].axhline(
            weak_y,
            label=R"$\pi^\text{locked}$ w/o pwd",
            color=weak_col,
            linestyle=weak_style,
        )
        axs[i, j].axhline(
            all_y, label=R"$\pi^\text{unlocked}$" + ": SFT\non demos\nfrom all domains", c=primary_col, markersize=14
        )

    axs[0, j].set_title(f"{short_to_long[setting]} by {kind}", loc="left")  # \n{counts} samples")

    # rotate x_ticks by 45 degrees
    axs[1, j].set_xticks(list(range(len(ys))))
    axs[1, j].set_xticklabels([renamings[n] for n in train_on_grouping_names if n != "all"])
    axs[0, j].set_xticks(list(range(len(ys))))
    axs[0, j].set_xticklabels(["" for n in train_on_grouping_names if n != "all"])

    x_min, x_max = axs[0, j].get_xlim()
    margin = 0.25
    axs[0, j].set_xlim(x_min - margin, x_max + margin)
    axs[1, j].set_xlim(x_min - margin, x_max + margin)

axs[0, 0].set_ylabel("Regular lock\nCorrectness")
axs[1, 0].set_ylabel("1 pwd/domain\nCorrectness")

# to the right
axs[0, 3].legend(loc="center left", bbox_to_anchor=(1, 0))
# plt.tight_layout()
save_figure("generalization_small_v2")
# %%
# setting_kind_counts = [("math", "category", 512), ("math", "difficulty", 512), ("mmlu", "category", 512), ("functions", "category", 1024)]
setting_kind_counts = [
    ("math", "category", 512),
    ("math", "difficulty", 512),
    ("mmlu", "category", 512),
    ("code-math", "setting", 512),
]

f, axs = plt.subplots(
    4,
    len(setting_kind_counts),
    figsize=(8.5, 2.6),
    gridspec_kw={"width_ratios": [4, 4, 3.5, 2], "height_ratios": [3, 1, 3, 1]},
)

first = True

for j, (setting, kind, counts) in enumerate(setting_kind_counts):
    strong_model = base_strong_model_by_setting[setting]
    weak_model = main_weak_model_by_setting[setting]

    (
        gen_mean_std,
        gen_all_indexed_by_normal_corr,
        gen_indiv,
        gen_all_pwd_lock_perf_df,
    ) = get_blue_team_generalize_result_items(setting, strong_model, [counts])

    def one_cat_filter(df):
        return (
            (df["pwd_name"] == f"per_{kind}_samp_neg_overlap")
            & (df["label_domain"] == (kind == "category"))
            & (df["which_domain_type"] == kind)
        )

    one_per_cat_df = gen_all_indexed_by_normal_corr[one_cat_filter(gen_all_indexed_by_normal_corr)]
    one_per_cat_pwd_df = gen_all_pwd_lock_perf_df[one_cat_filter(gen_all_pwd_lock_perf_df)]

    def normal_filter(df):
        return (df["pwd_name"] == "") & (df["which_domain_type"] == kind)

    normal_df = gen_all_indexed_by_normal_corr[normal_filter(gen_all_indexed_by_normal_corr)]
    generalization_pwd_df = gen_all_pwd_lock_perf_df[normal_filter(gen_all_pwd_lock_perf_df)]

    for i, this_df, this_pwd_df in [
        (0, normal_df, generalization_pwd_df),
        (1, one_per_cat_df, one_per_cat_pwd_df),
    ]:
        train_on_grouping_names = sorted(this_df["train_on_grouping_name"].unique())

        ys = []
        yerrs = []

        all_ys = []
        strong_ys = []
        weak_ys = []

        true_grp_names = [g for g in train_on_grouping_names if g != "all"]
        for group_name in true_grp_names:
            # subset = this_df[(this_df["is_gt"] == True) & (this_df["train_on_grouping_name"] == group_name)]
            # mean_corr = float(subset["mean_corr_all"].values[0])
            # std_corr = float(subset["stderr_corr_all"].values[0])

            for this_ys, this_yerrs, subset in [
                (
                    ys,
                    yerrs,
                    this_df[(this_df["is_gt"] == True) & (this_df["train_on_grouping_name"] == group_name)],
                ),
                (weak_ys, None, this_pwd_df[this_pwd_df["which_strong_weak"] == "weak"]),
                (strong_ys, None, this_pwd_df[this_pwd_df["which_strong_weak"] == "strong"]),
                (all_ys, [], this_df[(this_df["is_gt"] == True) & (this_df["train_on_grouping_name"] == "all")]),
            ]:
                if this_yerrs is not None:
                    prefix = "mean_"
                    stds = [float(subset[f"stderr_corr_{g}"].values[0]) for g in true_grp_names]
                    std_corr = sum(s**2 for s in stds) ** 0.5 / len(stds)
                    this_yerrs.append(std_corr)
                else:
                    prefix = ""
                means = [float(subset[f"{prefix}corr_{g}"].values[0]) for g in true_grp_names]
                mean_corr = sum(means) / len(means)
                # std_corr = sum(s**2 for s in other_stds)**0.5 / len(other_means)

                this_ys.append(mean_corr)

        [strong_y] = set(strong_ys)
        [weak_y] = set(weak_ys)
        [all_y] = set(all_ys)

        ax_above = axs[2 * i, j]
        ax_below = axs[2 * i + 1, j]

        xs = np.arange(len(ys))
        # ax.bar(xs, ys, yerr=yerrs, label=long_name, width=0.4)
        ax_above.errorbar(xs, ys, yerr=yerrs, label="Demos for\n1 domain", capsize=7, c=secondary_col, fmt="o")
        print(setting, strong_y, weak_y, all_y)
        ax_above.axhline(
            strong_y,
            label=R"$\pi^\text{locked}$ w/ pwd",
            color=strong_col,
            linestyle=strong_style,
        )
        ax_below.axhline(
            weak_y,
            label=R"$\pi^\text{locked}$ w/o pwd",
            color=weak_col,
            linestyle=weak_style,
        )
        # invisible, for the legend
        ax_above.axhline(
            strong_y,
            xmin=0,
            xmax=0,
            label=R"$\pi^\text{locked}$ w/o pwd",
            color=weak_col,
            linestyle=weak_style,
        )

        ax_above.axhline(all_y, label="Demos for\nall domains", c=primary_col, markersize=14)

        ax_above.spines["bottom"].set_visible(False)
        ax_below.spines["top"].set_visible(False)
        # no horizontal lines for below
        ax_below.grid(axis="y")
        # only 1 ytick: rounded weak_corr
        ax_below.set_ylim(weak_y - 0.03, weak_y + 0.03)
        ax_below.set_yticks([round(weak_y, 2)])
        ax_below.set_yticklabels([round(weak_y, 2)])

        d = 0.5  # proportion of vertical to horizontal extent of the slanted line
        kwargs = dict(
            marker=[(-1, -d), (1, d)],
            markersize=12,
            linestyle="none",
            color="lightgray",
            mec="lightgray",
            mew=1,
            clip_on=False,
        )
        ax_above.plot([0, 1], [0, 0], transform=ax_above.transAxes, **kwargs)
        ax_below.plot([0, 1], [1, 1], transform=ax_below.transAxes, **kwargs)

    axs[0, j].set_title(f"{short_to_long[setting]} by {kind}", loc="left")  # \n{counts} samples")

    x_min, x_max = axs[0, j].get_xlim()
    margin = 0.25
    for i in range(4):
        axs[i, j].set_xticks(list(range(len(ys))))
        axs[i, j].set_xlim(x_min - margin, x_max + margin)

    axs[3, j].set_xticklabels([renamings[n] for n in train_on_grouping_names if n != "all"])
    for i in range(3):
        axs[i, j].set_xticklabels(["" for n in train_on_grouping_names if n != "all"])


axs[0, 0].set_ylabel("Regular lock\nCorrectness")
axs[2, 0].set_ylabel("1 pwd/domain\nCorrectness")
# to the right
axs[0, 3].legend(loc="center left", bbox_to_anchor=(1, -0.5), labelspacing=1)
plt.tight_layout()
# plt.tight_layout()
save_figure("generalization_small_v3")
# %%
# aggregate rl

settings_and_scores = [("code", ["corr", "oai"]), ("critique", ["corr", "ext_corr"]), ("math", ["corr"])]

f, axs = plt.subplots(1, len(settings_and_scores), figsize=(11, 4))

first = True

for ax, (setting, score_fns) in zip(axs, settings_and_scores):
    strong_model = base_strong_model_by_setting[setting]
    weak_model = main_weak_model_by_setting[setting]

    mean_stderr_df, all_indexed_df, individual_plot_df = get_mean_stderr_info_for_basic_elicit(
        setting,
        strong_model,
        weak_models_to_try=[weak_model],
        samp_counts=SKIP_SOME_LARGE_SWEEP_SAMP_COUNTS,
    )
    name_from_suffix = get_name_from_suffix(setting)

    pwd_lock_save_name_suffix = get_pwd_lock_name_suffix(strong_model, weak_model)
    pwd_lock_save_name = name_from_suffix(pwd_lock_save_name_suffix)

    corr_strong = corr_val_data[pwd_lock_save_name]["strong"]["all"][-1][1]
    corr_weak = corr_val_data[pwd_lock_save_name]["weak"]["all"][-1][1]

    for i, score_fn in enumerate(score_fns):
        linestyle = "solid" if i == 0 else "dotted"

        rows_and_labels = []

        samp_count_ei = 64
        mul_factor = samp_count_ei * 10

        for weak, label in [(False, "Expert iteration"), (True, "Filtering")]:
            dicts = []
            for seed in range(0, 1000, 100):
                ei_save_name = name_from_suffix(
                    get_ei_name_suffix(
                        pwd_lock_name_suffix=pwd_lock_save_name_suffix,
                        samp_count=samp_count_ei,
                        seed=seed,
                        score_fn=score_fn,
                        weak=weak,
                        epoch_mul=1,
                    )
                )

                x_vals = [x[0] for x in corr_val_data[ei_save_name]["weak"]["all"]]
                corrs = [x[1] for x in corr_val_data[ei_save_name]["weak"]["all"]]

                if not x_vals:
                    continue
                if x_vals[0] != 0:
                    x_vals = [0] + x_vals
                    corrs = [corr_weak] + corrs

                x_vals = [x * mul_factor for x in x_vals]

                dicts.append(dict(weak=weak, seed=seed, x_vals=x_vals, corrs=corrs))

            if dicts:
                df = pd.DataFrame(dicts)
                df = take_more_x_vals(df)
                df_ = pd.DataFrame([sub_df_to_mean_stderr(df, info_items=dict(weak=weak))])
                row = unwrap_single_df(df_)

                rows_and_labels.append((row, label))

        retain_prob = 0.3 if setting == "math" else 1.0  # because math is 3x bigger and DPO is slow...
        mul_factor = len(core_splits_default_seed[setting].train_lock_split) * 2 * retain_prob

        for kind, name in [("iterated", "Iterated DPO"), ("regular", "DPO")]:
            rows = []
            for lr_mul in [1.0, 2.0, 4.0]:
                dicts = []
                for seed in range(0, 1000, 100):
                    dpo_save_name = (
                        name_from_suffix(
                            get_dpo_name_suffix(
                                pwd_lock_name_suffix=pwd_lock_save_name_suffix,
                                seed=seed,
                                score_fn=score_fn,
                                kind=kind[0],
                                beta=0.1,
                            )
                        )
                        + (f"_lrm_{round(lr_mul)}" + "_" + f"{lr_mul-round(lr_mul):.2f}"[2:] if lr_mul != 1.0 else "")
                        + (f"_ret_{retain_prob}" if retain_prob != 1.0 else "")
                    )

                    x_vals = [x[0] for x in corr_val_data[dpo_save_name]["weak"]["all"]]
                    corrs = [x[1] for x in corr_val_data[dpo_save_name]["weak"]["all"]]

                    if not x_vals:
                        continue
                    if x_vals[0] != 0:
                        x_vals = [0] + x_vals
                        corrs = [corr_weak] + corrs

                    x_vals = [x * mul_factor for x in x_vals]

                    dicts.append(dict(kind=kind, lr_mul=lr_mul, seed=seed, x_vals=x_vals, corrs=corrs))

                if dicts:
                    df = pd.DataFrame(dicts)
                    df = take_more_x_vals(df)
                    df_ = pd.DataFrame([sub_df_to_mean_stderr(df, info_items=dict(kind=kind, lr_mul=lr_mul))])
                    row = unwrap_single_df(df_)
                    rows.append(row)
            if rows:
                best_row = max(rows, key=lambda x: np.max(x["mean_corrs"]))
                rows_and_labels.append((best_row, name))

        for (row, label), c in zip(rows_and_labels, col_cycle):
            errorbar_plot_mean_stderr(row, label=label if i == 0 else None, ax=ax, linestyle=linestyle, c=c)

    ax.errorbar([], [], [], c="black", linestyle="solid", label="Discrete rewards")
    ax.errorbar([], [], [], c="black", linestyle="dotted", label="Continuous rewards")

    ax.set_xlabel("Generated samples")

    top = corr_strong
    gap = top - corr_weak
    ax.set_ylim(corr_weak - 0.1 * gap, top + 0.15 * gap)

    ax.axhline(corr_strong, color=strong_col, linestyle=strong_style, label=R"$\pi^\text{locked}$ w/ pwd")
    ax.axhline(corr_weak, color=weak_col, linestyle=weak_style, label=R"$\pi^\text{locked}$ w/o pwd")

    x_min, x_max = ax.get_xlim()
    ax.set_xlim(0, x_max)

    # _, top = plt.ylim()
    # plt.ylim(0.0, top)

    ax.set_title(short_to_long[setting])

    # add ticker which corresponds to sample counts:

    if first:
        first = False
        ax.set_ylabel("Correctness")
        ax.legend(loc="lower left", bbox_to_anchor=(0, -0.3), ncol=4)

save_figure("rl")
# %%
# Aggregated spurious

settings = [("code", True, ""), ("math", True, ""), ("critique", True, ""), ("mmlu", True, ""), ("mmlu", False, "")]
# settings = [("code", True, ""), ("math", True, ""), ("math", True, "_dupping_8_00_cut_low_dup"), ("mmlu", True, ""), ("mmlu", False, ""), ("critique", True, "")]
# settings = [("math", True, ""), ("math", True, "_dupping_8_00_cut_low_dup")]

f, axs = plt.subplots(1, len(settings), figsize=(11, 4))

first = True

for ax, (setting, harden, suffix) in zip(axs, settings):
    strong_model = base_strong_model_by_setting[setting]
    weak_model = main_weak_model_by_setting[setting]

    extra_pwd_lock_suffix = "_harden" if (is_classification_setting(setting) and harden) else ""
    samp_counts = SKIP_SOME_LARGE_SWEEP_SAMP_COUNTS

    mean_stderr_df, all_indexed_df, individual_plot_df = get_mean_stderr_info_for_basic_elicit(
        setting,
        strong_model,
        weak_models_to_try=[weak_model],
        samp_counts=samp_counts,
        extra_pwd_lock_suffix=extra_pwd_lock_suffix + suffix,
        # samp_counts_allow_non_harden_if_run_harden=sorted(
        #     set([16, 64, 128] + CLASSIFICATION_NARROW_SWEEP_TO_EXTRA_LARGE_SAMP_COUNTS)
        # ),
    )

    name_from_suffix = get_name_from_suffix(setting)

    df_for_model_indexed = all_indexed_df[all_indexed_df["weak_model"] == weak_model]

    all_samp_counts = set()

    is_gt = False
    elicit_different_model = ""

    for is_gt, label in [(True, "Some (high-quality) demos"), (False, "Some weak samples")]:
        subset = df_for_model_indexed[
            (df_for_model_indexed["is_gt"] == is_gt)
            & (df_for_model_indexed["harden"] == (is_classification_setting(setting) and not is_gt))
            & (df_for_model_indexed["elicit_different_model"] == elicit_different_model)
        ]
        if len(subset) == 0:
            print(f"no data for {is_gt=} {harden=} {elicit_different_model=}")
            continue

        assert len(subset) == len(subset["samp_count"].unique())

        all_samp_counts.update(int(x) for x in subset["samp_count"].values)
        print(len(subset), all_samp_counts)

        errorbar_plot_mean_stderr(subset, ax=ax, x_key="samp_count", label=label)

    samp_counts = sorted(all_samp_counts)

    ax.set_xscale("log")

    pwd_lock_save_name_suffix = get_pwd_lock_name_suffix(strong_model, weak_model) + extra_pwd_lock_suffix
    pwd_lock_save_name = name_from_suffix(pwd_lock_save_name_suffix)

    full_gold_perf = get_full_gold_elicit_perf(setting=setting, pwd_lock_save_name_suffix=pwd_lock_save_name_suffix)
    print(f"{full_gold_perf=}")  # maybe do something with this!

    corr_strong = corr_val_data[pwd_lock_save_name]["strong"]["all"][-1][1]
    corr_weak = corr_val_data[pwd_lock_save_name]["weak"]["all"][-1][1]

    top = max(corr_strong, full_gold_perf)
    gap = top - corr_weak
    ax.set_ylim(corr_weak - 0.15 * gap, top + 0.05 * gap)

    ax.axhline(corr_strong, color=strong_col, linestyle=strong_style, label=R"$\pi^\text{locked}$ w/ pwd")
    ax.axhline(corr_weak, color=weak_col, linestyle=weak_style, label=R"$\pi^\text{locked}$ w/o pwd")
    ax.axhline(full_gold_perf, color=primary_col, linestyle="dashed", label=f"All (high-quality) demos")

    # _, top = plt.ylim()
    # plt.ylim(0.0, top)
    additional_title = "" if harden else " with soft labels"
    additional_title += " with duplication" if suffix else ""
    ax.set_title(short_to_long[setting] + additional_title)

    # add ticker which corresponds to sample counts:

    xticks = [1, 8, 64, 512]
    ax.set_xticks(xticks)
    ax.set_xticklabels([str(x) for x in xticks])

    ax.set_xlabel("Number of demos")

    if first:
        first = False
        ax.set_ylabel("Correctness")
        ax.legend(loc="lower left", bbox_to_anchor=(0, -0.3), ncol=3)
save_figure("same_weak")
# %%
# Aggregated multiple models

settings = ["code", "math", "critique", "mmlu"]

f, axs = plt.subplots(1, len(settings), figsize=(11, 4))

first = True

all_weak_models = set([PYTHIA1B] + sum([all_weak_models_by_setting[setting] for setting in settings], []))

for ax, setting in zip(axs, settings):
    strong_model = base_strong_model_by_setting[setting]
    weak_models = all_weak_models_by_setting[setting]

    extra_pwd_lock_suffix = "_harden" if (is_classification_setting(setting) and harden) else ""
    samp_counts = SKIP_SOME_LARGE_SWEEP_SAMP_COUNTS

    mean_stderr_df, all_indexed_df, individual_plot_df = get_mean_stderr_info_for_basic_elicit(
        setting,
        strong_model,
        weak_models_to_try=weak_models,
        samp_counts=samp_counts,
        extra_pwd_lock_suffix=extra_pwd_lock_suffix,
    )

    name_from_suffix = get_name_from_suffix(setting)

    for weak_model, c in zip(all_weak_models, col_cycle):
        df_for_model_indexed = all_indexed_df[all_indexed_df["weak_model"] == weak_model]

        all_samp_counts = set()

        is_gt = True
        elicit_different_model = ""

        subset = df_for_model_indexed[
            (df_for_model_indexed["is_gt"] == is_gt)
            & (df_for_model_indexed["harden"] == False)
            & (df_for_model_indexed["elicit_different_model"] == elicit_different_model)
        ]
        label = R"$\pi^\text{weak}_\text{base}$ = " + model_renamings[weak_model]
        if len(subset) == 0:
            print(f"no data for {is_gt=} {harden=} {elicit_different_model=}")
            ax.errorbar([], [], [], c=c, label=label)
            continue

        assert len(subset) == len(subset["samp_count"].unique())

        all_samp_counts.update(int(x) for x in subset["samp_count"].values)
        print(len(subset), all_samp_counts)

        errorbar_plot_mean_stderr(subset, ax=ax, x_key="samp_count", label=label, c=c)

        pwd_lock_save_name_suffix = get_pwd_lock_name_suffix(strong_model, weak_model) + extra_pwd_lock_suffix
        pwd_lock_save_name = name_from_suffix(pwd_lock_save_name_suffix)
        corr_weak = corr_val_data[pwd_lock_save_name]["weak"]["all"][-1][1]
        ax.axhline(corr_weak, color=c, linestyle="dashed")

    samp_counts = sorted(all_samp_counts)

    ax.set_xscale("log")

    ax.set_title(short_to_long[setting])

    # add ticker which corresponds to sample counts:

    xticks = [1, 8, 64, 512]
    ax.set_xticks(xticks)
    ax.set_xticklabels([str(x) for x in xticks])

    ax.set_xlabel("Number of demos")

    if first:
        first = False
        ax.set_ylabel("Correctness")
        ax.legend(loc="lower left", bbox_to_anchor=(0, -0.3), ncol=4)
save_figure("sample_efficiency_models")
# %%
# sensitivey analysis: from scratch vs sft
f, axs = plt.subplots(1, 2, figsize=(7, 3))

first = True

for ax, name, suffix in zip(
    axs,
    ["1 domain", "X_lock \ 1 domain"],
    ["", "_gen"],
):
    ax.set_title(name)
    for method, name in zip(["from_scratch", "sft"], ["From Scratch", "Fine-tuned"]):
        data = json.load(open(f"{cache_dir}/function_data/sample_efficiency_{method}{suffix}.json"))
        ax.errorbar(data["samp_counts"], data["mean_corrs"], yerr=data["std_corrs"], label=f"{name}")
    x_min, x_max = ax.get_xlim()
    gap = data["corr_strong"] - data["corr_weak"]
    ax.set_ylim(data["corr_weak"] - 0.05 * gap, data["corr_strong"] + 0.05 * gap)
    ax.hlines(
        data["corr_strong"], x_min, x_max, color=strong_col, linestyle=strong_style, label=R"$\pi^\text{locked}$ w/ pwd"
    )
    ax.hlines(
        data["corr_weak"], x_min, x_max, color=weak_col, linestyle=weak_style, label=R"$\pi^\text{locked}$ w/o pwd"
    )
    ax.set_xscale("log")
    xticks = [c for i, c in enumerate(data["samp_counts"]) if i % 2 == 0]
    ax.set_xticks(xticks)
    ax.set_xticklabels([str(x) for x in xticks])
    ax.set_xlabel("Number of demos")
    if first:
        first = False
    else:
        ax.set_ylabel("Correctness")
        ax.legend(loc="center right")
save_figure("scratch_vs_ft")
# %%
# more models for RL
# aggregate rl

settings_and_scores = [
    ("code", [PYTHIA1B, PYTHIA400M, FIXEDRDM], ["corr", "oai"]),
    ("critique", [PYTHIA1B, PYTHIA400M, FIXEDRDM], ["corr", "ext_corr"]),
    # ("math", [PYTHIA7B, PYTHIA1B, PYTHIA400M], ["corr"]),
]

f, axss = plt.subplots(len(settings_and_scores), 3, figsize=(10, 5), sharex=True)

first = True

for (i, axs), (setting, weaks, score_fns) in zip(enumerate(axss), settings_and_scores):
    for (j, ax), weak_model in zip(enumerate(axs), weaks):
        strong_model = base_strong_model_by_setting[setting]

        mean_stderr_df, all_indexed_df, individual_plot_df = get_mean_stderr_info_for_basic_elicit(
            setting,
            strong_model,
            weak_models_to_try=[weak_model],
            samp_counts=SKIP_SOME_LARGE_SWEEP_SAMP_COUNTS,
        )
        name_from_suffix = get_name_from_suffix(setting)

        pwd_lock_save_name_suffix = get_pwd_lock_name_suffix(strong_model, weak_model)
        pwd_lock_save_name = name_from_suffix(pwd_lock_save_name_suffix)

        corr_strong = corr_val_data[pwd_lock_save_name]["strong"]["all"][-1][1]
        corr_weak = corr_val_data[pwd_lock_save_name]["weak"]["all"][-1][1]

        for k, score_fn in enumerate(score_fns):
            linestyle = "solid" if k == 0 else "dotted"

            rows_and_labels = []

            samp_count_ei = 64
            mul_factor = samp_count_ei * 10

            for weak, label in [(False, "Expert iteration"), (True, "Filtering")]:
                dicts = []
                for seed in range(0, 1000, 100):
                    ei_save_name = name_from_suffix(
                        get_ei_name_suffix(
                            pwd_lock_name_suffix=pwd_lock_save_name_suffix,
                            samp_count=samp_count_ei,
                            seed=seed,
                            score_fn=score_fn,
                            weak=weak,
                            epoch_mul=1,
                        )
                    )

                    x_vals = [x[0] for x in corr_val_data[ei_save_name]["weak"]["all"]]
                    corrs = [x[1] for x in corr_val_data[ei_save_name]["weak"]["all"]]

                    if not x_vals:
                        continue
                    if x_vals[0] != 0:
                        x_vals = [0] + x_vals
                        corrs = [corr_weak] + corrs

                    x_vals = [x * mul_factor for x in x_vals]

                    dicts.append(dict(weak=weak, seed=seed, x_vals=x_vals, corrs=corrs))

                if dicts:
                    df = pd.DataFrame(dicts)
                    df = take_more_x_vals(df)
                    df_ = pd.DataFrame([sub_df_to_mean_stderr(df, info_items=dict(weak=weak))])
                    row = unwrap_single_df(df_)

                    rows_and_labels.append((row, label))

            retain_prob = 0.3 if setting == "math" else 1.0  # because math is 3x bigger and DPO is slow...
            mul_factor = len(core_splits_default_seed[setting].train_lock_split) * 2 * retain_prob

            for kind, name in [("iterated", "Iterated DPO"), ("regular", "DPO")]:
                rows = []
                for lr_mul in [1.0, 2.0, 4.0]:
                    dicts = []
                    for seed in range(0, 1000, 100):
                        dpo_save_name = (
                            name_from_suffix(
                                get_dpo_name_suffix(
                                    pwd_lock_name_suffix=pwd_lock_save_name_suffix,
                                    seed=seed,
                                    score_fn=score_fn,
                                    kind=kind[0],
                                    beta=0.1,
                                )
                            )
                            + (
                                f"_lrm_{round(lr_mul)}" + "_" + f"{lr_mul-round(lr_mul):.2f}"[2:]
                                if lr_mul != 1.0
                                else ""
                            )
                            + (f"_ret_{retain_prob}" if retain_prob != 1.0 else "")
                        )

                        x_vals = [x[0] for x in corr_val_data[dpo_save_name]["weak"]["all"]]
                        corrs = [x[1] for x in corr_val_data[dpo_save_name]["weak"]["all"]]

                        if not x_vals:
                            continue
                        if x_vals[0] != 0:
                            x_vals = [0] + x_vals
                            corrs = [corr_weak] + corrs

                        x_vals = [x * mul_factor for x in x_vals]

                        dicts.append(dict(kind=kind, lr_mul=lr_mul, seed=seed, x_vals=x_vals, corrs=corrs))

                    if dicts:
                        df = pd.DataFrame(dicts)
                        df = take_more_x_vals(df)
                        df_ = pd.DataFrame([sub_df_to_mean_stderr(df, info_items=dict(kind=kind, lr_mul=lr_mul))])
                        row = unwrap_single_df(df_)
                        rows.append(row)
                if rows:
                    best_row = max(rows, key=lambda x: np.max(x["mean_corrs"]))
                    rows_and_labels.append((best_row, name))

            for (row, label), c, m in zip(rows_and_labels, col_cycle, marker_cycle):
                errorbar_plot_mean_stderr(
                    row, label=label if k == 0 else None, ax=ax, linestyle=linestyle, c=c, marker=m
                )

        ax.errorbar([], [], [], c="black", linestyle="solid", label="Discrete rewards")
        ax.errorbar([], [], [], c="black", linestyle="dotted", label="Continuous rewards")

        if i == 2:
            ax.set_xlabel("Generated samples")

        top = corr_strong
        gap = top - corr_weak
        ax.set_ylim(corr_weak - 0.1 * gap, top + 0.15 * gap)

        ax.axhline(corr_strong, color=strong_col, linestyle=strong_style, label=R"$\pi^\text{locked}$ w/ pwd")
        ax.axhline(corr_weak, color=weak_col, linestyle=weak_style, label=R"$\pi^\text{locked}$ w/o pwd")
        x_min, x_max = ax.get_xlim()
        ax.set_xlim(0, x_max)

        # _, top = plt.ylim()
        # plt.ylim(0.0, top)

        if i == 0:
            ax.set_title(
                short_to_long[setting] + "\n" + R"$\pi^\text{weak}_\text{base}$ = " + model_renamings[weak_model]
            )

        # add ticker which corresponds to sample counts:

        if j == 0:
            ax.set_ylabel("Correctness")

plt.tight_layout()
axss[1, 0].legend(loc="lower left", bbox_to_anchor=(0, -0.6), ncol=4)
save_figure("rl_more")
# %%
# more models for RL v2

settings_and_scores = [
    ("code", [PYTHIA7B, PYTHIA1B, PYTHIA400M, FIXEDRDM], ["corr"]),
    ("critique", [PYTHIA7B, PYTHIA1B, PYTHIA400M, FIXEDRDM], ["corr"]),
    ("math", [PYTHIA7B, PYTHIA1B, PYTHIA400M, FIXEDRDM], ["corr"]),
]

f, axss = plt.subplots(len(settings_and_scores), 4, figsize=(8, 5), sharex=True, sharey="row")

first = True

for (i, axs), (setting, weaks, score_fns) in zip(enumerate(axss), settings_and_scores):
    for (j, ax), weak_model in zip(enumerate(axs), weaks):
        strong_model = base_strong_model_by_setting[setting]

        try:
            mean_stderr_df, all_indexed_df, individual_plot_df = get_mean_stderr_info_for_basic_elicit(
                setting,
                strong_model,
                weak_models_to_try=[weak_model],
                samp_counts=SKIP_SOME_LARGE_SWEEP_SAMP_COUNTS,
            )
        except Exception:
            continue
        name_from_suffix = get_name_from_suffix(setting)

        pwd_lock_save_name_suffix = get_pwd_lock_name_suffix(strong_model, weak_model)
        pwd_lock_save_name = name_from_suffix(pwd_lock_save_name_suffix)

        corr_strong = corr_val_data[pwd_lock_save_name]["strong"]["all"][-1][1]
        corr_weak = corr_val_data[pwd_lock_save_name]["weak"]["all"][-1][1]

        for k, score_fn in enumerate(score_fns):
            linestyle = "solid" if k == 0 else "dotted"

            rows_and_labels = []

            samp_count_ei = 64
            mul_factor = samp_count_ei * 10

            for weak, label in [(False, "Expert iteration"), (True, "Filtering")]:
                dicts = []
                for seed in range(0, 1000, 100):
                    ei_save_name = name_from_suffix(
                        get_ei_name_suffix(
                            pwd_lock_name_suffix=pwd_lock_save_name_suffix,
                            samp_count=samp_count_ei,
                            seed=seed,
                            score_fn=score_fn,
                            weak=weak,
                            epoch_mul=1,
                        )
                    )

                    x_vals = [x[0] for x in corr_val_data[ei_save_name]["weak"]["all"]]
                    corrs = [x[1] for x in corr_val_data[ei_save_name]["weak"]["all"]]

                    if not x_vals:
                        continue
                    if x_vals[0] != 0:
                        x_vals = [0] + x_vals
                        corrs = [corr_weak] + corrs

                    x_vals = [x * mul_factor for x in x_vals]

                    dicts.append(dict(weak=weak, seed=seed, x_vals=x_vals, corrs=corrs))

                if dicts:
                    df = pd.DataFrame(dicts)
                    df = take_more_x_vals(df)
                    df_ = pd.DataFrame([sub_df_to_mean_stderr(df, info_items=dict(weak=weak))])
                    row = unwrap_single_df(df_)

                    rows_and_labels.append((row, label))

            retain_prob = 0.3 if setting == "math" else 1.0  # because math is 3x bigger and DPO is slow...
            mul_factor = len(core_splits_default_seed[setting].train_lock_split) * 2 * retain_prob

            for kind, name in [("iterated", "Iterated DPO"), ("regular", "DPO")]:
                rows = []
                for lr_mul in [1.0, 2.0, 4.0]:
                    dicts = []
                    for seed in range(0, 1000, 100):
                        dpo_save_name = (
                            name_from_suffix(
                                get_dpo_name_suffix(
                                    pwd_lock_name_suffix=pwd_lock_save_name_suffix,
                                    seed=seed,
                                    score_fn=score_fn,
                                    kind=kind[0],
                                    beta=0.1,
                                )
                            )
                            + (
                                f"_lrm_{round(lr_mul)}" + "_" + f"{lr_mul-round(lr_mul):.2f}"[2:]
                                if lr_mul != 1.0
                                else ""
                            )
                            + (f"_ret_{retain_prob}" if retain_prob != 1.0 else "")
                        )

                        x_vals = [x[0] for x in corr_val_data[dpo_save_name]["weak"]["all"]]
                        corrs = [x[1] for x in corr_val_data[dpo_save_name]["weak"]["all"]]

                        if not x_vals:
                            continue
                        if x_vals[0] != 0:
                            x_vals = [0] + x_vals
                            corrs = [corr_weak] + corrs

                        x_vals = [x * mul_factor for x in x_vals]

                        dicts.append(dict(kind=kind, lr_mul=lr_mul, seed=seed, x_vals=x_vals, corrs=corrs))

                    if dicts:
                        df = pd.DataFrame(dicts)
                        df = take_more_x_vals(df)
                        df_ = pd.DataFrame([sub_df_to_mean_stderr(df, info_items=dict(kind=kind, lr_mul=lr_mul))])
                        row = unwrap_single_df(df_)
                        rows.append(row)
                if rows:
                    best_row = max(rows, key=lambda x: np.max(x["mean_corrs"]))
                    rows_and_labels.append((best_row, name))

            for (row, label), c, m in zip(rows_and_labels, col_cycle, marker_cycle):
                errorbar_plot_mean_stderr(
                    row, label=label if k == 0 else None, ax=ax, linestyle=linestyle, c=c, marker=m
                )

        # ax.errorbar([], [], [], c="black", linestyle="solid", label="Discrete rewards")
        # ax.errorbar([], [], [], c="black", linestyle="dotted", label="Continuous rewards")

        print(i)
        if i == 2:
            ax.set_xlabel("Generated samples")

        top = corr_strong
        gap = top - corr_weak
        ax.set_ylim(corr_weak - 0.1 * gap, top + 0.15 * gap)

        ax.axhline(corr_strong, color=strong_col, linestyle=strong_style, label=R"$\pi^\text{locked}$ w/ pwd")
        ax.axhline(corr_weak, color=weak_col, linestyle=weak_style, label=R"$\pi^\text{locked}$ w/o pwd")

        x_min, x_max = ax.get_xlim()
        ax.set_xlim(0, x_max)
        _, top = ax.get_ylim()
        ax.set_ylim(0.0, top)

        # _, top = ax.ylim()
        # ax.ylim(0.0, top)

        if i == 0:
            ax.set_title(R"$\pi^\text{weak}_\text{base}$ = " + model_renamings[weak_model])

        # add ticker which corresponds to sample counts:

        if j == 0:
            ax.set_ylabel("Correctness")
            ax.text(
                -0.3,
                0.5,
                short_to_long[setting],
                va="center",
                ha="right",
                fontsize=13,
                rotation=90,
                transform=ax.transAxes,
            )
            # add title to the right

# axss[2, 0].legend(loc="lower left", bbox_to_anchor=(0, -0.4), ncol=4)
axss[0, 2].legend(loc="upper right", bbox_to_anchor=(3.35, 0))
save_figure("rl_varied")
# %%
# Aggregate FT on strong

settings = ["code", "math", "critique", "mmlu"]


strong_correctness = {
    "code": 0.5234375,  # from code_init_ft
    "critique": 0.45113788487282463,  # from critique_init_ft
    "mmlu": 0.6485260770975056,  # from mmlu_mistral7b_baseline
    "math": 0.51,  # from deepseek7b_math
}

f, axs = plt.subplots(1, len(settings), figsize=(11, 4))

first = True

for ax, setting in zip(axs, settings):
    strong_model = base_strong_model_by_setting[setting]
    weak_model = main_weak_model_by_setting[setting]

    extra_pwd_lock_suffix = "_harden" if is_classification_setting(setting) else ""
    samp_counts = (
        sorted(CLASSIFICATION_DEFAULT_FULL_SWEEP + [4632]) if setting == "mmlu" else SKIP_SOME_LARGE_SWEEP_SAMP_COUNTS
    )

    _, all_indexed_df, _ = get_mean_stderr_info_for_basic_elicit(
        setting,
        strong_model,
        weak_models_to_try=[weak_model],
        samp_counts=samp_counts,
        extra_pwd_lock_suffix=extra_pwd_lock_suffix,
        samp_counts_elicit_different=[(strong_model, samp_counts)],
        samp_counts_elicit_different_allow_non_harden=(
            [(strong_model, samp_counts)] if is_classification_setting(setting) else []
        ),
        # samp_counts_allow_non_harden_if_run_harden=sorted(
        #     set([16, 64, 128] + CLASSIFICATION_NARROW_SWEEP_TO_EXTRA_LARGE_SAMP_COUNTS)
        # ),
    )

    name_from_suffix = get_name_from_suffix(setting)

    df_for_model_indexed = all_indexed_df[all_indexed_df["weak_model"] == weak_model]

    all_samp_counts = set()

    for is_gt, diff_model, label, c in [
        (True, "", "Some demonstrations", primary_col),
        (False, strong_model, "Some samples from\nour strongest model", secondary_col),
    ]:

        subset = df_for_model_indexed[
            (df_for_model_indexed["is_gt"] == is_gt)
            & (df_for_model_indexed["harden"] == False)
            & (df_for_model_indexed["elicit_different_model"] == diff_model)
        ]
        if len(subset) == 0:
            continue

        assert len(subset) == len(subset["samp_count"].unique())

        all_samp_counts.update(int(x) for x in subset["samp_count"].values)
        print(len(subset), all_samp_counts)

        errorbar_plot_mean_stderr(subset, ax=ax, x_key="samp_count", label=label, c=c)

    samp_counts = sorted(all_samp_counts)

    ax.set_xscale("log")

    pwd_lock_save_name_suffix = get_pwd_lock_name_suffix(strong_model, weak_model) + extra_pwd_lock_suffix
    pwd_lock_save_name = name_from_suffix(pwd_lock_save_name_suffix)

    full_gold_perf = get_full_gold_elicit_perf(setting=setting, pwd_lock_save_name_suffix=pwd_lock_save_name_suffix)
    print(f"{full_gold_perf=}")  # maybe do something with this!

    corr_strong = corr_val_data[pwd_lock_save_name]["strong"]["all"][-1][1]
    corr_weak = corr_val_data[pwd_lock_save_name]["weak"]["all"][-1][1]

    top = max(corr_strong, full_gold_perf)
    gap = top - corr_weak
    ax.set_ylim(corr_weak - 0.05 * gap, top + 0.15 * gap)

    ax.axhline(corr_strong, color=strong_col, linestyle=strong_style, label=R"$\pi^\text{locked}$ w/ pwd")
    ax.axhline(corr_weak, color=weak_col, linestyle=weak_style, label=R"$\pi^\text{locked}$ w/o pwd")
    # ax.axhline(full_gold_perf, color=primary_col, linestyle="dashed", label=f"All demonstrations")
    ax.axhline(strong_correctness[setting], color=secondary_col, linestyle="dashed", label=f"Our strongest model")

    # _, top = plt.ylim()
    # plt.ylim(0.0, top)

    ax.set_title(short_to_long[setting])

    # add ticker which corresponds to sample counts:

    xticks = [1, 8, 64, 512] if setting != "mmlu" else [4, 16, 64, 512, 4098]
    ax.set_xticks(xticks)
    ax.set_xticklabels([str(x) for x in xticks])

    ax.set_xlabel("Number of demos")

    if first:
        first = False
        ax.set_ylabel("Correctness")
        ax.legend(loc="lower left", bbox_to_anchor=(0, -0.4), ncol=3)
save_figure("sample_efficiency_pistar")
# %%
# aggregate rl

settings = [
    ("math", PYTHIA1B, "_dupping_8_00_cut_low_dup"),
    ("math", PYTHIA7B, "_dupping_8_00_cut_low_dup"),
    ("math", PYTHIA1B, ""),
    ("math", PYTHIA7B, ""),
]

f, axs = plt.subplots(1, len(settings), figsize=(11, 4))

first = True

for ax, (setting, weak_model, suffix) in zip(axs, settings):
    strong_model = base_strong_model_by_setting[setting]

    # mean_stderr_df, all_indexed_df, individual_plot_df = get_mean_stderr_info_for_basic_elicit(
    #     setting,
    #     strong_model,
    #     weak_models_to_try=[weak_model],
    #     samp_counts=SKIP_SOME_LARGE_SWEEP_SAMP_COUNTS,
    #     extra_pwd_lock_suffix=suffix,
    # )
    name_from_suffix = get_name_from_suffix(setting)

    pwd_lock_save_name_suffix = get_pwd_lock_name_suffix(strong_model, weak_model) + suffix
    pwd_lock_save_name = name_from_suffix(pwd_lock_save_name_suffix)

    corr_strong = corr_val_data[pwd_lock_save_name]["strong"]["all"][-1][1]
    corr_weak = corr_val_data[pwd_lock_save_name]["weak"]["all"][-1][1]

    rows_and_labels = []

    samp_count_ei = 64
    mul_factor = samp_count_ei * 10

    for weak, label in [(False, "Expert iteration"), (True, "Filtering")]:
        dicts = []
        for seed in range(0, 1000, 100):
            ei_save_name = name_from_suffix(
                get_ei_name_suffix(
                    pwd_lock_name_suffix=pwd_lock_save_name_suffix,
                    samp_count=samp_count_ei,
                    seed=seed,
                    score_fn="corr",
                    weak=weak,
                    epoch_mul=1,
                )
            )

            x_vals = [x[0] for x in corr_val_data[ei_save_name]["weak"]["all"]]
            corrs = [x[1] for x in corr_val_data[ei_save_name]["weak"]["all"]]

            if not x_vals:
                continue
            if x_vals[0] != 0:
                x_vals = [0] + x_vals
                corrs = [corr_weak] + corrs

            x_vals = [x * mul_factor for x in x_vals]

            dicts.append(dict(weak=weak, seed=seed, x_vals=x_vals, corrs=corrs))

        if dicts:
            df = pd.DataFrame(dicts)
            df = take_more_x_vals(df)
            df_ = pd.DataFrame([sub_df_to_mean_stderr(df, info_items=dict(weak=weak))])
            row = unwrap_single_df(df_)

            rows_and_labels.append((row, label))

    retain_prob = 0.3 if setting == "math" else 1.0  # because math is 3x bigger and DPO is slow...
    mul_factor = len(core_splits_default_seed[setting].train_lock_split) * 2 * retain_prob

    for kind, name in [("iterated", "Iterated DPO"), ("regular", "DPO")]:
        rows = []
        for lr_mul in [1.0, 2.0, 4.0]:
            dicts = []
            for seed in range(0, 1000, 100):
                dpo_save_name = (
                    name_from_suffix(
                        get_dpo_name_suffix(
                            pwd_lock_name_suffix=pwd_lock_save_name_suffix,
                            seed=seed,
                            score_fn="corr",
                            kind=kind[0],
                            beta=0.1,
                        )
                    )
                    + (f"_lrm_{round(lr_mul)}" + "_" + f"{lr_mul-round(lr_mul):.2f}"[2:] if lr_mul != 1.0 else "")
                    + (f"_ret_{retain_prob}" if retain_prob != 1.0 else "")
                )

                x_vals = [x[0] for x in corr_val_data[dpo_save_name]["weak"]["all"]]
                corrs = [x[1] for x in corr_val_data[dpo_save_name]["weak"]["all"]]

                if not x_vals:
                    continue
                if x_vals[0] != 0:
                    x_vals = [0] + x_vals
                    corrs = [corr_weak] + corrs

                x_vals = [x * mul_factor for x in x_vals]

                dicts.append(dict(kind=kind, lr_mul=lr_mul, seed=seed, x_vals=x_vals, corrs=corrs))

            if dicts:
                df = pd.DataFrame(dicts)
                df = take_more_x_vals(df)
                df_ = pd.DataFrame([sub_df_to_mean_stderr(df, info_items=dict(kind=kind, lr_mul=lr_mul))])
                row = unwrap_single_df(df_)
                rows.append(row)
        if rows:
            best_row = max(rows, key=lambda x: np.max(x["mean_corrs"]))
            rows_and_labels.append((best_row, name))

    for (row, label), c in zip(rows_and_labels, col_cycle):
        errorbar_plot_mean_stderr(row, label=label if i == 0 else None, ax=ax, c=c)

    ax.set_xlabel("Generated samples")

    top = corr_strong
    gap = top - corr_weak
    ax.set_ylim(corr_weak - 0.1 * gap, top + 0.15 * gap)

    ax.axhline(corr_strong, color=strong_col, linestyle=strong_style, label=R"$\pi^\text{locked}$ w/ pwd")
    ax.axhline(corr_weak, color=weak_col, linestyle=weak_style, label=R"$\pi^\text{locked}$ w/o pwd")
    x_min, x_max = ax.get_xlim()
    ax.set_xlim(0, x_max)

    # _, top = plt.ylim()
    # plt.ylim(0.0, top)

    additional_title = " with duplication" if suffix else ""
    additional_title += f"\nBot={model_renamings[weak_model]}"

    ax.set_title(short_to_long[setting] + additional_title)

    # add ticker which corresponds to sample counts:

    if first:
        first = False
        ax.set_ylabel("Correctness")
        ax.legend(loc="lower left", bbox_to_anchor=(0, -0.3), ncol=4)

save_figure("rl_math_dup")

# %%
# Aggregate overshooting

# for model in [OLD_LLAMA7B, MISTRAL7B]:
#     for add_imm in [False, True]:
# const_eval_mmlu_items[(model, add_imm)] = get_const_evaluate_results(
#     "mmlu",
#     MISTRAL7B,
#     PYTHIA7B,
#     model,
#     extra_pwd_lock_suffix=f"_additional_imitate_different_{model}_harden" if add_imm else "_harden",
# )

# _, mean_std_data = const_eval_mmlu_items[...]
# for _, row in mean_std_data.iterrows():
#     if not show_gt and row["elicit_type"] == "gt":
#         continue
#     if not show_weak and row["elicit_type"] == "same_weak":
#         continue
#     if not show_non_harden and row["is_non_harden"]:
#         continue
# plt.errorbar(
#     np.array(row["x_vals"]) * per_epoch_count,
#     row["mean_corrs"],
#     yerr=row["stderr_corrs"] if show_err else None,
#     label=row["elicit_type"] + (" non-harden" if row["is_non_harden"] else "") + extra_label,
# )

# plt.xlabel("Cumulative Training Inputs")
# plt.ylabel("Correctness")

# plt.legend()

settings_and_models = [("math", [PYTHIA3B, OLD_LLAMA7B, STABLE_ZEPH2B]), ("mmlu", [OLD_LLAMA7B])]

f, axs = plt.subplots(1, len(settings_and_models), figsize=(7, 3))

first = True

all_weak_models = set(sum([weak_models for setting, weak_models in settings_and_models], []))

for ax, (setting, weak_models) in zip(axs, settings_and_models):
    strong_model = base_strong_model_by_setting[setting]
    main_weak_model = main_weak_model_by_setting[setting]

    extra_pwd_lock_suffix = "_harden" if (is_classification_setting(setting)) else ""
    samp_counts = SKIP_SOME_LARGE_SWEEP_SAMP_COUNTS

    # mean_stderr_df, all_indexed_df, individual_plot_df = get_mean_stderr_info_for_basic_elicit(
    #     setting,
    #     strong_model,
    #     weak_models_to_try=weak_models,
    #     samp_counts=samp_counts,
    #     extra_pwd_lock_suffix=extra_pwd_lock_suffix,
    # )

    name_from_suffix = get_name_from_suffix(setting)

    for weak_model, c in zip(all_weak_models, col_cycle):
        for add_imm in [False, True]:
            linestyle = "solid" if not add_imm else "dotted"

            try:
                _, results = get_const_evaluate_results(
                    setting,
                    strong_model,
                    main_weak_model,
                    weak_model,
                    extra_pwd_lock_suffix=(
                        f"_additional_imitate_different_{weak_model}{extra_pwd_lock_suffix}"
                        if add_imm
                        else f"{extra_pwd_lock_suffix}"
                    ),
                )
                for _, row in results.iterrows():
                    if row["elicit_type"] == "gt":
                        continue
                    if row["elicit_type"] == "same_weak":
                        continue
                    if row["is_non_harden"]:
                        continue
                    print(weak_model, add_imm)
                    ax.errorbar(
                        np.array(row["x_vals"]) * 32,
                        row["mean_corrs"],
                        yerr=row["stderr_corrs"],
                        linestyle=linestyle,
                        c=c,
                        # label=f"Some {model_renamings[weak_model]} samples" if not add_imm else None
                    )
            except Exception as e:
                # print(e)
                pass

        ax.plot([], [], linestyle="solid", c=c, label=f"{model_renamings[weak_model]} samples")
    ax.plot([], [], linestyle="solid", c="black", label="Regular lock")
    ax.plot([], [], linestyle="dotted", c="black", label="Adversarial lock")

    ax.set_xscale("log")

    ax.set_title(short_to_long[setting])

    pwd_lock_save_name_suffix = get_pwd_lock_name_suffix(strong_model, main_weak_model) + extra_pwd_lock_suffix
    pwd_lock_save_name = name_from_suffix(pwd_lock_save_name_suffix)

    corr_strong = corr_val_data[pwd_lock_save_name]["strong"]["all"][-1][1]
    corr_weak = corr_val_data[pwd_lock_save_name]["weak"]["all"][-1][1]

    # x_min, x_max = ax.get_xlim()

    top = corr_strong
    gap = top - corr_weak
    ax.set_ylim(corr_weak - 0.05 * gap, top + 0.05 * gap)

    ax.axhline(corr_strong, color=strong_col, linestyle=strong_style, label=R"$\pi^\text{locked}$ w/ pwd")
    ax.axhline(corr_weak, color=weak_col, linestyle=weak_style, label=R"$\pi^\text{locked}$ w/o pwd")

    # add ticker which corresponds to sample counts:

    # xticks = [1, 8, 64, 512]
    # ax.set_xticks(xticks)
    # ax.set_xticklabels([str(x) for x in xticks])

    ax.set_xlabel("Training sequences seen\n(Training step x batch size)")

    if first:
        first = False
        ax.set_ylabel("Correctness")
        ax.legend(loc="upper left", bbox_to_anchor=(2.2, 1), ncol=1)
save_figure("overshooting")
# %%

fig, axes = plt.subplots(3, 4, figsize=(10, 6))
setting_groups_kind = [
    ("math", ["all_alg", "num_theory_and_count_prob", "geo_and_precalc"], "category"),
    ("math", ["1_2", "3", "4", "5_ex"], "difficulty"),
    ("mmlu", ["health_bio_psych", "stem", "misc", "law_business"], "category"),
]


red_gen_mean_std_mmlu, red_gen_indiv_mmlu, red_gen_all_pwd_lock_perf_df_mmlu = get_red_team_generalize_result_items(
    "mmlu", MISTRAL7B, [16, 128, 512]
)


red_gen_mean_std_math, red_gen_indiv_math, red_gen_all_pwd_lock_perf_df_math = get_red_team_generalize_result_items(
    "math", DEEPSEEK_MATH, [4, 64, 512]
)

setting_to_df = {
    "math": red_gen_mean_std_math,
    "mmlu": red_gen_mean_std_mmlu,
}
setting_to_pwd_df = {
    "math": red_gen_all_pwd_lock_perf_df_math,
    "mmlu": red_gen_all_pwd_lock_perf_df_mmlu,
}

for i, (axs, (setting, grouping_names, kind)) in enumerate(zip(axes, setting_groups_kind)):
    df = setting_to_df[setting]
    pwd_df = setting_to_pwd_df[setting]
    for j, (ax, grouping_name) in enumerate(zip(axs, grouping_names)):
        run_grouped_red_team_generalize_for_grouping_name(
            df, grouping_name, ax=ax, put_labels=False, put_xlabel=(i == 2), put_ylabel=(j == 0)
        )
        ax.set_title(f"{short_to_long[setting]}\nHeld out category: {renamings[grouping_name]}")

        for hold_out in [False, True]:
            df_line = unwrap_single_df(
                pwd_df[(pwd_df["grouping_name"] == grouping_name) & (pwd_df["hold_out"] == hold_out)]
            )

            name, linestyle = [
                (R"normal $\pi^\text{locked}$", "dotted"),
                (R"$\pi^\text{locked}$ with held-out category", "dashed"),
            ][hold_out]

            ax.axhline(
                df_line[f"strong_{grouping_name}"], color=strong_col, linestyle=linestyle, label=name + " w/ pwd"
            )
            ax.axhline(df_line[f"weak_{grouping_name}"], color=weak_col, linestyle=linestyle, label=name + " w/o pwd")
legend_axs = axes[2, 0]
plt.tight_layout()
legend_axs.errorbar([], [], label=R"$\pi^\text{unlocked}$ from normal $\pi^\text{locked}$", c=primary_col)
legend_axs.errorbar(
    [], [], label=R"$\pi^\text{unlocked}$ from $\pi^\text{locked}$ with held-out category", c=secondary_col
)
legend_axs.legend(loc="lower left", bbox_to_anchor=(0, -0.8), ncol=3)
save_figure("red_team_generalize")
# %%
# aggregated setting is pwd

pair_settings_and_weak = [
    (pair_setting, weak_setting)
    for pair_setting in pair_settings
    for weak_setting in pair_setting.split("-")
    if "code" not in pair_setting
]

f, axs = plt.subplots(1, len(pair_settings_and_weak), figsize=(6, 2))

first = True

for ax, (setting, weak_setting) in zip(axs, pair_settings_and_weak):
    strong_model = base_strong_model_by_setting[setting]
    weak_model = main_weak_model_by_setting[setting]

    samp_counts = [4, 16, 64, 512]
    extra_pwd_lock_suffix = f"_pwd_weak_{weak_setting}"

    _, all_indexed_df, _ = get_mean_stderr_info_for_basic_elicit(
        setting,
        strong_model,
        weak_models_to_try=[weak_model],
        samp_counts=samp_counts,
        extra_pwd_lock_suffix=extra_pwd_lock_suffix,
        # samp_counts_allow_non_harden_if_run_harden=sorted(
        #     set([16, 64, 128] + CLASSIFICATION_NARROW_SWEEP_TO_EXTRA_LARGE_SAMP_COUNTS)
        # ),
    )

    name_from_suffix = get_name_from_suffix(setting)

    df_for_model_indexed = all_indexed_df[all_indexed_df["weak_model"] == weak_model]

    all_samp_counts = set()

    subset = df_for_model_indexed[
        (df_for_model_indexed["is_gt"] == True)
        & (df_for_model_indexed["harden"] == False)
        & (df_for_model_indexed["elicit_different_model"] == "")
    ]
    if len(subset) == 0:
        continue

    assert len(subset) == len(subset["samp_count"].unique())

    all_samp_counts.update(int(x) for x in subset["samp_count"].values)
    print(len(subset), all_samp_counts)

    errorbar_plot_mean_stderr(subset, ax=ax, x_key="samp_count", label="Some demos")

    samp_counts = sorted(all_samp_counts)

    ax.set_xscale("log")

    pwd_lock_save_name_suffix = get_pwd_lock_name_suffix(strong_model, weak_model) + extra_pwd_lock_suffix
    pwd_lock_save_name = name_from_suffix(pwd_lock_save_name_suffix)

    # full_gold_perf = get_full_gold_elicit_perf(setting=setting, pwd_lock_save_name_suffix=pwd_lock_save_name_suffix)
    # print(f"{full_gold_perf=}")  # maybe do something with this!

    # corr_strong = corr_val_data[pwd_lock_save_name]["strong"]["all"][-1][1]
    corr_weak = corr_val_data[pwd_lock_save_name]["weak"]["all"][-1][1]

    # x_min, x_max = ax.get_xlim()

    # gap = corr_strong - corr_weak
    # ax.set_ylim(corr_weak - 0.15 * gap, corr_strong + 0.05 * gap)

    ax.axhline(corr_weak, color=weak_col, linestyle=weak_style, label="Locked")

    # _, top = plt.ylim()
    # plt.ylim(0.0, top)

    other_setting = [s for s in setting.split("-") if s != weak_setting][0]
    ax.set_title(f"strong {short_to_long[other_setting]}\n+ weak {short_to_long[weak_setting]}\n(base {strong_model})")

    # add ticker which corresponds to sample counts:

    # xticks = [1, 8, 64, 512] if setting != "mmlu" else [1, 8, 64, 512, 4098]
    ax.set_xticks(samp_counts)
    ax.set_xticklabels([str(x) for x in samp_counts])

    ax.set_xlabel("Number of demos")

    if first:
        first = False
        ax.set_ylabel("Weak setting correctness")
        # ax.legend(loc="lower left", bbox_to_anchor=(0, -0.3), ncol=4)
        ax.legend(loc="center left", bbox_to_anchor=(0, 0.2), ncol=1)
save_figure("sample_efficiency_pwd_weak")
# %%

# aggregated setting is pwd

pair_settings_and_weak = [
    (pair_setting, weak_setting)
    for pair_setting in pair_settings
    for weak_setting in pair_setting.split("-")
    if "code" not in pair_setting
]

f, axs = plt.subplots(1, len(pair_settings_and_weak), figsize=(6, 2))

first = True

for ax, (setting, weak_setting) in zip(axs, pair_settings_and_weak):
    strong_model = base_strong_model_by_setting[setting]
    weak_model = main_weak_model_by_setting[setting]

    samp_counts = [4, 16, 64, 512]
    extra_pwd_lock_suffix = f"_pwd_weak_{weak_setting}"

    name_from_suffix = get_name_from_suffix(setting)

    pwd_lock_save_name_suffix = get_pwd_lock_name_suffix(strong_model, weak_model) + extra_pwd_lock_suffix
    pwd_lock_save_name = name_from_suffix(pwd_lock_save_name_suffix)

    corr_strong = corr_val_data[pwd_lock_save_name]["strong"]["all"][-1][1]
    corr_weak = corr_val_data[pwd_lock_save_name]["weak"]["all"][-1][1]

    for i, score_fn in enumerate(score_fns):
        linestyle = "solid" if i == 0 else "dotted"

        rows_and_labels = []

        samp_count_ei = 64
        mul_factor = samp_count_ei * 10

        # for weak, label in [(False, "Expert iteration"), (True, "Filtering")]:
        for weak, label in [(False, "Expert iteration")]:
            dicts = []
            for seed in range(0, 1000, 100):
                ei_save_name = name_from_suffix(
                    get_ei_name_suffix(
                        pwd_lock_name_suffix=pwd_lock_save_name_suffix,
                        samp_count=samp_count_ei,
                        seed=seed,
                        score_fn=score_fn,
                        weak=weak,
                        epoch_mul=1,
                    )
                )

                x_vals = [x[0] for x in corr_val_data[ei_save_name]["weak"]["all"]]
                corrs = [x[1] for x in corr_val_data[ei_save_name]["weak"]["all"]]

                # if first:
                #     print(corrs)

                if not x_vals:
                    continue
                if x_vals[0] != 0:
                    x_vals = [0] + x_vals
                    corrs = [corr_weak] + corrs

                x_vals = [x * mul_factor for x in x_vals]

                dicts.append(dict(weak=weak, seed=seed, x_vals=x_vals, corrs=corrs))

            if dicts:
                df = pd.DataFrame(dicts)
                df = take_more_x_vals(df)
                df_ = pd.DataFrame([sub_df_to_mean_stderr(df, info_items=dict(weak=weak))])
                row = unwrap_single_df(df_)

                rows_and_labels.append((row, label))

        for (row, label), c in zip(rows_and_labels, col_cycle):
            errorbar_plot_mean_stderr(row, label=label if i == 0 else None, ax=ax, linestyle=linestyle, c=c)

    # ax.errorbar([], [], [], c="black", linestyle="solid", label="Discrete rewards")
    # ax.errorbar([], [], [], c="black", linestyle="dotted", label="Continuous rewards")

    ax.set_xlabel("Generated samples")

    top = corr_strong
    gap = top - corr_weak
    # if first:
    #     ax.set_ylim(corr_weak - 0.1 * gap, top + 0.1 * gap)

    ax.axhline(corr_weak, color=weak_col, linestyle=weak_style, label="Locked")
    x_min, x_max = ax.get_xlim()
    ax.set_xlim(0, x_max)

    other_setting = [s for s in setting.split("-") if s != weak_setting][0]
    ax.set_title(f"strong {short_to_long[other_setting]}\n+ weak {short_to_long[weak_setting]}\n(base {strong_model})")

    # _, top = plt.ylim()
    # plt.ylim(0.0, top)

    # add ticker which corresponds to sample counts:

    if first:
        first = False
        ax.set_ylabel("Weak setting correctness")
        ax.legend(loc="lower left", bbox_to_anchor=(0, -0.5), ncol=4)
# ax.legend(loc="upper right", bbox_to_anchor=(2.2, 1))

save_figure("rl_pwd_weak")
# %%

# aggregate rl

setting = "math"

f, axs = plt.subplots(1, len(intermediate_weak_models_by_setting[setting]) + 2, figsize=(11, 2.5), sharey=True)

first = True

for ax, (inter_name, inter_model, mul_epochs) in zip(
    axs, [(None, PYTHIA1B, 0)] + intermediate_weak_models_by_setting[setting] + [(None, PYTHIA7B, 0)]
):

    epochs = int(64 * mul_epochs)

    strong_model = base_strong_model_by_setting[setting]
    weak_model = main_weak_model_by_setting[setting]

    mean_stderr_df, all_indexed_df, individual_plot_df = get_mean_stderr_info_for_basic_elicit(
        setting,
        strong_model,
        weak_models_to_try=[weak_model],
        samp_counts=SKIP_SOME_LARGE_SWEEP_SAMP_COUNTS,
    )
    name_from_suffix = get_name_from_suffix(setting)

    pwd_lock_save_name_suffix = get_pwd_lock_name_suffix(strong_model, weak_model)
    pwd_lock_save_name = name_from_suffix(pwd_lock_save_name_suffix)
    if inter_name is None:
        pwd_lock_save_name_suffix = get_pwd_lock_name_suffix(strong_model, inter_model)
        pwd_lock_save_name = name_from_suffix(pwd_lock_save_name_suffix)
        rl_pwd_lock_save_name_suffix = pwd_lock_save_name_suffix
        rl_pwd_lock_save_name = pwd_lock_save_name
    else:
        rl_pwd_lock_save_name_suffix = inter_name + "_elicit_" + pwd_lock_save_name_suffix + "_count_16_seed_0"
        rl_pwd_lock_save_name = name_from_suffix(rl_pwd_lock_save_name_suffix)

    corr_strong = corr_val_data[pwd_lock_save_name]["strong"]["all"][-1][1]
    corr_weak = corr_val_data[pwd_lock_save_name]["weak"]["all"][-1][1]
    rl_corr_weak = corr_val_data[rl_pwd_lock_save_name]["weak"]["all"][-1][1]

    for i, score_fn in enumerate(score_fns):
        linestyle = "solid" if i == 0 else "dotted"

        rows_and_labels = []

        samp_count_ei = 64
        mul_factor = samp_count_ei * 10

        for weak, label in [(False, "Expert iteration")]:
            dicts = []
            for seed in range(0, 1000, 100):
                ei_save_name = name_from_suffix(
                    get_ei_name_suffix(
                        pwd_lock_name_suffix=rl_pwd_lock_save_name_suffix,
                        samp_count=samp_count_ei,
                        seed=seed,
                        score_fn=score_fn,
                        weak=weak,
                        epoch_mul=1,
                    )
                )

                x_vals = [x[0] for x in corr_val_data[ei_save_name]["weak"]["all"]]
                corrs = [x[1] for x in corr_val_data[ei_save_name]["weak"]["all"]]

                if not x_vals:
                    continue
                if x_vals[0] != 0:
                    x_vals = [0] + x_vals
                    corrs = [rl_corr_weak] + corrs

                x_vals = [x * mul_factor for x in x_vals]

                dicts.append(dict(weak=weak, seed=seed, x_vals=x_vals, corrs=corrs))

            if dicts:
                df = pd.DataFrame(dicts)
                df = take_more_x_vals(df)
                df_ = pd.DataFrame([sub_df_to_mean_stderr(df, info_items=dict(weak=weak))])
                row = unwrap_single_df(df_)

                rows_and_labels.append((row, label))

        retain_prob = 0.3 if setting == "math" else 1.0  # because math is 3x bigger and DPO is slow...
        mul_factor = len(core_splits_default_seed[setting].train_lock_split) * 2 * retain_prob

        for kind, name in [("iterated", "Iterated DPO")]:
            rows = []
            for lr_mul in [1.0, 2.0, 4.0]:
                dicts = []
                for seed in range(0, 1000, 100):
                    dpo_save_name = (
                        name_from_suffix(
                            get_dpo_name_suffix(
                                pwd_lock_name_suffix=rl_pwd_lock_save_name_suffix,
                                seed=seed,
                                score_fn=score_fn,
                                kind=kind[0],
                                beta=0.1,
                            )
                        )
                        + (f"_lrm_{round(lr_mul)}" + "_" + f"{lr_mul-round(lr_mul):.2f}"[2:] if lr_mul != 1.0 else "")
                        + (f"_ret_{retain_prob}" if retain_prob != 1.0 else "")
                    )

                    x_vals = [x[0] for x in corr_val_data[dpo_save_name]["weak"]["all"]]
                    corrs = [x[1] for x in corr_val_data[dpo_save_name]["weak"]["all"]]

                    if not x_vals:
                        continue
                    if x_vals[0] != 0:
                        x_vals = [0] + x_vals
                        corrs = [rl_corr_weak] + corrs

                    x_vals = [x * mul_factor for x in x_vals]

                    dicts.append(dict(kind=kind, lr_mul=lr_mul, seed=seed, x_vals=x_vals, corrs=corrs))

                if dicts:
                    df = pd.DataFrame(dicts)
                    df = take_more_x_vals(df)
                    df_ = pd.DataFrame([sub_df_to_mean_stderr(df, info_items=dict(kind=kind, lr_mul=lr_mul))])
                    row = unwrap_single_df(df_)
                    rows.append(row)
            if rows:
                best_row = max(rows, key=lambda x: np.max(x["mean_corrs"]))
                rows_and_labels.append((best_row, name))

        for (row, label), c in zip(rows_and_labels, col_cycle):
            errorbar_plot_mean_stderr(row, label=label if i == 0 else None, ax=ax, linestyle=linestyle, c=c)

    # ax.errorbar([], [], [], c="black", linestyle="solid", label="Discrete rewards")
    # ax.errorbar([], [], [], c="black", linestyle="dotted", label="Continuous rewards")

    ax.set_xlabel("Generated samples")

    top = corr_strong
    gap = top - corr_weak
    if first:
        ax.set_ylim(corr_weak - 0.1 * gap, top + 0.1 * gap)

    ax.axhline(corr_strong, color=strong_col, linestyle=strong_style, label=R"$\pi^\text{locked}$ w/ pwd")
    ax.axhline(corr_weak, color=weak_col, linestyle=weak_style, label=R"$\pi^\text{locked}$ w/o pwd")
    x_min, x_max = ax.get_xlim()
    ax.set_xlim(0, x_max)

    # _, top = plt.ylim()
    # plt.ylim(0.0, top)
    fontsize = 10
    if inter_name is None:
        if inter_model == PYTHIA1B:
            ax.set_title(f"No FT\nStart from pwd-locked\nmodel directly", fontsize=fontsize)
        else:
            ax.set_title(
                f"Baseline: No FT\nStart from pwd-locked\nmodel directly, weak={model_renamings[inter_model]}",
                fontsize=fontsize,
            )
    else:
        ax.set_title(f"FT on 16 samples\nfrom {model_renamings[inter_model]}\nfor {epochs} epochs", fontsize=fontsize)

    # add ticker which corresponds to sample counts:

    if first:
        first = False
        ax.set_ylabel("Correctness")
        ax.legend(loc="lower left", bbox_to_anchor=(0, -0.4), ncol=4)
# ax.legend(loc="upper right", bbox_to_anchor=(2.2, 1))
save_figure("rl_intermediate")
# %%
