#yes
import os
import json
import numpy as np
import glob
from joblib import Parallel, delayed

def extract_entropy_from_json(file_path):
    """ Load JSON file and extract the logits_entropy value. """
    try:
        with open(file_path, 'r') as file:
            data = json.load(file)
            return data['logits_entropy']
    except (IOError, KeyError, json.JSONDecodeError):
        print(f"Error reading {file_path}. Skipping.")
        return None
    
import glob

def find_updated_results_path(base_path):
    search_pattern = os.path.join(base_path, '*updated*.json')
    files = glob.glob(search_pattern)
    if files:
        return files[0] 
    return None 


def process_obj_id(obj_path, obj_id, predictions):
    """ Process a specific obj_id directory and return the list of entropies. """
    entropies_correct = []
    entropies_incorrect = []
    if os.path.isdir(obj_path):
        json_files = glob.glob(os.path.join(obj_path, '*metrics*.json'))
        for file_path in json_files:
            entropy = extract_entropy_from_json(file_path)
            if entropy is not None:
                obj_id_directory = os.path.basename(os.path.dirname(obj_path))  
                image_directory = os.path.basename(os.path.dirname(os.path.dirname(obj_path)))
                # Construct the key
                key = f"{image_directory}/{obj_id_directory}"
                #print("key",key)
                if predictions.get(key, False):
                    entropies_correct.append(entropy)
                else:
                    entropies_incorrect.append(entropy)
    return obj_id, entropies_correct, entropies_incorrect


def load_prediction_results(results_path):
    with open(results_path, 'r') as file:
        data = json.load(file)
    predictions = {}
    for item in data:
        image_name = item['filename'].replace('.jpg', '.png') 
        for obj in item['objects']:
            key = f"{image_name}/obj_id{obj['bbox_number']}"
            predictions[key] = (obj['name'].lower() == obj['prediction'].lower())
            
    return predictions


def process_directory(base_path, predictions):
    entropy_sums = {f'obj_id{i}': {'correct': [], 'incorrect': []} for i in range(1, 6)}
    tasks = []
    for image_dir in os.listdir(base_path):
        image_path = os.path.join(base_path, image_dir)
        if os.path.isdir(image_path):
            for obj_id in entropy_sums:
                obj_path = os.path.join(image_path, obj_id, '0')
                tasks.append(delayed(process_obj_id)(obj_path, obj_id, predictions))
    all_entropies = Parallel(n_jobs=-1)(tasks)
    for obj_id, correct, incorrect in all_entropies:
        entropy_sums[obj_id]['correct'].extend(correct)
        entropy_sums[obj_id]['incorrect'].extend(incorrect)
    averages = {}
    for obj_id, data in entropy_sums.items():
        averages[obj_id] = {
            'correct': np.mean(data['correct']) if data['correct'] else float('nan'),
            'incorrect': np.mean(data['incorrect']) if data['incorrect'] else float('nan')
        }
    return averages


if __name__ == '__main__':
    base_path = ''
    results_path = find_updated_results_path(base_path)
    if results_path is None:
        print("No updated results json file found in the specified directory.")
        exit() 
    predictions = load_prediction_results(results_path)
    averages = process_directory(base_path, predictions)
    print("Average logits_entropy values for each obj_id:")
    for obj_id, avg in averages.items():
        print(f"{obj_id} - Correct: {avg['correct']:.4f}, Incorrect: {avg['incorrect']:.4f}")
