#include <fstream>
#include <limits>
#include <phat/boundary_matrix.h>

#include <scc/Scc.h>

#include <unordered_map>
#include <unordered_set>

typedef scc::Scc<> Scc;

typedef std::vector<std::vector<long>> Induced_matrix;

struct Generator {

  double gr[2];
  int pos_in_scc;
  int index;

  Generator(double d1, double d2,int pos) {
    gr[0]=d1;
    gr[1]=d2;
    pos_in_scc=pos;
    // index is set during algorithm
  }
};

struct Relation {

  double gr[2];
  std::vector<long> bd;

  Relation(double d1, double d2,std::vector<int>& bdy) {
    gr[0]=d1;
    gr[1]=d2;
    std::copy(bdy.begin(),bdy.end(),std::back_inserter(bd));
  }
};

struct Sort_by_secondary {
  
  int secondary_parameter;
  
  Sort_by_secondary(int primary_parameter) : secondary_parameter(1-primary_parameter) {}

  template<typename T>
  bool operator() (T& g1, T& g2) {
    return g1.gr[secondary_parameter] < g2.gr[secondary_parameter];
  }
};

template<typename Generator_output_iterator, typename Relation_output_iterator>
void compute_generators_and_relations_sorted_by_secondary_parameter(Scc& parser,
								    int level,
								    int primary_parameter,
								    Generator_output_iterator out_gen,
								    Relation_output_iterator out_rel,
								    double secondary_threshold) {
  std::vector<double> gr;
  std::vector<std::pair<int,int>> coeff;
  
  // First the generators:

  std::vector<Generator> gens;
  
  int count=0;
  while(parser.has_next_column(level+1)) {
    gr.clear();
    coeff.clear();
    parser.next_column(level+1,std::back_inserter(gr),std::back_inserter(coeff));
    if(gr[1-primary_parameter]<=secondary_threshold) {
      gens.push_back(Generator(gr[0],gr[1],count++));
    }
  }
  std::sort(gens.begin(),gens.end(),Sort_by_secondary(primary_parameter));
  
  std::vector<int> gens_reindexing;
  gens_reindexing.resize(gens.size());
  
  for(int i=0;i<gens.size();i++) {
    Generator& gen = gens[i];
    gen.index=i;
    gens_reindexing[gen.pos_in_scc]=i;
  }

  std::copy(gens.begin(),gens.end(),out_gen);

  // Now the relations:

  std::vector<Relation> rels;

  gr.clear();
  coeff.clear();
  
  while(parser.has_next_column(level)) {
    gr.clear();
    coeff.clear();
    parser.next_column(level,std::back_inserter(gr),std::back_inserter(coeff));
    if(gr[1-primary_parameter]<=secondary_threshold) {
      std::vector<int> coeff_reindexed;
      for(std::pair<int,int> c : coeff) {
	coeff_reindexed.push_back(gens_reindexing[c.first]);
      }
      std::sort(coeff_reindexed.begin(),coeff_reindexed.end());
      rels.push_back(Relation(gr[0],gr[1],coeff_reindexed));
    }
  }
  std::sort(rels.begin(),rels.end(),Sort_by_secondary(primary_parameter));
  
  std::copy(rels.begin(),rels.end(),out_rel);
			     
}


class Vertex_relevance_checker {

protected:
  std::vector<Generator>& gens;
  std::vector<Relation>& rels;
  int primary_parameter;
  double relevance_threshold;
  bool include_infinite_bars;

public:
  Vertex_relevance_checker(std::vector<Generator>& gens, std::vector<Relation>& rels,int primary_parameter, double relevance_threshold, bool include_infinite_bars) : gens(gens), rels(rels),primary_parameter(primary_parameter), relevance_threshold(relevance_threshold), include_infinite_bars(include_infinite_bars) {}

  bool operator() (long gen_idx, long rel_idx) {
    //std::cout << "Checking relevance for " << gen_idx << " " << rel_idx << std::endl;
    if(gen_idx==-1) {
      return false;
    }
    if(rel_idx >= rels.size()) {
      // This is encoding an infinite bar
      return include_infinite_bars;
    }
    double birth=gens[gen_idx].gr[1-primary_parameter];
    double death=rels[rel_idx].gr[1-primary_parameter];
    
    //std::cout << "Values " << birth << " " << death << " " << relevance_threshold << std::endl;
    return death-birth>0 && death-birth>=relevance_threshold;
  }

};

template< typename Matrix>
class Dynamic_boundary_matrix {

  typedef long index;

  typedef std::priority_queue<index,std::vector<index>,std::greater<index>> PQ;

protected:
  Matrix mat;

  index num_rels; // number of relations 

  PQ columns_to_consider;

  std::unordered_set<index> new_unpaired_generators;

  std::unordered_map< index,index > pivots;

  index relevant_index;

  Vertex_relevance_checker& vertex_relevance_check;

  std::vector<Generator>& gens;
  std::vector<Relation>& rels;

  int primary_parameter;

public:

  std::vector<std::pair<std::pair<double,double>, std::pair<int,double>> > vertices;
  std::vector<std::pair<index,index>> edges;

  bool filter_out_disjoint_pairs;
  bool include_infinite_bars;

  bool do_exhaustive_reduction;
  bool do_random_shuffles;
  
  int number_of_filtered_disjoint_pairs;

  Dynamic_boundary_matrix(std::vector<Generator>& gens,
			  std::vector<Relation>& rels, 
			  Vertex_relevance_checker& vertex_relevance_check, 
			  int primary_parameter, 
			  bool filter_out_disjoint_pairs,
			  bool include_infinite_bars,
			  bool do_exhaustive_reduction,
			  int do_random_shuffles=0) : vertex_relevance_check(vertex_relevance_check), gens(gens), rels(rels), primary_parameter(primary_parameter), filter_out_disjoint_pairs(filter_out_disjoint_pairs), include_infinite_bars(include_infinite_bars), do_exhaustive_reduction(do_exhaustive_reduction), do_random_shuffles(do_random_shuffles), number_of_filtered_disjoint_pairs(0) {
    num_rels = rels.size();
    mat.set_num_cols(num_rels+gens.size()); // We need up to m "articifical columns" for the bars going to infinity
    relevant_index=0;
  }

  void new_batch(std::vector<std::pair<int,Generator>>& gen_batch,
		 std::vector<std::pair<int,Relation>>& rel_batch,
		 int slice_index,
		 double slice_value) {

    new_unpaired_generators.clear();

    for(auto& gen_pair : gen_batch) {
      int idx = gen_pair.first;
      new_unpaired_generators.insert(idx);
      mat.set_dim(num_rels+idx,-1);
    }

    for(auto& rel_pair : rel_batch) {
      int idx = rel_pair.first;
      Relation& rel = rel_pair.second;
      if(! mat.is_empty(idx)) {
	std::cerr << "Newly inserted column " << idx << " was not zero!" << std::endl;
	std::exit(1);
      }
      mat.set_col(idx,rel.bd);
      mat.set_dim(idx,-1);
      if(! mat.is_empty(idx)) {
	columns_to_consider.push(idx);
      }
      
    }

    //std::cout << "Starting reducing non-zero columns: " << columns_to_consider.size() << std::endl;
    PQ new_columns_to_consider;
    while(!columns_to_consider.empty()) {
      index idx = columns_to_consider.top();
      // There might have been multiple instances of the same index pushed into the queue...take them all out
      while(!columns_to_consider.empty() && idx==columns_to_consider.top()) {
	columns_to_consider.pop();
      }
      /*
      if(idx >= num_rels) {
	std::cout << " NOW idx=" << idx << std::endl;
	std::cout << "There are currently " << new_unpaired_generators.size() << " unpaired gens" << std::endl;
      }
      */
      int piv = mat.get_max_index(idx);
      /*
      if(idx>=num_rels) {
	std::cout << "reducing " << idx << " , pivot is " << piv << std::endl;
      }
      */
      std::vector<int> added_columns;
      bool old_column_was_relevant = (!mat.is_empty(idx) && mat.get_dim(idx)>=0);
      
      double old_birth=old_column_was_relevant ? gens[piv].gr[1-primary_parameter] : 0.0;
      double old_death=old_column_was_relevant ? rels[idx].gr[1-primary_parameter] : 0.0;

      while(piv!=-1 && pivots.count(piv) && pivots[piv]<idx) {
	//std::cout << "Pivot is " << piv << " -- at column " << pivots[piv] << std::endl;
	/*
	if(idx>=num_rels) {
	  std::cout << "Adding column " << pivots[piv] << " to it " << std::endl;
	}
	*/
	mat.add_to(pivots[piv],idx);
	
	if(old_column_was_relevant) {
	  added_columns.push_back(pivots[piv]);
	}
	piv = mat.get_max_index(idx);
	/*
	if(idx>=num_rels) {
	  std::cout << "New pivot is " << piv << std::endl;
	}
	*/
      }
      if(do_exhaustive_reduction) {
	int old_piv=piv;
	std::vector<index> final_col;
	while(! mat.is_empty(idx)) {
	  piv=mat.get_max_index(idx);
	  while(piv!=-1 && (!pivots.count(piv) || pivots[piv]>=idx)) {
	    final_col.push_back(piv);
	    mat.remove_max(idx);
	    piv=mat.get_max_index(idx);
	  }
	  if(piv!=-1 && pivots.count(piv) && pivots[piv]<idx) {
	    mat.add_to(pivots[piv],idx);
	    if(old_column_was_relevant) {
	      added_columns.push_back(pivots[piv]);
	    }
	  }
	    
	}
	std::reverse(final_col.begin(),final_col.end());
	mat.set_col(idx,final_col);
	piv = mat.get_max_index(idx);
	assert(old_piv==piv);
      }
      
      /*
      if(idx>=num_rels) {
	std::cout << "Done, new pivot is " << piv << ", unpaired pivot: " << new_unpaired_generators.count(piv) << std::endl;
      }
      */

      if(do_random_shuffles) {
	index old_piv=piv;
	for(auto p : pivots) {
	  index saved_piv = p.first;
	  index saved_idx = p.second;
	  if(saved_piv<piv && saved_idx < idx) {
	    if(rand()%20==0) { // coin flip
	      mat.add_to(saved_idx,idx);
	      if(old_column_was_relevant) {
		added_columns.push_back(saved_idx);
	      }
	    }
	  }
	}
	assert(old_piv==piv);
      }

      //std::sort(added_columns.begin(),added_columns.end());
      if(!mat.is_empty(idx)) {
	bool has_pivot_at_piv = pivots.count(piv);
	if(has_pivot_at_piv && pivots[piv]>idx) {
	  columns_to_consider.push(pivots[piv]);
	}
	if(!has_pivot_at_piv || pivots[piv]>idx) {
	  pivots[piv]=idx;
	}
	if(pivots.count(piv)) {
	  new_unpaired_generators.erase(piv);
	}
      }

      bool new_column_is_relevant=vertex_relevance_check(piv,idx);

     

      if(old_column_was_relevant) {
	index old_idx = mat.get_dim(idx);
	for(index added : added_columns) {
	  // The dim-field in the matrix is abused to contain the vertex id of relevant bars
	  if(mat.get_dim(added)>=0) {
	    index other_idx=added;
	    index other_piv=mat.get_max_index(added);
	    double other_birth=gens[other_piv].gr[1-primary_parameter];
	    double other_death=rels[other_idx].gr[1-primary_parameter];

	    if(!filter_out_disjoint_pairs || (other_birth <= old_death && old_birth<=other_death)) {
	      index other_idx = mat.get_dim(added);
	      edges.push_back(std::make_pair(old_idx,other_idx));
	      edges.push_back(std::make_pair(other_idx,old_idx));
	    } else {
	      number_of_filtered_disjoint_pairs++;
	    }
	  }
	}
	if(new_column_is_relevant) {
	  // Old and new interval always overlap, no need to check for disjointness
	  edges.push_back(std::make_pair(old_idx,relevant_index));
	  edges.push_back(std::make_pair(relevant_index,old_idx));
	}
      }
	
      if(new_column_is_relevant) {
	double new_birth=gens[piv].gr[1-primary_parameter];
	double new_death=(idx<num_rels) ? rels[idx].gr[1-primary_parameter] : std::numeric_limits<double>::max();
	vertices.push_back(std::make_pair(std::make_pair(new_birth,new_death),std::make_pair(slice_index,slice_value)));
	mat.set_dim(idx,relevant_index);
	relevant_index++;
	new_columns_to_consider.push(idx);
      } else {
	mat.set_dim(idx,-1);
      }
	  
    }
    //std::cout << "At end of iteration, there are " << new_unpaired_generators.size() << " unpaired generators" << std::endl;
    if(include_infinite_bars) {
      for(index idx : new_unpaired_generators) {
	//std::cout << "NOW ON UNPAIRED " << idx << std::endl;
	if(mat.is_empty(num_rels+idx)) { //new infinite bar
	  double new_birth=gens[idx].gr[1-primary_parameter];
	  double new_death=std::numeric_limits<double>::max();
	  vertices.push_back(std::make_pair(std::make_pair(new_birth,new_death),std::make_pair(slice_index,slice_value)));
	  std::vector<index> new_col;
	  new_col.push_back(idx);
	  mat.set_col(num_rels+idx,new_col);
	  mat.set_dim(num_rels+idx,relevant_index);
	  relevant_index++;
	  new_columns_to_consider.push(num_rels+idx);
	}
      }
    }

    //std::cout << "Piv size: " << pivots.size() << std::endl;
    columns_to_consider.swap(new_columns_to_consider);
  }
  
};


void _determine_slice_values(Scc& parser,
			     int level,
			     int slices,
			     int primary_parameter,
			     std::set<double>& grades,
			     double& min_gr,
			     double& max_gr) {
  std::vector<double> gr;
  std::vector<std::pair<int,int>> coeff;
  
  while(parser.has_next_column(level)) {
    gr.clear();
    coeff.clear();
    parser.next_column(level,std::back_inserter(gr),std::back_inserter(coeff));
    double new_grade = gr[primary_parameter];
    if(slices==0) {
      grades.insert(new_grade);	
    } else {
      min_gr=std::min(min_gr,new_grade);
      max_gr=std::max(max_gr,new_grade);
    }
  }
  parser.reset(level);
}

template <typename OutputIterator> void determine_slice_values(Scc& parser,
							       int level,
							       int slices,
							       int primary_parameter,
							       OutputIterator out) {
  // Needed only for slices==0
  std::set<double> grades;
  // Needed only for slices>0
  double min_gr=std::numeric_limits<double>::max();
  double max_gr=std::numeric_limits<double>::min();

  _determine_slice_values(parser,level,slices,primary_parameter,grades,min_gr,max_gr);
  _determine_slice_values(parser,level+1,slices,primary_parameter,grades,min_gr,max_gr);

  if(slices==0) {
    std::copy(grades.begin(),grades.end(),out);
  } else {
    assert(max_gr>=min_gr);
    double delta = max_gr-min_gr;
    
    for(int i=0;i<slices;i++) {
      // Guard the case slices==1 (even though it is a useless case)
      if(i==0) {
	*out++=min_gr;
      }	else {
	*out++ = min_gr+(double)i/(slices-1)*delta;
      }
    }
    
  }

  parser.reset(level);
  
}

void print_help_message(char* arg0) {

  std::cout << "Usage: " << arg0 << " [OPTIONS] <INPUT_FILE> <OUTPUT_FILE>\n\n";

  std::cout << "Computes the graphcode for a chain complex in scc2020 format. \n\n";

  std::cout << "Options:\n";
  std::cout << "--primary-parameter d       - Should be 0 or 1 (default is 0)\n";
  std::cout << "--slices n                  - Computes barcodes at n slices, which are taken equidistantly in the range scale of the primary parameter. 0 means that all slices are computed. Default is n=10\n";
  std::cout << "--slice-file filename       - Computes barcodes at slices whose value is determined by the file filename. If used, parameter \"--slices\" is ignored.\n";
  std::cout << "--include-infinite-bars     - takes infinite bars into account in the output\n";
  std::cout << "--keep-disjoint-pairs       - does not filter out edges between bars with disjoint lifetimes\n";
  std::cout << "--do-exhaustive-reduction   - do matrix reduction exhaustively\n";
  std::cout << "--do-random-shuffles         - perform additional column operations on the reduced matrix. The outcome depends on randomization\n";
  std::cout << "--relevance-threshold v     - only include bars of persistence at least v in the output. v must be non-negative decimal\n";
  std::cout << "--secondary-threshold v     - only compute the barcodes at each slice values up to this threshold\n";
  std::cout << "--level i                   - Computes the graphcode on level i of the input chain complex (default is 1)\n";
  std::cout << "-h --help                   - prints this message\n\n";
}

bool is_non_negative_integer(std::string arg) {
  
  for(int i=0;i<arg.length();i++) {
    if(! std::isdigit(arg[i])) {
      return false;
    }
  }
  return true;
}

bool is_non_negative_float(std::string arg) {
  
  for(int i=0;i<arg.length();i++) {
    if(! (std::isdigit(arg[i]) || arg[i]=='.')) {
      return false;
    }
  }
  return true;
}



template<typename OutputIterator>
void slice_values_from_file(std::string filename,OutputIterator ofstr) {
  std::ifstream ifstr(filename.c_str());
  double next;
  ifstr >> next;
  while(ifstr.good()) {
    *ofstr++=next;
    ifstr>>next;
  }
}


int main(int argc, char** argv) {

  int level=1;

  int slices=10;


  // This is the parameter that determines the slice values
  // That is, the other parameter is the filtration value along which
  // the barcodes are computed
  int primary_parameter = 1;

  bool include_infinite_bars=false;  

  // -1 stands for not set
  double relevance_threshold=-1;

  double secondary_threshold = std::numeric_limits<double>::max();

  bool inputfile_read=false;
  bool outputfile_read=false;

  std::string infile,outfile;

  bool print_help=false;

  bool read_slice_values_from_file=false;
  std::string slice_value_file;

  bool filter_out_disjoint_pairs=true;
  
  int pos=1;
  
  bool do_exhaustive_reduction=false;

  int do_random_shuffles=false;
  
  while(pos<argc) {

    std::string arg(argv[pos]);
    
    if(arg=="-h" || arg=="--help") {
      print_help=true;
      break;
    } else if(arg=="--primary-parameter") {
      if(pos+1<argc && is_non_negative_integer(std::string(argv[pos+1]))) {
	primary_parameter=atoi(argv[++pos]);
      } else {
	std::cout << "--primary-parameter requires extra integer argument" << std::endl;
	print_help=true;
	break;
      }
    } else if(arg=="--level") {
      if(pos+1<argc && is_non_negative_integer(std::string(argv[pos+1]))) {
	level=atoi(argv[++pos]);
      } else {
	std::cout << "--level requires extra integer argument" << std::endl;
	print_help=true;
	break;
      }
    } else if(arg=="--slices") {
      if(pos+1<argc && is_non_negative_integer(std::string(argv[pos+1]))) {
	slices=atoi(argv[++pos]);
      } else {
	std::cout << "--primary-parameter requires extra integer argument" << std::endl;
	print_help=true;
	break;
      }
    } else if(arg=="--slice-file") {
      if(pos+1<argc) {
	read_slice_values_from_file=true;
	slice_value_file=std::string(argv[pos+1]);
	pos++;
      } else {
	std::cout << "--slice-file requires extra argument" << std::endl;
	print_help=true;
	break;
      }
    } else if(arg=="--include-infinite-bars") {
      include_infinite_bars=true;
    } else if(arg=="--keep-disjoint-pairs") {
      filter_out_disjoint_pairs=false;
    } else if(arg=="--secondary-threshold") {
      if(pos+1<argc && is_non_negative_float(std::string(argv[pos+1]))) {
	secondary_threshold=atof(argv[++pos]);
      } else {
	std::cout << "--secondary-threshold requires extra decimal argument" << std::endl;
	print_help=true;
	break;
      }
    } else if(arg=="--relevance-threshold") {
      if(pos+1<argc && is_non_negative_float(std::string(argv[pos+1]))) {
	relevance_threshold=atof(argv[++pos]);
      } else {
	std::cout << "--relevance-threshold requires extra decimal argument" << std::endl;
	print_help=true;
	break;
      }
    } else if(arg=="--do-exhaustive-reduction") {
      do_exhaustive_reduction=true;
    } else if(arg=="--do-random-shuffles") {
      srand(time(NULL));
      do_random_shuffles=true;
    } else {
      if(arg[0]=='-') {
	std::cout << "Unrecognized option: " << arg << std::endl;
	print_help=true;
	break;
      }
      if(! inputfile_read) {
	infile=arg;
	inputfile_read=true;
      } else if(! outputfile_read) {
	outfile=arg;
	outputfile_read=true;
      } else {
	std::cerr << "Ignoring argument " << arg << std::endl;
      }
    }
    pos++;
  }

  if(!print_help && !inputfile_read) {
    std::cout << "Input file missing" << std::endl;
    print_help=true;
  }

  if(!print_help && !outputfile_read) {
   std::cout << "Output file missing" << std::endl;
    print_help=true;
  } 


  if(print_help) {
    print_help_message(argv[0]);
    std::exit(0);
  }




  std::ifstream istr(infile);

  Scc parser(istr);

  istr.close();

  std::cout << "Number of parameters: " << parser.number_of_parameters() << std::endl;

  int levels = parser.number_of_levels();
  std::cout << "Number of levels: " << levels << std::endl;

  /*
  for(int i=1;i<=levels;i++) {
    std::cout << "Level " << i << ": " << parser.number_of_generators(i) << " generators" << std::endl;
  }
  */

  std::cout << "Relevance threshold: " << relevance_threshold << std::endl;

  std::vector<double> slice_values;

  if(read_slice_values_from_file) {
    slice_values_from_file(slice_value_file,std::back_inserter(slice_values));
    slices = slice_values.size();
  } else {
    determine_slice_values(parser, level, slices, primary_parameter, std::back_inserter(slice_values));
  }


  
  std::cout << "Considering level  " << level << std::endl;
  std::cout << "Number of slices:  " << slices << std::endl;
  std::cout << "Primary parameter: " << primary_parameter << std::endl;

  
  std::cout << "Number of slice values: " << slice_values.size() << std::endl;

  /*
  for(double gr : slice_values) {
    std::cout << "Grade: " << gr << std::endl;
  }
  */

  std::vector<Generator> gens;
  std::vector<Relation> rels;
  
  compute_generators_and_relations_sorted_by_secondary_parameter(parser,
								 level,
								 primary_parameter,
								 std::back_inserter(gens),
								 std::back_inserter(rels),
								 secondary_threshold);

  std::cout << "We have " << gens.size() << " generators and " << rels.size() << " relations" << std::endl;
  
  // Now re-organize the relations into buckets that are added one by one (one per slice-value)

  std::vector<std::vector<std::pair<int,Generator>>> gen_buckets;
  std::vector<std::vector<std::pair<int,Relation>>> rel_buckets;

  for(int i=0;i<slice_values.size();i++) {
    gen_buckets.push_back(std::vector<std::pair<int,Generator>>());
    rel_buckets.push_back(std::vector<std::pair<int,Relation>>());
  }

  // Map needed to avoid linear scan
  std::unordered_map<double,int> slice_value_to_index;
  for(int i=0;i<slice_values.size();i++) {
    slice_value_to_index[slice_values[i]]=i;
  }

  
  for(int i=0;i<gens.size();i++) {
    
    auto bucket_it = std::lower_bound(slice_values.begin(),slice_values.end(),gens[i].gr[primary_parameter]);
    if(bucket_it==slice_values.end()) {
      std::cerr << "Warning, relation was found that is added later as the last slice value" << std::endl;
      continue;
    }
    int bucket_idx = slice_value_to_index[*bucket_it];
    gen_buckets[bucket_idx].push_back(std::make_pair(i,gens[i]));
  }


  for(int i=0;i<rels.size();i++) {
    
    auto bucket_it = std::lower_bound(slice_values.begin(),slice_values.end(),rels[i].gr[primary_parameter]);
    if(bucket_it==slice_values.end()) {
      std::cerr << "Warning, relation was found that is added later as the last slice value" << std::endl;
      continue;
    }
    int bucket_idx = slice_value_to_index[*bucket_it];
    rel_buckets[bucket_idx].push_back(std::make_pair(i,rels[i]));
  }
   

  for(int i=0;i<slice_values.size();i++) {
    std::cout << "On " <<i << "th slice values, there are " << gen_buckets[i].size() << " generators and " << rel_buckets[i].size() << " relations to add" << std::endl;
  }

  Vertex_relevance_checker vertex_relevance_check(gens,rels,primary_parameter,relevance_threshold,include_infinite_bars);

  typedef phat::boundary_matrix<phat::vector_vector> Phat_boundary_matrix;

  Dynamic_boundary_matrix<Phat_boundary_matrix> dmat(gens,rels,vertex_relevance_check,primary_parameter, filter_out_disjoint_pairs,include_infinite_bars, do_exhaustive_reduction, do_random_shuffles);

  for(int i=0;i<slice_values.size();i++) {
    std::cout << "New batch i=" << i << std::endl;
    dmat.new_batch(gen_buckets[i],rel_buckets[i],i,slice_values[i]);
  }
  if(filter_out_disjoint_pairs) {
    std::cout << "Number of filtered-out edges between disjoint bars: " << dmat.number_of_filtered_disjoint_pairs << std::endl;
  }

  std::cout << "Generated " << dmat.vertices.size() << " vertices and " << dmat.edges.size()/2 << " edges." << std::endl;

  std::ofstream ofstr(outfile);
  
  ofstr.precision(std::numeric_limits<double>::max_digits10);
  
  ofstr << dmat.vertices.size() << std::endl;
  for(auto& vert : dmat.vertices) {
    ofstr << vert.first.first << " " << vert.first.second << " " << vert.second.first << " " << vert.second.second << std::endl;
  }
  for(auto& edge : dmat.edges) {
    ofstr << edge.first << " " << edge.second << std::endl;
  }

  ofstr.close();




  return 0;

}
