import os
from typing import List
import wandb
from dotenv import load_dotenv


def list_wandb_projects(api_key: str = "", entity: str = "") -> List[str]:
    """
    List all projects under a given entity in Weights & Biases.

    This function loads the API key and entity name from environment variables if they are not provided.

    Args:
        api_key (str): The API key for authenticating with Weights & Biases. If empty, the function attempts to load it from the WANDB_KEY environment variable.
        entity (str): The entity name whose projects are to be listed. If empty, the function attempts to load it from the WANDB_ENTITY environment variable.

    Returns:
        List[str]: A list of project names under the specified entity.
    """
    load_dotenv()  # Load environment variables from .env file

    # Use provided API key and entity, or load from environment variables
    api_key = api_key or os.getenv("WANDB_KEY")
    entity = entity or os.getenv("WANDB_ENTITY")

    if not api_key:
        raise ValueError(
            "WANDB_KEY is required. Provide as an argument or set in the .env file."
        )
    if not entity:
        raise ValueError(
            "WANDB_ENTITY is required. Provide as an argument or set in the .env file."
        )

    # Initialize the W&B API client
    wandb.login(key=api_key)
    api = wandb.Api()

    # Fetch all projects for the given entity
    projects = api.projects(entity=entity)

    # Return the list of project names
    return [project.name for project in projects]


# Example usage
if __name__ == "__main__":
    project_names = list_wandb_projects()
    for name in project_names:
        print(name)
