import logging

import torch
import torch.nn.functional as F

pylogger = logging.getLogger(__name__)


def train(args, model, device, train_loader, optimizer, epoch, softmax=False):
    model.train()
    correct = 0
    for batch_idx, (data, target) in enumerate(train_loader):
        data, target = data.to(device), target.to(device)
        optimizer.zero_grad()
        output = model(data)
        if softmax:
            output = F.log_softmax(output, dim=1)
        loss = F.nll_loss(output, target)
        pred = output.argmax(dim=1, keepdim=True)  # get the index of the max log-probability
        correct += pred.eq(target.view_as(pred)).sum().item()
        loss.backward()
        optimizer.step()

        if batch_idx % args.log_interval == 0:
            acc = 100.0 * correct / len(train_loader.dataset)
            pylogger.info(
                "Train Epoch: {} [{}/{} ({:.0f}%)]\tLoss: {:.6f}".format(
                    epoch,
                    batch_idx * len(data),
                    len(train_loader.dataset),
                    100.0 * batch_idx / len(train_loader),
                    loss.item(),
                )
            )
    acc = 100.0 * correct / len(train_loader.dataset)
    pylogger.info("Train Accuracy: ({:.0f}%) ".format(acc))


def test(model, device, test_loader, softmax=False):
    model.eval()
    test_loss = 0
    correct = 0

    with torch.no_grad():
        for data, target in test_loader:
            data, target = data.to(device), target.to(device)
            output = model(data)

            if softmax:
                output = F.log_softmax(output, dim=1)

            test_loss += F.nll_loss(output, target, reduction="sum").item()  # sum up batch loss
            pred = output.argmax(dim=1, keepdim=True)  # get the index of the max log-probability
            correct += pred.eq(target.view_as(pred)).sum().item()

    test_loss /= len(test_loader.dataset)
    acc = 100.0 * correct / len(test_loader.dataset)
    pylogger.info("\nAverage loss: {:.4f}, Accuracy: ({:.0f}%)\n".format(test_loss, acc))

    return test_loss, acc
