# Copyright 2022 DeepMind Technologies Limited.
# Copyright 2023 Bytedance Ltd. and/or its affiliate
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
# https://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""BH76 benchmark set"""

from lapnet import base_config
from lapnet.utils import system


systems = {
    'h_hf': [
        ['H', (20.0, 0, 0)],
        ['F', (0.0, 0.0, 0.864911)],
        ['H', (0.0, 0.0, -0.864911)],
    ],
    'hfhts': [
        ['H', (0.0, 0.0, 2.14903)],
        ['F', (0.0, 0.0, 0.0)],
        ['H', (0.0, 0.0, -2.14903)],
    ],
    'h_hcl': [
        ['H', (20.0, 0, 0)],
        ['Cl', (0.0, 0.0, 1.204179)],
        ['H', (0.0, 0.0, -1.204179)],
    ],
    'hclhts': [
        ['H', (0.0, 0.0, 2.80776)],
        ['Cl', (0.0, 0.0, 0.0)],
        ['H', (0.0, 0.0, -2.80776)],
    ],
    'h_ch3f': [
        ['H', (20.0, 0, 0)],
        ['C', (0.0, 0.0, 0.12390246548611)],
        ['F', (0.0, 0.0, -2.4861775343035)],
        ['H', (0.9730507764948, -1.68537338323335, 0.78742502293913)],
        ['H', (0.9730507764948, 1.68537338323335, 0.78742502293913)],
        ['H', (-1.94610155298961, 0.0, 0.78742502293913)],
    ],
    'hfch3ts': [
        ['H', (0.00070836096667, 4.275445, 0.0)],
        ['F', (-0.00020815643333, 1.935045, 0.0)],
        ['C', (-0.00020815643333, -1.218245, 0.0)],
        ['H', (1.99117798396667, -1.663095, 0.0)],
        ['H', (-0.99573501603333, -1.664575, 1.7241)],
        ['H', (-0.99573501603333, -1.664575, -1.7241)],
    ],
    'hf_ch3': [
        ['F', (20.0, 0.0, 0.864911)],
        ['H', (20.0, 0.0, -0.864911)],
        ['C', (0.0, 0.0, 0.0)],
        ['H', (1.01791624999999, 1.76308266284997, 0.0)],
        ['H', (1.01791624999999, -1.76308266284997, 0.0)],
        ['H', (-2.03583249999998, 0.0, 0.0)],
    ],
    'h_f2': [
        ['H', (20.0, 0, 0)],
        ['F', (0.0, 0.0, 1.31828)],
        ['F', (0.0, 0.0, -1.31828)],
    ],
    'f2': [
        #['H', (20.0, 0, 0)],
        ['F', (0.0, 0.0, 1.31828)],
        ['F', (0.0, 0.0, -1.31828)],
    ],
    'hf2ts': [
        ['H', (0.0, 0.0, -2.96716666666667)],
        ['F', (0.0, 0.0, 0.08485333333333)],
        ['F', (0.0, 0.0, 2.88231333333333)],
    ],
    'hf_f': [
        ['F', (20.0, 0.0, 0.864911)],
        ['H', (20.0, 0.0, -0.864911)],
        ['F', (0.0, 0.0, 0.0)],
    ],
    'oh_ch3f': [
        ['O', (20.0, 0.0, 0.909)],
        ['H', (20.0, 0.0, -0.909)],
        ['C', (0.0, 0.0, 0.12390246548611)],
        ['F', (0.0, 0.0, -2.4861775343035)],
        ['H', (0.9730507764948, -1.68537338323335, 0.78742502293913)],
        ['H', (0.9730507764948, 1.68537338323335, 0.78742502293913)],
        ['H', (-1.94610155298961, 0.0, 0.78742502293913)],
    ],
    'ch3oh_f': [
        ['C', (-0.26318086666667, 0.63253, 0.0)],
        ['O', (-0.26318086666667, -2.04735, 0.0)],
        ['H', (-2.22950406666667, 1.22377, 0.0)],
        ['H', (1.45082593333333, -2.61801, 0.0)],
        ['H', (0.65251993333333, 1.40453, 1.68099)],
        ['H', (0.65251993333333, 1.40453, -1.68099)],
        ['F', (20.0, 0.0, 0.0)],
    ],
    'hoch3fcomp1': [
        ['C', (-1.32939657142857, -0.29755242857143, -3.035084286e-05)],
        ['O', (0.22164242857143, 1.81429057142857, 0.00013594485714)],
        ['H', (-3.32102657142857, 0.28691357142857, -0.01629150014286)],
        ['H', (-1.03243657142857, -1.51888942857143, -1.65933610014286)],
        ['H', (-1.05572657142857, -1.50318942857143, 1.67501389985714)],
        ['H', (2.08832042857143, 1.13992157142857, 0.00049310385714)],
        ['F', (4.42862342857143, 0.07850557142857, 1.500255714e-05)],
    ],
    'hoch3fcomp2': [
        ['F', (4.25100082857143, -0.30544528571429, -0.01712943285714)],
        ['C', (1.58716782857143, 0.03837371428571, 0.00587236714286)],
        ['H', (0.66477502857143, -1.64660228571429, -0.68940133285714)],
        ['H', (0.98723582857143, 0.40285271428571, 1.92673266714286)],
        ['H', (1.13425882857143, 1.62949771428571, -1.19911733285714)],
        ['O', (-3.43734917142857, -0.29918228571429, -0.01130150285714)],
        ['H', (-5.18708917142857, 0.18050571428571, -0.01565543285714)],
    ],
    'hoch3fts': [
        ['F', (4.03967224285714, 0.15606831428571, -0.00037685118571)],
        ['C', (0.71420724285714, 0.20097761428571, 0.00037337081429)],
        ['H', (0.61981534285714, 2.22102301428571, -0.02233736618571)],
        ['H', (0.61274014285714, -0.81682698571429, -1.74424496618571)],
        ['H', (0.61335054285714, -0.77799298571429, 1.76700503381429)],
        ['O', (-3.03436775714286, 0.37611801428571, -0.00035795418571)],
        ['H', (-3.56541775714286, -1.35936698571429, -6.126688571e-05)],
    ],
    'ch3_clf': [
        ['C', (0.0, 0.0, -20.0)],
        ['H', (1.01791624999999, 1.76308266284997, -20.0)],
        ['H', (1.01791624999999, -1.76308266284997, -20.0)],
        ['H', (-2.03583249999998, 0.0, -20.0)],
        ['F', (0.0, 0.0, -1.54044)],
        ['Cl', (0.0, 0.0, 1.54044)]
    ],
    'ch3f_cl': [
        ['C', (0.0, 0.0, 0.12390246548611)],
        ['F', (0.0, 0.0, -2.4861775343035)],
        ['H', (0.9730507764948, -1.68537338323335, 0.78742502293913)],
        ['H', (0.9730507764948, 1.68537338323335, 0.78742502293913)],
        ['H', (-1.94610155298961, 0.0, 0.78742502293913)],
        ['Cl', (20, 0, 0)],
    ],
    'ch3fclts': [
        ['Cl', (5.50499183333333, -0.00028467333333, -0.00007838691667)],
        ['F', (2.14442083333333, 0.01080424666667, 0.00023152818333)],
        ['C', (-1.75565816666667, -0.00200432333333, -0.00014074781667)],
        ['H', (-1.95911816666667, -1.61434708333333, -1.22720279781667)],
        ['H', (-1.96332816666667, -0.26000108333333, 2.00903720218333)],
        ['H', (-1.97130816666667, 1.86583291666667, -0.78184679781667)],
    ],
    'hcn': [
        ['C', (0.0, 0.0, -0.054852)],
        ['N', (0.0, 0.0, 2.124011)],
        ['H', (0.0, 0.0, -2.069159)]
    ],
    'hnc': [
        ['C', (0.0, 0.0, -2.09982933333333)],
        ['N', (0.0, 0.0, 0.10989866666667)],
        ['H', (0.0, 0.0, 1.98993066666667)]
    ],
    'hcnts': [
        ['C', (0.70831033333333, 0.97855700000000, 0.00000000000000)],
        ['N', (0.70831033333333, -1.26710300000000, 0.00000000000000)],
        ['H', (-1.41662066666667, 0.28854600000000, 0.00000000000000)]
    ],
    'c5h8': [
        ['C', (-3.69558992307692, -1.01958446153846, 0.00001521455385)],
        ['C', (-2.13743992307692, 1.34771553846154, 0.00009458305385)],
        ['C', (0.38861907692308, 1.52502553846154, 0.00005111935385)],
        ['C', (2.18749007692308, -0.57008146153846, -0.00008116154615)],
        ['C', (4.69332007692308, -0.23848946153846, -0.00006604374615)],
        ['H', (-4.92286992307692, -1.07499446153846, 1.65754198645385)],
        ['H', (-4.92294992307692, -1.07482446153846, -1.65745801354615)],
        ['H', (-2.55361992307692, -2.72753446153846, -0.00010194854615)],
        ['H', (-3.20053992307692, 3.09975553846154, 0.00019662845385)],
        ['H', (1.22033007692308, 3.39843553846154, 0.00012292945385)],
        ['H', (1.44616007692308, -2.47650446153846, -0.00019643454615)],
        ['H', (5.98043007692308, -1.82356446153846, -0.00016430954615)],
        ['H', (5.51666007692308, 1.63464553846154, 0.00004545015385)]
    ],
    'RKT22': [
        ['C', (-2.45592700790231, -1.42989507692308, 0.02453686153846)],
        ['C', (-2.27890700790231, 1.23589092307692, 0.03727546153846)],
        ['C', (0.00000299209769, 2.51626492307692, 0.35232126153846)],
        ['C', (2.27891299209769, 1.23588492307692, 0.03726036153846)],
        ['C', (2.45594299209769, -1.42988507692308, 0.02455196153846)],
        ['H', (4.09843299209769, -2.22867507692308, -0.91190173846154)],
        ['H', (1.95028299209769, -2.46835507692308, 1.71267426153846)],
        ['H', (3.84962299209769, 2.33149492307692, -0.69043973846154)],
        ['H', (0.00000488182769, 4.55632492307692, 0.22496126153846)],
        ['H', (-3.84961700790231, 2.33149492307692, -0.69039573846154)],
        ['H', (-4.09842700790231, -2.22865507692308, -0.91193773846154)],
        ['H', (-0.00001779490231, -1.95351507692308, -0.92156073846154)],
        ['H', (-1.95030700790231, -2.46837507692308, 1.71265426153846)]
    ],
    'f_ch3f': [
        ['F', (20.0, 0, 0)],
        ['C', (0.0, 0.0, 0.12390246548611)],
        ['F', (0.0, 0.0, -2.4861775343035)],
        ['H', (0.9730507764948, -1.68537338323335, 0.78742502293913)],
        ['H', (0.9730507764948, 1.68537338323335, 0.78742502293913)],
        ['H', (-1.94610155298961, 0.0, 0.78742502293913)],
    ],
    'fch3fcomp': [
        ['F', (0.0, 0.0, -3.38520099999999)],
        ['C', (0.0, 0.0, -0.69091599999999)],
        ['H', (-0.967145, 1.6751442782862, -0.03323400000003)],
        ['H', (-0.967145, -1.6751442782862, -0.03323400000003)],
        ['H', (1.93429, 0.0, -0.03323400000003)],
        ['F', (0.0, 0.0, 4.17581900000001)],
    ],
    'fch3fts': [
        ['F', (3.44602644836333, 0.00098128583333, 0.00020547983333)],
        ['C', (0.00027242436333, -0.00016955716667, -0.00034443116667)],
        ['H', (0.00012124536333, -0.57615623316667, 1.93709997083333)],
        ['H', (0.00054265436333, 1.96565176683333, -0.47041402916667)],
        ['H', (-0.00017922081667, -1.39132823316667, -1.46676002916667)],
        ['F', (-3.44678355163667, 0.00102097083333, 0.00021303883333)],
    ],
    'nh2_c2h5': [
        ['N', (20.0, 0.0, 0.80170933333333)],
        ['H', (20.0-1.51615, 0.0, -0.40085466666667)],
        ['H', (20.0+1.51615, 0.0, -0.40085466666667)],
        ['C', (-0.04853221428571, -1.18090857142857, 0)],
        ['C', (-0.04853221428571, 1.63675142857143, 0)],
        ['H', (1.87834168571429, -1.94402857142857, 0)],
        ['H', (-0.98970131428571, -1.93608857142857, 1.66942000000000)],
        ['H', (-0.98970131428571, -1.93608857142857, -1.66942000000000)],
        ['H', (0.09906268571429, 2.68018142857143, -1.74671000000000)],
        ['H', (0.09906268571429, 2.68018142857143, 1.74671000000000)],
    ],
    'nh_c2h6': [
        ['N', (20.0, 0.0, 0.979571)],
        ['H', (20.0, 0.0, -0.979571)],
        ['C', (-1.44204, 6.25e-07, -1.75e-06)],
        ['C', (1.44204, 6.25e-07, -1.75e-06)],
        ['H', (2.18889, -0.838361375, -1.72739175)],
        ['H', (-2.18889, -1.914909375, -0.14105075)],
        ['H', (2.18889, -1.076789375, 1.58973825)],
        ['H', (2.18889, 1.915150625, 0.13766125)],
        ['H', (-2.18889, 0.835297625, 1.72887825)],
        ['H', (-2.18889, 1.079610625, -1.58783175)],
    ],
    'RKT19': [
        ['C', (-2.1803793, -1.0289018, 0.00734993)],
        ['C', (-0.3676803, 1.1702142, 0.00985193)],
        ['N', (4.0983407, -0.8941478, 0.01175866)],
        ['H', (-1.8120793, -2.2974558, -1.57319855)],
        ['H', (-2.0426393, -2.1215858, 1.74792145)],
        ['H', (-4.1338393, -0.3782528, -0.16279555)],
        ['H', (-0.3335553, 2.2935042, -1.70833855)],
        ['H', (-0.3778493, 2.3324642, 1.70199145)],
        ['H', (2.0214007, 0.1621052, -0.04795105)],
        ['H', (5.1282807, 0.7620562, 0.01341028)],
    ],
    'nh2_ch3': [
        ['N', (20.0, 0.0, 0.80170933333333)],
        ['H', (20.0-1.51615, 0.0, -0.40085466666667)],
        ['H', (20.0+1.51615, 0.0, -0.40085466666667)],
        ['C', (0.0, 0.0, 0.0)],
        ['H', (1.01791624999999, 1.76308266284997, 0.0)],
        ['H', (1.01791624999999, -1.76308266284997, 0.0)],
        ['H', (-2.03583249999998, 0.0, 0.0)],
    ],
    'nh_ch4': [
        ['N', (20, 0, 0.97957100000000)],
        ['H', (20, 0, -0.97957100000000)],
        ['C', (0, 0, 0)],
        ['H', (1.18644000000000, -1.18644000000000, 1.18644000000000)],
        ['H', (-1.18644000000000, 1.18644000000000, 1.18644000000000)],
        ['H', (-1.18644000000000, -1.18644000000000, -1.18644000000000)],
        ['H', (1.18644000000000, 1.18644000000000, -1.18644000000000)],
    ],
    'RKT18': [
        ['C', (-1.61723557142857, 0.17430391428571, -0.00007322295714)],
        ['N', (3.29660442857143, 0.44073301428571, 0.00001181474286)],
        ['H', (-2.04636557142857, -0.77397198571429, 1.76320346884286)],
        ['H', (-2.03360557142857, -0.92222098571429, -1.67836653115714)],
        ['H', (-2.22319557142857, 2.12814901428571, -0.08653393115714)],
        ['H', (1.00665342857143, 0.43512801428571, 0.00187508884286)],
        ['H', (3.61714442857143, -1.48212098571429, -0.00011668715714)],
    ],
    'h_c2h4': [
        ['H', (20, 0, 0)],
        ['C', (1.25779000000000, 0, 0)],
        ['C', (-1.25779000000000, 0, 0)],
        ['H', (2.32752000000000, -1.74137000000000, 0)],
        ['H', (2.32752000000000, 1.74137000000000, 0)],
        ['H', (-2.32752000000000, -1.74137000000000, 0)],
        ['H', (-2.32752000000000, 1.74137000000000, 0)],
    ],
    'c2h5': [
        ['C', (-0.04853221428571, -1.18090857142857, 0)],
        ['C', (-0.04853221428571, 1.63675142857143, 0)],
        ['H', (1.87834168571429, -1.94402857142857, 0)],
        ['H', (-0.98970131428571, -1.93608857142857, 1.66942000000000)],
        ['H', (-0.98970131428571, -1.93608857142857, -1.66942000000000)],
        ['H', (0.09906268571429, 2.68018142857143, -1.74671000000000)],
        ['H', (0.09906268571429, 2.68018142857143, 1.74671000000000)],
    ],
    'c2h5ts': [
        ['C', (-0.82576285714286, 0.00011692482857, -0.65272157142857)],
        ['C', (1.66681714285714, -0.00004748147143, -0.15971057142857)],
        ['H', (-2.57566285714286, -0.00090163827143, 2.65566942857143)],
        ['H', (-1.83453285714286, 1.73980054872857, -1.01114257142857)],
        ['H', (-1.83515285714286, -1.73896945127143, -1.01205657142857)],
        ['H', (2.70236714285714, -1.74295945127143, 0.08941442857143)],
        ['H', (2.70192714285714, 1.74296054872857, 0.09054742857143)],
    ],
    'fch3clcomp1': [
        ['Cl', (0.00000000000000, 0.00000000000000, 4.01182083333330)],
        ['C', (0.00000000000000, 0.00000000000000, 0.51488583333330 )],
        ['H', (-0.96973500000000, -1.67963028987781, -0.10452916666662)],
        ['H', (-0.96973500000000, 1.67963028987781, -0.10452916666662)],
        ['H', (1.93947000000001, 0.00000000000000, -0.10452916666662)],
        ['F', (0.00000000000000, 0.00000000000000, -4.21311916666670)],
    ],
    'fch3clcomp2': [
        ['F', (0.00000000000000, 0.00000000000000, -3.57146166666667)],
        ['C', (0.00000000000000, 0.00000000000000, -0.91002166666667)],
        ['H', (-0.96822000000000, 1.67700623290434, -0.24151166666666)],
        ['H', (-0.96822000000000, -1.67700623290434, -0.24151166666666)],
        ['H', (1.93644000000000, 0.00000000000000, -0.24151166666666)],
        ['Cl', (0.00000000000000, 0.00000000000000, 5.20601833333333)],
    ],
    'fch3clts': [
        ['F', (0.00000000000000, 0.00000000000000, -3.77342166666668)],
        ['C', (0.00000000000000, 0.00000000000000, 0.09967833333332)],
        ['H', (-1.00352500000000, 1.73815628666556, -0.13956166666664)],
        ['H', (-1.00352500000000, -1.73815628666556, -0.13956166666664)],
        ['H', (2.00705000000000, 0.00000000000000, -0.13956166666664)],
        ['Cl', (0.00000000000000, 0.00000000000000, 4.09242833333332)],
    ],
    'clch3clcomp': [
        ['Cl', (0.00000000000000, 0.00000000000000, -4.19154916666667)],
        ['C', (0.00000000000000, 0.00000000000000, -0.75525916666667)],
        ['H', (-0.96854500000000, 1.67756914941680, -0.10906416666665)],
        ['H', (-0.96854500000000, -1.67756914941680, -0.10906416666665)],
        ['H', (1.93709000000000, 0.00000000000000, -0.10906416666665)],
        ['Cl', (0.00000000000000, 0.00000000000000, 5.27400083333333)],
    ],
    'clch3clts': [
        ['Cl', (-0.00051899416667, 0.03636383216667, 4.38895919100000)],
        ['C', (0.00040319263333, 0.00038343916667, -0.00009899500000)],
        ['H', (1.43804376263333, -1.41921496783333, 0.01211999100000)],
        ['H', (-1.94783623736667, -0.53480596783333, 0.00412643100000)],
        ['H', (0.51103576263333, 1.95519503216667, -0.01636580900000)],
        ['Cl', (-0.00112748636667, -0.03792136783333, -4.38874080900000)],
    ],
    'oh_n2': [
        ['N', (0.00000000000000, 0.00000000000000, 1.03662000000000)],
        ['N', (0.00000000000000, 0.00000000000000, -1.03662000000000)],
        ['O', (20.00000000000000, 0.00000000000000, 0.91547450000000)],
        ['H', (20.00000000000000, 0.00000000000000, -0.91547450000000)],
    ],
    'h_n2o':[
        ['H', (20.0, 0.0, 0.0)],
        ['O', (0.0, 0.0,  2.20131966666667)],
        ['N', (0.0, 0.0, -2.15944033333333)],
        ['N', (0.0, 0.0, -0.04187933333333)],
    ],
    'n2ohts':[
        ['H', (-0.50842200000000, -2.90864275000000, 0.00000000000000)],
        ['O', (-1.56236600000000, -0.43463275000000, 0.00000000000000)],
        ['N', ( 0.06470400000000,  1.22558825000000, 0.00000000000000)],
        ['N', ( 2.00608400000000,  2.11768725000000, 0.00000000000000)],
    ],
}

tot_spin_configs = {
  'oh_ch3f': (14, 14),
  'ch3oh_f': (14, 14),
  'hoch3fcomp1': (14, 14),
  'hoch3fcomp2': (14, 14),
  'hoch3fts': (14, 14),
  'h_hf': (6, 5),
  'hfhts': (6, 5),
  'h_hcl': (10, 9),
  'hclhts': (10, 9),
  'h_ch3f': (10, 9),
  'hfch3ts': (10, 9),
  'hf_ch3': (10, 9),
  'h_f2': (10, 9),
  'f2': (9, 9),
  'hf2ts': (10, 9),
  'hf_f': (10, 9),
  'ch3_clf': (18, 17),
  'ch3f_cl': (18, 17),
  'ch3fclts': (18, 17),
  'hcn': (7, 7),
  'hnc': (7, 7),
  'hcnts': (7, 7),
  'c5h8': (19, 19),
  'RKT22': (19, 19),
  'f_ch3f': (14, 14),
  'fch3fcomp': (14, 14),
  'fch3fts': (14, 14),
  'nh2_c2h5': (14, 12),
  'nh_c2h6': (14, 12),
  'RKT19': (14, 12),
  'nh_ch4': (10, 8),
  'nh2_ch3': (10, 8),
  'RKT18': (8, 10),
  'h_c2h4': (9, 8),
  'c2h5': (9, 8),
  'c2h5ts': (9, 8),
  'fch3clcomp1': (18, 18),
  'fch3clcomp2': (18, 18),
  'fch3clts': (18, 18),
  'clch3clcomp': (22, 22),
  'clch3clts': (22, 22),
  'oh_n2': (12, 11),
  'h_n2o': (12, 11),
  'n2ohts': (11, 12),
}

atom_spin_configs = {
    'nh2_ch3': [
        (4, 3), (1, 0), (0, 1),
        (3, 3), (1, 0), (0, 1), (1, 0),
    ],
    'oh_ch3f': [
        (5, 4), (0, 1),
        (3, 3), (5, 4),
        (0, 1), (1, 0), (0, 1),
    ],
    'ch3oh_f': [
        (3, 3), (4, 4),
        (1, 0), (0, 1), (1, 0), (0, 1),
        (5, 5),
    ],
    'hoch3fcomp1': [
        (3, 3), (4, 4),
        (1, 0), (0, 1), (1, 0), (0, 1),
        (5, 5),
    ],
    'hoch3fcomp2': [
        (5, 5), (3, 3),
        (1, 0), (0, 1), (1, 0),
        (4, 4), (1, 0),
    ],
    'hoch3fts': [
        (5, 5), (3, 3),
        (1, 0), (0, 1), (1, 0),
        (4, 4), (1, 0),
    ],
    'h_hf': [
        (1, 0), (4, 5), (1, 0),
    ],
    'hfhts': [
        (1, 0), (4, 5), (1, 0),
    ],
    'h_hcl': [
        (1, 0), (8, 9), (1, 0),
    ],
    'hclhts': [
        (1, 0), (8, 9), (1, 0),
    ],
    'h_ch3f': [
        (1, 0), (3, 3),
        (5, 4), (1, 0), (0, 1), (1, 0),
    ],
    'hfch3ts': [
        (1, 0), (4, 5),
        (3, 3), (1, 0), (0, 1), (1, 0),
    ],
    'hf_ch3': [
        (5, 4), (0, 1),
        (3, 3), (1, 0), (0, 1), (1, 0),
    ],
    'h_f2': [
        (1, 0),
        (4, 5), (5, 4),
    ],
    'f2': [
        (4, 5), (5, 4),
    ],
    'hf2ts': [
        (1, 0),
        (4, 5), (5, 4),
    ],
    'hf_f': [
        (5, 4), (0, 1),
        (5, 4),
    ],
    'ch3_clf': [
       (4, 2),
       (0, 1), (0, 1), (1, 0),
       (5, 4), (8, 9),
    ],
    'ch3f_cl': [
        (3, 3), (5, 4),
        (1, 0), (0, 1), (0, 1), (9, 8),
    ],
    'ch3fclts': [
        (9, 8), (4, 5),
        (4, 2), (0, 1), (0, 1), (1, 0),
    ],
    'hcn': [
        (3, 3), (3, 4), (1, 0),
    ],
    'hnc': [
        (3, 3), (4, 3), (0, 1),
    ],
    'hcnts': [
        (3, 3), (4, 3), (0, 1),
    ],
    'c5h8': [
        (3, 3), (3, 3), (3, 3), (3, 3), (3, 3),
        (0, 1), (0, 1), (1, 0), (1, 0), (0, 1), (1, 0), (0, 1), (0, 1),
    ],
    'RKT22': [
        (3, 3), (3, 3), (3, 3), (3, 3), (3, 3),
        (0, 1), (0, 1), (1, 0), (1, 0), (0, 1), (1, 0), (0, 1), (0, 1),
    ],
    'f_ch3f': [
        (5, 5),
        (3, 3), (4, 5), (1, 0), (1, 0), (0, 1),
    ],
    'fch3fcomp': [
        (5, 5),
        (3, 3), (4, 5), (1, 0), (1, 0), (0, 1),
    ],
    'fch3fts': [
        (5, 4),
        (3, 4), (1, 0), (0, 1), (0, 1),
        (5, 4),
    ],
    'nh2_c2h5': [
        (4, 3), (1, 0), (0, 1),
        (3, 3), (4, 2),
        (1, 0), (0, 1), (1, 0), (0, 1), (0, 1)
    ],
    'nh_c2h6': [
        (5, 2), (0, 1),
        (3, 3), (3, 3),
        (1, 0), (1, 0), (0, 1),
        (0, 1), (0, 1), (1, 0),
    ],
    'RKT19': [
        (3, 3), (3, 3), (5, 2),
        (1, 0), (1, 0), (1, 0),
        (0, 1), (0, 1), (0, 1), (0, 1)
    ],
    'nh_ch4': [
        (5, 2), (0, 1),
        (3, 3), (1, 0), (0, 1), (1, 0), (0, 1),
    ],
    'RKT18': [
        (4, 2), (4, 3),
        (0, 1), (0, 1), (0, 1), (0, 1), (0, 1),
    ],
    'h_c2h4': [
        (1, 0),
        (3, 3), (3, 3),
        (1, 0), (0, 1), (1, 0), (0, 1),
    ],
    'c2h5': [
        (3, 3), (3, 3),
        (1, 0), (1, 0), (1, 0), (0, 1), (0, 1),
    ],
    'c2h5ts': [
        (3, 3), (3, 3),
        (1, 0), (1, 0), (1, 0), (0, 1), (0, 1),
    ],
    'fch3clcomp1': [
        (9, 8), (3, 3),
        (1, 0), (0, 1), (0, 1), (5, 5),
    ],
    'fch3clcomp2': [
        (5, 4), (3, 3),
        (1, 0), (0, 1), (0, 1), (9, 9),
    ],
    'fch3clts': [
        (5, 5), (3, 3),
        (1, 0), (0, 1), (0, 1), (9, 8),
    ],
    'clch3clcomp': [
        (9, 8), (3, 3),
        (0, 1), (1, 0), (0, 1),
        (9, 9)
    ],
    'clch3clts': [
        (9, 9), (3, 2),
        (0, 1), (1, 0), (0, 1),
        (9, 9)
    ],
    'oh_n2': [
        (4, 3), (3, 4), (5, 3), (0, 1)
    ],
    'h_n2o': [
        (1, 0), (4, 4), (4, 3), (3, 4)
    ],
    'n2ohts': [
        (0, 1), (4, 4), (4, 3), (3, 4)
    ],
}


def BH76set_static(cfg):
  """Sets molecule and electrons fields in ConfigDict."""
  if cfg.system.config_name not in systems.keys():
    raise ValueError(f'Unrecognized molecule: {cfg.system.config_name}')

  units = 'bohr'
  molecule = []
  for element, coords in systems[cfg.system.config_name]:
    molecule.append(system.Atom(symbol=element, coords=coords, units=units))
  cfg.system.molecule = molecule
  cfg.system.atom_spin_configs = atom_spin_configs[cfg.system.config_name] if \
               atom_spin_configs.__contains__(cfg.system.config_name) else None
  if tot_spin_configs.__contains__(cfg.system.config_name):
    cfg.system.electrons = tot_spin_configs[cfg.system.config_name]
  else:
    electrons = sum(int(round(atom.charge)) for atom in cfg.system.molecule)
    nbeta = electrons // 2
    cfg.system.electrons = (electrons-nbeta, nbeta)

  return cfg


def get_config():
  """
  Returns config for running BH76.
  Using config.system.config_name to set the system.
  """
  name = 'BH76'
  cfg = base_config.default()
  cfg.system.config_name = name
  cfg.system.atom_spin_configs = None
  with cfg.ignore_type():
    cfg.system.set_molecule = BH76set_static
    cfg.config_module = '.bh76'
  return cfg
