from math import log, exp
import numpy as np
from abc import ABC, abstractmethod
import matplotlib.pyplot as plt
import scipy
from scipy.special import lambertw
import seaborn as sns
import sys
from profiling import Profiling


class BaseTrade(ABC):
    def __init__(self, M):
        self.M = M
        self.s = 0
        self.w = 0
        self.p_max = 1
        self.exchanges = []

    @abstractmethod
    def trade(self, p):
        return

    def buy(self, p, qty):
        if qty:
            self.w += qty
            self.s += p * qty
        self.exchanges.append(
            (self.w, p, qty, p / (self.s + 1 - self.w), self.s))

    def trade_seq(self, seq):
        last = 1
        for p in seq:
            if p > last:
                self.trade(p)
                last = p
        assert (self.w <= 1)
        if self.w < 1:
            self.s += (1 - self.w)
            self.exchanges.append((self.w, 1, 1 - self.w, p / self.s, self.s))
            self.w = 1

    def plot_exchanges(self, title):
        self.exchanges = np.array(self.exchanges)
        self.exchanges = self.exchanges[self.exchanges[:, 3] != None]

        plt.figure(1)
        plt.plot(self.exchanges[:, 0], self.exchanges[:, 3], 'bo-')
        plt.xlabel('utilization')
        plt.ylabel('exchanged')

        plt.figure(2)
        plt.plot(self.exchanges[:, 1], self.exchanges[:, 3], 'ro-')
        plt.xlabel('price')
        plt.ylabel('exchanged')

        plt.title(title)

        plt.show()

    def reset(self):
        self.s = 0
        self.w = 0
        self.p_max = 1
        self.exchanges = []


class Adaptive(BaseTrade):
    def __init__(self, M):
        super().__init__(M)
        # competitive ratio before seeing any price
        self.res_prices = [lambertw((M-1)/np.exp(1))+1]
        self.prices_not_exchanged = []

    def get_reservation_price(self, p, min_error=1):
        h = self.M
        l = p
        error = self.M
        while error > min_error and (h - l) > min_error:
            res = (h + l)/2
            # print(h,l,res)
            error = (res + (p-1)*log((self.M-1)/(res-1))) * \
                log((self.M-1)/(res-1)) - self.w*(self.s+p*(1-self.w))
            if error > 0:
                h = res
            else:
                l = res
            error = abs(error)
        return res

    def trade(self, p):
        if p > self.p_max:
            self.p_max = p
            self.a = (log((self.M-1)/(p-1)) * (p-1) + p) / \
                (p*(1-self.w) + self.s)  # pas = p
            x = min(1 - log((self.M-1) / (p-1)) / self.a, 1)
            if x-self.w > 0:
                self.res_prices.append(p)
                if x-self.w < 0.00001:
                    print(p, 'aaaaaaa')
            else:
                self.prices_not_exchanged.append(p)
                self.res_prices.append(self.get_reservation_price(p))
            self.buy(p, max(0, x-self.w))


class Threshold(BaseTrade):
    def __init__(self, M):
        super().__init__(M)
        self.opt_comp_rat = float(scipy.special.lambertw((M-1)/np.exp(1)) + 1)

    def inv_phi(self, p):
        return (np.log(p) + 1) / self.opt_comp_rat

    def trade(self, p):
        if p > self.p_max:
            # if self.w <= 1/self.opt_comp_rat:
            self.p_max = p
            new_w = min(self.inv_phi(p), 1)
            self.buy(p, max(0, new_w - self.w))


class Threat(BaseTrade):
    def __init__(self, M):
        super().__init__(M)
        self.a = float(scipy.special.lambertw((M-1)/np.exp(1)) + 1)

    def trade(self, p):
        # aha, and when we adapt the ratio at each step, the algorithms are identical
        # self.a = (log((self.M-1)/(p-1)) * (p-1) + p) / \
        #     (p*(1-self.w) + self.s)  # pas = p
        if p > self.p_max:
            self.p_max = p
            lower = max(0, (p - self.a * (self.s + 1 - self.w))/(self.a * p))
            if self.w + lower > 1:
                lower = max(0, 1 - self.w)
            self.buy(p, lower)


class Prediction(BaseTrade):
    '''
    given a prediction pred and robustness rob
    exchanges using algorithm 2 in our paper
    '''

    def __init__(self, M, pred, rob):
        super().__init__(M)
        self.pred = pred
        self.rob = rob

    def get_max_res_price(self, p, min_error=1):
        h = self.M
        l = p
        error = self.M
        res = (h + l)/2
        while error > min_error and (h - l) > min_error:
            res = (h + l)/2
            # print(h,l,res)
            error = res - (1-log((self.M-1)/(res-1))) * \
                (p-self.rob) - self.rob*(self.s+1-self.w*p)
            if error > 0:
                h = res
            else:
                l = res
            error = abs(error)
        return res

    def trade(self, p):
        if p < self.pred:
            new_uti = 1 - log((self.M-1)/(p-1)) / self.rob
        else:
            r = self.get_max_res_price(p)
            if r <= p:
                print(r, p)
            new_uti = 1 - log((self.M-1)/(r-1)) / self.rob
        self.buy(p, max(0, new_uti-self.w))


class OTAPrediction(BaseTrade):
    def __init__(self, M, pred, lam, cons, rob):
        super().__init__(M)
        self.pred = pred
        self.rob = rob
        self.lam = lam
        self.cons = cons
        self.b = (cons*(exp(rob)*(M-1)-1)+rob) / \
            (exp(rob)*(M-1)*cons-rob*exp(cons)*(cons-1))
        self.th = 1 + (cons-1)*exp(cons*self.b)
        # getting the last 4 values?

    def trade(self, p):
        if p < self.pred:
            new_uti = 1 - log((self.M-1)/(p-1)) / self.rob
        else:
            r = self.get_max_res_price(p)
            assert (r > p)
            new_uti = 1 - log((self.M-1)/(r-1)) / self.rob
        self.buy(p, max(0, new_uti-self.w))


class AdaptiveWithPrediction(BaseTrade):
    def __init__(self, M, pred, r, w_star_error=0.001):
        super().__init__(M)
        self.pred = pred
        self.r = r
        self.pred = pred
        self.p_max = 1
        self.w_star_error = w_star_error

    def get_w_star(self, p):
        '''
        does bijection to find w_star with self.w_star_error
        following equation
        w_star = 1 - log((M-1)/(r*(s+1-p*w+w_star*(p-1)-1)))/r
        '''
        h = 1
        l = self.w
        error = 1
        if self.w >= 1 - self.w_star_error:
            return 1
        while error > self.w_star_error and (h - l) > self.w_star_error:
            w_star = (h + l)/2
            error = 1 - \
                log((self.M-1)/(self.r*(self.s + 1 - p*self.w + w_star*(p-1) - 1))) / \
                self.r - w_star
            if error < 0:
                h = w_star
            else:
                l = w_star
            error = abs(error)
        if (h - l) < self.w_star_error:
            w_star = h
        return w_star

    def trade(self, p):
        if p > self.p_max:
            self.p_max = p
            if p < self.pred:
                new_w = (p - self.r*(self.s + 1 - p*self.w)) / (self.r*(p-1))
            else:
                w_star = self.get_w_star(p)
                if self.r*(self.s + 1 - p*self.w + w_star) >= self.M:
                    new_w = 1
                else:
                    new_w = w_star
            self.buy(p, max(0, new_w-self.w))


class ParetoOptimal(BaseTrade):
    def __init__(self, M, pred, r, w_star_error=0.0000001):
        super().__init__(M)
        self.p_max = 1
        self.pred = pred
        self.r = r
        # phi function is exponential starting at r ending at pred
        # ends at utilization ln((pred-1)/(r-1))/r
        # then exponential from wstar to 1 (with a reservation price)
        self.w_star_error = w_star_error
        pred_inv = log((pred-1)/(r-1))/r
        self.pred_inv = pred_inv
        w_star = self.get_w_star()
        self.w_star = w_star
        self.start_second_exp = self.get_constant_second_exp(self.w_star) + 1
        # print("start_second ", self.start_second_exp)
        if self.start_second_exp >= M:
            self.w_star = 1
        # print("pred_inv ", pred_inv, "w_star ", w_star)
        self.constant_second_exp = self.get_constant_second_exp(w_star)
        # print("phi m: ", self.invert_phi_p_more_pred(self.M))
        profit_worst_case = (self.pred - self.r + log((self.pred-1)/(self.r-1)))/self.r + pred * (self.w_star - self.pred_inv) + self.constant_second_exp * exp(
            1 - self.w_star) + r - self.constant_second_exp - r * self.w_star
        # print("profit worst case ", profit_worst_case, "ratio ", M/profit_worst_case)

    def get_constant_second_exp(self, w_star):
        return self.r * ((self.pred - self.r + log((self.pred-1)/(self.r-1)))/self.r + self.pred*(w_star - self.pred_inv) + 1 - w_star) - 1
        # return self.r * (self.pred/self.r + 1 - self.pred*self.pred_inv + w_star*(self.pred-1)) - 1

    def get_w_star(self):
        '''
        does bijection to find w_star with self.w_star_error
        following equation
        w_star = 1 - log((M-1)/(r*(s+1-p*w+w_star*(p-1)-1)))/r
        '''
        h = 1
        l = self.pred_inv
        error = 1
        if self.pred_inv >= 1 - self.w_star_error:
            return 1
        while error > self.w_star_error and (h - l) > self.w_star_error:
            w_star = (h + l)/2
            # error = 1 - log((self.M-1)/(self.r*(self.pred/self.r + 1 -
            #                 self.pred*self.pred_inv + w_star*(self.pred-1) - 1))) / self.r - w_star
            phi_1 = self.get_constant_second_exp(
                w_star) * exp(self.r*(1-w_star)) + 1
            error = self.M - phi_1
            if error > 0:
                h = w_star
            else:
                l = w_star
            error = abs(error)
        # if (h - l) < self.w_star_error:
            # print("lower", h, l)
            # w_star = l
        return w_star

    def invert_phi_p_less_pred(self, p):
        return log((p-1) / (self.r-1)) / self.r

    def invert_phi_p_more_pred(self, p):
        return log((p-1)/self.constant_second_exp)/self.r + self.w_star

    def trade(self, p):
        if p > self.p_max:
            self.p_max = p
            if p < self.pred:
                new_w = self.invert_phi_p_less_pred(p)
            elif (p >= self.pred) and (p < self.start_second_exp):
                new_w = self.w_star
                # print("middle price ", p, "ex", new_w - self.w)
            else:
                new_w = self.invert_phi_p_more_pred(p)
                # print("above price ", p, p/self.s)
            new_w = min(1, new_w)
            self.buy(p, max(0, new_w-self.w))

    def plot(self):
        def phi_before_pred(x):
            return (self.r - 1) * np.exp(self.r * x) + 1

        def phi_after_pred(x):
            return self.constant_second_exp * np.exp(self.r * (x - self.w_star)) + 1

        def f(x):
            conditions = [x < self.pred_inv, (self.pred_inv < x) & (
                x < self.w_star), x >= self.w_star]
            return np.piecewise(x, conditions, [phi_before_pred, np.nan, phi_after_pred])

        sns.set_theme()
        sns.set_context("paper")
        sns.set_style("white")
        # Plotting
        plt.figure(figsize=(8, 6))
        x_values = np.linspace(0, 1, 1000)
        y_values = f(x_values)
        y_values = np.where(y_values > self.M, self.M, y_values)
        x_values = x_values[:len(y_values)]
        plt.plot(x_values, y_values, linewidth=2, color="darkblue")
        plt.show()


class ExchangeFromGivenPhi(BaseTrade):
    def __init__(self, M, phis):
        super().__init__(M)
        # phi = ["line", start_w, end_w, price, price, alpha]
        # phi = ["exp",  start_w, end_w, start_price, end_price, res_price, alpha]
        self.phis = phis
        self.current_phi_ind = 0

    def trade(self, p):
        if p > self.p_max:
            self.p_max = p
            while (p > self.phis[self.current_phi_ind][4]):
                self.current_phi_ind += 1
            if self.current_phi_ind < (len(self.phis)-1) and p == self.phis[self.current_phi_ind][4] and self.phis[self.current_phi_ind][0] == "exp":
                self.current_phi_ind += 1  # i want to spend as much as the line
            phi = self.phis[self.current_phi_ind]
            # print("traded with: ", phi, " price: ", p)
            if phi[0] == "line":
                new_w = phi[2]
            else:
                if p <= phi[3]:
                    new_w = phi[1]
                else:
                    new_w = log((p-1)/(phi[5]-1)) / phi[6] + phi[1]
            self.buy(p, max(0, new_w - self.w))

    def reset(self):
        super().reset()
        self.current_phi_ind = 0


def plot(list_to_plot, x, y, x_ticks, x_ticks_names, x_name, y_name, name="", save=True, scatter=True):
    '''
    list_to_plot : [(algorithm : child of baseTrade after processing, color : to plot)]
    x, y : number between 0 and 3, matching resource usage, price, traded qty, ratio, left-over money
    '''
    sns.set_theme()
    sns.set_context("paper")
    sns.set_style("white")
    label = ["resource usage", "price", "traded qty",
             "ratio", "profit", "left-over money"]
    plt.figure(1)
    # use latex in text in plot
    # plt.rc('text', usetex=True)
    # for L, color in list_to_plot:
    #     plt.scatter([t[x] for t in L], [t[y] for t in L], c=color)
    if x == 5 or y == 5:
        for L, _ in list_to_plot:
            for i in range(len(L)):
                L[i] = L[i] + (1 - L[i][0],)
    if scatter:
        for L, color, linestyle, label in list_to_plot:
            plt.plot([t[x] for t in L], [t[y]
                     for t in L], c=color, marker=linestyle, linewidth=0.5, label=label)
    else:
        for L, color, linestyle, label in list_to_plot:
            plt.plot([t[x] for t in L], [t[y] for t in L], c=color, linewidth=2, linestyle=linestyle, label=label)
    plt.grid(True)
    fontsize = 18
    plt.xticks(x_ticks, x_ticks_names,fontsize=fontsize-6)
    # add legend
    l = plt.legend(fontsize=fontsize-8)
    plt.gcf().set_facecolor('white')
    plt.xlabel(x_name, fontsize=fontsize)
    plt.ylabel(y_name, fontsize=fontsize)
    # plt.title(name, fontsize=fontsize)
    # plt.show()
    if save:
        plt.savefig(f'./pdfs/{name}.pdf', format='pdf', dpi=200,bbox_inches='tight')
    else:
        plt.show()


def get_phi_list_from_profile(prices, r):
    profile = Profiling(prices, None)
    consistency = profile.do_binary_search_3_intervals_symmetric(r)
    profile.alphas = [r, consistency, r]
    return profile.phi


def worst_case_pareto_vs_profile():
    M = 100
    pred = np.random.uniform(1, M)
    dist_pred = pred * 0.1
    step = 0.01
    r = 4
    po = ParetoOptimal(M, pred, r)
    # po.plot()
    # return
    phis = get_phi_list_from_profile([pred-dist_pred, pred+dist_pred, M], r)
    print(phis)
    profiled = ExchangeFromGivenPhi(M, phis)
    prices = np.arange(1+step, M, step)
    # f = lambda x : round(x,2)
    # prices = np.array(list(map(f, prices)))
    for p in prices:
        po.trade(p)
        profiled.trade(p)
        # x.append(pred - p)
        # y.append(profiled.s - po.s)
    # label = ["resource usage", "price", "traded qty", "ratio", "profit", "left-over money"]
    plot([(po.exchanges, "darkred", "dashed", "PO"), (profiled.exchanges, "darkblue", "solid", "Profile")], 1, 3,
         [1, pred-dist_pred, pred, pred+dist_pred, M], ["$1$", "$0.9\hat{p}$", "$\hat{p}$", "$1.1\hat{p}$", "$M$"], "max rate", "performance ratio", "worst-case pareto optimal vs profiled", True, False) 
    # po.reset()
    # profiled.reset()
    # plt.plot(prices,y)
    # plt.show()


def worst_case_aggregate_pareto_vs_profile():
    M = 100
    step = 0.01
    r = 4
    max_rates = []
    ratio_of_ratios = []

    for _ in range(100):
        pred = np.random.uniform(1, M)
        dist_pred = pred * 0.1
        max_rate = np.random.uniform(pred-dist_pred, pred+dist_pred)
        po = ParetoOptimal(M, pred, r)
        phis = get_phi_list_from_profile(
            [pred-dist_pred, pred+dist_pred, M], r)
        profiled = ExchangeFromGivenPhi(M, phis)
        prices = np.arange(1+step, max_rate, step)
        for p in prices:
            po.trade(p)
            profiled.trade(p)
        max_rates.append(max_rate - pred)
        ratio_of_ratios.append((po.exchanges[-1][3] - profiled.exchanges[-1][3]) / po.exchanges[-1][3])
        # print(pred, dist_pred, max_rate, profiled.exchanges[-1][4], po.exchanges[-1][4])
        po.reset()
        profiled.reset()
    # label = ["resource usage", "price", "traded qty", "ratio", "profit", "left-over money"]
    # plot([(po.exchanges, "red"), (profiled.exchanges, "blue")], 1, 3, [pred], "worst-case pareto optimal vs profiled", False, False)
    sns.set_theme()
    sns.set_context("paper")
    sns.set_style("white")
    plt.figure(1)
    fontsize = 18
    plt.xlabel("$p^* - \hat{p}$", fontsize=fontsize)
    plt.ylabel("relative improvement", fontsize=fontsize)
    plt.scatter(max_rates, ratio_of_ratios, c="blue", marker=".")
    print(np.mean(ratio_of_ratios))
    # plt.show()
    plt.savefig('./pdfs/agregate_worst_case_profile_vs_pareto.pdf',
                format='pdf', dpi=200, bbox_inches='tight')


def btc_pareto_vs_profile():
    seq = []
    f = open("prediction.txt", 'r')
    for line in f:
        seq.append(float(line))
    r = 4
    len_pre = int(len(seq) / 5)
    M = max(seq)
    pred = max(seq[:len_pre])
    dist_pred = pred * 0.1
    seq = seq[len_pre:]
    po = ParetoOptimal(M, pred, r)
    phis = get_phi_list_from_profile([pred-dist_pred, pred+dist_pred, M], r)
    # print([pred-dist_pred,pred+dist_pred,M], phis)
    profiled = ExchangeFromGivenPhi(M, phis)
    for p in seq:
        po.trade(p)
        profiled.trade(p)
    plot([(po.exchanges, "darkred", "o", "PO"), (profiled.exchanges, "darkblue", "o", "Profile")], 1,
         3, [pred-dist_pred,pred,M], ["$0.9\hat{p}$","$\hat{p}$", "M"], "max rate", "performance ratio", "bitc_profile_new", True, True)


def btc_pareto_vs_profile_aggregate():
    r = 4
    inds = []
    ratios = []
    sns.set_theme()
    sns.set_context("paper")
    sns.set_style("white")
    fontsize = 18
    file = open('btc_sequences.txt', 'r')
    i = 1
    ind = 0
    seqs = [[] for _ in range(20)]
    for line in file:
        i += 1
        seqs[ind].append(float(line))
        if i % 1000 == 0:
            ind += 1
        if ind > 19:
            break
    for i, seq in enumerate(seqs):
        rates = []
        ratio_ratios = []
        len_pre = int(len(seq) / 5)
        M = max(seq)
        pred = max(seq[:len_pre])
        dist_pred = pred * 0.1
        seq = seq[len_pre:]
        po = ParetoOptimal(M, pred, r)
        phis = get_phi_list_from_profile(
            [pred-dist_pred, pred+dist_pred, M], r)
        # print([pred-dist_pred,pred+dist_pred,M], phis)
        profiled = ExchangeFromGivenPhi(M, phis)
        for p in seq:
            po.trade(p)
            profiled.trade(p)
            if p > pred-dist_pred and p < pred+dist_pred:
                rates.append(p)
                ratio_ratios.append(
                    (po.exchanges[-1][3] - profiled.exchanges[-1][3]) / po.exchanges[-1][3])
        # plt.xlabel("Rate", fontsize=fontsize)
        # plt.ylabel("Ratio of performance ratios", fontsize=fontsize)
        # plt.title("BTC pareto vs profile", fontsize=fontsize)
        # plt.scatter(rates, ratio_ratios, c="blue", marker=".")
        # plt.show()

        inds.append(i)
        ratios.append(np.mean(ratio_ratios))
    plt.xticks([i for i in range(1,21)], [str(i) for i in range(1,21)])
    plt.xlabel("sequence", fontsize=fontsize)
    plt.ylabel("relative improvement", fontsize=fontsize)
    # remove previous scatter plot
    plt.scatter(inds, ratios, c="darkblue",  marker="o")
    plt.grid(True, axis='y')
    plt.savefig('./pdfs/btc_avg_profile.pdf',
                format='pdf', dpi=200)
    ratios = np.array(ratios)
    ratios = ratios[~np.isnan(ratios)]
    print(ratios)
    print(np.mean(ratios))
    ratios = ratios[ratios > 0]
    print(len(ratios))


def btc_pareto_vs_adaptive(left_over_money=False):
    seq = []
    f = open("prediction.txt", 'r')
    for line in f:
        seq.append(float(line))
    r = 4
    len_pre = int(len(seq) / 5)
    M = max(seq)
    pred = max(seq[:len_pre])
    seq = seq[len_pre:]
    po = ParetoOptimal(M, pred, r)
    adap = AdaptiveWithPrediction(M, pred, r)
    for p in seq:
        po.trade(p)
        adap.trade(p)
    if left_over_money:
        plot([(po.exchanges, "red"), (adap.exchanges, "blue")], 1, 5, [
            pred,M], ["\hat{p}"], "rate", "left-over-money", "bitcoin pareto optimal vs adaptive left-over", False, True)
    else:
        plot([(po.exchanges, "darkred", "o", "PO"), (adap.exchanges, "darkblue", "o", "Adap-PO")], 1, 3, [
            pred,M], ["$\hat{p}$","M"], "max rate", "performance ratio", "btc_adap_ratio_new", True, True)


def seq_jumps_pareto_vs_adaptive_plot_profit():
    M = 400
    seq = np.arange(1, M, 10)
    r = 4.5
    pred = 120
    po = ParetoOptimal(M, pred, r)
    adap = AdaptiveWithPrediction(M, pred, r)
    for p in seq:
        po.trade(p)
        adap.trade(p)
    plot([(po.exchanges, "red"), (adap.exchanges, "blue")], 1,
         4, [pred], "seq_jumps pareto optimal vs adaptive", True)


def seq_jumps_pareto_vs_adaptive_plot_money_at_phat():
    M = 400
    seq = np.arange(1, M, 10)
    r = 4.5
    pred = 120
    po = ParetoOptimal(M, pred, r)
    adap = AdaptiveWithPrediction(M, pred, r)
    for p in seq:
        po.trade(p)
        adap.trade(p)
    plot([(po.exchanges, "red"), (adap.exchanges, "blue")], 1, 5, [pred],
         "seq_jumps pareto optimal vs adaptive how much money used to remain r-robust", True)


def plot_profile(prices, alphas):
    sns.set_theme()
    sns.set_context("paper")
    sns.set_style("white")
    plt.figure(figsize=(8, 6))
    for i in range(len(prices)-1):
        x = np.linspace(prices[i], prices[i+1], 1000)
        y = [alphas[i]]*len(x)
        plt.plot(x, y, linewidth=2, color="darkblue")
    x_labels = []
    y_labels = []
    for i in range(len(alphas)):
        y_labels.append(f"t{i+1}: {alphas[i]}")
    for i in range(len(prices)):
        x_labels.append(f"q{i+1}: {prices[i]}")
    plt.xticks(prices, x_labels)
    plt.yticks(alphas, y_labels)
    plt.xlabel("Performance ratios")
    plt.ylabel("Price intervals")
    plt.title("Profile")
    plt.grid(True)
    plt.savefig(f'profile_btc.pdf', format='pdf', dpi=200)


def pareto_opt():
    M = 200
    pred = 70
    r = 4.5
    po = ParetoOptimal(M, pred, r)
    # po.plot()
    step = 0.01
    prices = np.arange(1+step, M+step, step)
    for p in prices:
        po.trade(p)
    plot([(po.exchanges, "red")], 1, 3, [pred], "pareto optimal", False, False)


if __name__ == "__main__":
    arg = sys.argv[1]
    dic_arg = {
        "wc_profile": worst_case_pareto_vs_profile,
        "btc_profile": btc_pareto_vs_profile,
        "btc_adap": btc_pareto_vs_adaptive,
        "wc_avg_profile": worst_case_aggregate_pareto_vs_profile,
        "btc_avg_profile": btc_pareto_vs_profile_aggregate,
    }
    if arg in dic_arg:
        dic_arg[arg]()
    else:
        print("Usage: oneway.py [mode]")
        print("where mode can be one of the following strings")
        print(*dic_arg.keys())
