import pandas as pd
import os
import json
import numpy as np

from src.utils import (
    get_hyperparameters,
)

def save_results(result_dict, attack_config, model_name, dataset_name, shuffle, seed, test_split):
    
    config = get_config(attack_config, model_name, dataset_name, shuffle, seed, test_split)  
    ignore_keys = ["model_name", "dataset_name"]  
    result_name = model_name + "_" + dataset_name[:5] + "_" + '_'.join([str(key)[0] + str(value)[0:2] for key, value in config.items() if key not in ignore_keys])
    path = get_experiment_path(model_name, dataset_name)
    
    results_path, new_result_name = check_file_name_already_exists(path, result_name) 
    config["result_name"] = new_result_name
    df = pd.DataFrame(result_dict)
    df.to_json(results_path + ".json", index=False)
    
    with open(results_path + "_config.json", 'w') as fp:
        json.dump(config, fp)
    print("Results saved to: " + results_path)
    

def get_config(attack_config, model_name, dataset_name, shuffle, seed, test_split):
    config = {}
    config["model_name"] = model_name
    config["dataset_name"] = dataset_name
    config["shuffle"] = shuffle
    config["seed"] = seed
    config["test_split"] = test_split
    config.update(attack_config)
    return config


def get_experiment_path(model_name, dataset_name):
    path = "results/" 
    nested_path = [dataset_name, model_name]
    for folder in nested_path:
        path += folder + "/"
        if not os.path.exists(path):
            os.mkdir(path)
    return path


def check_experiment_already_exists(attack_config, model_name, dataset_name, shuffle, seed, test_split):
    config = get_config(attack_config, model_name, dataset_name, shuffle, seed, test_split)
    df = pd.DataFrame([config])

    hyperparameters = get_hyperparameters()
    # temperature is only a relevant hyperparameter for no_attack; it should not be used for attacks
    if df["attack_type"][0] != "no_attack" and "temperature" in hyperparameters:
        hyperparameters.remove("temperature")
    hyperparameters = df.columns.intersection(set(hyperparameters))
    print(hyperparameters)

    path = get_experiment_path(model_name, dataset_name)
    experiments = os.listdir(path)
    for experiment in experiments:
        if experiment.endswith("_config.json"):
            existing_config = json.load(open(path + experiment))
            existing_df = pd.DataFrame([existing_config])
            
            # check if the same set of hyperpparameters were used
            same_set_of_hyperparameters = set(existing_df.columns).issuperset(set(hyperparameters))
            if not same_set_of_hyperparameters:
                print("Not same set of hyperparams. Difference is:", set(hyperparameters).difference(existing_df.columns))
                continue
            
            current = df[hyperparameters].values
            existing = existing_df[hyperparameters].values
            equal = (current == existing)
            if equal.all():
                print("found existing experiment with same config:", experiment)
                return True
    return False
            

def check_file_name_already_exists(path, result_name, i=0):
    if i > 0:
        new_result_name = result_name + "_" + str(i)
    else:
        new_result_name = result_name

    if os.path.exists(path + new_result_name + ".json"):
        return check_file_name_already_exists(path, result_name, i+1)
    
    return path + new_result_name, new_result_name