import os
import mysql.connector
import yaml
import pandas as pd
import sys

# load global config
file_path = os.path.dirname(__file__)
project_path = os.path.dirname(os.path.dirname(file_path))
global_config = yaml.safe_load(open(os.path.join(project_path, "config/global.yaml"), "r"))

DATABASE_USER = global_config.get("mysql").get("username")
DATABASE_PASSWD = global_config.get("mysql").get("password")
HOST = global_config.get("mysql").get("host")

DATABASE = "Schedule"

path_to_csv = 'dialogue.csv'

# MySQL database configuration
db_config = {'host': HOST, 'user': DATABASE_USER, 'password': str(DATABASE_PASSWD)}

# Establish MySQL connection
conn = mysql.connector.connect(**db_config)
cursor = conn.cursor(buffered=True)


def create_database(cursor):
    try:
        cursor.execute("CREATE DATABASE IF NOT EXISTS {}".format(DATABASE))
        print("Database '{}' created successfully".format(DATABASE))
    except mysql.connector.Error as err:
        print("Error creating database '{}':".format(DATABASE), err)


def create_users_table(cursor):
    try:
        cursor.execute("""
            CREATE TABLE IF NOT EXISTS users (
                id INT AUTO_INCREMENT PRIMARY KEY,
                name VARCHAR(255) NOT NULL UNIQUE,
                password VARCHAR(255) NOT NULL
            )
        """)
        print("Table 'users' created successfully")
    except mysql.connector.Error as err:
        print("Error creating table 'users':", err)


def create_friendships_table(cursor):
    try:
        cursor.execute("""
            CREATE TABLE IF NOT EXISTS friendships (
                user_id INT,
                friend_id INT,
                PRIMARY KEY (user_id, friend_id),
                FOREIGN KEY (user_id) REFERENCES users(id),
                FOREIGN KEY (friend_id) REFERENCES users(id)
            )
        """)
        print("Table 'friendships' created successfully")
    except mysql.connector.Error as err:
        print("Error creating table 'friendships':", err)


def create_chats_table(cursor):
    try:
        cursor.execute("""
            CREATE TABLE IF NOT EXISTS chats (
                id INT AUTO_INCREMENT PRIMARY KEY,
                sender VARCHAR(255) NOT NULL,
                receiver VARCHAR(255) NOT NULL,
                message TEXT NOT NULL,
                timestamp TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            )
            CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        """)
        print("Table 'chats' created successfully")
    except mysql.connector.Error as err:
        print("Error creating table 'chats':", err)


def insert_chat(cursor, sender, receiver, message):
    try:
        query = """
            INSERT INTO chats (sender, receiver, message) 
            VALUES (%s, %s, %s)
        """
        data = (sender, receiver, message)
        cursor.execute(query, data)
        print("Chat inserted successfully: {}, {}".format(sender, receiver))
    except mysql.connector.Error as err:
        print("Error inserting chat:", err)


def insert_friendship_data(cursor, user_id, friend_id):
    try:
        cursor.execute(
            """
            INSERT INTO friendships (user_id, friend_id)
            VALUES (%s, %s)
        """, (user_id, friend_id))
        print("Friendship data inserted successfully: {}, {}".format(user_id, friend_id))
    except mysql.connector.Error as err:
        print("Error inserting Friendship data:", err)


def insert_user_data(cursor, id, name, password):
    try:
        cursor.execute(
            """
            INSERT INTO users (id, name, password)
            VALUES (%s, %s, %s)
        """, (id, name, password))
        print("User data inserted successfully: {}, {}, {}".format(id, name, password))
    except mysql.connector.Error as err:
        print("Error inserting user data:", err)


def fetch_all_users(cursor):
    try:
        cursor.execute("SELECT * FROM users")
        rows = cursor.fetchall()
        for row in rows:
            print(row)
    except mysql.connector.Error as err:
        print("Error fetching user data:", err)

def print_table_summary(cursor, table_name):
    try:
        cursor.execute("DESCRIBE {}".format(table_name))
        print("-" * 10)
        print("Table Structure for '{}':".format(table_name))
        structure_rows = cursor.fetchall()
        for row in structure_rows:
            print(row)
        
        cursor.execute("SELECT * FROM {} LIMIT 5".format(table_name))
        print("\nSample Data for '{}':".format(table_name))
        data_rows = cursor.fetchall()
        for row in data_rows:
            print(row)
        print("-" * 10)
        
    except mysql.connector.Error as err:
        print("Error printing table summary for '{}':".format(table_name), err)


concat_df = pd.read_csv(path_to_csv)

all_character = set(concat_df['sender']) | set(concat_df['receiver'])
id2cha = dict()
cha2id = dict()
for user_id, character in enumerate(list(all_character)):
    id2cha[user_id + 1000] = character
    cha2id[character] = user_id + 1000

create_database(cursor)
cursor.execute("USE {}".format(DATABASE))

# init table
create_users_table(cursor)
create_friendships_table(cursor)
create_chats_table(cursor)
conn.commit()

# insert user
for id in id2cha:
    insert_user_data(cursor, id, id2cha[id], id2cha[id])
conn.commit()

# insert friendship&message
exist_friendship = set()
for _, line in concat_df.iterrows():
    insert_chat(cursor, line['sender'], line['receiver'], line['message'])
    id_a = cha2id[line['sender']]
    id_b = cha2id[line['receiver']]
    if str(id_a) + "_" + str(id_b) not in exist_friendship:
        exist_friendship.add(str(id_a) + "_" + str(id_b))
        exist_friendship.add(str(id_b) + "_" + str(id_a))
        insert_friendship_data(cursor, id_a, id_b)
        if id_a != id_b:
            insert_friendship_data(cursor, id_b, id_a)
conn.commit()

print("all character in this conversation: ", all_character)
print_table_summary(cursor, "users")
print_table_summary(cursor, "friendships")
print_table_summary(cursor, "chats")

cursor.close()
conn.close()