import json
import logging
import os
from abc import ABC, abstractmethod
import re

import yaml

from avatarchat.tool import SqlTool, JsonFormatTool, MindFillTool, FaissTool
from avatarchat.util import AvatarLogger
from backend.gemini import query_gemini
from backend.gpt import query_gpt, query_gpt4, query_claude

# load global config
file_path = os.path.dirname(__file__)
project_path = os.path.dirname(file_path)
global_config = yaml.safe_load(open(os.path.join(project_path, "config/global.yaml"), "r"))

class Agent(ABC):
    """The Base class for Agent
    It defines the backend llm of agent, 
    and how agent get the context for assembling the query prompt.    
    """

    def __init__(self, master, backend, task, is_assistant=False) -> None:
        """init

        Args:
            master (str): name of the human master of agent (username in the chat database)
            backend (str): name of the backend llm (gemini/gpt/gpt4)
            task (str): task prompt
            is_assistant (bool, optional): the agent who raises task is the instructor, the other is the assistant. Defaults to False.

        """
        super().__init__()
        # meta info
        self.master = master
        self.task = task
        self.agent_chat_history = []

        # backend llm
        self.backend = backend
        if self.backend == "gemini":
            self.query_func = query_gemini
        elif self.backend == "gpt":
            self.query_func = query_gpt
        elif self.backend == "gpt4":
            self.query_func = query_gpt4
        elif self.backend == "claude":
            self.query_func = query_claude
        else:
            raise ValueError("{} backend not implemented".format(backend))
        self.is_assistant = is_assistant

        # load profile prompt
        prompt_path = os.path.join(project_path, "prompts")
        if self.is_assistant:
            system_prompt_filepath = os.path.join(prompt_path, "assistant_system_prompt.json")  
        else:
            system_prompt_filepath = os.path.join(prompt_path, "instructor_system_prompt.json")  
        with open(system_prompt_filepath, "r") as f:
            self.system_prompt = json.load(f)

        # load tool prompt
        with open(os.path.join(project_path, "prompts", "tool_prompt.json"), "r") as f:
            self.tool_prompt = json.load(f)

        # returned json formats
        self.json_return_format = {
            "utterance": ""
        }


        # Tool
        self.sql_tool = SqlTool()
        self.json_tool = JsonFormatTool(self.query_func)
        self.mindfill_tool = MindFillTool(self.query_func)
    
    def set_master(self, master):
        self.master = master

    @abstractmethod
    def get_other_chat_history(self, receiver, communication_history=None) -> str:
        """defines how the agent get the context from chatting with other friends

        Args:
            receiver (str): the name of user in current chatting 

        Returns:
            str: retrieved chat history as the context for assembling the query prompt.
        """
        pass

    @abstractmethod
    def get_current_chat_history(self, receiver, communication_history) -> str:
        """defines how the agent get the context from current chatting

        Args:
            receiver (str): the name of user in current chatting 

        Returns:
            str: retrieved chat history as the context for assembling the query prompt.
        """
        pass

    def assemble_prompt(self, receiver, communication_history) -> str:
        """defines the process of assembling query prompt

        Args:
            receiver (str): the name of user in current chatting 
            communication_history (list[str]): the chat history between two agents

        Returns:
            str: assembled prompt
        """
        current_chat_history = self.get_current_chat_history(receiver, communication_history)
        other_chat_history = self.get_other_chat_history(receiver, communication_history)

        # the prompt is assembled in the order of 
        # 1. define the role
        # 2. give the human chat history (retrieved context)
        # 3. define the task
        # 4. give the current progress of agents' communication
        # 5. define the return format
        system_prompt = "\n".join([
            "\n".join(self.system_prompt['role']).format(master=self.master, 
                                                         contact=receiver),
            "\n".join(self.system_prompt['chat_history']).format(master=self.master,
                                                                 contact=receiver,
                                                                 current_chat_history=current_chat_history,
                                                                 other_chat_history=other_chat_history),
            "\n".join(self.system_prompt['task']).format(contact=receiver, 
                                                         task=self.task),
            "\n".join(self.system_prompt['agent_chat_history']).format(contact=receiver, 
                                                                       agent_chat_history="\n".join(communication_history), 
                                                                       master=self.master),
            "\n".join(self.system_prompt['return_format']),
            str(self.json_return_format),
        ])

        return system_prompt

    def _query(self, receiver, communication_history) -> str:
        """send the query to backend llm

        Args:
            receiver (str): the name of user in current chatting 
            communication_history (list[str]): the chat history between two agents

        Returns:
            str: llm response
        """
        query_str = self.assemble_prompt(receiver, communication_history)
        response = self.query_func(query_str)
        AvatarLogger.log(query_str, response, "Query to generate message from {} to {}".format(self.master, receiver))
        return response

    def query(self, receiver, communication_history) -> str:
        """Wrap on the _query method to ensure the return format is correct

        Args:
            receiver (str): the name of user in current chatting 
            communication_history (list[str]): the chat history between two agents

        Returns:
            str: llm response in json string
        """
        raw_response = self._query(receiver, communication_history)
        response = self.json_tool.json_reformat(raw_response, self.json_return_format)
        AvatarLogger.log(instruction="response generated from {} to {}\nthe reformatted message:\n{}".format(self.master, receiver, response))
        return response

    def conclusion(self, communication_history) -> str:
        """summarize the agents' communication and give the final answer to the task

        Args:
            communication_history (list[str]): the chat history between two agents

        Returns:
            str: final answer to the task
        """
        query_str = "\n".join(self.tool_prompt['conclusion']).format(agent_communication="\n".join(communication_history), 
                                                                     task=self.task)
        response = self.query_func(query_str)
        AvatarLogger.log(query_str, response, "【Conclusion】")
        return response

    def get_friends(self) -> str:
        """used in OnlineMultiCommunication
        get all friends of one user

        Returns:
            str: friend list
        """        
        result_str = "\n"
        sql_execute_results = self.sql_tool.get_friends(self.master)
        for message in sql_execute_results:
            result_str += f"{message[0]}\n"
        return result_str


class OnlineAgent(Agent):

    def __init__(self, master, backend, task, is_assistant=False) -> None:
        """the agent used in the online chatting demo
        It get the context from chatting history by sql.
        """
        super().__init__(master, backend, task, is_assistant)

    def get_other_chat_history(self, receiver, communication_history) -> str:
        result_str = "\n"
        sql_execute_results = self.sql_tool.get_other_chat_history(self.master, receiver)
        for message in sql_execute_results:
            result_str += f"from {message[1]} to {message[2]}: {message[3]}\n"
        return result_str

    def get_current_chat_history(self, receiver, communication_history) -> str:
        result_str = "\n"
        sql_execute_results = self.sql_tool.get_current_chat_history(self.master, receiver)
        for message in sql_execute_results:
            result_str += f"from {message[1]} to {message[2]}: {message[3]}\n"
        return result_str
    
        # # WARNING: cancel current chat history for experiments
        # return ""


class OnlineThinkAgent(OnlineAgent):
    """the agent used in online chatting demo, with mindpin mechanism activated
    """

    def __init__(self, master, backend, task, is_assistant=False) -> None:
        super().__init__(master, backend, task, is_assistant)
        self.mindpin = None
        self.mindpin_status = 0 # 0 for init; 1 for mark; 2 for update

    def assemble_prompt_think(self, receiver, communication_history) -> str:
        """assemble the prompt for initializing/updating the mindpin(mindmap with pinned facts)

        Args:
            receiver (str): the name of user in current chatting 
            communication_history (list[str]): the chat history between two agents

        Returns:
            str: assembled query prompt
        """
        # init the mindpin
        if self.mindpin_status == 0:
            prompt_mindpin = "\n".join(self.tool_prompt['mindpin_init'])
            system_prompt = "\n".join([
                "\n".join(self.system_prompt['role']).format(master=self.master, 
                                                             contact=receiver),
                "\n".join(self.system_prompt['task']).format(contact=receiver,
                                                             task=self.task), 
                prompt_mindpin
            ])
            self.mindpin_status += 1
        # mark the mindpin
        elif self.mindpin_status == 1:
            prompt_mindpin = "\n".join(self.tool_prompt['mindpin_mark'])
            system_prompt = "\n".join([
                "\n".join(self.system_prompt['role']).format(master=self.master, 
                                                             contact=receiver),
                prompt_mindpin.format(task=self.task, mindpin=self.mindpin)
            ])
            self.mindpin_status += 1
        # update the mindpin
        else:
            prompt_mindpin = "\n".join(self.tool_prompt['mindpin_update']).format(mindpin=self.mindpin, 
                                                                                  known_facts=self.mindfill_tool.get_known_facts(), 
                                                                                  unknown_facts=self.mindfill_tool.get_unknown_facts())
            system_prompt = "\n".join([
                "\n".join(self.system_prompt['role']).format(master=self.master, 
                                                             contact=receiver),
                "\n".join(self.system_prompt['task']).format(contact=receiver, 
                                                             task=self.task),
                "\n".join(self.system_prompt['agent_chat_history']).format(contact=receiver, 
                                                                           agent_chat_history="\n".join(communication_history),
                                                                           master=self.master), 
                prompt_mindpin
            ])

        return system_prompt

    def assemble_prompt(self, receiver, communication_history, current_chat_history, other_chat_history) -> str:
        # add the mindpin to the query prompt
        system_prompt = "\n".join([
            "\n".join(self.system_prompt['role']).format(master=self.master, 
                                                         contact=receiver),
            "\n".join(self.system_prompt['chat_history']).format(master=self.master,
                                                                 contact=receiver,
                                                                 current_chat_history=current_chat_history,
                                                                 other_chat_history=other_chat_history),
            "\n".join(self.system_prompt['task']).format(contact=receiver, 
                                                         task=self.task),
            "\n".join(self.system_prompt['agent_chat_history']).format(contact=receiver, 
                                                                       agent_chat_history="\n".join(communication_history), 
                                                                       master=self.master),
            "\n".join(self.system_prompt['return_format_withmindpin']).format(mindpin=self.mindpin, 
                                                                              unknown_facts=self.mindfill_tool.get_unknown_facts()),
            str(self.json_return_format),
        ])

        return system_prompt

    def _query(self, receiver, communication_history) -> str:
        """first update/init the mindpin, 
        then incorporate the mindpin in the final query prompt

        Args:
            receiver (str): the name of user in current chatting 
            communication_history (list[str]): the chat history between two agents

        Returns:
            str: llm response
        """
        current_chat_history = self.get_current_chat_history(receiver, communication_history)
        other_chat_history = self.get_other_chat_history(receiver, communication_history)


        # init and update the mindpin in the first utterance
        if self.mindpin_status < 2:
            query_think = self.assemble_prompt_think(receiver, communication_history)
            self.mindpin = self.query_func(query_think)
            AvatarLogger.log(query_think, self.mindpin, "【Init mindpin from {} to {}:】".format(self.master, receiver))
            
            # mark the mindpin (invoke assemble_prompt_think again, but self.mindpin_status changed)
            query_think = self.assemble_prompt_think(receiver, communication_history)
            self.mindpin = self.query_func(query_think)
            AvatarLogger.log(query_think, self.mindpin,  "【Mark mindpin from {} to {}:】".format(self.master, receiver))

            # list all unknown facts in the mindpin
            self.mindfill_tool.set_unknown_facts(self.mindpin)

        # update the mindpin
        else:
            query_think = self.assemble_prompt_think(receiver, communication_history)
            updated_facts = self.query_func(query_think)
            AvatarLogger.log(query_think, updated_facts, "【Updated facts from {} to {}:】".format(self.master, receiver))
            self.mindpin = self.mindfill_tool.fill_mind(self.mindpin, updated_facts)
            
        # get the final query prompt    
        query_action = self.assemble_prompt(receiver, 
                                            communication_history, 
                                            current_chat_history, 
                                            other_chat_history)
        response = self.query_func(query_action)
        AvatarLogger.log(query_action, response, "【Query to generate message from {} to {}】".format(self.master, receiver))
        return response


class OnlineMemoryAgent(OnlineThinkAgent):

    def __init__(self, 
                 master, 
                 backend, 
                 task, 
                 is_assistant=False, 
                 enable_distinct_memory=True, 
                 enable_fuzzy_memory=False,
                 memory_type="") -> None:
        """the agent used in the online chatting demo
        It decide and generate sql/faiss command to retrieve chat history.
        """
        super().__init__(master, backend, task, is_assistant)
        self.previous_sql_result = "None"
        self.previous_sql_params = "None"
        self.previous_sql_result_cur = "None"
        self.previous_sql_params_cur = "None"
        self.previous_faiss_params = "None"
        self.previous_faiss_result = "None"
        self.enable_distinct_memory = enable_distinct_memory
        self.enable_fuzzy_memory = enable_fuzzy_memory
        if "anonymous" in global_config.get("mysql").get("database"):
            memory_type = "summary_by_scene_v3_anonymous"
        else:
            memory_type = "summary_by_scene_v3"
        self.memory_type = memory_type
        assert self.enable_distinct_memory or self.enable_fuzzy_memory, "For OnlineMemoryAgent, either distinct memory or fuzzy memory should be enabled"
        self.memory_file_path = os.path.join(project_path, "memory", self.memory_type, master + ".tsv")
        self.faiss_tool = FaissTool(self.memory_file_path)
        self.stopwords = set()
        with open(os.path.join(project_path, "avatarchat", "stopwords.txt"), "r") as f:
            for line in f:
                self.stopwords.add(line.strip())
    
    def set_master(self, master):
        self.master = master
        self.memory_file_path = os.path.join(project_path, "memory", self.memory_type, master + ".tsv")
        self.faiss_tool = FaissTool(self.memory_file_path)


    def get_current_chat_history(self, receiver, communication_history) -> str:
        result_str = "\nRetrieved Distinct Memory:\n"

        # add distinct memory to here
        if self.enable_distinct_memory:
            response_json_format = {
                "keyword":"ring/alice/steal",
                "window":3,
                "limit":10
            }  
            system_prompt = "\n".join(["\n".join(self.system_prompt['role']).format(master=self.master, 
                                                                                    contact=receiver),
                                        "\n".join(self.system_prompt['task']).format(contact=receiver, 
                                                                                     task=self.task)])
            query_prompt = system_prompt + "\n".join(self.tool_prompt['sql_react']).format(example_json=str(response_json_format),
                                                                                           previous_params=self.previous_sql_params_cur,
                                                                                           previous_sql_result=self.previous_sql_result_cur,
                                                                                           agent_communication="\n".join(communication_history)) 
            response = self.query_func(query_prompt)
            AvatarLogger.log(query_prompt, response, "【generate sql query by {}:】".format(self.master))
            response_json = self.json_tool.json_reformat(response, response_json_format)
            response_json = eval(response_json)
            sql_keywords = set(re.split("/| |'|\"", response_json['keyword'].lower())) - self.stopwords
            AvatarLogger.log(instruction="【SQL Keywords Set:】 {}".format(str(sql_keywords)))
            distinct_memories = []
            for keyword in sql_keywords:
                sql_execute_results = self.sql_tool.get_context_bykeyword_current(keyword,
                                                                                self.master,
                                                                                receiver,
                                                                                response_json['limit'],
                                                                                response_json['window']) 
                distinct_memories += sql_execute_results
            # TODO: Total Max Limit
            for message in distinct_memories[:50]:
                result_str += f"from {message[2]} to {message[3]}: {message[4]}\n"
            self.previous_sql_result_cur = result_str
            self.previous_sql_params_cur = str(response_json)
            AvatarLogger.log(instruction="【Distinct Memory (with current contact) Retrieved results of {}:】 \n{}".format(self.master, result_str))

        return result_str
    
    def get_other_chat_history(self, receiver, communication_history) -> str:
        result_str = "\n"

        # add distinct memory to here
        if self.enable_distinct_memory:
            result_str += "Retrieved Distinct Memory:\n"
            response_json_format = {
                "keyword":"ring/alice/steal",
                "window":3,
                "limit":10
            }  
            system_prompt = "\n".join(["\n".join(self.system_prompt['role']).format(master=self.master, 
                                                                                    contact=receiver),
                                        "\n".join(self.system_prompt['task']).format(contact=receiver, 
                                                                                     task=self.task)])
            query_prompt = system_prompt + "\n".join(self.tool_prompt['sql_react']).format(example_json=str(response_json_format),
                                                                                           previous_params=self.previous_sql_params,
                                                                                           previous_sql_result=self.previous_sql_result,
                                                                                           agent_communication="\n".join(communication_history)) 
            response = self.query_func(query_prompt)
            AvatarLogger.log(query_prompt, response, "【sql query prompt to {}:】".format(self.master))
            response_json = self.json_tool.json_reformat(response, response_json_format)
            response_json = eval(response_json)
            sql_keywords = set(re.split("/| |'|\"", response_json['keyword'].lower())) - self.stopwords
            AvatarLogger.log(instruction="【SQL Keywords Set:】 {}".format(str(sql_keywords)))
            distinct_memories = []
            for keyword in sql_keywords:
                sql_execute_results = self.sql_tool.get_context_bykeyword(keyword,
                                                                          self.master,
                                                                          receiver,
                                                                          response_json['limit'],
                                                                          response_json['window']) 
                distinct_memories += sql_execute_results
            # TODO: Total Max Limit
            for message in distinct_memories[:50]:
                result_str += f"from {message[2]} to {message[3]}: {message[4]}\n"
            self.previous_sql_result = result_str
            self.previous_sql_params = str(response_json)
            AvatarLogger.log(instruction="【Distinct Memory Retrieved results of {}:】 \n{}".format(self.master, result_str))
        
        # add fuzzy memory to here
        if self.enable_fuzzy_memory:
            response_json_format = {
                "query": "{}".format(self.task),
                "topk": 3
            }
            system_prompt = "\n".join(["\n".join(self.system_prompt['role']).format(master=self.master, 
                                                                                    contact=receiver),
                            "\n".join(self.system_prompt['task']).format(contact=receiver, 
                                                                         task=self.task)])
            query_prompt = system_prompt + "\n".join(self.tool_prompt['faiss_react']).format(example_json=str(response_json_format),
                                                                                             task=self.task,
                                                                                             previous_params=self.previous_faiss_params,
                                                                                             previous_faiss_result=self.previous_faiss_result,
                                                                                             agent_communication="\n".join(communication_history)) 
            response = self.query_func(query_prompt)
            AvatarLogger.log(query_prompt, response, "【faiss query prompt to {}:】".format(self.master))
            response_json = self.json_tool.json_reformat(response, response_json_format)
            response_json = eval(response_json)
            query = response_json['query']
            topk = response_json['topk']
            ret_dis, ret_indices, ret_text = self.faiss_tool.query(query, topk)
            result_str += "Retrieved Fuzzy Memory :\n{}".format("\n".join(ret_text))
            AvatarLogger.log(instruction="【Fuzzy Memory Retrieved results of {}:】 \n{}".format(self.master, "\n".join(ret_text)))
            self.previous_faiss_params = str(response_json)
            self.previous_faiss_result = "\n".join(ret_text)

        return result_str


class OnlineMemoryVanillaAgent(OnlineAgent):

    def __init__(self, 
                 master, 
                 backend, 
                 task, 
                 is_assistant=False, 
                 enable_distinct_memory=True, 
                 enable_fuzzy_memory=False,
                 memory_type="") -> None:
        """the agent used in the online chatting demo
        It decide and generate sql/faiss command to retrieve chat history.
        The reason part does not use MindPin, so it inheritate OnlineAgent instead of OnlineThinkAgent
        """
        super().__init__(master, backend, task, is_assistant)
        self.previous_sql_result = "None"
        self.previous_sql_params = "None"
        self.previous_sql_result_cur = "None"
        self.previous_sql_params_cur = "None"
        self.previous_faiss_params = "None"
        self.previous_faiss_result = "None"
        self.enable_distinct_memory = enable_distinct_memory
        self.enable_fuzzy_memory = enable_fuzzy_memory
        if "anonymous" in global_config.get("mysql").get("database"):
            memory_type = "summary_by_scene_v3_anonymous"
        else:
            memory_type = "summary_by_scene_v3"
        self.memory_type = memory_type
        assert self.enable_distinct_memory or self.enable_fuzzy_memory, "For OnlineMemoryAgent, either distinct memory or fuzzy memory should be enabled"
        self.memory_file_path = os.path.join(project_path, "memory", self.memory_type, master + ".tsv")
        self.faiss_tool = FaissTool(self.memory_file_path)
        self.stopwords = set()
        with open(os.path.join(project_path, "avatarchat", "stopwords.txt"), "r") as f:
            for line in f:
                self.stopwords.add(line.strip())
    
    def set_master(self, master):
        self.master = master
        self.memory_file_path = os.path.join(project_path, "memory", self.memory_type, master + ".tsv")
        self.faiss_tool = FaissTool(self.memory_file_path)


    def get_current_chat_history(self, receiver, communication_history) -> str:
        result_str = "\nRetrieved Distinct Memory:\n"

        # add distinct memory to here
        if self.enable_distinct_memory:
            response_json_format = {
                "keyword":"ring/alice/steal",
                "window":3,
                "limit":10
            }  
            system_prompt = "\n".join(["\n".join(self.system_prompt['role']).format(master=self.master, 
                                                                                    contact=receiver),
                                        "\n".join(self.system_prompt['task']).format(contact=receiver, 
                                                                                     task=self.task)])
            query_prompt = system_prompt + "\n".join(self.tool_prompt['sql_react']).format(example_json=str(response_json_format),
                                                                                           previous_params=self.previous_sql_params_cur,
                                                                                           previous_sql_result=self.previous_sql_result_cur,
                                                                                           agent_communication="\n".join(communication_history)) 
            response = self.query_func(query_prompt)
            AvatarLogger.log(query_prompt, response, "【generate sql query by {}:】".format(self.master))
            response_json = self.json_tool.json_reformat(response, response_json_format)
            response_json = eval(response_json)
            sql_keywords = set(re.split("/| |'|\"", response_json['keyword'].lower())) - self.stopwords
            AvatarLogger.log(instruction="【SQL Keywords Set:】 {}".format(str(sql_keywords)))
            distinct_memories = []
            for keyword in sql_keywords:
                sql_execute_results = self.sql_tool.get_context_bykeyword_current(keyword,
                                                                                self.master,
                                                                                receiver,
                                                                                response_json['limit'],
                                                                                response_json['window']) 
                distinct_memories += sql_execute_results
            # TODO: Total Max Limit
            for message in distinct_memories[:50]:
                result_str += f"from {message[2]} to {message[3]}: {message[4]}\n"
            self.previous_sql_result_cur = result_str
            self.previous_sql_params_cur = str(response_json)
            AvatarLogger.log(instruction="【Distinct Memory (with current contact) Retrieved results of {}:】 \n{}".format(self.master, result_str))

        return result_str
    
    def get_other_chat_history(self, receiver, communication_history) -> str:
        result_str = "\n"

        # add distinct memory to here
        if self.enable_distinct_memory:
            result_str += "Retrieved Distinct Memory:\n"
            response_json_format = {
                "keyword":"ring/alice/steal",
                "window":3,
                "limit":10
            }  
            system_prompt = "\n".join(["\n".join(self.system_prompt['role']).format(master=self.master, 
                                                                                    contact=receiver),
                                        "\n".join(self.system_prompt['task']).format(contact=receiver, 
                                                                                     task=self.task)])
            query_prompt = system_prompt + "\n".join(self.tool_prompt['sql_react']).format(example_json=str(response_json_format),
                                                                                           previous_params=self.previous_sql_params,
                                                                                           previous_sql_result=self.previous_sql_result,
                                                                                           agent_communication="\n".join(communication_history)) 
            response = self.query_func(query_prompt)
            AvatarLogger.log(query_prompt, response, "【sql query prompt to {}:】".format(self.master))
            response_json = self.json_tool.json_reformat(response, response_json_format)
            response_json = eval(response_json)
            sql_keywords = set(re.split("/| |'|\"", response_json['keyword'].lower())) - self.stopwords
            AvatarLogger.log(instruction="【SQL Keywords Set:】 {}".format(str(sql_keywords)))
            distinct_memories = []
            for keyword in sql_keywords:
                sql_execute_results = self.sql_tool.get_context_bykeyword(keyword,
                                                                          self.master,
                                                                          receiver,
                                                                          response_json['limit'],
                                                                          response_json['window']) 
                distinct_memories += sql_execute_results
            # TODO: Total Max Limit
            for message in distinct_memories[:50]:
                result_str += f"from {message[2]} to {message[3]}: {message[4]}\n"
            self.previous_sql_result = result_str
            self.previous_sql_params = str(response_json)
            AvatarLogger.log(instruction="【Distinct Memory Retrieved results of {}:】 \n{}".format(self.master, result_str))
        
        # add fuzzy memory to here
        if self.enable_fuzzy_memory:
            response_json_format = {
                "query": "{}".format(self.task),
                "topk": 3
            }
            system_prompt = "\n".join(["\n".join(self.system_prompt['role']).format(master=self.master, 
                                                                                    contact=receiver),
                            "\n".join(self.system_prompt['task']).format(contact=receiver, 
                                                                         task=self.task)])
            query_prompt = system_prompt + "\n".join(self.tool_prompt['faiss_react']).format(example_json=str(response_json_format),
                                                                                             task=self.task,
                                                                                             previous_params=self.previous_faiss_params,
                                                                                             previous_faiss_result=self.previous_faiss_result,
                                                                                             agent_communication="\n".join(communication_history)) 
            response = self.query_func(query_prompt)
            AvatarLogger.log(query_prompt, response, "【faiss query prompt to {}:】".format(self.master))
            response_json = self.json_tool.json_reformat(response, response_json_format)
            response_json = eval(response_json)
            query = response_json['query']
            topk = response_json['topk']
            ret_dis, ret_indices, ret_text = self.faiss_tool.query(query, topk)
            result_str += "Retrieved Fuzzy Memory :\n{}".format("\n".join(ret_text))
            AvatarLogger.log(instruction="【Fuzzy Memory Retrieved results of {}:】 \n{}".format(self.master, "\n".join(ret_text)))
            self.previous_faiss_params = str(response_json)
            self.previous_faiss_results = "\n".join(ret_text)

        return result_str