"""
Copyright (c) ANONYMOUS
All rights reserved.

MIT License

Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the “Software”), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
"""

import haiku as hk
import jax
import jax.numpy as jnp


class ActionValueTabular(hk.Module):
    def __init__(self, mdp, name="action_value_tabular"):
        super().__init__(name=name)
        self.mdp = mdp

    def __call__(self, x):
        action_value = hk.get_parameter(
            "action_value", [self.mdp.num_state, self.mdp.num_actions], init=jnp.zeros
        )
        state = self.mdp.observation_to_state(x)
        return state @ action_value


class HindsightTabular(hk.Module):
    def __init__(self, mdp, name="hindsight_tabular", reward_based=False, reward_values=None):
        super().__init__(name=name)
        self.mdp = mdp
        self.reward_based = reward_based
        self.reward_values = reward_values

    def __call__(self, current_observation, future_observation):
        contribution = hk.get_parameter(
            "contribution",
            [
                self.mdp.num_state,
                self.mdp.num_state if not self.reward_based else len(self.reward_values),
                self.mdp.num_actions,
            ],
            init=jnp.zeros,
        )

        current_state = self.mdp.observation_to_state(current_observation)
        if not self.reward_based:
            future_state = self.mdp.observation_to_state(future_observation)
        else:
            future_state = future_observation

        return current_state @ jax.vmap(lambda a, b: a @ b, in_axes=(None, 0))(
            future_state, contribution
        )


class PolicyTabular(hk.Module):
    def __init__(self, mdp, name="policy_tabular"):
        super().__init__(name=name)
        self.mdp = mdp

    def __call__(self, x):
        logit = hk.get_parameter(
            "logit", [self.mdp.num_state, self.mdp.num_actions], init=jnp.zeros
        )
        state = jax.vmap(self.mdp.observation_to_state)(x)
        return state @ logit


class ValueTabular(hk.Module):
    def __init__(self, mdp, name="value_tabular"):
        super().__init__(name=name)
        self.mdp = mdp

    def __call__(self, x):
        value = hk.get_parameter("value", [self.mdp.num_state, 1], init=jnp.zeros)
        state = self.mdp.observation_to_state(x)
        return state @ value
