"""
Copyright (c) ANONYMOUS
All rights reserved.

MIT License

Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the “Software”), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
"""
import chex
import flax
import jax
import jax.numpy as jnp
import jax.tree_util as jtu
from typing import Dict

@flax.struct.dataclass
class ReplayBufferState:
    trajectories: chex.PyTreeDef
    next_slot: int
    full: bool


class ReplayBuffer:
    """
    Replay buffer that stores a batch of trajectories and allows to sample from them.
    """

    def __init__(self, max_size):
        self.max_size = max_size

    def reset(self, sample_trajectory: chex.PyTreeDef) -> ReplayBufferState:
        next_slot = 0
        full = False

        def batch_empty_like(x):
            return jnp.empty_like(x, shape=(self.max_size, *x.shape))

        trajectories = jtu.tree_map(batch_empty_like, sample_trajectory)

        return ReplayBufferState(trajectories, next_slot, full)

    def add(
        self, state_buffer: ReplayBufferState, trajectory_batch: chex.PyTreeDef
    ) -> ReplayBufferState:
        """
        Add a batch of trajectories to the store of trajectories and overwrite if full (FIFO).
        """
        chex.assert_equal_shape(jtu.tree_leaves(trajectory_batch), dims=0)
        batch_size = len(jtu.tree_leaves(trajectory_batch)[0])

        # If buffer is filled, start replacing values FIFO
        next_slots = jnp.mod(state_buffer.next_slot + jnp.arange(batch_size), self.max_size)

        def set_at_next_slots(x, y):
            return x.at[next_slots].set(y)

        trajectories = jtu.tree_map(set_at_next_slots, state_buffer.trajectories, trajectory_batch)

        # Check if buffer as been filled at least once
        full = jax.lax.cond(
            ((state_buffer.next_slot + batch_size) >= self.max_size),
            lambda _: True,
            lambda _: state_buffer.full,
            None,
        )

        return ReplayBufferState(trajectories, next_slots[-1] + 1, full)

    def sample(
        self, rng: chex.PRNGKey, state_buffer: ReplayBufferState, batch_size: int
    ) -> chex.PyTreeDef:
        """
        Sample a batch of trajectories.
        """
        # Determine range of indeces to sample from
        # NOTE: it is not possible to conditionally sample without replacement in a jit-compatible way
        idx = jax.lax.cond(
            state_buffer.full,
            lambda _: jax.random.randint(rng, (batch_size,), 0, maxval=self.max_size),
            lambda _: jax.random.randint(rng, (batch_size,), 0, maxval=state_buffer.next_slot),
            None,
        )

        return jtu.tree_map(lambda x: x[idx], state_buffer.trajectories)


@flax.struct.dataclass
class MultiReplayBufferState:
    state: Dict

class MultiReplayBuffer:
    def __init__(self, max_size, reward_values):
        self.max_size = max_size
        self.reward_values = reward_values
        self.buffers=dict()
        for value in self.reward_values:
            self.buffers[value] = ReplayBuffer(max_size)

    def reset(self, sample_trajectory: chex.PyTreeDef) -> ReplayBufferState:
        sample_trajectory = jtu.tree_map(lambda x: jnp.expand_dims(x[0], axis=0), sample_trajectory)
        sample_trajectory = sample_trajectory.replace(dones = jnp.concatenate((sample_trajectory.dones, sample_trajectory.dones)))
        buffers_state = dict()
        for value in self.reward_values:
            buffers_state[value] = self.buffers[value].reset(sample_trajectory)

        return MultiReplayBufferState(buffers_state)

    def add(
        self, state_buffer: ReplayBufferState, trajectory_batch: chex.PyTreeDef
    ) -> ReplayBufferState:
        """
        Add a batch of trajectories to the store of trajectories and overwrite if full (FIFO).
        """
        chex.assert_equal_shape(jtu.tree_leaves(trajectory_batch), dims=0)
        batch_size = len(jtu.tree_leaves(trajectory_batch)[0])
        dones = trajectory_batch.dones[:, :-1]
        next_dones = trajectory_batch.dones[:, 1:]
        trajectory_batch = trajectory_batch.replace(dones = dones)
        trajectory_batch = jtu.tree_map(lambda x: jnp.reshape(x, (x.shape[0]*x.shape[1], 1, 1, *x.shape[2:])), trajectory_batch)
        next_dones = jnp.reshape(next_dones, (batch_size*next_dones.shape[1], 1, 1))
        trajectory_batch = trajectory_batch.replace(dones = jnp.concatenate([trajectory_batch.dones,next_dones], axis=2))

        def _add(curr_state, trajectory):
            next_state = dict()
            for value in self.reward_values:
                next_state[value] = jax.lax.cond(trajectory.rewards[0,0] == value,
                                     lambda _: self.buffers[value].add(curr_state[value], trajectory),
                                     lambda _: curr_state[value], None)
            return next_state, None
        next_state_dict, _ = jax.lax.scan(_add, state_buffer.state, trajectory_batch)

        return MultiReplayBufferState(next_state_dict)

    def sample(
        self, rng: chex.PRNGKey, state_buffer: ReplayBufferState, batch_size: int
    ) -> chex.PyTreeDef:
        """
        Sample a batch of trajectories.
        """
        batches = None
        for value in self.reward_values:
            curr_rng, rng = jax.random.split(rng)
            next_batch = (self.buffers[value].sample(curr_rng, state_buffer.state[value], batch_size // len(self.reward_values)))
            if batches is None:
                batches = next_batch
            else:
                batches = jtu.tree_map(lambda x,y : jnp.concatenate([x,y], axis=0), batches, next_batch)

        return batches