"""Modified from https://github.com/facebookresearch/DiT/blob/main/models.py"""
# Copyright (c) Meta Platforms, Inc. and affiliates.
# All rights reserved.

# This source code is licensed under the license found in the
# LICENSE file in the root directory of this source tree.
# --------------------------------------------------------
# References:
# GLIDE: https://github.com/openai/glide-text2im
# MAE: https://github.com/facebookresearch/mae/blob/main/models_mae.py
# --------------------------------------------------------

import torch
import torch.nn as nn
import numpy as np
import math
from timm.models.vision_transformer import  Attention, Mlp
from .configs import GINRDiTConfig


def modulate(x, shift, scale):
    return x * (1 + scale.unsqueeze(1)) + shift.unsqueeze(1)


#################################################################################
#               Embedding Layers for Timesteps and Class Labels                 #
#################################################################################

class TimestepEmbedder(nn.Module):
    """
    Embeds scalar timesteps into vector representations.
    """
    def __init__(self, hidden_size, frequency_embedding_size=256):
        super().__init__()
        self.mlp = nn.Sequential(
            nn.Linear(frequency_embedding_size, hidden_size, bias=True),
            nn.SiLU(),
            nn.Linear(hidden_size, hidden_size, bias=True),
        )
        self.frequency_embedding_size = frequency_embedding_size

    @staticmethod
    def timestep_embedding(t, dim, max_period=10000):
        """
        Create sinusoidal timestep embeddings.
        :param t: a 1-D Tensor of N indices, one per batch element.
                          These may be fractional.
        :param dim: the dimension of the output.
        :param max_period: controls the minimum frequency of the embeddings.
        :return: an (N, D) Tensor of positional embeddings.
        """
        # https://github.com/openai/glide-text2im/blob/main/glide_text2im/nn.py
        half = dim // 2
        freqs = torch.exp(
            -math.log(max_period) * torch.arange(start=0, end=half, dtype=torch.float32) / half
        ).to(device=t.device)
        args = t[:, None].float() * freqs[None]
        embedding = torch.cat([torch.cos(args), torch.sin(args)], dim=-1)
        if dim % 2:
            embedding = torch.cat([embedding, torch.zeros_like(embedding[:, :1])], dim=-1)
        return embedding

    def forward(self, t):
        t_freq = self.timestep_embedding(t, self.frequency_embedding_size)
        t_emb = self.mlp(t_freq)
        return t_emb


class LabelEmbedder(nn.Module):
    """
    Embeds class labels into vector representations. Also handles label dropout for classifier-free guidance.
    """
    def __init__(self, num_classes, hidden_size, dropout_prob):
        super().__init__()
        use_cfg_embedding = dropout_prob > 0
        self.embedding_table = nn.Embedding(num_classes + use_cfg_embedding, hidden_size)
        self.num_classes = num_classes
        self.dropout_prob = dropout_prob

    def token_drop(self, labels, force_drop_ids=None):
        """
        Drops labels to enable classifier-free guidance.
        """
        if force_drop_ids is None:
            drop_ids = torch.rand(labels.shape[0], device=labels.device) < self.dropout_prob
        else:
            drop_ids = force_drop_ids == 1
        labels = torch.where(drop_ids, self.num_classes, labels) # 마지막 idx가 unconditional 셋팅
        return labels

    def forward(self, labels, train, force_drop_ids=None):
        use_dropout = self.dropout_prob > 0
        if (train and use_dropout) or (force_drop_ids is not None):
            labels = self.token_drop(labels, force_drop_ids)
        embeddings = self.embedding_table(labels)
        return embeddings


#################################################################################
#                                 Core DiT Model                                #
#################################################################################

class GINRDiTBlock(nn.Module):
    """
    A DiT block with adaptive layer norm zero (adaLN-Zero) conditioning.
    """
    def __init__(self, hidden_size, num_heads, mlp_ratio=4.0, **block_kwargs):
        super().__init__()
        self.norm1 = nn.LayerNorm(hidden_size, elementwise_affine=False, eps=1e-6)
        self.attn = Attention(hidden_size, num_heads=num_heads, qkv_bias=True, **block_kwargs)
        self.norm2 = nn.LayerNorm(hidden_size, elementwise_affine=False, eps=1e-6)
        mlp_hidden_dim = int(hidden_size * mlp_ratio)
        approx_gelu = lambda: nn.GELU(approximate="tanh")
        self.mlp = Mlp(in_features=hidden_size, hidden_features=mlp_hidden_dim, act_layer=approx_gelu, drop=0)
        self.adaLN_modulation = nn.Sequential(
            nn.SiLU(),
            nn.Linear(hidden_size, 6 * hidden_size, bias=True)
        )

    def forward(self, x, c):
        shift_msa, scale_msa, gate_msa, shift_mlp, scale_mlp, gate_mlp = self.adaLN_modulation(c).chunk(6, dim=1)
        x = x + gate_msa.unsqueeze(1) * self.attn(modulate(self.norm1(x), shift_msa, scale_msa))
        x = x + gate_mlp.unsqueeze(1) * self.mlp(modulate(self.norm2(x), shift_mlp, scale_mlp))
        return x


class FinalLayer(nn.Module):
    """
    The final layer of GINRDiT.
    """
    def __init__(self, hidden_size, out_channels):
        super().__init__()
        self.norm_final = nn.LayerNorm(hidden_size, elementwise_affine=False, eps=1e-6)
        self.linear = nn.Linear(hidden_size, out_channels, bias=True)
        self.adaLN_modulation = nn.Sequential(
            nn.SiLU(),
            nn.Linear(hidden_size, 2 * hidden_size, bias=True)
        )

    def forward(self, x, c):
        shift, scale = self.adaLN_modulation(c).chunk(2, dim=1)
        x = modulate(self.norm_final(x), shift, scale)
        x = self.linear(x)
        return x


class GINRDiT(nn.Module):
    """
    Diffusion model with a Transformer backbone.
    """
    Config = GINRDiTConfig
    def __init__(self, config: GINRDiTConfig):
        super().__init__()
        self.config = config = config.copy()
        self.type = config.type
        self.input_size = input_size = config.input_size
        self.in_channels = in_channels = config.in_channels
        self.hidden_size = hidden_size = config.hidden_size
        self.depth = depth = config.depth
        self.num_heads = num_heads = config.num_heads
        self.mlp_ratio = mlp_ratio = config.mlp_ratio
        self.class_dropout_prob = class_dropout_prob = config.class_dropout_prob
        self.num_classes = num_classes = config.num_classes
        self.learn_sigma = learn_sigma = config.learn_sigma

        self.out_channels = out_channels = in_channels * 2 if learn_sigma else in_channels

        self.x_embedder = nn.Linear(in_channels, hidden_size, bias=True)
        self.t_embedder = TimestepEmbedder(hidden_size)
        self.y_embedder = LabelEmbedder(num_classes, hidden_size, class_dropout_prob)
        
        # the positional embeddings are trainable for transformer inputs
        self.pos_embed = nn.Parameter(torch.zeros(1, input_size, hidden_size))

        self.blocks = nn.ModuleList([
            GINRDiTBlock(hidden_size, num_heads, mlp_ratio=mlp_ratio) for _ in range(depth)
        ])
        self.final_layer = FinalLayer(hidden_size, out_channels)
        self.initialize_weights()

    def initialize_weights(self):
        # Initialize transformer layers:
        def _basic_init(module):
            if isinstance(module, nn.Linear):
                torch.nn.init.xavier_uniform_(module.weight)
                if module.bias is not None:
                    nn.init.constant_(module.bias, 0)
        self.apply(_basic_init)

        # Initialize trainable pos_embed by normal
        self.pos_embed.data.normal_(mean=0.0, std=0.02)

        # Initialize patch_embed like nn.Linear (instead of nn.Conv2d):
        nn.init.zeros_(self.x_embedder.bias)

        # Initialize label embedding table:
        nn.init.normal_(self.y_embedder.embedding_table.weight, std=0.02)

        # Initialize timestep embedding MLP:
        nn.init.normal_(self.t_embedder.mlp[0].weight, std=0.02)
        nn.init.normal_(self.t_embedder.mlp[2].weight, std=0.02)

        # Zero-out adaLN modulation layers in GINRDiT blocks:
        for block in self.blocks:
            nn.init.constant_(block.adaLN_modulation[-1].weight, 0)
            nn.init.constant_(block.adaLN_modulation[-1].bias, 0)

        # Zero-out output layers:
        nn.init.constant_(self.final_layer.adaLN_modulation[-1].weight, 0)
        nn.init.constant_(self.final_layer.adaLN_modulation[-1].bias, 0)
        nn.init.constant_(self.final_layer.linear.weight, 0)
        nn.init.constant_(self.final_layer.linear.bias, 0)


    def forward(self, x, t, y, convert_channel_first=True):
        """
        Forward pass
        x: (N, D, T) tensor of latent latent representations for INRs
        t: (N,) tensor of diffusion timesteps
        y: (N,) tensor of class labels
        """
        x = torch.permute(x, (0, 2, 1)) # (N, D, T) -> (N, T, D)
        x = self.x_embedder(x) + self.pos_embed 
        t = self.t_embedder(t)                   # (N, C)
        y = self.y_embedder(y, self.training)    # (N, C)
        c = t + y                                # (N, C)
        for block in self.blocks:
            x = block(x, c)                      # (N, T, C)
        x = self.final_layer(x, c)                # (N, T, D)
        
        x = torch.permute(x, (0, 2, 1)) # (N, D, T)
        return x

    def forward_with_cfg(self, x, t, y, cfg_scale):
        """
        Forward pass of DiT, but also batches the unconditional forward pass for classifier-free guidance.
        # https://github.com/openai/glide-text2im/blob/main/notebooks/text2im.ipynb
        """
        half = x[: len(x) // 2]
        combined = torch.cat([half, half], dim=0)
        model_out = self.forward(combined, t, y)
        
        eps, rest = model_out[:, :self.in_channels], model_out[:, self.in_channels:]
        cond_eps, uncond_eps = torch.split(eps, len(eps) // 2, dim=0)
        
        half_eps = uncond_eps + cfg_scale * (cond_eps - uncond_eps)
        eps = torch.cat([half_eps, half_eps], dim=0)
        return torch.cat([eps, rest], dim=1)



"""
#################################################################################
############################## Possible Model Specs #############################
#################################################################################
def GINRDiT_XL_2(**kwargs):
    return GINRDiT(depth=28, hidden_size=1152, num_heads=16, **kwargs)

def GINRDiT_XL_4(**kwargs):
    return GINRDiT(depth=28, hidden_size=1152, num_heads=16, **kwargs)

def GINRDiT_XL_8(**kwargs):
    return GINRDiT(depth=28, hidden_size=1152, num_heads=16, **kwargs)

def GINRDiT_L_2(**kwargs):
    return GINRDiT(depth=24, hidden_size=1024, num_heads=16, **kwargs)

def GINRDiT_L_4(**kwargs):
    return GINRDiT(depth=24, hidden_size=1024, num_heads=16, **kwargs)

def GINRDiT_L_8(**kwargs):
    return GINRDiT(depth=24, hidden_size=1024, num_heads=16, **kwargs)

def GINRDiT_B_2(**kwargs):
    return GINRDiT(depth=12, hidden_size=768, num_heads=12, **kwargs)

def GINRDiT_B_4(**kwargs):
    return GINRDiT(depth=12, hidden_size=768, num_heads=12, **kwargs)

def GINRDiT_B_8(**kwargs):
    return GINRDiT(depth=12, hidden_size=768, num_heads=12, **kwargs)

def GINRDiT_S_2(**kwargs):
    return GINRDiT(depth=12, hidden_size=384, num_heads=6, **kwargs)

def GINRDiT_S_4(**kwargs):
    return GINRDiT(depth=12, hidden_size=384, num_heads=6, **kwargs)

def GINRDiT_S_8(**kwargs):
    return GINRDiT(depth=12, hidden_size=384, num_heads=6, **kwargs)

"""