
% Depth
L = 1;

% Power law exponents (length L+1)
b0 = 1;
b1 = 1;

n0 = 1000;

% Label noise
eta = 0.5;

% Number of realizations
numR = 100;


%% Set plot points

g0 = (0.05:0.05:2)';

g1 = (0.5:0.25:2);


%% Set color scheme

% Check for custom colormaps
if exist('viridis.m', 'file') == 2
    corder = magma(length(g1));
else
    warning('Custom colormaps not found; using MATLAB default.')
    cmp = parula(2^16);
end

%% Set up parallel pool

poolObj = gcp('nocreate');
if isempty(poolObj)
    poolObj = parpool('local');
end

%% Run numerical simulations

tAll = tic;
genErrEx = nan(length(g0),length(g1),numR);

for indP = 1:length(g0)
    for indN = 1:length(g1)

        p = round(n0 * g0(indP));
        n1 = round(n0 * g1(indN));

        S0 = diag(sqrt(n0.^(1+b0) ./ (1:n0).^(1+b0)));
        S1 = diag(sqrt(n1.^(1+b1) ./ (1:n1).^(1+b1)));

        tic;
        parfor indR = 1:numR


            X = randn(p, n0) * S0;
            F = randn(n0,n1) * S1 / sqrt(n1);
            w = randn(n0,1);

            y = X*w/sqrt(n0) + eta * randn(p,1);

            v = lsqminnorm(X*F/sqrt(n0), y);

            genErrEx(indP, indN, indR) = vecnorm(S0 * (F*v - w),2).^2 / n0;

        end

        fprintf('\tLoad %d of %d, width %d of %d: completed %d repeats in %f seconds.\n', indP, length(g0), indN, length(g1), numR, toc);
    end
end
fprintf('\nCompleted simulation in %f seconds.\n', toc(tAll));

%% Compute theoretical prediction

g0Th = (0.005:0.005:2)';

k0 = ((pi/(1+b0))/sin(pi/(1+b0)))^(1+b0);
k1 = ((pi/(1+b1))/sin(pi/(1+b1)))^(1+b1);

chi = @(z) ( k0 .* (z.^b0-1) + (2 + b0*(1-k0)).*(1-1./z) ) .* (z>1);

genErrTh = ((1 + b1 + 1./(g1./g0Th.*(g1>g0Th) - 1)) .* chi(1./g0Th) + eta^2 * (b0+b1+ 1./(g1./g0Th.*(g1>g0Th) - 1) + 1./(1./g0Th.*(g0Th<1)-1))) .* (g0Th < min(1,g1))...
    + (1./(1-g1./g0Th.*(g1<g0Th)) .* chi(1./g1) + eta^2*g1./g0Th./(1-g1./g0Th.*(g1<g0Th))).* (g1 < 1 & g0Th > g1) ...
    + (eta^2 * 1./(g0Th.*(g0Th ~= 1)-1)) .* (g0Th > 1 & g1 >= 1) ...
    + 1e8*(g1 < 1 & g0Th == g1) + 1e8*(g1 >= 1 & g0Th == 1 & eta>0);

%% Plot results

figure('Position',[200,500,500,700],'WindowStyle','docked');
hold on;
set(gca, 'colororder',corder);
plot(g0Th, genErrTh, 'linewidth', 2);
for ind = 1:length(g1)
    errorbar(g0, mean(genErrEx(:,ind,:),3), std(genErrEx(:,ind,:),0,3)/sqrt(numR), 'o', 'MarkerFaceColor', corder(ind,:), 'MarkerEdgeColor','none','CapSize',0,'Color', corder(ind,:),'linewidth',2);
end
axis('square');
ylim([0 100]);
xlabel('1/\alpha_{0}');
ylabel('\epsilon');
legend(cellstr(num2str(g1', '\\alpha_{1}/\\alpha_{0} = %0.2f')))
title(sprintf('L = %d, \\gamma_{0} = %0.2f, \\gamma_{1} = %0.2f, \\eta = %0.2f', L, b0, b1,eta));
set(gca, 'FontSize', 16, 'LineWidth', 2, 'Box','off','tickdir','out');
