# =================================================================================================#
# Description: Plots the coefficient function for illustrating the synthetic data
# Author: Ryan Thompson
# =================================================================================================#

import Cairo, Compose, DataFrames, Distributions, Fontconfig, Gadfly, LinearAlgebra, Pipe, 
    Random, Statistics

rng = Random.MersenneTwister(34); Random.default_rng() = rng

# Generate contextual features
z = vcat([[i j] for i in range(- 1, 1, 100), j in range(- 1, 1, 100)]...)
n = size(z, 1)

# Generate coefficients
s = rand(Distributions.Uniform(0.05, 0.15), 3)
c = rand(Distributions.Uniform(- 1, 1), 3, 2)
r = map(j -> Statistics.quantile([LinearAlgebra.norm(z[i, :] - c[j, :], 2) 
    for i in 1:n], s[j]), 1:3)
beta(z, c, r) = (1 - 0.5 * LinearAlgebra.norm(z - c, 2) / r) * (LinearAlgebra.norm(z - c, 2) <= r)
beta_mat = [beta(z[i, :], c[j, :], r[j]) for i in 1:n, j in 1:3]

# Plot coefficients
Pipe.@pipe DataFrames.DataFrame(
    z₁ = z[:, 1], 
    z₂ = z[:, 2], 
    β₁ = beta_mat[:, 1], 
    β₂ = beta_mat[:, 2], 
    β₃ = beta_mat[:, 3]
    ) |> 
    DataFrames.stack(_, 3:5) |>
    DataFrames.filter(:value => x -> x .!= 0, _) |>
    Gadfly.plot(
        _,
        x = :z₁,
        y = :z₂,
        color = :value,
        Gadfly.Geom.point,
        Gadfly.Guide.colorkey(title = ""),
        Gadfly.Guide.xlabel("z₁"),
        Gadfly.Guide.annotation(Compose.compose(Compose.context(), 
            Compose.text(0.56Gadfly.w, 0.2Gadfly.h, "β₁"))),
        Gadfly.Guide.annotation(Compose.compose(Compose.context(), 
            Compose.text(0.3Gadfly.w, 0.4Gadfly.h, "β₂"))),
        Gadfly.Guide.annotation(Compose.compose(Compose.context(), 
            Compose.text(0.15Gadfly.w, 0.7Gadfly.h, "β₃"))),
        Gadfly.Theme(highlight_width = 0Gadfly.mm, default_color = "black", 
            plot_padding = [0Gadfly.mm]),
    ) |> 
    Gadfly.PDF("Figures/demo.pdf", 5Gadfly.inch, 2Gadfly.inch)