
import pennylane as qml
from pennylane.ops import CNOT, RX, RY, RZ, CZ
from pennylane import numpy as np
import random
import numpy.linalg as la
import math
from math import pi
from datetime import datetime
import os

def f_n(weights, ansatz=None):

    return np.sum(ansatz(weights))

def gd_optimizer(ansatz, weights, noise_gamma, lr, iteration, n_check):
    grad_func = qml.grad(f_n)
    n_params = len(weights)
    grad_norm = np.zeros(iteration)
    loss = np.zeros(iteration)
    gradient_list = np.zeros((iteration, n_params))
    t1 = datetime.now()
    for j in range(iteration):
        weights.requires_grad = True
        noise = np.random.normal(0,noise_gamma, n_params)
        loss[j] = f_n(weights, ansatz=ansatz)
        gradient_list[j] = grad_func(weights, ansatz=ansatz)
        grad_norm[j] = la.norm(gradient_list[j])
        gradient = gradient_list[j] + noise
        weights = weights - lr * gradient
        if(j%n_check==0):
            t2=datetime.now()
            print( j, " loss : %.6f" % loss[j], "gradnorm: %.6f" % grad_norm[j], "noise: %.6f" % la.norm(noise), "time: ", (t2-t1).seconds)
            t1=t2
    return loss, grad_norm, weights, gradient_list

def gd_gaussian(ansatz, circuit_name, qubits, n_params, gamma, gau_rate, noise_gamma, noise_rate, lr, iteration, n_time, n_check):
    

    folder = "./"+circuit_name +"_"+ str(qubits) +"_"+ str(n_params)+"_gd_gaussian"+"_"+str(gau_rate)+"_"+str(noise_rate)
    a = np.random.normal(0, gamma*gau_rate, n_params*n_time).reshape(n_time,n_params)
    if not os.path.exists(folder):
        os.makedirs(folder)
    for time in range(n_time):
        np.save(folder+"/weights_init_"+str(time)+".npy", a[time])
        loss, grad_norm, weights, gradient = gd_optimizer(ansatz, a[time], noise_gamma*noise_rate, lr, iteration, n_check)
        np.save(folder+"/loss_"+str(time)+".npy", loss)
        np.save(folder+"/grad_norm_"+str(time)+".npy",  grad_norm)
        np.save(folder+"/weights_final_"+str(time)+".npy", weights)
        np.save(folder+"/gradient_"+str(time)+".npy", gradient)
        print("the training with gd gaussian ends", gau_rate, noise_rate)
    return 0

def gd_zero(ansatz, circuit_name, qubits, n_params, gamma, gau_rate, noise_gamma, noise_rate, lr, iteration, n_time, n_check):
    

    folder = "./"+circuit_name +"_"+ str(qubits) +"_"+ str(n_params)+"_gd_zero"+"_"+str(noise_rate)
    a = np.random.normal(0, 0, n_params*n_time).reshape(n_time,n_params)
    if not os.path.exists(folder):
        os.makedirs(folder)
    for time in range(n_time):
        np.save(folder+"/weights_init_"+str(time)+".npy", a[time])
        loss, grad_norm, weights, gradient = gd_optimizer(ansatz, a[time], noise_gamma*noise_rate, lr, iteration, n_check)
        np.save(folder+"/loss_"+str(time)+".npy", loss)
        np.save(folder+"/grad_norm_"+str(time)+".npy",  grad_norm)
        np.save(folder+"/weights_final_"+str(time)+".npy", weights)
        np.save(folder+"/gradient_"+str(time)+".npy", gradient)
        print("the training with gd zero ends", noise_rate)
    return 0

def gd_uniform(ansatz, circuit_name, qubits, n_params, gamma, gau_rate, noise_gamma, noise_rate, lr, iteration, n_time, n_check):
    

    folder = "./"+circuit_name +"_"+ str(qubits) +"_"+ str(n_params)+"_gd_uniform"+"_"+str(noise_rate)

    a = np.random.uniform(0, 2*pi, size=n_time*n_params).reshape(n_time,n_params)
    if not os.path.exists(folder):
        os.makedirs(folder)
    for time in range(n_time):
        np.save(folder+"/weights_init_"+str(time)+".npy", a[time])
        loss, grad_norm, weights, gradient = gd_optimizer(ansatz, a[time], noise_gamma*noise_rate, lr, iteration, n_check)
        np.save(folder+"/loss_"+str(time)+".npy", loss)
        np.save(folder+"/grad_norm_"+str(time)+".npy",  grad_norm)
        np.save(folder+"/weights_final_"+str(time)+".npy", weights)
        np.save(folder+"/gradient_"+str(time)+".npy", gradient)
        print("the training with gd uniform ends", noise_rate)
    return 0

def adam_optimizer(ansatz, weights, noise_gamma, lr, iteration, n_check):
    beta_1 = 0.9
    beta_2 = 0.99
    epsilon = 0.00000001
    grad_func = qml.grad(f_n)
    n_params = len(weights)
    grad_norm = np.zeros(iteration)
    loss = np.zeros(iteration)
    gradient_list = np.zeros((iteration, n_params))
    m = np.zeros(n_params)
    v = np.zeros(n_params)
    t1 = datetime.now()
    for j in range(iteration):
        weights.requires_grad = True
        noise = np.random.normal(0,noise_gamma, n_params)
        loss[j] = f_n(weights, ansatz=ansatz)
        gradient_list[j] = grad_func(weights, ansatz=ansatz)
        grad_norm[j] = la.norm(gradient_list[j])
        gradient = gradient_list[j] + noise
        m = beta_1*m + (1-beta_1)*gradient
        v = beta_2*v + (1-beta_2)*np.power(gradient,2)
        m_hat = m / (1-np.power(beta_1, j+1))
        v_hat = v / (1-np.power(beta_2, j+1))
        weights = weights - lr * m_hat/(np.sqrt(v_hat)+epsilon)
        if(j%n_check==0):
            t2=datetime.now()
            print( j, " loss : %.6f" % loss[j], "gradnorm: %.6f" % grad_norm[j], "noise: %.6f" % la.norm(noise), "time: ", (t2-t1).seconds)
            t1=t2
    return loss, grad_norm, weights, gradient_list



def adam_gaussian(ansatz, circuit_name, qubits, n_params, gamma, gau_rate, noise_gamma, noise_rate, lr, iteration, n_time, n_check):
    
    folder = "./"+circuit_name +"_"+ str(qubits) +"_"+ str(n_params)+"_adam_gaussian"+"_"+str(gau_rate)+"_"+str(noise_rate)
    a = np.random.normal(0, gamma*gau_rate, n_params*n_time).reshape(n_time,n_params)
    if not os.path.exists(folder):
        os.makedirs(folder)
    for time in range(n_time):
        np.save(folder+"/weights_init_"+str(time)+".npy", a[time])
        loss, grad_norm, weights, gradient = adam_optimizer(ansatz, a[time], noise_gamma*noise_rate, lr, iteration, n_check)
        np.save(folder+"/loss_"+str(time)+".npy", loss)
        np.save(folder+"/grad_norm_"+str(time)+".npy",  grad_norm)
        np.save(folder+"/weights_final_"+str(time)+".npy", weights)
        np.save(folder+"/gradient_"+str(time)+".npy", gradient)
        print("the training with adam gaussian ends", gau_rate, noise_rate)
    return 0

def adam_zero(ansatz, circuit_name, qubits, n_params, gamma, gau_rate, noise_gamma, noise_rate, lr, iteration, n_time, n_check):
    

    folder = "./"+circuit_name +"_"+ str(qubits) +"_"+ str(n_params)+"_adam_zero"+"_"+str(noise_rate)
    a = np.random.normal(0, 0, n_params*n_time).reshape(n_time,n_params)
    if not os.path.exists(folder):
        os.makedirs(folder)
    for time in range(n_time):
        np.save(folder+"/weights_init_"+str(time)+".npy", a[time])
        loss, grad_norm, weights, gradient = adam_optimizer(ansatz, a[time], noise_gamma*noise_rate, lr, iteration, n_check)
        np.save(folder+"/loss_"+str(time)+".npy", loss)
        np.save(folder+"/grad_norm_"+str(time)+".npy",  grad_norm)
        np.save(folder+"/weights_final_"+str(time)+".npy", weights)
        np.save(folder+"/gradient_"+str(time)+".npy", gradient)
        print("the training with adam zero ends", noise_rate)
    return 0

def adam_uniform(ansatz, circuit_name, qubits, n_params, gamma, gau_rate, noise_gamma, noise_rate, lr, iteration, n_time, n_check):
    
    folder = "./"+circuit_name +"_"+ str(qubits) +"_"+ str(n_params)+"_adam_uniform"+"_"+str(noise_rate)
    a = np.random.uniform(0, 2*pi, size=n_time*n_params).reshape(n_time,n_params)
    if not os.path.exists(folder):
        os.makedirs(folder)
    for time in range(n_time):
        np.save(folder+"/weights_init_"+str(time)+".npy", a[time])
        loss, grad_norm, weights, gradient = adam_optimizer(ansatz, a[time], noise_gamma*noise_rate, lr, iteration, n_check)
        np.save(folder+"/loss_"+str(time)+".npy", loss)
        np.save(folder+"/grad_norm_"+str(time)+".npy",  grad_norm)
        np.save(folder+"/weights_final_"+str(time)+".npy", weights)
        np.save(folder+"/gradient_"+str(time)+".npy", gradient)
        print("the training with adam uniform ends", noise_rate)
    return 0


