from singledispatchmethod import singledispatchmethod

from .state import WorldState
import rideshare_simulator.events as events


class Simulator(object):
    def __init__(self, request_generator, driver_generator,
                 dispatch_policy, pricing_policy, **kwargs):
        self.request_generator = request_generator
        self.driver_generator = driver_generator
        self.dispatch_policy = dispatch_policy
        self.pricing_policy = pricing_policy
        self.state = WorldState(**kwargs)
        self.state.push_event(
            self.driver_generator.generate(self.state)[0])
        self.state.push_event(
            self.request_generator.generate(self.state)[0])

    def run(self, T=float("Inf")):
        while not len(self.state.event_queue) == 0 and self.state.ts < T:
            event = self.state.pop_event() ##get the next nearest event
            self.state.step(event.ts) ##Worldstate forwards to the time of the event
            new_events = self.handle_event(event) ##handle event (new_events may occur)
            for new_event in new_events: 
                self.state.push_event(new_event) ##add new event
            yield (self.state, event)

    @singledispatchmethod
    def handle_event(self, event):
        """
        Returns a tuple (updates, new_events), where updates represent
        state changes to be applied to self.state, and new_events represents
        future events to be added to self.state.event_queue.

        This is the main point at which state mutation occurs.
        """
        raise NotImplementedError("No handler for event of type {cls}"
                                  .format(cls=type(event)))

    @handle_event.register(events.DriverOnlineEvent)
    def _(self, event: events.DriverOnlineEvent):
        """
            a new driver is added 
        """
        self.state.add_driver(event.driver) ## add this driver to WorldState
        offline_event = events.DriverOfflineEvent(
            event.ts + event.shift_length, event.driver.id)  ## add the event that the driver leaves the network
        return self.driver_generator.generate(self.state) + [offline_event] ## also add the event for the next new driver 

    @handle_event.register(events.DriverOfflineEvent)
    def _(self, event: events.DriverOfflineEvent):
        """
            a driver is leaving the network
        """
        self.state.drivers[event.driver_id].go_offline()
        return []

    @handle_event.register(events.RequestEvent)
    def _(self, event: events.RequestEvent):
        self.state.riders[event.rider.id] = event.rider
        next_request = self.request_generator.generate(self.state)
        offer = self.pricing_policy.make_offer(self.state, event)
        is_accepted = event.rider.respond_to_offer(offer)
        response_event = events.OfferResponseEvent(
            ts=event.ts,
            rider_id=event.rider.id,
            offer=offer,
            accepted=is_accepted)
        return [response_event] + next_request

    @handle_event.register(events.OfferResponseEvent)
    def _(self, event: events.OfferResponseEvent):
        if event.accepted:
            new_events = self.dispatch_policy.dispatch(self.state, event)
        else:
            new_events = []
        return new_events

    @handle_event.register(events.DispatchEvent)
    def _(self, event: events.DispatchEvent):
        self.state.drivers[event.driver_id].route = event.route
        return []
