# Supplimentary Code: Generating High-Quality Explanations for Navigation in Partially-Revealed Environments

We use Docker (with the Nvidia runtime) and GNU Make to run our code, so both are required to run our code. First, docker must be installed by following [the official docker install guide](https://docs.docker.com/engine/install/ubuntu/)
(the official docker install guide). Second, our docker environments will require that the NVIDIA docker runtime is installed (via `nvidia-container-toolkit`. Follow the install instructions on the [nvidia-docker GitHub page](https://github.com/NVIDIA/nvidia-docker#quickstart) to get it.

## Generating Explanations
	
We have provided a make target that generates two explanations that correspond to those included in the paper. Running the following make targets in a command prompt will generate these:

```bash
# Build the repo
make build
# Generate explanation plots
make xai-explanations
```

For each, the planner is run for a set number of steps and an explanation is generated by the agent and its learned model to justify its behavior compared to what the oracle planner specifies as the action known to lead to the unseen goal. A plot will be generated for each of the explanations and added to `./data/explanations`.

# Re-Running Results Experiments

We also provide targets for re-running the results for each of our simulated experimental setups:

```bash
# Build the repo
make build

# Ensure data timestamps are in the correct order
# Only necessary on the first pass
make fix-target-timestamps

# Maze Environments
make xai-maze EXPERIMENT_NAME=base_allSG
make xai-maze EXPERIMENT_NAME=base_4SG SP_LIMIT_NUM=4
make xai-maze EXPERIMENT_NAME=base_0SG SP_LIMIT_NUM=0

# University Building (floorplan) Environments
make xai-floorplan EXPERIMENT_NAME=base_allSG
make xai-floorplan EXPERIMENT_NAME=base_4SG SP_LIMIT_NUM=4
make xai-floorplan EXPERIMENT_NAME=base_0SG SP_LIMIT_NUM=0

# Results Plotting
make xai-process-results
```

(This can also be done by running `./run.sh`)

This code will build the docker container, **do nothing** (since the results already exist), and then print out the results. GNU Make is clever: it recognizes that the plots already exist in their respective locations for each of the experiments and, as such, it does not run any code. To save on space to meet the 100MB size requirements, the results images for each experiment have been downsampled to thumbnail size. If you would like to reproduce any of our results, delete the plots of interest in the results folder and rerun the above code; make will detect which plots have been deleted and reproduce them. All results plots can be found in their respective folder in `./data/results`.

The make commands above can be augmented to run the trials in parallel, by adding `-jN` (where `N` is the number of trials to be run in parallel) to each of the Make commands. On our NVIDIA 2060 SUPER, we are limited by GPU RAM, and so we limit to `N=4`. Running with higher N is *possible* but sometimes our simulator tries to allocate memory that does not exist and will crash, requiring that the trial be rerun. It is in principle possible to also generate data and train the learned planners from scratch, though (for now) this part of the pipeline has not been as extensively tested; data generation consumes roughly 1.5TB of disk space, so be sure to have that space available if you wish to run that part of the pipeline. Even with 4 parallel trials, we estimate that running all the above code from scratch (including data generation, training, and evaluation) will take roughly 2 weeks, half of which is evaluation.

## Code Organization

The `src` folder contains a number of python packages necessary for this paper. Most of the *algorithmic* code that reflects our primary *research* contributions is predominantly spread across three files:

- `xai.planners.subgoal_planner` The `SubgoalPlanner` class is the one which encapsulates much of the logic for deciding where the robot should go including its calculation of which action it should take and what is the "next best" action. This class is the primary means by which the agent collects information and dispatches it elsewhere to make decisions.
- `xai.learning.models.exp_nav_vis_lsp` The `ExpVisNavLSP` defines the neural network along with its loss terms used to train it. Also critical are the functions included in this and the `xai.utils.data` file for "updating" the policies to reflect the newly estimated subgoal properties even after the network has been retrained. This class also includes the functionality for computing the delta subgoal properties that primarily define our counterfactual explanations. Virtuall all of this functionality heavily leverages PyTorch, which makes it easy to compute the gradients of the expected cost for each of the policies.
- `xai.planners.explanation` This file defines the `Explanation` class that stores the subgoal properties and their deltas (computed via `ExpVisNavLSP`) and composes these into a natural language explanation and a helpful visualization showing all the information necessary to understand the agent's decision-making process.
