import torch
import matplotlib
import matplotlib.pyplot as plt
import numpy as np
import imageio
from cal_constraints import *
from proj_io_adapter_io_adapter import *

DEMO = "collision_1"

def plot_gif(before_project, after_project, arrow, boundary_list, name = '0'):
    def generate_one_frame(data, pred, arrow, boundary, xy_max = 2):

        fig, ax = plt.subplots(figsize=(10,10))
        
        circle1 = plt.Circle((0, 0), 2.0, color='r', fill=False)
        ax.add_artist(circle1)

        X, Y, U, V = zip(*[arrow])
        ax.quiver(X, Y, U, V, angles='xy', scale_units='xy', scale=1)

        ax.scatter(boundary[:,0], boundary[:,1], c = 'r')    
        ax.scatter(data[:,0], data[:,1], c = 'y')
        ax.scatter(pred[:,0], pred[:,1], c = 'b')          
        ax.grid()
        ax.set(xlabel='X', ylabel='Y', title='yellow: points before projection; blue: after projection.')
        ax.set_xlim(-xy_max, xy_max)
        ax.set_ylim(-xy_max, xy_max)
        fig.canvas.draw()     

        image = np.frombuffer(fig.canvas.tostring_rgb(), dtype='uint8')
        image  = image.reshape(fig.canvas.get_width_height()[::-1] + (3,))
        
        return image
    
    imageio.mimsave('./gif/' + name + '.gif', [
        generate_one_frame(before_project[i], after_project[i], arrow[i], boundary_list[i]) 
        for i in range(len(before_project))
        ], fps=10)

class Proj():
    def __init__(self):
        model_path = 'models/colli.pt'
        self.model = torch.load(model_path).cpu()
        print(self.model.iter)
    def project(self, data):
        d = torch.Tensor(data[None, :, :])
        pred = self.model(d)
        pred = pred[0,:,:]
        return pred.detach().numpy()


def get_boundary(new_data):
    center1 = (new_data[0,:]+new_data[1,:]+new_data[2,:]+new_data[3,:])/4
    boundary_data = np.zeros([4,2])
    boundary_data[0, :] = 0.6 * (new_data[3,:]-center1) - 1.2 * (new_data[1,:]-center1) + center1
    boundary_data[1, :] = 1.2 * (new_data[3,:]-center1) + 0.6 * (new_data[1,:]-center1) + center1
    boundary_data[2, :] = - 0.6 * (new_data[3,:]-center1) + 1.2 * (new_data[1,:]-center1) + center1
    boundary_data[3, :] = - 1.2 * (new_data[3,:]-center1) - 0.6 * (new_data[1,:]-center1) + center1
    return boundary_data

def get_full_data(boundary_data):
    full_data = np.zeros([16,2])
    x1 = boundary_data[3,0]
    y1 = boundary_data[3,1]
    x2 = boundary_data[1,0]
    y2 = boundary_data[1,1]
    # print(boundary_data)
    for i in range(4):
        for j in range(4):
            temp1 = (j * boundary_data[3,:] + (3-j) * boundary_data[2,:] ) / 3
            temp2 = (j * boundary_data[0,:] + (3-j) * boundary_data[1,:] ) / 3
            full_data[i*4+j, :] = (i * temp1 + (3-i) * temp2) / 3
    return full_data

def test(te = 0):
    data = np.array([[ 0.130514,   0.437662 ],
                        [ 0.0862001,  0.882675 ],
                        [ 0.575526,   0.481976 ],
                        [ 0.531213,   0.926988 ],
                        [-0.457831,  -0.12947  ],
                        [-0.5583,     0.306311 ],
                        [-0.0220487, -0.0290008],
                        [-0.122518,   0.406781 ]])
    timestamp = 0.1
    vel = np.array([[5.0, 0.0],[5.0, 0.0],[5.0, 0.0],[5.0, 0.0],
                    [5.0, 0.0],[5.0, 0.0],[5.0, 0.0],[5.0, 0.0]])
    vel *= 0.4

    full_data = np.zeros([32,2])
    boundary_data = np.zeros([8,2])
    boundary_data[0:4, :] = get_boundary(data[0:4,:])
    boundary_data[4:8, :] = get_boundary(data[4:8,:])
    full_data[0:16, :] = get_full_data(boundary_data[0:4,:])
    full_data[16:32, :] = get_full_data(boundary_data[4:8,:])
    print(boundary_data)
    print(full_data)

    g = np.array([0, -5])

    initial_boundary = np.array([[ 0.6,  0.0],
        [ 0.6, 0.6],
        [0.0,  0.6],
        [0.0, 0.0],[ 0.6,  0.0],
        [ 0.6, 0.6],
        [0.0,  0.6],
        [0.0, 0.0]])

    data_list = [np.array(data)]
    proj_list = [np.array(data)]
    boundary_list = [boundary_data]
    full_list = [full_data]
    force_list = [np.array([0, 0, 0, 0])]

    pr = Proj()
    for ite in range(100):
        new_data = np.array(data)
        for i in range (0, len(data)):
            vel[i, :] += timestamp * g
            new_data[i, :] = data[i,:] + vel[i,:] * timestamp 

        data_list.append(np.array(new_data))
        force_list.append(np.array([new_data[3,0], new_data[3,1], 0, 0]))

        new_data = pr.project(new_data)
        boundary_data = np.zeros([8,2])
        full_data = np.zeros([32,2])
        boundary_data[0:4, :] = get_boundary(new_data[0:4,:])
        boundary_data[4:8, :] = get_boundary(new_data[4:8,:])
        full_data[0:16, :] = get_full_data(boundary_data[0:4,:])
        full_data[16:32, :] = get_full_data(boundary_data[4:8,:])
        
        proj_list.append(np.array(new_data))
        boundary_list.append(np.array(boundary_data))
        full_list.append(np.array(full_data))

        for i in range (len(data)):
           vel[i,:] =  (new_data[i,:] - data[i,:]) / timestamp 

        data = (np.array(new_data))

    plot_gif(data_list, proj_list, force_list, full_list, DEMO)
    run(boundary_list, [0,0,0,0,1], initial_boundary, DEMO)
    run_e(boundary_list, [1,0,0,0,0], initial_boundary, DEMO)
    write_viewer(full_list, output_dir = "output_" + DEMO)
    cal_mse(full_list, 9, DEMO)





if __name__ == '__main__':
    test()
