clear;
load datasets/citeseer_label.txt;
load datasets/citeseer_adj.txt;
mex d_norm_err_stat.cpp;
mex d_nonorm_err_stat.cpp;


% Rename the data
label = citeseer_label;
Adj = citeseer_adj;

% Generate the adjacent matrix
n = size(label);
n = n(1);

A = zeros(n, n);
for i = 1:length(Adj(:,1)),
    A(Adj(i, 1), Adj(i, 2)) = 1;
end

% Extract the connected component based on a vertex 
index = extract_comp(Adj, 815);
map = (n+1) * ones(n, 1);
map(index) = (1:length(index))';

% Relabel the vertices of this connected component
label = label(index, 2);
Adj = map(Adj);

Adj(Adj(:, 1) == n+1, :)  = [];
Adj(Adj(:, 2) == n+1, :)  = [];
A = A(index, index);

% Set parameters for experiments
% Number of seeds
seed_num = 20;
% Parameter for Inverse PR
lambda = 0.99;
types = length(lambda) + 3;
% Parameter for Personalized PR
PRalpha = 0.95;
% Parameter for Heat-kernal PR
heatPRbeta = 10;
% The number of independent experiments 
times = 10;
% The number of steps of random walks to be recorded  
step = 100;
% A parameter to decide when to record the ranking of all vertices:
% Accumulate the first ROC_point many of steps and we may get a ranking of
% all vertices. The full ranking list typically provides more information
% than that based on recall
ROC_point = 50;

% to record the result: 
% realerr_record: Record the number of classification errors in the first |C|
% ranking vertices, where |C| is the size of the ground-truth community
% Dimension explanation --- 
% number of steps * number of different methods (IPR 0.99,
% IPR 0.95, IPR 0.9, Single-step LP, PPR, HPR) * 
% number of different communities * number of experiments * 
% degree normalization or not

realerr_record = zeros(step, types, max(label)+1, times,2);

% ROC_record: Record the ranking of vertices by accumulating the first 
% ROC_point many steps
% Dimension explanation ---
% Recalls at different percentiles of budgets  * 
% number of different methods (IPR 0.99, IPR 0.95, IPR 0.9, Single-step LP, PPR, HPR) * 
% number of different communities * number of experiments * 
% degree normalization or not 
ROC_record = zeros(101, types, max(label)+1, times, 2);

% Run experiments with degree normalization
[realerr, ROC] = d_norm_err_stat(Adj, label, step, seed_num, lambda, PRalpha, heatPRbeta, times, ROC_point);
realerr = reshape(realerr, [step, types, max(label)+1, times]);
realerr_record(:,:,:,:, 1) = realerr;
ROC = reshape(ROC, [101, types, max(label)+1, times]);
ROC_record(:,:,:,:, 1) = ROC;

% Run experiments without degree normalization
[realerr, ROC] = d_nonorm_err_stat(Adj, label, step, seed_num, lambda, PRalpha, heatPRbeta, times, ROC_point);
realerr = reshape(realerr, [step, types, max(label)+1, times]);
realerr_record(:,:,:,:, 2) = realerr;
ROC = reshape(ROC, [101, types, max(label)+1, times]);
ROC_record(:,:,:,:, 2) = ROC;


realerr_mean = sum(mean(realerr_record, 4), 3);
ROC_mean = mean(mean(ROC_record, 4), 3);

% draw figures for the case with degree normalization
precision_mean_re = 1-reshape(realerr_mean(:,:,:,:,1), [step, types])/length(index);
ROC_mean_re = reshape(ROC_mean(:,:,:,:,1), [101, types]);
figure(1);
plot(1:step, precision_mean_re(:,1), 1:step, precision_mean_re(:,2), 1:step, precision_mean_re(:,3), 1:step, precision_mean_re(:,4));

axis([0 100 0.2 0.7])
xlabel('steps (k)');
ylabel('recall');
h_legend = legend('IPR 0.99', 'one-step LP', 'PPR', 'HPR', '');
set(h_legend,'fontsize',20,'Location','southeast');


