package solver;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;

import ilog.concert.IloNumExpr;
import ilog.concert.IloNumVar;
import ilog.concert.IloRange;
import ilog.cplex.IloCplex;

public class MinCostFlowSolver
{
	private HashSet<Object> vertSet;
	private HashMap<Object, Integer> vertMap;
	private HashMap<Object, ArrayList<FlowEdgeEntry>> edge;
	private HashMap<Object, ArrayList<FlowEdgeEntry>> revEdge;
	
	public MinCostFlowSolver()
	{
		vertSet = new HashSet<Object>();
		vertMap = new HashMap<Object, Integer>();
		edge = new HashMap<Object, ArrayList<FlowEdgeEntry>>();
		revEdge = new HashMap<Object, ArrayList<FlowEdgeEntry>>();
	}
	
	public void addVert(Object v)
	{
		assert(!vertSet.contains(v));
		vertSet.add(v);
	}
	
	public void finishAddingVert()
	{
		int cnt = 0;
		for (Object v : vertSet)
		{
			vertMap.put(v, cnt++);
			edge.put(v, new ArrayList<FlowEdgeEntry>());
			revEdge.put(v, new ArrayList<FlowEdgeEntry>());
		}
	}
	
	public void addEdge(Object u, Object v, double cost, double cap)
	{
		assert(vertSet.contains(u));
		assert(vertSet.contains(v));
		edge.get(u).add(new FlowEdgeEntry(v, cost, cap));
		revEdge.get(v).add(new FlowEdgeEntry(u, cost, cap));
	}
	
	private String getEdgeVar(Object u, Object v)
	{
		return "x" + vertMap.get(u) + "," + vertMap.get(v);
	}
	
	public double LPSolveCplex(Object source, Object sink)
	{
		double res = 0;
		try
		{
			IloCplex cplex = new IloCplex();
			// cplex.setOut(null);
			cplex.setWarning(null);

			HashMap<String, Integer> edgeMap = new HashMap<String, Integer>();
			int cnt = 0;
			for (Object u : vertSet)
			{
				ArrayList<FlowEdgeEntry> adj = edge.get(u);
				for (FlowEdgeEntry e : adj)
				{
					edgeMap.put(getEdgeVar(u, e.u), cnt++);
				}
			}

			String[] varName = new String[edgeMap.size()];
			double[] lb = new double[edgeMap.size()];
			double[] ub = new double[edgeMap.size()];
			for (String e : edgeMap.keySet())
			{
				varName[edgeMap.get(e)] = e;
			}
			// capacity constraint
			Arrays.fill(lb, 0);
			for (Object u : vertSet)
			{
				ArrayList<FlowEdgeEntry> adj = edge.get(u);
				for (FlowEdgeEntry e : adj)
				{
					ub[edgeMap.get(getEdgeVar(u, e.u))] = e.cap;
				}
			}
			IloNumVar[] x = cplex.numVarArray(edgeMap.size(), lb, ub, varName);

			// build objective
			double[] objCoe = new double[edgeMap.size()];
			for (Object u : vertSet)
			{
				ArrayList<FlowEdgeEntry> adj = edge.get(u);
				for (FlowEdgeEntry e : adj)
				{
					objCoe[edgeMap.get(getEdgeVar(u, e.u))] = e.cost;
					// lpw.plus(getEdgeVar(u, e.u), e.cost);
				}
			}
			cplex.addMinimize(cplex.scalProd(x, objCoe));
			// balance constraint
			ArrayList<IloRange> cons = new ArrayList<IloRange>();
			int consCount = 0;
			for (Object u : vertSet)
			{
				ArrayList<FlowEdgeEntry> adj = null;
				double balance = 0;
				if (u.equals(source) || u.equals(sink))
				{
					adj = u.equals(source) ? edge.get(u) : revEdge.get(u);
					for (FlowEdgeEntry e : adj)
					{
						balance += e.cap;
					}
					double sign = u.equals(source) ? -1.0 : 1.0;
					balance *= sign;
				}
				ArrayList<IloNumExpr> term = new ArrayList<IloNumExpr>();
				// LPWizardConstraint cons = lpw.addConstraint("c" + consCounter, balance, "=");
				adj = revEdge.get(u);
				for (FlowEdgeEntry e : adj)
				{
					term.add(cplex.prod(1.0, x[edgeMap.get(getEdgeVar(e.u, u))]));
					// cons.plus(getEdgeVar(e.u, u), 1.0);
				}
				adj = edge.get(u);
				for (FlowEdgeEntry e : adj)
				{
					term.add(cplex.prod(-1.0, x[edgeMap.get(getEdgeVar(u, e.u))]));
					//cons.plus(getEdgeVar(u, e.u), -1.0);
				}
				IloNumExpr[] expr = new IloNumExpr[term.size()];
				for (int i = 0; i < term.size(); i++)
				{
					expr[i] = term.get(i);
				}
				cplex.addEq(cplex.sum(expr), balance, "c" + consCount);
				consCount++;
			}
			// solve		
			cplex.solve();
			res = cplex.getObjValue();
			cplex.end();
		}
		catch (Exception e)
		{
			e.printStackTrace();
		}
		return res;

        //IloNumVar[][] var = new IloNumVar[1][];
        //IloRange[][]  rng = new IloRange[1][];
	}
}

class FlowEdgeEntry
{
	Object u;
	double cost;
	double cap;
	
	public FlowEdgeEntry(Object u, double cost, double cap)
	{
		this.u = u;
		this.cost = cost;
		this.cap = cap;
	}
}
