#ifndef CROWD_H_INCLUDED
#define CROWD_H_INCLUDED

#include <random>
#include <vector>
#include "distro.h"

// holds a single (worker_id, task_id, label) tuple
class data_point {
public:
    unsigned int task_id;
    unsigned int work_id;
    double label;

    data_point(unsigned int task_id, unsigned int work_id, double label);
    void set_task_id(unsigned int task_id);
};

// crowd data (abstract class)
class crowd {
public:
    std::vector<data_point> data_seq;
    std::vector<double> work_acc;

    virtual ~crowd() = default;

    // call this function for incremental values of t
    // creates the data in batches under the hood
    virtual void create_data_point(std::mt19937& ran_gen, unsigned int t) = 0;
};

// fixed budget of (worker_id, label) tuples, all shuffled at the start
class crowd_budget: public crowd {
private:
    distro* d_work_acc;
    distro* d_work_quota;
    unsigned int n_workers;
public:
    crowd_budget(distro* d_work_acc, distro* d_work_quota, unsigned int n_workers);
    ~crowd_budget();
    void create_data_point(std::mt19937& ran_gen, unsigned int t);
};

// the workers come one by one, each bringing a limited amount of labels
class crowd_quota: public crowd {
private:
    distro* d_work_acc;
    distro* d_work_quota;
public:
    crowd_quota(distro* d_work_acc, distro* d_work_quota);
    ~crowd_quota();
    void create_data_point(std::mt19937& ran_gen, unsigned int t);
};

// a fixed pool of workers keep producing labels in turn
class crowd_workers: public crowd {
private:
    distro* d_work_acc;
    unsigned int n_workers;
public:
    crowd_workers(distro* d_work_acc, unsigned int n_workers);
    ~crowd_workers();
    void create_data_point(std::mt19937& ran_gen, unsigned int t);
};

// read the data from a file
class crowd_file: public crowd {
public:
    unsigned int n_tasks;
    unsigned int n_workers;
    unsigned int n_labels;

    crowd_file(const char *filename);
    ~crowd_file();
    void create_data_point(std::mt19937& ran_gen, unsigned int t);
};

// crowd parser
crowd* crowd_parse(int *argc, char **argv[], unsigned int n_labels);

// home-made unit tests
void crowd_budget_test();
void crowd_quota_test();
void crowd_workers_test();
void crowd_parse_test();

#endif // CROWD_H_INCLUDED
