#include "gtest/gtest.h"

#include <iostream>
// define constants like M_PI and C keywords for MSVC
#ifdef _MSC_VER
#define _USE_MATH_DEFINES
#include <math.h>
#endif
#include "ATen/ATen.h"
#include "ATen/Dispatch.h"
#include "test_seed.h"

using std::cout;
using namespace at;

constexpr auto Float = ScalarType::Float;

template<typename scalar_type>
struct Foo {
  static void apply(Tensor a, Tensor b) {
    scalar_type s = 1;
    std::stringstream ss;
    ss << "hello, dispatch: " << a.type().toString() << s << "\n";
    auto data = (scalar_type*)a.data_ptr();
    (void)data;
  }
};
template<>
struct Foo<Half> {
  static void apply(Tensor a, Tensor b) {}
};

void test_overflow() {
  auto s1 = Scalar(M_PI);
  ASSERT_EQ(s1.toFloat(), static_cast<float>(M_PI));
  s1.toHalf();

  s1 = Scalar(100000);
  ASSERT_EQ(s1.toFloat(), 100000.0);
  ASSERT_EQ(s1.toInt(), 100000);

  ASSERT_THROW(s1.toHalf(), std::domain_error);

  s1 = Scalar(NAN);
  ASSERT_TRUE(std::isnan(s1.toFloat()));
  ASSERT_THROW(s1.toInt(), std::domain_error);

  s1 = Scalar(INFINITY);
  ASSERT_TRUE(std::isinf(s1.toFloat()));
  ASSERT_THROW(s1.toInt(), std::domain_error);
}

TEST(TestScalar, TestScalar) {
  manual_seed(123, at::kCPU);
  manual_seed(123, at::kCUDA);

  Scalar what = 257;
  Scalar bar = 3.0;
  Half h = bar.toHalf();
  Scalar h2 = h;
  cout << "H2: " << h2.toDouble() << " " << what.toFloat() << " "
       << bar.toDouble() << " " << what.isIntegral() << "\n";
  Generator& gen = at::globalContext().defaultGenerator(at::kCPU);
  ASSERT_NO_THROW(gen.seed());
  auto&& C = at::globalContext();
  if (at::hasCUDA()) {
    auto t2 = zeros({4, 4}, at::kCUDA);
    cout << &t2 << "\n";
  }
  auto t = ones({4, 4});

  auto wha2 = zeros({4, 4}).add(t).sum();
  ASSERT_EQ(wha2.item<double>(), 16.0);

  ASSERT_EQ(t.sizes()[0], 4);
  ASSERT_EQ(t.sizes()[1], 4);
  ASSERT_EQ(t.strides()[0], 4);
  ASSERT_EQ(t.strides()[1], 1);

  Type& T = CPU(Float);
  Tensor x = randn({1, 10}, T);
  Tensor prev_h = randn({1, 20}, T);
  Tensor W_h = randn({20, 20}, T);
  Tensor W_x = randn({20, 10}, T);
  Tensor i2h = at::mm(W_x, x.t());
  Tensor h2h = at::mm(W_h, prev_h.t());
  Tensor next_h = i2h.add(h2h);
  next_h = next_h.tanh();

  ASSERT_ANY_THROW(at::_local_scalar(Tensor{}));

  test_overflow();

  if (at::hasCUDA()) {
    auto r = CUDA(Float).copy(next_h);
    ASSERT_TRUE(CPU(Float).copy(r).equal(next_h));
  }
  ASSERT_NO_THROW(randn({10, 10, 2}, T));

  // check Scalar.toTensor on Scalars backed by different data types
  ASSERT_EQ(scalar_to_tensor(bar).type().scalarType(), kDouble);
  ASSERT_EQ(scalar_to_tensor(what).type().scalarType(), kLong);
  ASSERT_EQ(
      scalar_to_tensor(ones({})._local_scalar()).type().scalarType(), kDouble);

  if (x.type().scalarType() != ScalarType::Half) {
    AT_DISPATCH_ALL_TYPES(x.type(), "foo", [&] {
      scalar_t s = 1;
      std::stringstream ss;
      ASSERT_NO_THROW(
          ss << "hello, dispatch" << x.type().toString() << s << "\n");
      auto data = (scalar_t*)x.data_ptr();
      (void)data;
    });
  }

  // test direct C-scalar type conversions
  {
    auto x = ones({1, 2}, T);
    ASSERT_ANY_THROW(x.item<float>());
  }
  auto float_one = ones({}, T);
  ASSERT_EQ(float_one.item<float>(), 1);
  ASSERT_EQ(float_one.item<int32_t>(), 1);
  ASSERT_EQ(float_one.item<at::Half>(), 1);
}
