% Blind Calibration solver using a GAMP algorithm
% Inputs are the matrix Y and the matrix G
% such that D*Y = G*X with:
% X a sparse P*N signal matrix
% D a diagonal M*M matrix with random entries drawn from a given distribution
% G a M*N random gaussian measurement matrix
% The algorithm tries to recover X and D from Y and G and the parameters given in the option:
% the structure opt contains following fields:
%   P           number of samples used for calibration
%	rho 		sparsity of the signal
%	alpha		measurement rate
%	Delta		variance of the additive white noise
%	d           mean of the calibration coefficients in D
%	delta		variance of the calibration coefficients in D
%   nb_iter     maximum number of iterations of the algorithm
%   conv_criterion  below we which we consider the algorithm has coverged
%   damp_mes    damping coefficient used for suppressing oscillations
%   mute        0: display everything,  1:display nothing
%   verbose_n   dislpay current estimate MSE every verbose_n iterations



function [X, D_guess,parameters] = BC(Y,G,opt)

%%Get parameters
[m P] = size(Y);
[m N] = size(G);
rho = opt.rho;
Delta = opt.Delta;
delta = opt.delta;
d = opt.d;
affichage = opt.verbose_n;
conv_criterion = opt.conv_criterion;
damp_mes = opt.damp_mes;
P = opt.P;
mute = opt.mute;


%% Initialization

a = zeros([N P]);
v = 10*repmat(rho,[N P]);
w = Y;
K = d;
L = delta;
V = (G.^2)*v;
S=0;

%% Main loop
for iter =1:opt.nb_iter
    
    %save previous variables
    V_old = V;
    w_old = w;
    a_old = a;
    
    %compute new variables
    V = G.^2*double(v);
    w = G*a - (K*Y - w).*((G.^2)*v)./(Delta+V_old);
    
    %adding damping helps to suppress oscillations
    w = damping(w,w_old,damp_mes);
    V = damping(V,V_old,damp_mes);
    C = (sum(Y.^2./(Delta+V), 2)).^(-1);
    D = (C.*sum(Y.*w./(Delta+V),2));
    K = Kmu(C,D,P,d,delta);
    K = diag(K);
    L = Lmu(C,D,P,d,delta);
    L = diag(L);
    S = squeeze(sum(repmat(G.^2,[1 1 P]).*permute(repmat((1-L*(Y.^2)./(Delta+V))./(Delta+V),[1 1 N]),[1 3 2]),1)).^(-1);
    
    
    if P==1
        S = permute(S,[2 1]);
        R = a + S.*(sum(repmat(G,[1 1 P]).*repmat((K*Y - w)./(Delta+V),[1 N]),1))';
    else
        R = a + S.*(squeeze(sum(repmat(G,[1 1 P]).*permute(repmat((K*Y - w)./(Delta+V),[1 1 N]),[1 3 2]),1)));
    end;
    
    a = f_a(S,R,rho,0,1);
    v = f_c(S,R,rho,0,1);
    
    
    
    X=a;
    
    
    %% Stopping conditions
    diff(iter) = norm(a-a_old)/(norm(a));
    % E is an approximation of the corrected mean square error
    E(iter) = 1/(m*P) * sum(sum((K*Y - G*X).^2));
    
    % A few stopping conditions for when the algorithm finds the right
    % solution
    if (E(iter) < 10*Delta) & iter>25
        disp('Reached estimated MSE of the order of 10 Deltas');
        % If the MSE is of the order of Delta, it will not decrease further
        break;
    end
    if (E(iter) < conv_criterion) & iter>25
        disp('Target estimated MSE reached');
        break;
    end
    if  (diff(iter) < conv_criterion) & iter>25
        disp('Target relative variation reached');
        break;
    end
    
    % A few stopping conditions for when the algorithm does not find the right
    % solution
    if ~isfinite(diff(iter))
        % something divergered...
        disp('Inifinites appearing...');
        break;
    end
    if iter>100 && E(iter)>0.9995*E(iter-99)
        % the MSE does not seem to decrease on a larger number of
        % iterations: the algorithm has probably reached its steady state.
        disp('Non-decreasing MSE');
        break;
    end
    if iter>25 && E(iter)>10^3
        disp('estimated MSE bigger than 10 000');
        break;
    end
    
    %% Display what is happening
    if mute==0
        if  mod(iter, affichage)==0
            disp(sprintf('Iteration %d , difference %d, approx MSE: %d',iter,diff(iter),E(iter)));
        end
    end
    
    
end

parameters.iterations=iter;
D_guess = diag(K);
X=a;


end
