function [aux_std, gg] = logistic_aux(labels, log_prior_var, prior_mean)
%LOGISTIC_AUX return effective Gaussian likelihood noise level (and centres)
%
%     [aux_std, gg] = logistic_aux(labels, log_prior_var, prior_mean)
%
% Inputs:
%             labels     Nx1    values are in {-1,+1}
%      log_prior_var 1x1 or Nx1
%         prior_mean 1x1 or Nx1
%
% Outputs:
%            aux_std Nx1 
%                 gg Nx1 (if needed) 

% Iain Murray, April 2010

if (nargin > 2) && (prior_mean ~= 0)
    error('Non-zero prior means are not implemented yet.');
else
    prior_mean = 0;
end

prior_var = exp(log_prior_var);
prior_precision = 1./prior_var;

% The approx that's special to the logistic likelihood:
post_var = prior_var.*(1 - 1./(pi/2 + 4./prior_var));

post_precision = 1./post_var;
mask = (post_precision > prior_precision);
aux_std = zeros(size(mask));
aux_std(mask) = sqrt(1 ./ (post_precision(mask) - msk(prior_precision, mask)));
aux_std(~mask) = Inf;

if numel(aux_std) == 1
    aux_std = repmat(aux_std, size(labels));
end

if nargout > 1
    mu = sqrt(prior_var./(pi/2 + 4./prior_var));
    gg = (aux_std.^2).*(mu.*post_precision - prior_mean.*prior_precision);
    % Get rid of infinities, which disappear in sensible limits anyway:
    BIG = 1e100;
    gg = min(gg, BIG);
    gg = labels.*gg;
end


function xx = msk(A, mask)
%MSK msk(A, mask) returns A(mask), or just A if A is a scalar.
%
%This is useful for when A is a scalar standing in for an array with all
%elements equal.
if numel(A) == 1
    xx = A;
else
    xx = A(mask);
end
