% Copyright (C) 2009  Arno Onken
%
% This program is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program.  If not, see <http://www.gnu.org/licenses/>.

% -------------------------------------------------------------------------
% Estimates the parameter theta of the Frank copula family for which
% Kendall's tau of the Frank copula is equal to the function argument tau.
%
% Arguments:
%  tau   - Kendall's tau of the Frank copula
%
% Returns:
%  theta - Parameter of the Frank copula family
% -------------------------------------------------------------------------
function theta = frankparam (tau)
    if (nargin ~= 1)
        error ('frankparam: usage theta = frankparam (tau)');
    end

    if (~isscalar (tau))
        error ('frankparam: tau must be a scalar');
    end

    tau_nn = abs (tau);
    if (tau_nn < sqrt (eps))
        theta = 0;
    elseif (tau_nn >= 1)
        theta = inf;
    else
        fkt_diff = @(theta) frank_kendall_tau (theta) - tau_nn;
        theta = fzero (fkt_diff, 1);
    end
    if (tau < 0)
        theta = -theta;
    end
end

% Calculates Kendall's tau for the Frank copula with paramter theta using
% the Debye function
function tau = frank_kendall_tau (theta)
    if (theta < sqrt (eps))
        tau = 0;
    else
        tau = 1 - 4 / theta * (1 - debye_1 (theta));
    end
end

% Debye function D_n for n = 1
% Adopted from the GNU Scientific Library (GSL) version 1.9
% D_1(x) := 1/x Integrate[t/(e^t - 1), {t,0,x}]
% Note: This function could use some vectorization
function result = debye_1 (x)
    log_dbl_epsilon = -3.6043653389117154e+01;
    sqrt_dbl_epsilon = 1.4901161193847656e-08;
    val_infinity = 1.64493406684822644;
    xcut = 7.0839641853226408e+02;

    if (x < 2.0*sqrt_dbl_epsilon)
        result = 1.0 - 0.25*x + x*x/36.0;
    elseif(x <= 4.0)
        t = x*x/8.0 - 1.0;
        c = cheb_eval_e (t);
        result = c - 0.25 * x;
    elseif(x < -(log (2) + log_dbl_epsilon))
        nexp = floor (xcut/x);
        ex  = exp (-x);
        sum = 0.0;
        xk  = nexp * x;
        rk  = nexp;
        i = nexp;
        while (i>=1)
            sum = sum * ex;
            sum = sum + (1.0 + 1.0/xk)/rk;
            rk = rk - 1.0;
            xk = xk - x;
            i = i - 1;
        end
        result = val_infinity/x - sum*ex;
    elseif(x < xcut)
        result = (val_infinity - exp (-x) * (x+1.0)) / x;
    else
        result = val_infinity/x;
    end
end

% Chebychev approximation
function result = cheb_eval_e (x)
    adeb1_data = [
       2.4006597190381410194
       0.1937213042189360089
      -0.62329124554895770e-02
       0.3511174770206480e-03
      -0.228222466701231e-04
       0.15805467875030e-05
      -0.1135378197072e-06
       0.83583361188e-08
      -0.6264424787e-09
       0.476033489e-10
      -0.36574154e-11
       0.2835431e-12
      -0.221473e-13
       0.17409e-14
      -0.1376e-15
       0.109e-16
      -0.9e-18
    ];
    order = 16;
    a = -1.0;
    b = 1.0;

    d  = 0;
    dd = 0;

    y  = (2*x - a - b) / (b - a);
    y2 = 2 * y;

    j = order;
    while (j >= 1)
      temp = d;
      d = y2*d - dd + adeb1_data(j+1);
      dd = temp;
      j = j - 1;
    end

    d = y*d - dd + 0.5 * adeb1_data(1);

    result = d;
end