// Standard deep Gaussian process prior implementation of the model

data {
  int<lower = 1> N;
  int<lower = 1> N_s; 
  vector[3] m;                   // parameters prior mean
  vector[3] v;                   // parameters prior variance
  vector[N_s] mu_do;             // first layer prior mean
  matrix[N_s, N_s] Sigma_do;     // covariance matrix of first layer
  matrix[N, N_s] A;              // matching from measurements to first hidden layer
  matrix[N_s, N_s] noise_matrix; // nugget for the covariance of f
  vector[N] y;                   // outcome
}

parameters {
  vector[3] theta;    // parameter set
  vector[N_s] f_obs;  // function from observational distribution
}

transformed parameters {
  real<lower = 0> sf2;
  real<lower = 0> ell;
  real<lower = 0> likv;
  sf2  <- exp(theta[1]);
  ell  <- exp(theta[2]);
  likv <- exp(theta[3]);
}

model {

  matrix[N_s, N_s] Sigma; // covariance matrix of second layer
  vector[N]    m_y;       // observable mean
  matrix[N, N] S_y;       // observable covariance
  
  // Second layer
  
  for (i in 1:(N_s - 1)) {
   for (j in (i + 1):N_s) {
     Sigma[i, j] <- sf2 * exp(-0.5 * (f_obs[i] - f_obs[j])^2 / ell);
     Sigma[j, i] <- Sigma[i, j];
   }
  }
  for (i in 1:N_s) {
    Sigma[i, i] <- sf2;
  }

  // Output layer

  m_y <- A * f_obs;
  S_y <- A * (Sigma + noise_matrix) * A';
  for (i in 1:N) {
    S_y[i, i] <- S_y[i, i] + likv;
  }
  
  // Model

  f_obs ~ multi_normal(mu_do, Sigma_do);
  
  theta ~ normal(m, v);
	
  y ~ multi_normal(m_y, S_y);
  
}
